/****************************************************************************
*
*						      PM/Lite Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: pmlite.c $
* Version:		$Revision: 1.1 $
*
* Language:		ANSI C
* Environment:	IBM PC (MSDOS) Real mode and 16/32 bit Protected Mode
*
* Description:	Module implementing DOS extender independant protected
*				mode programming. This module will need to be included in
*				all programs that use SciTech Software's products that
*				are to be compiled in protected mode, and will need to be
*				compiled with the correct defines for the DOS extender that
*				you will be using (or with no defines for real mode
*				emulation of these routines).
*
*				The Lite version of this library is freely redistributable
*				in source code form.
*
* $Id: pmlite.c 1.1 1994/08/22 07:46:45 kjb release $
*
****************************************************************************/

#include <stdlib.h>
#include <string.h>
#include "pmode.h"

#pragma pack(1)				/* Pack all structures on 1 byte boundary	*/

/*--------------------------- Global variables ----------------------------*/

/* Set the type of target being compiled for so that we can do switching at
 * runtime as well as compile time.
 */

#if	defined(REALMODE)
int	_PM_modeType = PM_realMode;
#elif defined(PM286)
int _PM_modeType = PM_286;
#elif defined(PM386)
int _PM_modeType = PM_386;
#endif

/*----------------------------- Implementation ----------------------------*/

/*-------------------------------------------------------------------------*/
/* Generic routines common to all extenders								   */
/*-------------------------------------------------------------------------*/

unsigned PM_getBIOSSelector(void)
{
	static unsigned BIOS_sel = 0;
	if (!BIOS_sel)
		BIOS_sel = PM_createSelector(0x400,0xFFFF);
	return BIOS_sel;
}

unsigned PM_getVGASelector(void)
{
	static long	VGA_sel = 0;
	static int 	oldMode = 0;
	int			mode = PM_getByte(PM_getBIOSSelector(),0x49);

	if (!VGA_sel || oldMode != mode) {
		if (VGA_sel)
			PM_freeSelector(VGA_sel);
		switch (oldMode = mode) {
			case 0x3:
				VGA_sel = 0xB8000L;
				break;
			case 0x7:
				VGA_sel = 0xB0000L;
				break;
			default:
				VGA_sel = 0xA0000L;
				break;
			}
		VGA_sel = PM_createSelector(VGA_sel,0xFFFF);
		}
	return (unsigned)VGA_sel;
}

/*-------------------------------------------------------------------------*/
/* DOS Real Mode support.												   */
/*-------------------------------------------------------------------------*/

#ifdef REALMODE

#ifndef	MK_FP
#define MK_FP(s,o)  ( (void far *)( ((unsigned long)(s) << 16) + \
					(unsigned long)(o) ))
#endif

unsigned char _cdecl PM_getByte(unsigned s, unsigned o)
{ return *((unsigned char*)MK_FP(s,o)); }

unsigned short _cdecl PM_getWord(unsigned s, unsigned o)
{ return *((unsigned*)MK_FP(s,o)); }

unsigned long _cdecl PM_getLong(unsigned s, unsigned o)
{ return *((unsigned long*)MK_FP(s,o)); }

void _cdecl PM_setByte(unsigned s, unsigned o, unsigned char v)
{ *((unsigned char*)MK_FP(s,o)) = v; }

void _cdecl PM_setWord(unsigned s, unsigned o, unsigned short v)
{ *((unsigned short *)MK_FP(s,o)) = v; }

void _cdecl PM_setLong(unsigned s, unsigned o, unsigned long v)
{ *((unsigned long*)MK_FP(s,o)) = v; }

void _cdecl PM_memcpynf(void *dst,unsigned src_s,unsigned src_o,unsigned n)
{ memcpy(dst,MK_FP(src_s,src_o),n); }

void _cdecl PM_memcpyfn(unsigned dst_s,unsigned dst_o,void *src,unsigned n)
{ memcpy(MK_FP(dst_s,dst_o),src,n); }

void PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{ *sel = r_seg; *off = r_off; }

unsigned PM_createSelector(unsigned long base,unsigned limit)
{ limit = limit; return (unsigned)(base >> 4); }

void PM_freeSelector(unsigned sel)
{ sel = sel; }

int PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	/* Call malloc() to allocate the memory for us */
	void *p = malloc(size);
	*sel = *r_seg = FP_SEG(p);
	*off = *r_off = FP_OFF(p);
	return 1;
}

void PM_freeRealSeg(unsigned sel,unsigned off)
{
	free(MK_FP(sel,off));
}

int PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	return int86(intno,in,out);
}

int PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	return int86x(intno,in,out,sregs);
}

#endif

/*-------------------------------------------------------------------------*/
/* Windows 3.1 16 bit DPMI and Borland DPMI16 DOS Power Pack support.	   */
/*-------------------------------------------------------------------------*/

#if defined(WINDPMI16) || defined(DPMI16)

unsigned char _cdecl PM_getByte(unsigned s, unsigned o)
{ return *((unsigned char*)MK_FP(s,o)); }

unsigned short _cdecl PM_getWord(unsigned s, unsigned o)
{ return *((unsigned*)MK_FP(s,o)); }

unsigned long _cdecl PM_getLong(unsigned s, unsigned o)
{ return *((unsigned long*)MK_FP(s,o)); }

void _cdecl PM_setByte(unsigned s, unsigned o, unsigned char v)
{ *((unsigned char*)MK_FP(s,o)) = v; }

void _cdecl PM_setWord(unsigned s, unsigned o, unsigned short v)
{ *((unsigned short *)MK_FP(s,o)) = v; }

void _cdecl PM_setLong(unsigned s, unsigned o, unsigned long v)
{ *((unsigned long*)MK_FP(s,o)) = v; }

void _cdecl PM_memcpynf(void *dst,unsigned src_s,unsigned src_o,unsigned n)
{ memcpy(dst,MK_FP(src_s,src_o),n); }

void _cdecl PM_memcpyfn(unsigned dst_s,unsigned dst_o,void *src,unsigned n)
{ memcpy(MK_FP(dst_s,dst_o),src,n); }

void PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{
	static	unsigned	staticSel = 0;

	if (staticSel)
		PM_freeSelector(staticSel);
	staticSel = PM_createSelector(MK_PHYS(r_seg,r_off), 0xFFFF);
	*sel = staticSel;
	*off = 0;
}

unsigned PM_createSelector(unsigned long base,unsigned limit)
{
	unsigned	sel;
	union REGS	r;

	/* Allocate 1 descriptor */
	r.x.ax = 0;
	r.x.cx = 1;
	int86(0x31, &r, &r);
	if (r.x.cflag) return 0;

	/* Set base address */
	sel = r.x.bx = r.x.ax;
	r.x.ax = 7;
	r.x.cx = base >> 16;
	r.x.dx = base & 0xFFFF;
	int86(0x31, &r, &r);
	if (r.x.cflag) return 0;

	/* Set limit */
	r.x.ax = 8;
	r.x.cx = limit >> 16;
	r.x.dx = limit & 0xFFFF;
	int86(0x31, &r, &r);
	if (r.x.cflag) return 0;

	return sel;
}

void PM_freeSelector(unsigned sel)
{
	union REGS	r;

	r.x.ax = 1;
	r.x.bx = sel;
	int86(0x31, &r, &r);
}

int PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	union REGS		r;

	r.x.ax = 0x100;					/* DPMI allocate DOS memory		*/
	r.x.bx = (size + 0xF) >> 4;		/* number of paragraphs 		*/
	int86(0x31, &r, &r);
	if (r.x.cflag) return NULL;		/* DPMI call failed				*/

	*sel = r.x.dx;					/* Protected mode selector		*/
	*off = 0;
	*r_seg = r.x.ax;				/* Real mode segment			*/
	*r_off = 0;
	return 1;
}

void PM_freeRealSeg(unsigned sel,unsigned off)
{
	union REGS	r;

	r.x.ax = 0x101;					/* DPMI free DOS memory			*/
	r.x.dx = sel;					/* DX := selector from 0x100	*/
	off = off;
	int86(0x31, &r, &r);
}

typedef struct {
	long	edi;
	long	esi;
	long	ebp;
	long	reserved;
	long	ebx;
	long	edx;
	long	ecx;
	long	eax;
	short	flags;
	short	es,ds,fs,gs,ip,cs,sp,ss;
	} _RMREGS;

#define IN(reg)     rmregs.e##reg = in->x.reg
#define OUT(reg)    out->x.reg = rmregs.e##reg
#define OUT1(reg)   in->x.reg = rmregs.e##reg

void _cdecl PM_callRealMode(unsigned seg,unsigned off, RMREGS *in,
	RMSREGS *sregs)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;
	rmregs.cs = seg;
	rmregs.ip = off;

	memset(&sr, 0, sizeof(sr));
	r.x.ax = 0x301;					/* DPMI call real mode			*/
	r.h.bh = 0;
	r.x.cx = 0;
	sr.es = FP_SEG(&rmregs);
	r.x.di = FP_OFF(&rmregs);
	int86x(0x31, &r, &r, &sr);		/* Issue the interrupt			*/

	OUT1(ax); OUT1(bx); OUT1(cx); OUT1(dx); OUT1(si); OUT1(di);
	sregs->es = rmregs.es;
	sregs->ds = rmregs.ds;
}

int PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);

	memset(&sr, 0, sizeof(sr));
	r.x.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.x.cx = 0;
	sr.es = FP_SEG(&rmregs);
	r.x.di = FP_OFF(&rmregs);
	int86x(0x31, &r, &r, &sr);		/* Issue the interrupt			*/

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;

	memset(&sr, 0, sizeof(sr));
	r.x.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.x.cx = 0;
	sr.es = FP_SEG(&rmregs);
	r.x.di = FP_OFF(&rmregs);
	int86x(0x31, &r, &r, &sr);		/* Issue the interrupt */

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	sregs->es = rmregs.es;
	sregs->cs = rmregs.cs;
	sregs->ss = rmregs.ss;
	sregs->ds = rmregs.ds;
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

#endif

/*-------------------------------------------------------------------------*/
/* Phar Lap TNT DOS Extender support.									   */
/*-------------------------------------------------------------------------*/

#ifdef TNT

#include <pldos32.h>
#include <pharlap.h>
#include <hw386.h>

void PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{
	CONFIG_INF	config;

	_dx_config_inf(&config, (UCHAR*)&config);
	*sel = config.c_dos_sel;
	*off = MK_PHYS(r_seg,r_off);
}

unsigned _cdecl _PL_allocsel(void);

unsigned PM_createSelector(unsigned long base,unsigned limit)
{
	USHORT		sel;
	CD_DES		desc;

	if ((sel = _PL_allocsel()) == 0)
		return 0;
	if (_dx_ldt_rd(sel,(UCHAR*)&desc) != 0)
		return 0;
	if (limit >= 0x100000) {
		limit >>= 12;			/* Page granular for > 1Mb limit	*/
		desc.limit0_15 = limit & 0xFFFF;
		desc.limit16_19 = ((limit >> 16) & 0xF) | DOS_32 | SG_PAGE;
		}
	else {						/* Byte granular for < 1Mb limit	*/
		desc.limit0_15 = limit & 0xFFFF;
		desc.limit16_19 = ((limit >> 16) & 0xF) | DOS_32 | SG_BYTE;
		}
	desc.base0_15 = base & 0xFFFF;
	desc.base16_23 = (base >> 16) & 0xFF;
	desc.base24_31 = (base >> 24) & 0xFF;
	desc.arights = AR_DATA | AR_DPL3;
	if (_dx_ldt_wr(sel,(UCHAR*)&desc) != 0)
		return 0;
	return sel;
}

void PM_freeSelector(unsigned sel)
{
	CD_DES		desc;

	/* Set the selector's values to all zero's, so that Phar Lap will
	 * re-use the selector entry.
	 */
	if (_dx_ldt_rd(sel,(UCHAR*)&desc) != 0)
		return;
	desc.limit0_15 = 0;
	desc.limit16_19 = 0;
	desc.base0_15 = 0;
	desc.base16_23 = 0;
	desc.base24_31 = 0;
	desc.arights = 0;
	_dx_ldt_wr(sel,(UCHAR*)&desc);
}

int PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	CONFIG_INF	config;
	USHORT      addr,t;

	if (_dx_real_alloc((size + 0xF) >> 4,&addr,&t) != 0)
		return 0;
	_dx_config_inf(&config, (UCHAR*)&config);
	*sel = config.c_dos_sel;		/* Map with DOS 1Mb selector	*/
	*off = (ULONG)addr << 4;
	*r_seg = addr;					/* Real mode segment address	*/
	*r_off = 0;						/* Real mode segment offset		*/
	return 1;
}

void PM_freeRealSeg(unsigned sel,unsigned off)
{
	sel = sel;
	_dx_real_free(off >> 4);
}

/* Even though Phar Lap TNT does provide a _dx_call_real() routine, this
 * routine is limited in the fact that you cannot specify the values of
 * all the machine registers. However we can simulate what we need
 * temporarily hooking the INT 2Fh vector with a small real mode stub
 * that will call our real mode code for us.
 */

static unsigned char int2FHandler[] = {
	0x00,0x00,0x00,0x00,		/* 	__PMODE_callReal variable		*/
	0xFB,						/*	sti								*/
	0x2E,0xFF,0x1E,0x00,0x00,	/*  call	[cs:__PMODE_callReal]	*/
	0xCF,						/*	iretf							*/
	};

void _cdecl PM_callRealMode(unsigned seg,unsigned off, RMREGS *in,
	RMSREGS *sregs)
{
	ULONG			oldHandler;
	unsigned		psel,poff,r_seg,r_off;

	PM_allocRealSeg(sizeof(int2FHandler), &psel, &poff, &r_seg, &r_off);
	PM_memcpyfn(psel,poff,int2FHandler,sizeof(int2FHandler));
	PM_setWord(psel,poff,off);			/* Plug in address to call	*/
	PM_setWord(psel,poff+2,seg);
	_dx_rmiv_get(0x2F, &oldHandler);	/* Save old handler			*/
	_dx_rmiv_set(0x2F, (r_seg << 16) | (r_off+4));
	PM_int86x(0x2F, in, in, sregs);		/* Call real mode			*/
	_dx_rmiv_set(0x2F, oldHandler);		/* Restore old handler		*/
	PM_freeRealSeg(psel,poff);
}

#define IN(reg)     rmregs.e##reg = in->x.reg
#define OUT(reg)    out->x.reg = rmregs.e##reg

int PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	SWI_REGS	rmregs;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);

    _dx_real_int(intno,&rmregs);

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	SWI_REGS	rmregs;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;

    _dx_real_int(intno,&rmregs);

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	sregs->es = rmregs.es;
	sregs->cs = rmregs.cs;
	sregs->ss = rmregs.ss;
	sregs->ds = rmregs.ds;
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

#endif

/*-------------------------------------------------------------------------*/
/* Symantec C++ DOSX support (also X32-VM support).						   */
/*-------------------------------------------------------------------------*/

#ifdef	DOSX

void PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{
	*sel = _x386_zero_base_selector;
	*off = MK_PHYS(r_seg,r_off);
}

unsigned PM_createSelector(unsigned long base,unsigned limit)
{
	void _far *p = _x386_mk_protected_ptr(base);
	limit = limit;
	return FP_SEG(p);
}

void PM_freeSelector(unsigned sel)
{
	_x386_free_protected_ptr(MK_FP(sel,0));
}

int PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	union REGS	r;

	r.h.ah = 0x48;					/* DOS function 48h - allocate mem	*/
	r.x.bx = (size + 0xF) >> 4;		/* Number of paragraphs to allocate	*/
	int86(0x21, &r, &r);			/* Call DOS extender				*/
	if (r.x.cflag)
		return 0;					/* Could not allocate the memory	*/
	*sel = _x386_zero_base_selector;
	*off = r.e.eax << 4;
	*r_seg = r.e.eax;
	*r_off = 0;
	return 1;
}

void PM_freeRealSeg(unsigned sel,unsigned off)
{
	/* Cannot de-allocate this memory */
	sel = sel; off = off;
}

/* DOSX does not directly support calling a real mode procedure from
 * protected mode. However we can simulate what we need temporarily
 * hooking the INT 2Fh vector with a small real mode stub that will call
 * our real mode code for us.
 */

static unsigned char int2FHandler[] = {
	0x00,0x00,0x00,0x00,		/* 	__PMODE_callReal variable		*/
	0xFB,						/*	sti								*/
	0x2E,0xFF,0x1E,0x00,0x00,	/*  call	[cs:__PMODE_callReal]	*/
	0xCF,						/*	iretf							*/
	};

void _cdecl PM_callRealMode(unsigned seg,unsigned off, RMREGS *in,
	RMSREGS *sregs)
{
	unsigned		psel,poff,r_seg,r_off;
	unsigned short	far *p = MK_FP(_x386_zero_base_selector,0x2F * 4);
	unsigned    	oldSeg,oldOff;

	PM_allocRealSeg(sizeof(int2FHandler), &psel, &poff, &r_seg, &r_off);
	PM_memcpyfn(psel,poff,int2FHandler,sizeof(int2FHandler));
	PM_setWord(psel,poff,off);			/* Plug in address to call	*/
	PM_setWord(psel,poff+2,seg);
	oldOff = *p;	oldSeg = *(p+1);	/* Save old handler			*/
	*p = r_off+4;	*(p+1) = r_seg;		/* Hook 2F handler			*/
	PM_int86x(0x2F, in, in, sregs);		/* Call real mode code		*/
	*p = oldOff;	*(p+1) = oldSeg;	/* Restore old handler		*/
	PM_freeRealSeg(psel,poff);
}

int PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	return int86_real(intno,in,out);
}

int PM_int86x(int intno, RMREGS *in, RMREGS *out, RMSREGS *sregs)
{
    return int86x_real(intno,in,out,sregs);
}

#endif

/*-------------------------------------------------------------------------*/
/* Borland's DPMI32 DOS Power Pack Extender support.					   */
/*-------------------------------------------------------------------------*/

#ifdef	DPMI32

void PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{
	static unsigned lowMem = 0;
	if (!lowMem)
		lowMem = PM_createSelector(0, 0xFFFFF);
	*sel = lowMem;
	*off = MK_PHYS(r_seg,r_off);
}

unsigned PM_createSelector(unsigned long base,unsigned limit)
{
	unsigned	sel;
	union REGS	r;

	/* Allocate 1 descriptor */
	r.w.ax = 0;
	r.w.cx = 1;
	int386(0x31, &r, &r);
	if (r.w.cflag) return 0;

	/* Set base address */
	sel = r.w.bx = r.w.ax;
	r.w.ax = 7;
	r.w.cx = base >> 16;
	r.w.dx = base & 0xFFFF;
	int386(0x31, &r, &r);
	if (r.w.cflag) return 0;

	/* Set limit */
	r.w.ax = 8;
	r.w.cx = limit >> 16;
	r.w.dx = limit & 0xFFFF;
	int386(0x31, &r, &r);
	if (r.w.cflag) return 0;

	return sel;
}

void PM_freeSelector(unsigned sel)
{
	union REGS	r;

	r.w.ax = 1;
	r.w.bx = sel;
	int386(0x31, &r, &r);
}

int PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	union REGS		r;

	r.w.ax = 0x100;					/* DPMI allocate DOS memory		*/
	r.w.bx = (size + 0xF) >> 4;		/* number of paragraphs 		*/
	int386(0x31, &r, &r);
	if (r.w.cflag) return NULL;		/* DPMI call failed				*/

	*sel = r.w.dx;					/* Protected mode selector		*/
	*off = 0;
	*r_seg = r.w.ax;				/* Real mode segment			*/
	*r_off = 0;
	return 1;
}

void PM_freeRealSeg(unsigned sel,unsigned off)
{
	union REGS	r;

	r.w.ax = 0x101;					/* DPMI free DOS memory			*/
	r.w.dx = sel;					/* DX := selector from 0x100	*/
	off = off;
	int386(0x31, &r, &r);
}

typedef struct {
	long	edi;
	long	esi;
	long	ebp;
	long	reserved;
	long	ebx;
	long	edx;
	long	ecx;
	long	eax;
	short	flags;
	short	es,ds,fs,gs,ip,cs,sp,ss;
	} _RMREGS;

#define IN(reg)     rmregs.e##reg = in->x.reg
#define OUT(reg)    out->x.reg = rmregs.e##reg
#define OUT1(reg)   in->x.reg = rmregs.e##reg

void _cdecl PM_callRealMode(unsigned seg,unsigned off, RMREGS *in,
	RMSREGS *sregs)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;
	rmregs.cs = seg;
	rmregs.ip = off;

	segread(&sr);
	r.w.ax = 0x301;					/* DPMI call real mode			*/
	r.h.bh = 0;
	r.w.cx = 0;
	sr.es = sr.ds;
	r.x.edi = (unsigned)&rmregs;
	int386x(0x31, &r, &r, &sr);		/* Issue the interrupt			*/

	OUT1(ax); OUT1(bx); OUT1(cx); OUT1(dx); OUT1(si); OUT1(di);
	sregs->es = rmregs.es;
	sregs->ds = rmregs.ds;
}

int PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);

	segread(&sr);
	r.w.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.w.cx = 0;
	sr.es = sr.ds;
	r.x.edi = (unsigned)&rmregs;
	int386x(0x31, &r, &r, &sr);		/* Issue the interrupt			*/

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;

	segread(&sr);
	r.w.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.w.cx = 0;
	sr.es = sr.ds;
	r.x.edi = (unsigned)&rmregs;
	int386x(0x31, &r, &r, &sr);		/* Issue the interrupt */

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	sregs->es = rmregs.es;
	sregs->cs = rmregs.cs;
	sregs->ss = rmregs.ss;
	sregs->ds = rmregs.ds;
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

#endif

/*-------------------------------------------------------------------------*/
/* Watcom C/C++ with Rational DOS/4GW support.							   */
/*-------------------------------------------------------------------------*/

#ifdef	DOS4GW

void PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{
	*sel = FP_SEG(sel);
	*off = MK_PHYS(r_seg,r_off);
}

unsigned PM_createSelector(unsigned long base,unsigned limit)
{
	unsigned	sel;
	union REGS	r;

	/* Allocate 1 descriptor */
	r.w.ax = 0;
	r.w.cx = 1;
	int386(0x31, &r, &r);
	if (r.w.cflag) return 0;

	/* Set base address */
	sel = r.w.bx = r.w.ax;
	r.w.ax = 7;
	r.w.cx = base >> 16;
	r.w.dx = base & 0xFFFF;
	int386(0x31, &r, &r);
	if (r.w.cflag) return 0;

	/* Set limit */
	r.w.ax = 8;
	r.w.cx = limit >> 16;
	r.w.dx = limit & 0xFFFF;
	int386(0x31, &r, &r);
	if (r.w.cflag) return 0;

	return sel;
}

void PM_freeSelector(unsigned sel)
{
	union REGS	r;

	r.w.ax = 1;
	r.w.bx = sel;
	int386(0x31, &r, &r);
}

int PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	union REGS		r;

	r.w.ax = 0x100;					/* DPMI allocate DOS memory		*/
	r.w.bx = (size + 0xF) >> 4;		/* number of paragraphs 		*/
	int386(0x31, &r, &r);
	if (r.w.cflag) return NULL;		/* DPMI call failed				*/

	*sel = r.w.dx;					/* Protected mode selector		*/
	*off = 0;
	*r_seg = r.w.ax;				/* Real mode segment			*/
	*r_off = 0;
	return 1;
}

void PM_freeRealSeg(unsigned sel,unsigned off)
{
	union REGS	r;

	r.w.ax = 0x101;					/* DPMI free DOS memory			*/
	r.w.dx = sel;					/* DX := selector from 0x100	*/
	off = off;
	int386(0x31, &r, &r);
}

typedef struct {
	long	edi;
	long	esi;
	long	ebp;
	long	reserved;
	long	ebx;
	long	edx;
	long	ecx;
	long	eax;
	short	flags;
	short	es,ds,fs,gs,ip,cs,sp,ss;
	} _RMREGS;

#define IN(reg)     rmregs.e##reg = in->x.reg
#define OUT(reg)    out->x.reg = rmregs.e##reg

/* DOS4GW does not directly support calling a real mode procedure from
 * protected mode (DPMI service 301 is missing). However we can simulate
 * what we need temporarily hooking the INT 2Fh vector with a small real
 * mode stub that will call our real mode code for us.
 */

static unsigned char int2FHandler[] = {
	0x00,0x00,0x00,0x00,		/* 	__PMODE_callReal variable		*/
	0xFB,						/*	sti								*/
	0x2E,0xFF,0x1E,0x00,0x00,	/*  call	[cs:__PMODE_callReal]	*/
	0xCF,						/*	iretf							*/
	};

void _cdecl PM_callRealMode(unsigned seg,unsigned off, RMREGS *in,
	RMSREGS *sregs)
{
	unsigned		psel,poff,r_seg,r_off;
	unsigned short	*p = (unsigned short*)(0x2F * 4);
	unsigned    	oldSeg,oldOff;

	PM_allocRealSeg(sizeof(int2FHandler), &psel, &poff, &r_seg, &r_off);
	PM_memcpyfn(psel,poff,int2FHandler,sizeof(int2FHandler));
	PM_setWord(psel,poff,off);			/* Plug in address to call	*/
	PM_setWord(psel,poff+2,seg);
	oldOff = *p;	oldSeg = *(p+1);	/* Save old handler			*/
	*p = r_off+4;	*(p+1) = r_seg;		/* Hook 2F handler			*/
	PM_int86x(0x2F, in, in, sregs);		/* Call real mode code		*/
	*p = oldOff;	*(p+1) = oldSeg;	/* Restore old handler		*/
	PM_freeRealSeg(psel,poff);
}

int PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);

	memset(&sr, 0, sizeof(sr));
	r.w.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.w.cx = 0;
	sr.es = FP_SEG(&rmregs);
    r.x.edi = FP_OFF(&rmregs);
	int386x(0x31, &r, &r, &sr);		/* Issue the interrupt			*/

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;

	memset(&sr, 0, sizeof(sr));
	r.w.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.w.cx = 0;
	sr.es = FP_SEG(&rmregs);
	r.x.edi = FP_OFF(&rmregs);
	int386x(0x31, &r, &r, &sr);		/* Issue the interrupt */

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	sregs->es = rmregs.es;
	sregs->cs = rmregs.cs;
	sregs->ss = rmregs.ss;
	sregs->ds = rmregs.ds;
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

#endif

/*-------------------------------------------------------------------------*/
/* DJGPP port of GNU C++ support.										   */
/*-------------------------------------------------------------------------*/

#ifdef DJGPP

#include <dpmi.h>

void PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{
	static unsigned lowMem = 0;
	if (!lowMem)
		lowMem = PM_createSelector(0, 0xFFFFF);
	*sel = lowMem;
	*off = MK_PHYS(r_seg,r_off);
}

unsigned PM_createSelector(unsigned long base,unsigned limit)
{
	unsigned	sel;
	union REGS	r;

	/* Allocate 1 descriptor */
	r.x.ax = 0;
	r.x.cx = 1;
	int86(0x31, &r, &r);
	if (r.x.cflag) return 0;

	/* Set base address */
	sel = r.x.bx = r.x.ax;
	r.x.ax = 7;
	r.x.cx = base >> 16;
	r.x.dx = base & 0xFFFF;
	int86(0x31, &r, &r);
	if (r.x.cflag) return 0;

	/* Set limit */
	r.x.ax = 8;
	r.x.cx = limit >> 16;
	r.x.dx = limit & 0xFFFF;
	int86(0x31, &r, &r);
	if (r.x.cflag) return 0;

	return sel;
}

void PM_freeSelector(unsigned sel)
{
	union REGS	r;

	r.x.ax = 1;
	r.x.bx = sel;
	int86(0x31, &r, &r);
}

int PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	union REGS		r;

	r.x.ax = 0x100;					/* DPMI allocate DOS memory		*/
	r.x.bx = (size + 0xF) >> 4;		/* number of paragraphs 		*/
	int86(0x31, &r, &r);
	if (r.x.cflag) return NULL;		/* DPMI call failed				*/

	*sel = r.x.dx;					/* Protected mode selector		*/
	*off = 0;
	*r_seg = r.x.ax;				/* Real mode segment			*/
	*r_off = 0;
	return 1;
}

void PM_freeRealSeg(unsigned sel,unsigned off)
{
	union REGS	r;

	r.x.ax = 0x101;					/* DPMI free DOS memory			*/
	r.x.dx = sel;					/* DX := selector from 0x100	*/
	off = off;
	int86(0x31, &r, &r);
}

typedef struct {
	long	edi;
	long	esi;
	long	ebp;
	long	reserved;
	long	ebx;
	long	edx;
	long	ecx;
	long	eax;
	short	flags;
	short	es,ds,fs,gs,ip,cs,sp,ss;
	} _RMREGS;

#define IN(reg)     rmregs.e##reg = in->x.reg
#define OUT(reg)    out->x.reg = rmregs.e##reg
#define OUT1(reg)   in->x.reg = rmregs.e##reg

void _cdecl PM_callRealMode(unsigned seg,unsigned off, RMREGS *in,
	RMSREGS *sregs)
{
	_RMREGS			rmregs;
	union REGS		r;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;
	rmregs.cs = seg;
	rmregs.ip = off;

	r.x.ax = 0x301;					/* DPMI call real mode			*/
	r.x.bx = 0;
	r.x.cx = 0;
	r.x.di = (unsigned)&rmregs;
	int86(0x31, &r, &r);			/* Issue the interrupt			*/

	OUT1(ax); OUT1(bx); OUT1(cx); OUT1(dx); OUT1(si); OUT1(di);
	sregs->es = rmregs.es;
	sregs->ds = rmregs.ds;
}

int PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	_RMREGS			rmregs;
	union REGS		r;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);

	r.x.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.x.bx = intno;
	r.x.cx = 0;
	r.x.di = (unsigned)&rmregs;
	int86(0x31, &r, &r);			/* Issue the interrupt			*/

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	_RMREGS			rmregs;
	union REGS		r;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;

	r.x.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.x.bx = intno;
	r.x.cx = 0;
	r.x.di = (unsigned)&rmregs;
	int86(0x31, &r, &r);			/* Issue the interrupt 			*/

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	sregs->es = rmregs.es;
	sregs->cs = rmregs.cs;
	sregs->ss = rmregs.ss;
	sregs->ds = rmregs.ds;
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

#endif
