

//------------------------------VOBJ.CPP-----------------------------------//
//
// author: Daniel W. Moore 
//
// Implements virtual objects
//
//

#include "vobj.h"

// compile with "fast huge pointers" option off!

// -------- overload operator new for VObject & derived types -------- 

void* VObject::operator new (size_t sz) {
  VHANDLE hnd;
  VObject* p = (VObject*) new char[sz];         // allocate buffer for object
  if (p) {                                      // If buffer exists,
    p->size = sz;                               // fill in size and handle
    p->handle = hnd = AVMem.AllocMem(sz);       // information.
    if (!hnd) {                                 // zero handle means
      delete  (char*)p;                       // virtual array is full !!
      p=NULL;
      //virtual allocation failure, return null pointer
      // (solution: use more Disk/expanded memory space)
    }
  }
  return p;
}

// -- operator=(): splice out of chain, assign to new object (via handle) --

VPtrBase& VPtrBase::operator=(VHANDLE hnd)
{
  if (PLast) PLast->PNext=PNext;  // assign a pointer object using a
  if (PNext) PNext->PLast=PNext;  // already existing handle
  PLast = PNext = NULL;
  VEntry *E = AVMem.HeadEntry;
  while (E) {
    if (E->handle == hnd) break;
    E=E->down;
  }
  if (E)
    *this = *E;    // will call VPtrBase::operator=(VPtrBase&)
  else  {
    handle = hnd;
    AVMem.LoadObject(this);
  }
  return *this;
}

// -------- overload dereferencing operators --------

VObject& VPtrBase::operator*()
{
  if (!PNext)
  {
    if (OPtr) {
      AVMem.RemoveEntry((VEntry*)this);         // bring entry to  head of
      AVMem.AddEntryAtHead((VEntry*)this);      // linked list (for LRU)
      return *OPtr;
    }
    else
    {
      AVMem.LoadObject(this); // will modify (this->PNext)
	///// if (!PNext) exit(1);  //  if this happens, maxnEntries in
    }                               // VMemory object needs to be smaller.
  }
  return PNext->operator*();
}

VObject* VPtrBase::operator->()
{
  if (!PNext)
  {
    if (OPtr) {
      AVMem.RemoveEntry((VEntry*)this);         // bring entry to head of
      AVMem.AddEntryAtHead((VEntry*)this);      // linked list (for LRU)
      return OPtr;
    }
    else
    {
      AVMem.LoadObject(this); // will modify (this->PNext)
	///// if (!PNext) exit(1);
    }
  }
  return PNext->operator->();
}

// --- assignment from plain C++ pointer: same as constructor, but take out
//      of current chain 

VPtrBase& VPtrBase::operator=(VObject* p)
{
  if (PNext) PNext->PLast=PLast; // if already in a pointer chain, remove it
  if (PLast) PLast->PNext=PNext;
  handle = p ? p->handle : 0;
  OPtr = NULL;
  PLast = PNext = NULL;
  if (p)
  {
    AVMem.CreateEntryFor(this,p); // create entry referencing VObject
  }
  return *this;
}

// --- usage: p->Lock(), p->Unlock() -- locking mechanism for VPtr  ---

VObject* VPtrBase::Lock(int s) // (Un-)Lock an object in memory (default s=1)
{
  VEntry* p = AVMem.LoadObject(this);
  if (p) {
    if (p->OPtr) (((VEntry*)p)->locked)=s;
    return p->OPtr;
  } else
  return 0;
}

// -- constructor for creating a VPtr from an object handle

VPtrBase::VPtrBase(VHANDLE hnd)
{
  OPtr = NULL;
  PNext = PLast = NULL;
  (*this) = hnd;  // will call operator= (hnd)"
}

// --- activate a VEntry with the address of a resident object ---

void VEntry::activate(VPtrBase* pO, VObject* ptr)
{
    InsertRightOf(*pO);                 // make VPtr reference a VObject
    OPtr = ptr;                         // in memory
}

// --- beginning with "this" entry, find the L.R.U. unlocked entry ---

VEntry* VEntry::FindUnlocked()
{
  VEntry* temp = this;    // search from this entry up
  while (temp->locked) {  // til unlocked entry found
    temp =  temp->up;
    if (!temp) break;
  }
  return (temp);
}

//-- destructor:   When an LRU stack entry is deleted, store its object. --

VEntry::~VEntry()
{
    if (OPtr) AVMem.StoreObject(this);
}

// Create global Virtual memory manager object (there should only be one)
//   based on a flat-model Disk/EMS array object

VMemory::VMemory(FlatArray* fa, size_t numblk, size_t blksize, size_t maxne)
  : VtlArray( *fa )
{
  if (!VtlArray) {
    bSiz=nBlk=0;
    return;
  } else {
    nBlk= align( numblk+1 , 2 );   // make these even numbers (enable
    bSiz= align( blksize , 2 );    //  word-aligned access to virtual array)
    long arraysize = VtlArray.get_size();         // if virtual array size too
    if (nBlk*(bSiz+sizeof(size_t)) > arraysize) { // small, calculate new
      nBlk=arraysize/(bSiz+sizeof(size_t));       // number of blocks
      nBlk/=2;
      nBlk*=2;
    }
    for (int i=0; i<nBlk; i++) VtlArray(i)=0;     // all blocks start empty
  }
  nEntries=0;                                   // start with nothing resident
  HeadEntry = TailEntry = NULL;
  maxnEntries = maxne;
  iCurrent=1;           // index used for free block search
}

// Load object if not present in memory; otherwise, just
// return pointer to the corresponding stack entry

VEntry *VMemory::LoadObject(VPtrBase* pO)
{
  long i;
  if (!pO->handle) return NULL;
  while (pO->PNext)
    pO=pO->PNext;                       // if object already swapped in,
  if (pO->OPtr) return (VEntry*)pO;     // return pointer to its entry...

  VHANDLE hnd=pO->handle;               // if not, load it first.
  VHANDLE offshnd = offs(hnd);
  size_t sz =  VtlArray(hnd);

  char *p = new char [ sz ];            // 'create' object in memory w/o
  if (!p) return 0;                     // invoking constructor!
  for (i=0; i<sz; i++) {
    p[i] = VtlArray[offshnd+i];
  }
  VEntry* NewEntry = CreateEntryFor(pO,(VObject*)p);

  return NewEntry;
}

// Create a stack entry, with memory pointer, so that the
// object resides in conventional memory.

VEntry* VMemory::CreateEntryFor(VPtrBase* pO,VObject *p)
{
  int OldnEntries;                      // as long as can find excess unlocked
  while (nEntries>maxnEntries) {        // entries, delete them
    OldnEntries = nEntries;
    if (OldnEntries == Delete_LRU_Entry()) break;
  }
  VEntry* NewEntry = new VEntry;        // add new entry for this object
  if (NewEntry) {
    AddEntryAtHead(NewEntry);
    NewEntry->activate(pO,p);           // attach VPtr to object 
  }
  return NewEntry;
}

// -- remove a VEntry from the LRU stack 

void VMemory::RemoveEntry(VEntry* p)
{
  if (p->down) p->down->up = p->up;     // modify linked list pointers
	else TailEntry = p->up;
  if (p->up) p->up->down = p->down; 
	else HeadEntry = p->down;
  p->down = NULL;                       // detach internal pointers
  p->up = NULL;
  nEntries--;
}

// -- add a VEntry ad the LRU top-of-stack( head )

void VMemory::AddEntryAtHead(VEntry * p)
{
  p->down = HeadEntry;                  // add new "in-swapped" entry
  if (HeadEntry) HeadEntry->up = p;     // at head of list (for LRU algorithm)
  HeadEntry = p;
  if (!TailEntry) TailEntry = HeadEntry;  // linked list management items
  nEntries++;
}

// -- find the least recently used unlocked VEntry and remove it --
//    effect is to swap object out

VMemory::Delete_LRU_Entry()     // returns remaining number of entries
{
  if (!TailEntry) return 0;     // empty list, nothing to do.
  VEntry* TossEntry 
	= TailEntry->FindUnlocked();
  if (TossEntry) {              // if a L.R.U. unlocked entry found,
    RemoveEntry(TossEntry);     // remove it (effectively swapping object out;
    delete TossEntry;           // see VEntry destructor)
  }
  return nEntries;
}

// -- flush all cached objects from memory

void VMemory::flush() {         // gets rid of all entries, even if locked
  while (HeadEntry) {
    VEntry* temp = HeadEntry->down;
    delete HeadEntry;           // results in storing objects to virt. array
    HeadEntry=temp;
  }
  TailEntry=NULL;
  nEntries=0;
}

// -- deallocate blocks in virtual memory heap

void VMemory::DeAllocMem(VHANDLE hnd)     // deallocate object
{                                         //  in virtual array
   size_t sz = VtlArray( hnd );
   sz=sz?sz:1;
   size_t nb=((sz-1)/bSiz+1);             // get # blocks occupied
   for (size_t cnt = 0; cnt<nb; cnt++) {
     VtlArray(hnd+cnt) = 0;               // mark array blocks as unused
   }
}

// -- remove object from conventional memory, store into permanent address
// --  in virtual heap

void VMemory::StoreObject(VPtrBase *poptr)
{
    while (poptr->PNext) poptr=poptr->PNext;
    VObject* pobj = poptr->OPtr;
    char *cpobj = (char*) pobj;         // read buffer ptr from last in
    if (!pobj) {                        // linked list; if null, then
     return;                            //  not resident; return
    }
    size_t sz = pobj->size;             // get size and handle from resident
    VHANDLE hnd = pobj->handle;         //  object
    long offset = offs(hnd);
    for (long i=0; i<sz; i++) {         // swap object's contents out
       VtlArray[offset+i] = cpobj[i];
    }
    delete  cpobj; // destroy buffer but don't destroy object
}

// -- allocate however many blocks needed for object of size 'sz'

VHANDLE VMemory::AllocMem(size_t sz)
{
    sz=sz?sz:1;
    size_t bsz=((sz-1)/bSiz+1); // convert sz to #blocks
    size_t count=0;
    unsigned i,j;
    for (i=1; i<nBlk; i++)
    {                           // Search all available blocks starting from
      if (iCurrent>=nBlk) {     // current placeholder.
	iCurrent=1;
	count=0;                // wrap-around to beginning if necessary
      }
      if (VtlArray(iCurrent++)==0)  // check if block used
      {
	if (++count==bsz)      // required # consecutive blocks found?
	{
	   for (j=count; j>0; j--) VtlArray(iCurrent-j)=1; // mark as used
	   VHANDLE hnd = iCurrent-count;
	   VtlArray(hnd) =  sz;          // note size and return beginning
	   return hnd;                   // block number as handle
	}
      }
      else {
	count = 0;              // found break in consecutive free blocks;
      }                                 // reset the count
    }
    return (VHANDLE)(0);        // fell through loop: return zero handle
};                              //    (i.e. failure)

// -- virtual memory equivalent of C++ operator delete: 
//    'Delete(p);' replaces 'delete p;' syntax for VPtr objects

void Delete(VPtrBase& p)      
{
    VEntry *PVE = AVMem.LoadObject(&p); // must load object first...
    if (PVE) {
      VObject* PO = PVE->OPtr;
      VHANDLE hnd = PO->handle;
      p.Lock();                         // lock object in memory in case
      if (PO) delete PO;                // it contains VPtrs used in
      p.Unlock();                       // destructor!
      AVMem.RemoveEntry(PVE);           // remove entry from VMemory's list
      PVE->OPtr = NULL;
      delete PVE;
      AVMem.DeAllocMem(hnd);            // deallocate blocks in virt. array
    }
}

// -- compile with fast huge pointers off!

int PTRS_EQUAL( void far* p0, void far* q0 )
{
  char huge *p = (1+(char huge *)p0);   // compare two far pointers
  char huge *q = (1+(char huge *)q0);   // ( not always as easy as a
  return (q==p);                        // simple == )

}

// lock function as applied to a simple C++ pointer (see VPtrBase::Lock)

void Lock(VObject* op, int s)           // lock a VObject so it can't
{                                       // be swapped out til we unlock

  VEntry* E = AVMem.HeadEntry;
  while (E) {                           // .. must find pointer value in list
    if (PTRS_EQUAL(op,E->OPtr)) { E->Lock(s); break; }
    E=E->down;
  }
}

void Unlock(VObject* op)
{
  Lock(op,0);
}

