/* PLFILES.C : File I/O routines for PICLAB.  These routines are used in
** place of STDIO to improve speed a little.  The major improvement is
** that the program has direct access to the file I/O buffers, rather than
** having to copy bytes from the I/O buffers into user memory.	Other
** routines involving file I/O are here as well.  No command handlers.
** This code is very MSDOS specific.
*/

#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dos.h>
#include <io.h>

#include "piclab.h"

static struct _ioblk *checkio(int, int);

/* The makepath() routine is a general-purpose routine for making path
** names out of path, name, and extension fields.  If a path or extension
** is present in the filename, it is not changed; otherwise the path or
** extension specified is used.  This is done so that default paths and
** extensions can be overridden.
*/
char *makepath(char *dirname, char *fname, char *ext)
{
    register char *cp;
    static char path[64];

	if (strchr(fname, '\\') == NULL && strchr(fname, '/') == NULL &&
	  strchr(fname, ':') == NULL) {
		strcpy(path, dirname);
		cp = path+strlen(path)-1;
		if (cp >= path && *cp != ':' && *cp != '/' && *cp != '\\') {
			*++cp = '/';
			*++cp = '\0';
		}
	} else *path = '\0';

    strcat(path, fname);
    if (ext != NULL && (cp = strchr(fname, '.')) == NULL) {
        cp = path + strlen(path);
        *cp++ = '.';
        strcpy(cp, ext);
    }
    return path;
}

/* Utility function to check the validity of a file handle; returns a pointer
** to the _ioblk structure if valid, NULL otherwise.
*/
static struct _ioblk *checkio(int h, int mode)
{
    register struct _ioblk *bp;

	if (h >= 0 && h < NFILES) {
        bp = &ioblocks[h];
		if ((mode & bp->mode) != 0) return bp;
    }
	return NULL;
}

/* Close all open files.  Returns 1 if any open files were found.
*/
int p_closeall()
{
    int i, r;

    r = 0;
    for (i=0; i<NFILES; ++i) if (ioblocks[i].handle != 0) {
        r = 1;
        p_close(i);
    }
    return r;
}

/* Open a file.  Fname specifies final pathname, buffer must point to a
** block of memory (bufsize+width) bytes long, and mode must be READ or
** WRITE.  File handle (or -1 on error) is returned.  If mode is WRITE,
** linebuf(h) points to buffer where (width) bytes may be written before
** issuing p_putline().  File cannot be opened for simultaneous read and
** write with these calls.
*/
int p_open(char *fname, char *buffer, int bufsize, int width, int mode)
{
    int i, h;
    struct _ioblk *bp;

    for (i=0; i<NFILES; ++i) if (ioblocks[i].handle == 0) break;
	if (i == NFILES) return -1;

    bp = &ioblocks[i];
	if (width < 1) return -1; else bp->width = width;
    bp->mode = mode;
	bp->linebuf = bp->iobuf = buffer;
    bp->bfull = bp->bufp = 0;
    bp->bsize = bufsize;
	strcpy(bp->fname, fname);

	if (mode == WRITE) {
		if (_dos_creat(fname, _A_NORMAL, &h)) return -1;
    } else {
		if (_dos_open(fname, O_RDONLY, &h)) return -1;
    }
	bp->handle = h;
    return i;
}

/* Close existing file, flush any pending writes, then open in new mode.
*/
int p_reopen(int handle, int mode)
{
	struct _ioblk *bp;
	unsigned bw;
	int h;

	if ((bp = checkio(handle, READ|WRITE)) == NULL) return -1;
	if (bp->mode == WRITE && bp->bfull != 0) {
		_dos_write(bp->handle, bp->iobuf, bp->bfull, &bw);
    }
	_dos_close(bp->handle);

	bp->mode = mode;
	bp->linebuf = bp->iobuf;
    bp->bfull = bp->bufp = 0;
	if (mode == WRITE) {
		if (_dos_creat(bp->fname, _A_NORMAL, &h)) return -1;
    } else {
		if (_dos_open(bp->fname, O_RDONLY, &h)) return -1;
	}
	bp->handle = h;
	return handle;
}

/* Flush any pending writes, then seek to new position.
*/
int p_seek(int handle, long position)
{
	struct _ioblk *bp;
	unsigned bw;


	if ((bp = checkio(handle, READ|WRITE)) == NULL) return -1;

	if (bp->mode == WRITE && bp->bfull != 0) {
		_dos_write(bp->handle, bp->iobuf, bp->bfull, &bw);
    }
	lseek(bp->handle, position, 0);

	bp->linebuf = bp->iobuf;
    bp->bfull = bp->bufp = 0;
	return handle;
}

/* Close file opened with p_open.  Flushes any pending writes.
*/
void p_close(int h)
{
    struct _ioblk *bp;
    unsigned bw;

	if ((bp = checkio(h, READ|WRITE)) == NULL) return;

	if (bp->mode == WRITE && bp->bfull != 0) {
		_dos_write(bp->handle, bp->iobuf, bp->bfull, &bw);
    }
	_dos_close(bp->handle);
	bp->handle = 0;
}

/* Get one byte from file.	Returns byte or -1 on error.
*/
int p_getc(int h)
{
	struct _ioblk *bp;
	unsigned br;
	int r;

	if ((bp = checkio(h, READ)) == NULL) return -1;
	if (bp->bfull < 1) {
        if (bp->bfull) memcpy(bp->iobuf, bp->iobuf+bp->bufp, bp->bfull);
        bp->bufp = 0;
		_dos_read(bp->handle, bp->iobuf+bp->bfull, bp->bsize, &br);
		bp->bfull += br;
		if (bp->bfull < 1) return -1;
    }
    bp->linebuf = bp->iobuf+bp->bufp;
	r = *bp->linebuf;
	++bp->bufp;
	--bp->bfull;
	return r;
}

/* Get one line from file.  Returns pointer or NULL on error.
*/
char *p_gets(char *str, int width, int h)
{
	int i, r;

	for (i=0; i<width-1; ++i) {
		if ((r = p_getc(h)) == '\n' || r == -1) break;
		else str[i] = (char)r;
	}
	if (r == -1) return NULL;
	str[i] = '\0';
	return str;
}

/* Get one line from file.	If w is 0, use width specified in p_open.  At the
** end of this function, linebuf(h) points to the file i/o buffer where w
** bytes can be found.
*/
int p_getline(int h, int w)
{
	struct _ioblk *bp;
    unsigned br;

	if ((bp = checkio(h, READ)) == NULL) return 0;
    if (w == 0) w = bp->width;
    while (bp->bfull < w) {
        if (bp->bfull) memcpy(bp->iobuf, bp->iobuf+bp->bufp, bp->bfull);
        bp->bufp = 0;
		_dos_read(bp->handle, bp->iobuf+bp->bfull, bp->bsize, &br);
        bp->bfull += br;
        if (br == 0) break;
    }
    if (bp->bfull < w) w = bp->bfull;
    bp->linebuf = bp->iobuf+bp->bufp;
    bp->bufp += w;
    bp->bfull -= w;
    return w;
}

/* Put one line to file.  If w is 0, width specified in p_open is used.  At
** the end of this function, linebuf(h) points to I/O buffer where (width)
** bytes may be written before the next p_putline() call.
*/
int p_putline(int h, int w)
{
    struct _ioblk *bp;
    unsigned bw;

	if ((bp = checkio(h, WRITE)) == NULL) return 0;
    if (w == 0) w = bp->width;
    bp->bfull += w;
    bp->bufp += w;
    while (bp->bfull >= bp->bsize) {
		_dos_write(bp->handle, bp->iobuf, bp->bsize, &bw);

        bp->bfull -= bw;
        if (bp->bfull) memcpy(bp->iobuf, bp->iobuf+bw, bp->bfull);
        bp->bufp = bp->bfull;
        if (bw == 0) break;
    }
    bp->linebuf = bp->iobuf+bp->bufp;
    return w;
}

/* Open an image plane for read or write.  File and width specified in image
** buffer are used, and all reads and writes must be one line at a time.
*/
struct _plane *openplane(int plane, struct _imgbuf *ib, int mode)
{
    static char *fileext[] = { "R8", "G8", "B8" };
	int h;
	U8 FAR *bp;
    char *path;

	if ((bp = (U8 *)talloc(BUFSIZE + ib->width)) == NULL) return NULL;
    path = makepath(tempdir, ib->filename, fileext[plane]);
	if ((h = p_open(path, bp, BUFSIZE, ib->width, mode)) < 0) return NULL;
    planes[h].linebuf = linebuf(h);
    planes[h].handle = h;
    planes[h].width = ib->width;
	return &planes[h];
}

/* Close image plane opened with openplane().
*/
void closeplane(struct _plane *pl)
{
    p_close(pl->handle);
}

/* Get one line from image plane.
*/
int getline(struct _plane *pl)
{
    int r;

    r = p_getline(pl->handle, pl->width);
    pl->linebuf = linebuf(pl->handle);
	if (r < pl->width) {
		memset(pl->linebuf+r, 0, pl->width-r);
		r = pl->width;
	}
    return r;
}

/* Write one line to image plane.
*/
int putline(struct _plane *pl)
{
    int r;

    r = p_putline(pl->handle, pl->width);
    pl->linebuf = linebuf(pl->handle);
    return r;
}

/* Seek to line in plane.
*/
int seekline(struct _plane *pl, int line)
{
	int r;

	r = p_seek(pl->handle, (long)line * (long)pl->width);
	pl->linebuf = linebuf(pl->handle);
	return r;
}

/* This is used at the beginning of most transformations.  NEW buffer is
** marked OLD, OLD buffer is thrown out, and new NEW buffer is made to
** agree with new OLD buffer is size, depth, and flags.  If there is a
** point transformation pending, return an error value so that the calling
** function can abort.	 Mark the histogram as invalid.
*/
int begintransform()
{
    struct _imgbuf *temp;
	char *tc;

	if (transpend) return 8;
	histvalid = 0;
	temp = old; old = new; new = temp;
	tc = old->bufname; old->bufname = new->bufname; new->bufname = tc;
	new->flags = old->flags;
	new->planes = old->planes;
	new->width = old->width;
	new->height = old->height;
	return 0;
}
