/* PLGIF.C : GIF file encoding and decoding for PICLAB.  This code was
** combined from two different programs, one for encoding and one for
** decoding.  Although I wrote both, I did so at different times and the
** code shows signs of inconsistency.  I hope that doesn't cause too much
** confusion (LZW compression is confusing enough).
*/

#include <stdlib.h>
#include <string.h>
#include <memory.h>

#include "piclab.h"
#include "hash"

#define MAXBITS         12
#define LZSTRINGSIZE    512
#define TBLSIZE         (1<<(MAXBITS+1))
#define TBL(x)          ((x)&(TBLSIZE-1))
/* #define hash1(p,l)      TBL(crctable[l^LOBYTE(p)]^HIBYTE(p)) */
#define hash1(p,l)      (U16)TBL(((U32)p+l) * 41000L)
#define hash2(p,l)      primetable[(l^p)&0xFF]

typedef U8 (*CMAP)[256];
static CMAP lcmap, cmap;

static struct _plane *ip[3], *op[3];

static int gfile,   /* GIF file handle.                             */
    ifile[4];       /* De-interlacing buffer files.                 */
static U8 *imgline, /* Buffer into which image data is decoded.     */
    **scnbuf[3],    /* Screen buffer for multiimage/map files.      */
    *ilp, *ilend;   /* Pointers to start and end of imgline.        */

static U16 ileft, itop, /* X and Y offsets from GIF image header.   */
    swidth, sheight,    /* Screen H&W from screen header.           */
    iwidth, iheight,    /* Image H&W from image header.             */
    gcolors, lcolors,   /* Global, local, and current number of     */
    colors, crmask,     /*   colors; crez mask.                     */
    imagex, imagey,     /* Working variables for decoding.          */
    scnx, scny, nexty,  /*   "                                      */
    intpass, intinc,    /* Interlace pass and current increment.    */
    clearcode, eofcode, /* LZW working variables.                   */
    nextcode, nextlimit, csmask,
    totallines,         /* Total lines loaded from file for trace.  */
    *code;              /* LZW code table 'code' field.             */

static U8 bground,      /* Screen background color index.           */
    lost,               /* Information lost flag.                   */
    gcmapf, gcrez,      /* Various values and flags from the GIF    */
    gcbits, gsortf,     /*   screen and image headers.  Those which */
    lcmapf,             /*   begin with 'g' are from the GIF screen */
    lcbits, lintf,      /*   header; those beginning with 'l' are   */
    lsortf, cbits,      /*   from the image header; others are from */
    delay, transf,      /*   extension blocks.                      */
    transparent, clear,
    gflags, gaspect,
    lflags, pbits,
    ncmaps, nimages,    /* Number of color maps and images found.   */
    overflow,           /* Used to mark end of image.               */
    initialcodesize,    /* LZW working variables.                   */
    codesize,
    delayedclear = 0,   /*   Wait before sending clear code?        */
    eofflag,
    *block, startpack = 0,
    *suffix;            /* LZW code table 'suffix' field.           */

static int fullbits,    /* Working variables used for deblocking.   */
    bcount, bptr, blen;

static S16 *prefix, cp; /* LZW code table 'prefix' field, and       */
                        /*   'current' prefix working variable.     */

U32 codebuf = 0L,       /* LZW working variables.                   */
    ratio = 0L, inbits = 0L, outbits = 0L;

static char ifn[] = "INTPASS0";
CONST int intstart[] = { 0, 4, 2, 1, 0 },   /* Starting lines and           */
          intincs[] =  { 8, 8, 4, 2, 1 };   /* increments for interlacing.  */

extern int remove(const char *);    /* Not declared in stdlib.h */

static int getgifheader(void);
static int getcode(void);
static int g_putline(void);
static int decompress(void);
static int getimage(void);
static int getextension(void);
static int packline(U8 *);
static void lzw_clear(void);
static int lzw_putcode(U16);
static int endpack(void);
static int putgifheader(void);
static int putimageheader(void);
static int putcomment(char *);

/* This function is called for every line to be output to the GIF file.  The
** caller must set the variable startpack to 1 before calling the first time
** and must call endpack() when done.  Argument is a pointer to an array of
** color indices.  I won't attemp to explain LZW encoding here--this file is
** already the largest in the package without adding pages of comments.
*/
static int packline(U8 *ndx)
{
    int i, r;
    register S16 c;
    static S16 cl, fc;
    static U16 h1, h2;
    static int tablefull;

    if (startpack) {
        startpack = 0;
        inbits = outbits = codebuf = 0L;
        tablefull = fullbits = bcount = 0;
        lzw_clear();
        if ((r = lzw_putcode(clearcode)) != 0) return r;
		cp = 0xFFF;
    }
    for (i=0; i<iwidth; ++i) {
        cl = ndx[i];        inbits += pbits;
        h1 = hash1(cp, cl); h2 = hash2(cp, cl);

        /* NOTE: The following loop is guaranteed to terminate only if
        ** h2 is relatively prime to the hash table size, and there is
        ** at least one zero entry in the hash table.  These conditions
        ** are met by the program as it is; be cautious about changing
        ** hash table stuff.
        */
		fc = 0xFFF;
		if (cp == 0xFFF) {	   /* No prefix; roots are in table implicitly */
            fc = cp = cl;
        } else while (c = code[h1]) {
            if (cp == prefix[h1] && cl == suffix[h1]) {
                fc = cp = c; break;     /* Code found in table */
            }
            h1 = TBL(h1 + h2);
        }

		if (fc == 0xFFF) {				/* Code not found in table. 	  */
			if ((r = lzw_putcode(cp)) != 0) return r;

            if (tablefull) {            /* This section is only used when */
                if (inbits > 2000L) {   /* delayedclear = 1.              */
                    if (outbits > ratio) {
                        if ((r = lzw_putcode(clearcode)) != 0) return r;
                        lzw_clear();
                        tablefull = 0;
                    }
                    inbits = outbits = 0L;
                }
            } else {
                code[h1] = nextcode;
                prefix[h1] = cp;
                suffix[h1] = (U8)cl;
                if (++nextcode > nextlimit) {
                    if (++codesize > MAXBITS) {
                        --codesize;
                        if ((!delayedclear) || (inbits < outbits)) {
                            if ((r = lzw_putcode(clearcode)) != 0) return r;
                            lzw_clear();
                        } else {
                            tablefull = 1;
                            ratio = (outbits * 2000L) / inbits;
                            ratio = (105L * ratio) / 100L;
                        }
                        outbits = inbits = 0L;
                    }
                    else nextlimit <<= 1;
                }
            }
            cp = cl;
        }
    }
    return 0;
}

/* Clear code tables.  Note that sizeof() cannot be used in the memset()
** functions because these tables are dynamically allocated.
*/
static void lzw_clear()
{
    codesize = initialcodesize+1;
    nextlimit = 1 << codesize;
    clearcode = nextlimit >> 1;
    eofcode = clearcode + 1;
    nextcode = clearcode +2;

    memset(code, 0, 2 * TBLSIZE);
    memset(prefix, 0, 2 * TBLSIZE);
    memset(suffix, 0, TBLSIZE);
}

/* Put one code to the GIF file.  Handles blocking as well.
*/
static int lzw_putcode(register U16 c)
{
    codebuf |= ((U32)c << fullbits);
    fullbits += codesize;
    outbits += codesize;

    while (fullbits >= 8) {
        block[++bcount] = (U8)(codebuf & 0xFF);
        codebuf >>= 8;
        fullbits -= 8;
        if (bcount == 255) {
            block[0] = (U8)bcount;
            if (p_putline(gfile, bcount+1) != bcount+1) return 3;
            bcount = 0;
            block = linebuf(gfile);
        }
    }
    return 0;
}

/* Must be called after packline() has been called for the last line.  Flushes
** any remaining bits and writes the zero-length block.
*/
static int endpack()
{
    int r;

    if (cp >= 0) if ((r = lzw_putcode(cp)) != 0) return r;
    if ((r = lzw_putcode(eofcode)) != 0) return r;

    while (fullbits > 0) {
        block[++bcount] = (U8)(codebuf & 0xFF);
        codebuf >>= 8;
        fullbits -= 8;
        if (bcount == 255) {
            block[0] = (U8)bcount;
            if (p_putline(gfile, bcount+1) != bcount+1) return 3;
            bcount = 0;
            block = linebuf(gfile);
        }
    }
    block[bcount+1] = 0;
    block[0] = (U8)bcount;
    if (p_putline(gfile, bcount+2) != bcount+2) return 3;
    return 0;
}

/* The monstrosity below calculates in which of the 4096 subcubes (8x8x8) of
** the RGB cube a particular set of R, G, and B values lies.  This is used as
** sort of a hash function for RGB values.  Believe it or not, every single
** parenthesis down there (26!) is necessary.
*/
#define IVAL(r,g,b) (((U16)((r)&0xF0)<<4)+(U16)((g)&0xF0)+((U16)((b)&0xF0)>>4))

/* Write GIF screen header and global color map based on global info.  If
** image is color, the new palette is calculated here for dithering.  The
** method currently used to calculate the new palette is rather primitive.
** It works marvelously for small fixed-palette machines but does not take
** full advantage of those more capable.  This will probably be one of the
** first areas of major improvement.
*/
static int putgifheader()
{
    int i, j, cmsize;
    U8 *head, c;

    head = linebuf(gfile);
    strcpy(head, "GIF87a");
    PUT16(head[6], swidth);
    PUT16(head[8], sheight);
    head[10] = gflags;
    head[11] = bground;
    head[12] = gaspect;
    if (p_putline(gfile, 13) != 13) return 3;

    head = linebuf(gfile);
	j = (1 << pbits) - 1;
    cmsize = 3 * (1 << pbits);

	if ((new->flags & 2) == 0) {
		for (i=0; i<=j; ++i) {
			c = (U8)((i * 255L) / j);
			*head++ = c;    *head++ = c;    *head++ = c;
		}
	} else {
		for (i=0; i<=j; ++i) {
			*head++ = gcmap[0][i];
			*head++ = gcmap[1][i];
			*head++ = gcmap[2][i];
		}
	}

    if (p_putline(gfile, cmsize) != cmsize) return 3;
    return 0;
}

/* Write GIF image header based on global info.
*/
static int putimageheader()
{
    U8 *ihead;

    ihead = linebuf(gfile);
    ihead[0] = ',';
    PUT16(ihead[1],ileft);
    PUT16(ihead[3],itop);
    PUT16(ihead[5],iwidth);
    PUT16(ihead[7],iheight);
    ihead[9] = lflags;
    ihead[10] = initialcodesize;
    if (p_putline(gfile, 11) != 11) return 3;
    return 0;
}

/* Put comment into file.  This is currently unused pending its
** approval by CompuServe.
*/
static int putcomment(char *text)
{
    int s, l;
    char *buf;

    buf = linebuf(gfile);
    *buf++ = '!'; *buf++ = 0xFE;
    if (p_putline(gfile, 2) != 2) return 3;

    s = strlen(text);
    while (s > 0) {
        l = (s>255) ? 255 : s;
        *buf++ = (char)l;
        memcpy(buf, text, l);
        if (p_putline(gfile, l+1) != l+1) return 3;
        text += l;
        s -= l;
        buf = linebuf(gfile);
    }
    *buf = 0;
    if (p_putline(gfile, 1) != 1) return 3;
    return 0;
}

/* Read GIF screen header and global color map and set global variables
** appropriately.  If all of the colors in the global color map are grays
** and multimap is not set, then the GIF is assumed grayscale.  This
** function also sets the fields in the NEW buffer to reflect the file.
*/
static int getgifheader()
{
    /* I know the lowercase Ks and uppercase Ms look strange here, but I'm
    ** picky about details like that.  It's correct, take my word for it.
    */
    static char *crtext[]
        = { "8", "64", "512", "4096", "32k", "256k", "2M", "16M" };
    U32 needed, np;
    int i, j, p;
    U8 c, *header, *mp;

    if (p_getline(gfile, 13) != 13) return 4;
    header = linebuf(gfile);
    header[3] = '\0';
    if (strcmp((char *)header, "GIF")) return 6;

    GET16(header[6], swidth);   new->width = swidth;
    GET16(header[8], sheight);  new->height = sheight;
    if (multimap) new->planes = 3; else new->planes = 1;
    new->flags = 0;

    c = header[10];
    gcrez = ((c & 0x70) >> 4) + 1;
    gcbits = (c & 0x07) + 1;
    gcolors = 1 << gcbits;
    gcmapf = (U8)((c & 0x80) == 0x80);
    bground = header[11];
    if ((int)bground >= gcolors) bground = 0;

    if (multiimage) {
        if (multimap) np = 3L; else np = 1L;

        needed = (U32)swidth + np * sheight * sizeof(U8 *) + np * sheight * swidth;
        if (freemem() < needed) return 2;
        for (p=0; p<(int)np; ++p) {
            scnbuf[p] = (U8 **)talloc(sheight * sizeof(U8 *));
            for (i=0; i<sheight; ++i) {
                scnbuf[p][i] = (U8 *)talloc(swidth);
                memset(scnbuf[p][i], (char)bground, swidth);
            }
        }
    }
    imgline = (U8 *)talloc(swidth);
    pl_printf(" Screen: %u x %u, ", swidth, sheight);
    pl_printf("%u of %s colors\r\n", gcolors, crtext[gcrez-1]);

    if (gcmapf) {
        ++ncmaps;
        if (p_getline(gfile, 3 * gcolors) != 3 * gcolors) return 4;
		mp = linebuf(gfile);
		for (i=0; i<gcolors; ++i) {
			gcmap[0][i] = *mp++;
			gcmap[1][i] = *mp++;
			gcmap[2][i] = *mp++;
		}
		if (!multimap) new->flags |= 2;
    } else {
        j = gcolors-1;
        for (i=0; i<=j; ++i) {
			c = (U8)((i * 255L) / j);
            gcmap[0][i] = gcmap[1][i] = gcmap[2][i] = c;
        }
    }
    return 0;
}

static short bit_offset;
static short byte_offset, bits_left;
static U8 code_buffer[260];

int getcode()
{
    static int i;

    bptr = bit_offset >> 3;
    bits_left = bit_offset & 7;

    if (blen - bptr < 3) {
        if ((i = blen - bptr) > 0) {
            *(U32 *)code_buffer = *(U32 *)(code_buffer+bptr);
        }
        bptr = 0;
        bit_offset = bits_left;

        if ((blen = p_getc(gfile)) < 1) eofflag = 1;
        else {
            if (p_getline(gfile, blen) != blen) eofflag = 1;
            else {
                memcpy(code_buffer+i, linebuf(gfile), blen);
                blen += i;
            }
        }
    }
    if (eofflag) return eofcode;
    else {
        bit_offset += codesize;
        return (int)((*(U32*)(code_buffer+bptr)) >> bits_left) & csmask;
    }
}

/* This is called once for every whole line read from the GIF file.  Various
** combinations of multiimage, multimap, and interlace require different
** actions here.  The decompression code is not burdened with this problem.
*/
static int g_putline()
{
    int i, p, np;
    register U16 x;
    U8 *sp, *dp[3];

    if (overflow) return 0;
    np = new->planes;
    sp = imgline;

    if (multiimage) {
        if (multimap) {
            for (p=0; p<3; ++p) dp[p] = scnbuf[p][scny] + ileft;
            for (x=0; x<iwidth; ++x) {
                *dp[0]++ = cmap[0][*sp];
                *dp[1]++ = cmap[1][*sp];
                *dp[2]++ = cmap[2][*sp++];
            }
        } else {
            dp[0] = scnbuf[0][scny] + ileft;
            for (x=0; x<iwidth; ++x) *dp[0]++ = *sp++;
        }
    } else {
        if (lintf) {
            if (intpass < 3) {
                /* On the first three passes, just store the incoming data
                ** into interlace buffer files.
                */
				dp[0] = linebuf(ifile[intpass]);
                memcpy(dp[0], imgline, iwidth);
                p_putline(ifile[intpass], iwidth);
            } else {
                /* On last interlace pass, print the stored even-numbered
                ** line before each odd-numbered line read from the file.
                */
                if (imagey == 1) {
                    for (i=0; i<3; ++i) {
                        if ((ifile[i] = p_reopen(ifile[i], READ)) < 0) return 3;
                    }
                }
                if (((imagey-1)&7) == 0) i = 0;
                else if (((imagey-1)&3) == 0) i = 1;
                else i = 2;
                if (p_getline(ifile[i], iwidth) == 0) return 4;

				memcpy(op[0]->linebuf, linebuf(ifile[i]), iwidth);
                if (putline(op[0]) == 0) return 3;

				memcpy(op[0]->linebuf, imgline, iwidth);
                if (putline(op[0]) == 0) return 3;

                /* It's amazing how much code is used up by handling special
                ** cases.  The following block is executed only when loading
                ** the last line of an interlaced GIF file with an odd number
                ** of lines with multiimage and multimap set to FALSE.  Yes, I
                ** actually had to use this feature once.
                */
                if (imagey == iheight-2) {
                    if (((imagey+1)&7) == 0) i = 0;
                    else if (((imagey+1)&3) == 0) i = 1;
                    else i = 2;
                    if (p_getline(ifile[i], iwidth) == 0) return 4;

					memcpy(op[0]->linebuf, linebuf(ifile[i]), iwidth);
                    if (putline(op[0]) == 0) return 3;
                }
            }
        } else {
			memcpy(op[0]->linebuf, imgline, iwidth);
            if (putline(op[0]) == 0) return 3;
        }
    }

    ilp = imgline;
    if ((imagey += intinc) >= iheight) {
        if (++intpass >= 4) overflow = 1;
        else {
            imagey = intstart[intpass];
            scny = imagey + itop;
            intinc = intincs[intpass];
        }
    } else scny += intinc;
    pl_trace(++totallines);
    return 0;
}

/* This is called immediately after the image header is read, and it
** deblocks and decompresses the whole image, calling g_putline() each time
** a line's worth of data has been read.
*/
static int decompress()
{
    static U8 buf[LZSTRINGSIZE];
    register U8 *bufp;
    register int c;
    int finchar, code, oldcode, maxcode, maxcol, r;

    codebuf = 0L;
    bit_offset = bptr = blen = finchar = oldcode =
    fullbits = eofflag = 0;

    if ((c = p_getc(gfile)) == -1) return 4;
    else {
        initialcodesize = (U8)c;
        codesize = (U8)(c + 1);
    }

    if (codesize < 3 || codesize > 9) return 5;

    maxcol = (1 << cbits)-1;
    clearcode = 1 << (codesize-1);
    eofcode = clearcode + 1;
    nextcode = clearcode + 2;
    maxcode = clearcode << 1;
    csmask = maxcode - 1;
    bufp = buf+sizeof(buf);

    while (1) {
        if ((c = getcode()) == eofcode) break;
        if (c == clearcode) {
            codesize = initialcodesize+1;
            nextcode = clearcode + 2;
            maxcode = 1 << codesize;
            csmask = maxcode - 1;

            while ((c = getcode()) == clearcode)
                ;
            if (c == eofcode) break;
            else if (c >= nextcode) c = 0;

            *ilp++ = (U8)(oldcode = finchar = c);
            if (ilp >= ilend) {
                r = g_putline();
                if (r != 0) return r;
            }
        } else {
            code = c;
            if (code == nextcode) {
                *--bufp = (U8)finchar;
                code = oldcode;
            } else if (code > nextcode) return 5;

            while (code > maxcol) {
                if (bufp <= buf) {
                    while (bufp < buf + sizeof(buf)) {
                        *ilp++ = *bufp++;
                        if (ilp > ilend) {
                            r = g_putline();
                            if (r != 0) return r;
                        }
                    }
                }
                *--bufp = suffix[code];
                code = prefix[code];
            }
            *--bufp = (U8)(finchar = code);
            while (bufp < buf + sizeof(buf)) {
                *ilp++ = *bufp++;
                if (ilp >= ilend) {
                    r = g_putline();
                    if (r != 0) return r;
                }
            }

            if (nextcode < maxcode) {
                prefix[nextcode] = oldcode;
                suffix[nextcode] = (U8)(finchar = code);
                oldcode = c;
            }

            if (++nextcode >= maxcode) {
                if (++codesize > MAXBITS) {
                    --codesize;
                    --nextcode;
                } else {
                    maxcode <<= 1;
                    csmask = maxcode - 1;
                }
            }
        }
    }
	pl_printf("\r\n");
    return 0;
}

/* Get GIF image header and set appropriate global variables, allocate
** interlace buffer if necessary, call decompress().
*/
static int getimage()
{
    U32 mem;
    int i, r;
    char *path;
    U8 c, *iheader, *buf, *mp;

    if (p_getline(gfile, 9) != 9) return 4;
    iheader = linebuf(gfile);

    GET16(iheader[0], ileft);
    GET16(iheader[2], itop);
    GET16(iheader[4], iwidth);
    GET16(iheader[6], iheight);
    c = iheader[8];

    if (!multiimage) {
        new->width = iwidth;
        new->height = iheight;
    }
    lintf = (U8)((c & 0x40) == 0x40);
    if (lcmapf = (U8)((c & 0x80) == 0x80)) {
        if (++ncmaps == 2 && !multimap) {
            pl_printf("Multiple color maps found in file with MULTIMAP set to FALSE.\r\n");
            pl_printf("Colors in graphic will likely be incorrect.\r\n");
        }

        if (lcmap == NULL) lcmap = (CMAP)talloc(768);
        if (lcmap == NULL) return 2;
        lcbits = (c & 0x07) + 1;
        lcolors = 1 << lcbits;

        if (p_getline(gfile, 3 * lcolors) == 0) return 4;
		mp = linebuf(gfile);
		for (i=0; i<lcolors; ++i) {
			lcmap[0][i] = *mp++;
			lcmap[1][i] = *mp++;
			lcmap[2][i] = *mp++;
		}
        cmap = lcmap;
        colors = lcolors;
        cbits = lcbits;
    } else {
        cmap = gcmap;
        colors = gcolors;
        cbits = gcbits;
    }
    pl_printf("  Image: %d x %d at (%d,%d)\r\n", iwidth, iheight, ileft, itop);

    imagex = imagey = overflow = 0;
    scny = itop;
    ilp = imgline; ilend = ilp + iwidth;
    if (lintf) intpass = 0; else intpass = 4;
    intinc = intincs[intpass];

    if (lintf && !multiimage) {
        mem = mark();
        for (i=0; i<3; ++i) {
            if ((buf = (U8 *)talloc(BUFSIZE+iwidth)) == NULL) return 2;
            ifn[strlen(ifn)-1] = (char)(i+'0');
            path = makepath(tempdir, ifn, "TMP");
            if ((ifile[i] = p_open(path, buf, BUFSIZE, iwidth, WRITE)) < 0) return 3;
        }
    }
    r = decompress();
    if (lintf && !multiimage) {
        for (i=0; i<3; ++i) {
            p_close(ifile[i]);
            ifn[strlen(ifn)-1] = (char)(i+'0');
            path = makepath(tempdir, ifn, "TMP");
            if (remove(path) < 0) pl_printf("Could not remove %s\r\n", path);
        }
        release(mem);
    }
    return r;
}

/* Get GIF extension block.  If comment, print it; otherwise, ignore it.
*/
static int getextension()
{
    int fcode;

    if ((fcode = p_getc(gfile)) == -1) return 4;
    else while ((blen = p_getc(gfile)) > 0) {
        if (p_getline(gfile, blen) != blen) return 4;
        block = linebuf(gfile);
        if (fcode == 0xFE) {
            block[blen] = '\0';
            pl_printf("Comment: %s\r\n", (char *)block);
        } else {
            pl_printf("Extension block skipped, code %d\r\n", fcode);
        }
    }
    return 0;
}

/* This is the main entry point for the GLOAD command.	Open the file,
** allocate the LZW code tables, and call the above function to do all the
** work.
*/
int loadgif(int ac, argument *av)
{
    char *path;
    U8 *gbuf, *sp[3], *dp[3];
    int i, c, np, lines, x, r;

    if (ac < 2) {
        pl_printf("Syntax is [G]LOAD <filename>\r\n");
        return 0;
    }
    if (ac > 2) pl_warn(1);

    code = (U16 *)talloc(2 * TBLSIZE);      /* Allocate LZW code tables */
    prefix = (S16 *)talloc(2 * TBLSIZE);
    suffix = (U8 *)talloc(TBLSIZE);
    if ((gbuf = (U8 *)talloc(BUFSIZE + 768)) == NULL) return 2;
    path = makepath(picdir, av[1].cval, "GIF");
    if ((gfile = p_open(path, gbuf, BUFSIZE, 768, READ)) < 0) return 10;

    pl_printf(reading, path);
    if ((r = begintransform()) != 0) return r;
    if ((r = getgifheader()) != 0) return r;

    np = new->planes;		/* Was set by getgifheader() */
    for (i=0; i<np; ++i) {
        if ((op[i] = openplane(i, new, WRITE)) == NULL) return 3;
    }

    totallines = nimages = ncmaps = 0;
    while ((c = p_getc(gfile)) != ';') {
        if (c == ',') {
            if (++nimages > 1 && !multiimage) {
                pl_printf("Multiple images found with MULTIIMAGE set to FALSE.\r\n");
                pl_printf("Remainder of file will be skipped.\r\n");
                break;
            }
            if ((r = getimage()) != 0) return r;
        } else if (c == '!') {
            if ((r = getextension()) != 0) return r;
        } else if (c == -1) {
            pl_printf("\rGIF terminator not found.\r\n");
            break;
        }
    }
    pl_printf(crlf);
    p_close(gfile);

    /* If in multiimage mode, the GIF file has been read into scnbuf[].  We
    ** now write it out to the edit buffer.
    */
    if (multiimage) {
        for (lines=0; lines<sheight; ++lines) {
            for (i=0; i<np; ++i) dp[i] = op[i]->linebuf;
            for (i=0; i<np; ++i) sp[i] = scnbuf[i][lines];
			for (i=0; i<np; ++i) {
				for (x=0; x<swidth; ++x) *dp[i]++ = *sp[i]++;
			}
            for (i=0; i<np; ++i) if (putline(op[i]) == 0) return 3;
        }
    }
    for (i=0; i<np; ++i) closeplane(op[i]);
    pl_printf(done);
    return 0;
}

/* Main entry point for the GSAVE command.	If image is in grayscale, then
** we just make a gray color color map and write out the data.  If it is a
** color image, we must dither.
*/
int savegif(int ac, argument *av)
{
    U32 needed;
    char *path, *gbuf, *ibuf;
    int i, r, lines;
    U8 *indices, *rgb[3], *sp, *dp;

    itop = ileft = lintf = lost = 0;
    if (transpend) return 8;
    if (new->planes == 0) return 7;
    if ((new->flags & 1) == 1) {
        pl_printf("Can only save top-down raster to GIF file.  Issue\r\n");
        pl_printf("REVERSE command before continuing.\r\n");
        return 0;
    }
	if (new->planes != 1) {
		pl_printf("Can only save mono or mapped image ro GIF file.  Issue\r\n");
		pl_printf("MAP or GRAY command before continuing.\r\n");
		return 0;
	}
    iwidth = swidth = new->width;
    iheight = sheight = new->height;

    if (ac < 2) {
        pl_printf("Syntax is [G]SAVE <filename> [INTERLACE].\r\n");
        return 0;
    }
    if (ac == 3 && *av[2].cval == 'I') {
        lintf = 1;
        for (i=0; i<4; ++i) {
            if ((ibuf = (U8 *)talloc(BUFSIZE+iwidth)) == NULL) return 2;
            ifn[strlen(ifn)-1] = (char)(i+'0');
            path = makepath(tempdir, ifn, "TMP");
            if ((ifile[i] = p_open(path, ibuf, BUFSIZE, iwidth, WRITE)) < 0) return 3;
        }
    } else if (ac > 3) pl_warn(1);

    needed = BUFSIZE + 768 + (5L * TBLSIZE) + new->width +
	  (long)BUFSIZE + new->width;
    if (freemem() < needed) return 2;

    code = (U16 *)talloc(2 * TBLSIZE);      /* Allocate LZW code tables */
    prefix = (S16 *)talloc(2 * TBLSIZE);
    suffix = (U8 *)talloc(TBLSIZE);
    gbuf = talloc(BUFSIZE + 768);

    path = makepath(picdir, av[1].cval, "GIF");
    pl_printf(writing, path);
    if ((gfile = p_open(path, gbuf, BUFSIZE, 768, WRITE)) < 0) return 3;

	if (palette > 128) pbits = 8;
	else if (palette > 64) pbits = 7;
	else if (palette > 32) pbits = 6;
	else if (palette > 16) pbits = 5;
	else if (palette > 8) pbits = 4;
	else if (palette > 4) pbits = 3;
	else if (palette > 2) pbits = 2;
	else pbits = 1;

    gflags = (U8)(0x80 + ((crez-1)<<4) + (pbits-1));
    if (gaspect != 0) gaspect -= 31;
    lflags = (U8)((lintf ? 0x40 : 0) + (pbits-1));

    if ((r = putgifheader()) != 0) return r;
    initialcodesize = (U8)((pbits < 2) ? 2 : pbits);
    if ((r = putimageheader()) != 0) return r;

    startpack = 1;
    block = linebuf(gfile);
    if ((indices = (U8 *)talloc(iwidth)) == NULL) return 2;
    pl_printf(lintf ? "Interlacing...\r\n" : "Encoding...\r\n");

    if ((ip[0] = openplane(0, new, READ)) == NULL) return 3;
    for (lines=0; lines < new->height; ++lines) {
        if (getline(ip[0]) == 0) return 4;
        if (((new->flags & 2) == 0) && crez < 8) {
            sp = ip[0]->linebuf;
            dp = indices;
            for (i=0; i<iwidth; ++i) *dp++ = (*sp++ >> (8-crez));
            rgb[0] = indices;
        } else rgb[0] = ip[0]->linebuf;

        if (lintf) {
            if ((lines&7)==0) i = 0;
            else if ((lines&3)==0) i = 1;
            else if ((lines&1)==0) i = 2;
            else i = 3;
            memcpy(linebuf(ifile[i]), rgb[0], iwidth);
            if (p_putline(ifile[i], iwidth) == 0) return 3;
        } else if ((r = packline(rgb[0])) != 0) return r;
        pl_trace(lines);
    }
	closeplane(ip[0]);
    if (lintf) {
        pl_printf("\rEncoding...\r\n");
        for (i=0; i<4; ++i) {
            if ((ifile[i] = p_reopen(ifile[i], READ)) < 0) return 3;
        }
        imagey = intpass = 0;
        intinc = intincs[intpass];
        for (lines=0; lines<iheight; ++lines) {
            if (p_getline(ifile[intpass], iwidth) == 0) return 4;
            if ((r = packline(linebuf(ifile[intpass]))) != 0) return r;
            pl_trace(lines);
            if ((imagey += intinc) >= iheight) {
                if (++intpass >= 4) break;
                else {
                    imagey = intstart[intpass];
                    intinc = intincs[intpass];
                }
            }
        }
        for (i=0; i<4; ++i) {
            p_close(ifile[i]);
            ifn[strlen(ifn)-1] = (char)(i+'0');
            path = makepath(tempdir, ifn, "TMP");
            if (remove(path) < 0) pl_printf("\rCould not remove %s\r\n", path);
        }
    }
    if ((r = endpack()) != 0) return r;
    *linebuf(gfile) = ';';
    if (p_putline(gfile, 1) != 1) return 3;
    p_close(gfile);

    pl_printf(done);
    if (lost) pl_warn(3);
    return 0;
}
