/* PLMAP.C : Palette map and unmap functions.  The primary functions
** here are makepal() which performs color space reduction and map()
** which performs the mapping.
*/

#include <stdlib.h>
#include <memory.h>
#include <string.h>
#include <float.h>

#include "piclab.h"
#include "sierra"

extern U8 _nearest(int *);

/* Mapping input colors to the selected palette is currently the slow
** part--several algorithms are in the works here.  The one used in
** the public Piclab 1.50 is the one currently set to compile.
*/

static U8 *rgbmap, *rp, *disc, *part, *pp, *mp;
static U16 *precalc, masks[] = {
    0x8000, 0x4000, 0x2000, 0x1000, 0x0800, 0x0400, 0x0200, 0x0100,
    0x0080, 0x0040, 0x0020, 0x0010, 0x0008, 0x0004, 0x0002, 0x0001
};

#if 0
static U8 nearest(int *cv)
{
	int i, n;
	long dist, ldist, rl, gl, bl, rd, gd, bd;

	rl = (cv[0] & 0xF8) + 4;
	gl = (cv[1] & 0xF8) + 4;
	bl = (cv[2] & 0xF8) + 4;

	ldist = 0x7FFFFFFFL;
	n = 0;

	for (i=0; i<palette; ++i) {
		rd = (long)gcmap[0][i] - rl;
		gd = (long)gcmap[1][i] - gl;
		bd = (long)gcmap[2][i] - bl;

		dist = rd * rd + gd * gd + bd * bd;
		if (dist < ldist) {
			n = i;
			ldist = dist;
		}
	}
	return (U8)n;
}
#endif

void doline(U16 pwidth, U8 *rgb[3], U8 *indices)
{
    static S16 *this[3], *next[3], *next2[3], *temp[3];
    static U8 *up, c;
    static int p, x, c1[3], e[3], e1[3], e2[3], r, g, b;
    int w2, t3, t4, t5;
    register int i, d;
    S8 *sp;

    if (dither == 0) {
        for (x=0; x < new->width; ++x) {
            c1[0] = ((int)*rgb[0]++); r = c1[0] >> 3;
            c1[1] = ((int)*rgb[1]++); g = c1[1] >> 3;
            c1[2] = ((int)*rgb[2]++); b = c1[2] >> 3;
            i = (((r<<5)+g)<<5)+b;

            if ((precalc[i>>4] & masks[i&15]) == 0) {
                precalc[i>>4] |= masks[i&15];
                rgbmap[i] = _nearest(c1);
            }
			*indices++ = rgbmap[i];
        }
    } else {
        if (startdither) {          /* Here for the first time, intialize   */
            startdither = 0;
            w2 = 2 * pwidth + 8;
            for (p=0; p<3; ++p) {
                this[p] = (S16 *)talloc(w2) + 2;    memset(this[p]-2, 0, w2);
                next[p] = (S16 *)talloc(w2) + 2;    memset(next[p]-2, 0, w2);
                next2[p] = (S16 *)talloc(w2) + 2;   memset(next2[p]-2, 0, w2);
            }
            memset(indices, 0, pwidth);
        }

        e2[0] = e2[1] = e2[2] = 0;
        up = indices;
        for (p=0; p<3; ++p) {
            next[p][0] = next[p][1] = next2[p][0] = next2[p][1] = 0;
        }
        for (x=0; x<pwidth; ++x) {
            for (p=0; p<3; ++p) {
                c1[p] = rgb[p][x] + this[p][x] + e1[p];
                if (c1[p] < 0) c1[p] = 0;
                if (c1[p] > 255) c1[p] = 255;
            }
            r = c1[0] >> 3;
            g = c1[1] >> 3;
            b = c1[2] >> 3;
			i = (((r<<5)+g)<<5)+b;

            if ((precalc[i>>4] & masks[i&15]) == 0) {
                precalc[i>>4] |= masks[i&15];
                rgbmap[i] = _nearest(c1);
            }
			*up++ = c = rgbmap[i];

            e[0] = c1[0] - gcmap[0][c];
            e[1] = c1[1] - gcmap[1][c];
            e[2] = c1[2] - gcmap[2][c];

            for (p=0; p<3; ++p) if (e[p] != 0) {
                d = e[p]+255;   sp = sierra[d];
                t3 = *sp++;     t4 = *sp++;     t5 = *sp++;
                e1[p] = e2[p] + *sp++;
                e2[p] = t4;
                next[p][x-2] += t5;
                next[p][x-1] += t3;     next2[p][x-1] += t5;
                next[p][x] += *sp;      next2[p][x] += t4;
                next[p][x+1] += t3;     next2[p][x+1] = t5;
                next[p][x+2] = t5;
            }
        }
        for (p=0; p<3; ++p) {
            temp[p] = this[p]; this[p] = next[p];
            next[p] = next2[p]; next2[p] = temp[p];
        }
    }
}

int map(int ac, argument *av)
{
    U8 *sp[3], *dp;
    int p, r, lines;
    U32 mem, needed;
    struct _plane *ip[3], *op;

    mem = mark();
    needed = 32768L + 4L * BUFSIZE + 6L * (long)new->width + 4120L;
    if (needed > freemem()) return 2;

    if ((rgbmap = (U8 *)talloc(32768)) == NULL) return 2;
    if ((precalc = (U16 *)talloc(4096)) == NULL) return 2;
    memset(precalc, 0, 4096);

    pl_printf("Mapping image...\r\n");
    if ((r = begintransform()) != 0) return r;
    new->planes = 1;
    new->flags |= 2;

    for (p=0; p<3; ++p) if ((ip[p] = openplane(p, old, READ)) == NULL) return 3;
    if ((op = openplane(0, new, WRITE)) == NULL) return 3;

    startdither = 1;
    for (lines=0; lines < new->height; ++lines) {
        for (p=0; p<3; ++p) {
            if (getline(ip[p]) == 0) return 4;
            sp[p] = ip[p]->linebuf;
        }
        dp = op->linebuf;
        doline(new->width, sp, dp);
        if (putline(op) == 0) return 3;
        pl_trace(lines);
    }
    for (p=0; p<3; ++p) closeplane(ip[p]);
    closeplane(op);

    pl_printf(done);
    release(mem);
    return 0;
}

int unmap(int ac, argument *av)
{
    
    int r, i;
    U8 *rp, *gp, *bp, *sp;
    U16 lines, x;
    struct _plane *ip, *op[3];

    if ((new->flags & 2) == 0) {
        pl_printf("Image is not mapped.\r\n");
        return 0;
    }
    if (ac > 1) pl_warn(1);

    if ((r = begintransform()) != 0) return r;
    new->planes = 3;
    new->flags &= ~2;

    if ((ip = openplane(0, old, READ)) == NULL) return 3;
    for (i=0; i<3; ++i) if ((op[i] = openplane(i, new, WRITE)) == NULL) return 3;

    pl_printf(working);
    for (lines=0; lines < new->height; ++lines) {
        if (getline(ip) == 0) return 4;
        rp = op[0]->linebuf;
        gp = op[1]->linebuf;
        bp = op[2]->linebuf;
        sp = ip->linebuf;
        for (x=0; x < new->width; ++x) {
            *rp++ = gcmap[0][*sp];
            *gp++ = gcmap[1][*sp];
            *bp++ = gcmap[2][*sp++];
        }
        for (i=0; i<3; ++i) if (putline(op[i]) == 0) return 3;
        pl_trace(lines);
    }
    closeplane(ip);
    for (i=0; i<3; ++i) closeplane(op[i]);
    pl_printf(done);
    return 0;
}

int loadpal(int ac, argument *av)
{
    char *path, *buf;
    U32 mem;
    int h, i, j, c, v, offset, count;

    if (ac < 2) {
        pl_printf("Syntax is PLOAD <filename> <offset> <count>\r\n");
        return 0;
    } else {
        path = makepath(picdir, av[1].cval, "MAP");
        if (ac > 2) offset = (int)av[2].fval; else offset = 0;
        if (ac > 3) count = (int)av[3].fval; else count = palette;
        if (offset + count > 256) count = 256 - offset;
        if (ac > 4) pl_warn(1);
    }

    mem = mark();
    if ((buf = talloc(BUFSIZE+80)) == NULL) return 2;
    if ((h = p_open(path, buf, BUFSIZE, 80, READ)) < 0) return 3;

    pl_printf(reading, path);

    for (i = offset; i < offset+count; ++i) {
        for (j=0; j<3; ++j) {
            v = 0;
            do {
                if ((c = p_getc(h)) == -1) goto lpexit;
            } while (c < '0' || c > '9');
            do {
                v *= 10;
                v += (c - '0');
                if ((c = p_getc(h)) == -1) goto lpexit;
            } while (c >= '0' && c <= '9');
            gcmap[j][i] = (U8)v;
        }
        while (c != '\n') if ((c = p_getc(h)) == -1) goto lpexit;
    }
lpexit:
    p_close(h);
    release(mem);
    return 0;
}

int savepal(int ac, argument *av)
{
    char *path, *buf, line[80];
    U32 mem;
    int h, i, offset, count;

    if (ac < 2) {
        pl_printf("Syntax is PSAVE <filename> <offset> <count>\r\n");
        return 0;
    } else {
        path = makepath(picdir, av[1].cval, "MAP");
        if (ac > 2) offset = (int)av[2].fval; else offset = 0;
        if (ac > 3) count = (int)av[3].fval; else count = palette;
        if (offset + count > 256) count = 256 - offset;
        if (ac > 4) pl_warn(1);
    }

    mem = mark();
    if ((buf = talloc(BUFSIZE+80)) == NULL) return 2;
    if ((h = p_open(path, buf, BUFSIZE, 80, WRITE)) < 0) return 3;

    pl_printf(writing, path);
    for (i = offset; i < offset + count; ++i) {
        pl_sprintf(line, "%d %d %d\r\n", gcmap[0][i], gcmap[1][i], gcmap[2][i]);
        strcpy(linebuf(h), line);
        p_putline(h, strlen(line));
    }
    p_close(h);
    release(mem);
    return 0;
}

int egapal(int ac, argument *av)
{
    U32 mem, freq, *hist;
    int i, j, c, p, r, g, b;
    U8 *lp[3];
    U16 lines, x;
    struct _plane *ip[3];

    if (new->planes == 1) {
        pl_printf("Can only make palette for full-color image.\r\n");
        return 0;
    }
    mem = mark();
    hist = (U32 *)talloc(256);
    memset(hist, 0, 256);

    for (p=0; p<3; ++p) if ((ip[p] = openplane(p, new, READ)) == NULL) return 3;
    pl_printf("Building histogram...\r\n");

    for (lines=0; lines < new->height; ++lines) {
        for (p=0; p<3; ++p) {
            if (getline(ip[p]) == 0) return 4;
            else lp[p] = ip[p]->linebuf;
        }
        for (x=0; x < new->width; ++x) {
            r = (((int)*lp[0]++ + 32) >> 6);
            g = (((int)*lp[1]++ + 32) >> 6);
            b = (((int)*lp[2]++ + 32) >> 6);
            ++hist[(((r<<2)+g)<<2)+b];
        }
        pl_trace(lines);
    }
    for (p=0; p<3; ++p) closeplane(ip[p]);

	palette = 16;
	crez = 2;
	gcmap[0][0] = gcmap[1][0] = gcmap[2][0] = 0;
	gcmap[0][1] = gcmap[1][1] = gcmap[2][1] = 0xFF;

	for (i=2; i<16; ++i) {
		freq = 0L;
		c = 0;
		for (j=1; j<63; ++j) if (freq < hist[j]) {
			freq = hist[j];
			c = j;
		}
		hist[c] = 0L;
		gcmap[0][i] = (U8)(((c & 0x30) >> 4) * 0x55);
		gcmap[1][i] = (U8)(((c & 0x0C) >> 2) * 0x55);
		gcmap[2][i] = (U8)((c & 0x03) * 0x55);
	}

    pl_printf(done);
    release(mem);
    return 0;
}

/**************************************
**
** Here begins the Wan, Wong, and Prusinkiewicz color reduction.
*/

typedef struct {
	double variance, mean[3];
    U32 weight;
    U16 freq[3][32];
    U8  low[3], high[3];
} BOX;

static BOX *box;
static U16 *hist1;
static U32 npixels;

void boxstats(BOX *thisbox)
{
    int i, p;
    U16 *freq;
    double mean, var, tvar, fm, f;

    thisbox->variance = tvar = 0.0;
    if (thisbox->weight == 0L) return;

    for (p=0; p<3; ++p) {
        var = mean = 0.0;
        i = thisbox->low[p];
        freq = &thisbox->freq[p][i];
        for (; i < thisbox->high[p]; ++i, ++freq) {
			f = (double)(*freq);
            mean += i * f;
            var += i*i * f;
        }
		fm = mean / (double)thisbox->weight;
        thisbox->mean[p] = fm;
        tvar += var - fm * fm * (double)thisbox->weight;
    }
    thisbox->variance = tvar;
}

void updatefreq(BOX *box1, BOX *box2)
{
    U16 myfreq, *h, roff;
    int i, r, g, b;

    for (i=0; i<3; ++i) memset(box1->freq[i], 0, 64);

    for (r = box1->low[0]; r < box1->high[0]; ++r) {
        roff = r << 5;
        for (g = box1->low[1]; g < box1->high[1]; ++g) {
            b = box1->low[2];
            h = hist1 + (((roff | g) << 5) | b);
            for (; b < box1->high[2]; ++b) {
                if ((myfreq = *h++) == 0) continue;
                box1->freq[0][r] += myfreq;
                box1->freq[1][g] += myfreq;
                box1->freq[2][b] += myfreq;
                box2->freq[0][r] -= myfreq;
                box2->freq[1][g] -= myfreq;
                box2->freq[2][b] -= myfreq;
            }
        }
    }
}

void findcutpoint(BOX *thisbox, int p, BOX *new1, BOX *new2)
{
    double u, v, max, iw, t;
    int i, maxindex, minindex, cutpoint;
    U32 optweight, curweight, f;

    if ((thisbox->high[p] - thisbox->low[p]) <= 1) {
        new1->variance = FLT_MAX;
        return;
    }

    minindex = (U16)((thisbox->low[p] + thisbox->mean[p]) * 0.5);
    maxindex = (U16)((thisbox->high[p] + thisbox->mean[p]) * 0.5);

    cutpoint = minindex;
    optweight = thisbox->weight;
    iw = 1.0 / (double)thisbox->weight;

    curweight = 0L;
    for (i=thisbox->low[p]; i<minindex; ++i) curweight += thisbox->freq[p][i];
    u = 0.0;
    max = -1.0;

    for (i = minindex; i <= maxindex; ++i) {
        curweight += (f = thisbox->freq[p][i]);
        if (curweight == thisbox->weight) break;

        u += (double)(i * f) * iw;
        t = thisbox->mean[p] - u;
        v = ((double)curweight / (double)(thisbox->weight - curweight)) * t * t;
        if (v > max) {
            max = v;
            cutpoint = i;
            optweight = curweight;
        }
    }
    ++cutpoint;
    *new1 = *new2 = *thisbox;
    new1->weight = optweight;
    new2->weight -= optweight;
    new1->high[p] = (U8)cutpoint;
    new2->low[p] = (U8)cutpoint;
    updatefreq(new1, new2);
    boxstats(new1);
    boxstats(new2);
}

int cutbox(BOX *thisbox, BOX *newbox)
{
    int i, m;
    double totalvar[3];

    if (thisbox->variance == 0.0 || thisbox->weight == 0L) return 0;

    m = 0;
    for (i=0; i<3; ++i) {
        findcutpoint(thisbox, i, &box[256+2*i], &box[256+2*i+1]);
        totalvar[i] = box[256+2*i].variance;
        if (totalvar[i] != FLT_MAX) totalvar[i] += box[256+2*i+1].variance;
        else ++m;
    }
    if (m == 3) return 0;

    if (totalvar[0] <= totalvar[1]) {
        if (totalvar[0] <= totalvar[2]) i = 0;
        else i = 2;
    } else {
        if (totalvar[1] <= totalvar[2]) i = 1;
        else i = 2;
    }

    *thisbox = box[256+2*i];
    *newbox = box[256+2*i+1];
    return 1;
}

int greatestvar(int nboxes)
{
    int i, gv;
    double max;

    max = 0.0;
    gv = -1;
    for (i=0; i<nboxes; ++i) {
        if (box[i].variance >= max) {
            max = box[i].variance;
            gv = i;
        }
    }
	if (max == 0.0) return -1;
    else return gv;
}

int makepal(int ac, argument *av)
{
    register U16 *hp, *rf, *gf, *bf;
    U32 mem, needed, total;
    int i, p, r, g, b, curbox;
    U8 *lp[3];
    U16 lines, x;
    struct _plane *ip[3];

    if (new->planes == 1) {
        pl_printf("Can only make palette for full-color image.\r\n");
        return 0;
    }
    mem = mark();
    needed = (3L * BUFSIZE) + 65536L + (262L * sizeof(BOX));
    if (needed > freemem()) return 2;

    box = (BOX *)talloc(262 * sizeof(BOX));
    memset(box, 0, 262 * sizeof(BOX));
    hist1 = (U16 *)talloc(65535);
    memset(hist1, 0, 65535);
    hist1[32767] = 0;
    for (p=0; p<3; ++p) if ((ip[p] = openplane(p, new, READ)) == NULL) return 3;

    pl_printf("Building histogram...\r\n");
    rf = box[0].freq[0];
    gf = box[0].freq[1];
    bf = box[0].freq[2];
    total = npixels = 0L;

    for (lines=0; lines < new->height; ++lines) {
        for (p=0; p<3; ++p) {
            if (getline(ip[p]) == 0) return 4;
            else lp[p] = ip[p]->linebuf;
        }
        for (x=0; x < new->width; ++x) {
            r = (((int)*lp[0]++) >> 3);
            g = (((int)*lp[1]++) >> 3);
            b = (((int)*lp[2]++) >> 3);
            hp = hist1 + ((((U16)(r<<5)|g)<<5)|b);

            if ((*hp < 65535) && (rf[r] < 65535) &&
				(gf[g] < 65535) && (bf[b] < 65535)) {

                ++npixels;
                if (++*hp == 1) ++total;
                ++rf[r];
                ++gf[g];
                ++bf[b];
            }
        }
        pl_trace(lines);
    }
    for (p=0; p<3; ++p) closeplane(ip[p]);

    pl_printf("\r%d Total colors.\r\n", total);
    if (total <= palette) {
        for (r=31; r>=0; --r) {
            for (g=31; g>=0; --g) {
                for (b=31; b>=0; --b) {
                    if (hist1[(((r<<5)+g)<<5)+b]) {
                        gcmap[0][--total] = (U8)((255L * r) / 31L);
                        gcmap[1][total] = (U8)((255L * g) / 31L);
                        gcmap[2][total] = (U8)((255L * b) / 31L);
                    }
                }
            }
        }
    } else {
	    pl_printf("Dividing color space...\r\n");

    	box[0].low[0] = box[0].low[1] = box[0].low[2] = 0;
	    box[0].high[0] = box[0].high[1] = box[0].high[2] = 32;
    	box[0].weight = npixels;
	    boxstats(&box[0]);

    	for (curbox = 1; curbox < palette; ) {
        	if ((g = greatestvar(curbox)) == -1) break;
	        if (cutbox(&box[g], &box[curbox]) == 0) box[g].variance = 0.0;
    	    else {
				++curbox;
				pl_printf("\r%d", curbox);
			}
    	}
		pl_printf("\n");

	    for (i=0; i<curbox; ++i) {
    	    for (p=0; p<3; ++p) {
        	    gcmap[p][i] = (U8)(box[i].mean[p] * (255.0/31.0));
	        }
    	}
	}
    pl_printf(done);

    release(mem);
    return 0;
}
