/* PLAREA.C : Miscellaneous area process commands for PICLAB.  Commands
** here include MEDIAN, SHARPEN, SMOOTH.
**
** The command handlers here just parse their arguments for flags, then
** set the variable process to point to the function that does the work.
** The function doarea() is the shell of all these processes.  It steps
** through the file always keeping a buffer of five lines available to the
** process functions.  It also pads each line with two extra values at the
** start and end so that the process routines never have to think about
** boundary conditions.  None of the processes here currently use more than
** a 3x3 neighborhood, but a 5x5 is accounted for.	Larger neighborhoods
** could be handled by changing doarea() a little, but it didn't seem to be
** worth the effort.
*/

#include <stdlib.h>
#include <memory.h>

#include "piclab.h"

static int sharpness, weighted, smoothness;
static U8 *_line[5], **line, *temp;
static int doarea(void);
static void domedian(U8 *);
static void dosharpen(U8 *);
static void dosmooth(U8 *);
static void (*process)(U8 *);

/* The MEDIAN command replaces each value with the median of the 9 values
** in its immediate neighborhood.  If the first argument begins with a 'W'
** the median calculation is weighted in favor of the original value, i.e.,
** the center value is included three times and the median of the eleven
** values is used.
*/
int median(int ac, argument *av)
{
	if (new->flags & 2) {
		pl_printf(maperr);
		return 0;
	}
	if (ac == 2 && *av[1].cval == 'W') weighted = 1;
	else {
		if (ac != 1) pl_warn(1);
		weighted = 0;
	}
	process = domedian;
	return doarea();
}

/* The SHARPEN command performs a standard LaPlace-filter transformation,
** then adds the result (weighted by the variable sharpness) back to the
** original picture.  The first argument, if any, is interpreted as the
** sharpness value, which is kept accurate to 1/16.  If no argument is
** given, sharpness defaults to 1.
*/
int sharpen(int ac, argument *av)
{
	if (new->flags & 2) {
		pl_printf(maperr);
		return 0;
	}
	if (ac > 1) {
		if (ac > 2) pl_warn(1);
		sharpness = (int)(av[1].fval * 16.0F);
		if (sharpness == 0) {
			pl_printf("Sharpness of 0 specified; this has no effect.\r\n");
			return 0;
		}
	} else sharpness = 16;

	process = dosharpen;
	return doarea();
}

/* The SMOOTH command replaces each value with the mean of the 9 values in
** its neighborhood.  No options.
*/
int smooth(int ac, argument *av)
{
	if (new->flags & 2) {
		pl_printf(maperr);
		return 0;
	}
	if (ac > 1) {
		if (ac > 2) pl_warn(1);
		smoothness = (int)(av[1].fval * 16.0F);
		if (smoothness == 0) {
			pl_printf("Smoothness of 0 specified; this has no effect.\r\n");
			return 0;
		}
	} else smoothness = 16;

	process = dosmooth;
	return doarea();
}

static void domedian(U8 *dp)
{
	register U8 v;
	int i;
	U16 x;
	static U8 v0, v1, v2, v3, v4, v5;

	if (weighted) {
		for (x=0; x < new->width; ++x) {
			v0 = v1 = v2 = line[0][x];

			if ((v = line[-1][x-1]) >= v2) v3 = v;
			else { v3 = v2; v0 = v; }

			if ((v = line[-1][x]) >= v1) {
				if (v >= v3) v4 = v;
				else if (v >= v2) { v4 = v3; v3 = v; }
				else { v4 = v3; v3 = v2; v2 = v; }
			} else {
				v4 = v3; v3 = v2; v2 = v1;
				if (v >= v0) v1 = v;
				else { v1 = v0; v0 = v; }
			}

			if ((v = line[-1][x+1]) >= v2) {
				if (v >= v4) v5 = v;
				else if (v >= v3) { v5 = v4; v4 = v; }
				else { v5 = v4; v4 = v3; v3 = v; }
			} else {
				v5 = v4; v4 = v3;
				if (v >= v1) {
					if (v >= v2) v3 = v;
					else { v3 = v2; v2 = v; }
				} else {
					v3 = v2; v2 = v1;
					if (v >= v0) v1 = v;
					else { v1 = v0; v0 = v; }
				}
			}

			for (i=0; i<5; ++i) {
				if (i==0) v = line[0][x-1];
				else if (i==1) v = line[0][x+1];
				else v = line[1][x+i-3];

				if (v >= v2) {
					if (v >= v4) {
						if (v < v5) v5 = v;
					} else {
						v5 = v4; v4 = v3;
						if (v <= v3) v3 = v; else v4 = v;
					}
				} else {
					v5 = v4; v4 = v3;
					if (v >= v1) {
						if (v >= v2) v3 = v;
						else { v3 = v2; v2 = v; }
					} else {
						v3 = v2; v2 = v1;
						if (v >= v0) v1 = v;
						else { v1 = v0; v0 = v; }
					}
				}
			}
			*dp++ = v5;
		}
	} else {
		for (x=0; x < new->width; ++x) {
			v0 = line[-1][x-1];

			if ((v = line[-1][x]) >= v0) v1 = v;
			else { v1 = v0; v0 = v; }

			if ((v = line[-1][x+1]) >= v1) v2 = v;
			else if (v >= v0) { v2 = v1; v1 = v; }
			else { v2 = v1; v1 = v0; v0 = v; }

			if ((v = line[0][x-1]) >= v1) {
				if (v >= v2) v3 = v;
				else { v3 = v2; v2 = v; }
			} else {
				v3 = v2; v2 = v1;
				if (v >= v0) v1 = v;
				else { v1 = v0; v0 = v; }
			}

			if ((v = line[0][x]) >= v1) {
				if (v >= v3) v4 = v;
				else if (v >= v2) { v4 = v3; v3 = v; }
				else { v4 = v3; v3 = v2; v2 = v; }
			} else {
				v4 = v3; v3 = v2; v2 = v1;
				if (v >= v0) v1 = v;
				else { v1 = v0; v0 = v; }
			}

			for (i=0; i<4; ++i) {
				if (i==0) v = line[0][x+1]; else v = line[1][x+i-2];

				if (v >= v2) {
					if (v >= v4) continue;
					else if (v >= v3) v4 = v;
					else { v4 = v3; v3 = v; }
				} else {
					v4 = v3; v3 = v2;
					if (v >= v1) v2 = v;
					else if (v >= v0) { v2 = v1; v1 = v; }
					else { v2 = v1; v1 = v0; v0 = v; }
				}
			}
			*dp++ = v4;
		}
	}
}

static void dosharpen(U8 *dp)
{
	register int orig, delta;
	U16 x;
	U8 *sp;

	sp = line[0];
	for (x=0; x < new->width; ++x) {
		orig = *sp++;
		delta = (orig << 3) - line[0][x-1] - line[0][x+1];
		delta -= (line[-1][x-1] + line[-1][x] + line[-1][x+1]);
		delta -= (line[1][x-1] + line[1][x] + line[1][x+1]);
		orig += ((delta * sharpness) >> 4);
		if (orig < 0) orig = 0; else if (orig > 255) orig = 255;
		*dp++ = (U8)orig;
	}
}

static void dosmooth(U8 *dp)
{
	register int total, orig;
	U16 x;

	for (x=0; x < new->width; ++x) {
		total = line[-1][x-1] + line[-1][x] + line[-1][x+1] +
				line[0][x-1] + (orig = line[0][x]) + line[0][x+1] +
				line[1][x-1] + line[1][x] + line[1][x+1];
		orig += (int)(((long)(total - 9 * orig) * smoothness) / 144L);
		if (orig < 0) orig = 0; else if (orig > 255) orig = 255;
		*dp++ = (U8)orig;
	}
}

static int doarea()
{
	U32 mem;
	int r, p, j, w;
	U8 *dp;
	U16 lines;
	struct _plane *ip, *op;

	if ((r = begintransform()) != 0) return r;
	if (new->planes == 0) return 7;
	pl_printf(working);
	w = new->width;
	for (j=0; j<5; ++j) {
		_line[j] = (U8 *)talloc(w + 4);
		if (_line[j] == NULL) return 2;
		else _line[j] += 2;
	}
	line = &_line[2];

	mem = mark();
	for (p=0; p < new->planes; ++p) {
		if ((ip = openplane(p, old, READ)) == NULL) return 3;
		if ((op = openplane(p, new, WRITE)) == NULL) return 3;
		if (getline(ip) == 0) return 4;
		memcpy(line[-2], ip->linebuf, w);
		memcpy(line[-1], ip->linebuf, w);
		memcpy(line[0], ip->linebuf, w);
		if (getline(ip) == 0) return 4;
		memcpy(line[1], ip->linebuf, w);
		if (getline(ip) == 0) return 4;
		memcpy(line[2], ip->linebuf, w);

		for (j=-2; j<3; ++j) {
			line[j][-2] = line[j][-1] = line[j][0];
			line[j][w+1] = line[j][w] = line[j][w-1];
		}
		for (lines=0; lines < old->height; ++lines) {
			dp = op->linebuf;
			(*process)(dp);
			if (putline(op) == 0) return 3;

			temp = line[-2]; line[-2] = line[-1]; line[-1] = line[0];
			line[0] = line[1]; line[1] = line[2]; line[2] = temp;

			if (lines+3 < old->height) {
				if (getline(ip) == 0) return 4;
				memcpy(line[2], ip->linebuf, w);
			} else {
				memcpy(line[2], line[1], w);
			}
			line[2][-2] = line[2][-1] = line[2][0];
			line[2][w+1] = line[2][w] = line[2][w-1];

			pl_trace(lines);
		}

		closeplane(ip);
		closeplane(op);
		release(mem);
	}
	pl_printf(done);
	pl_warn(3);
	return 0;
}
