/* PL1.C : Miscellaneous command handlers for PICLAB.  Commands
** include SET, LIST, PAUSE, QUIT, UNDO.
*/

#include <stdlib.h>
#include <string.h>

#include "piclab.h"

static int listvars(void);
static struct _vars *getvar(char *);

/* Given variable name, return pointer to _vars structure or NULL
** if name is not found.
*/
static struct _vars *getvar(char *vname)
{
    int i;

    for (i=0; vartable[i].varname != NULL; ++i) {
		if (strncmp(vname, vartable[i].varname, strlen(vname)) == 0)
		  return &vartable[i];
    }
    return NULL;
}

/* List variables and their current values.
*/
static int listvars()
{
    int i;
    struct _vars *vp;

    for (i=0; vartable[i].varname != NULL; ++i) {
        vp = &vartable[i];
		if (vp->visible == 0) continue;

        pl_printf("%-12.12s= ", vp->varname);
        switch (vp->vartype) {
			case BTYPE: if (*(U16 *)vp->varaddr) pl_printf("TRUE\r\n");
						else pl_printf("FALSE\r\n");				break;
            case UTYPE: pl_printf("%u\r\n", *(U16 *)vp->varaddr);	break;
/*			case FTYPE: pl_printf("%g\r\n", *(float *)vp->varaddr);	break; */
            case STYPE: pl_printf("\"%s\"\r\n", vp->varaddr);		break;
			default:	return 9;
        }
    }
	return 0;
}

/* Procedure for SET command.  If no arguments, list all variables and their
** current values as well as free memory.  With one argument, clear the
** variable it names.  With two, set the variable named by the second argu-
** ment to the value specified by the third.
*/
int setvar(int ac, argument *av)
{
	static char *true[] = { "TRUE", "ON", "YES" };
	int r;
    struct _vars *vp;
	char *cv;
	float fv;

    if (ac == 1) {
		if ((r = listvars()) != 0) return r;
        pl_printf("%-12.12s= %lu\r\n", "FREE MEMORY", freemem());
	} else {
		if (ac == 2) {
			cv = "";
			fv = 0.0F;
		} else {
			cv = av[2].cval;
			fv = av[2].fval;
		}
		if ((vp = getvar(av[1].cval)) == NULL) return 9;
        else switch (vp->vartype) {
			case BTYPE: if (pl_match(cv, true)) fv = 1.0F; else fv = 0.0F;
			case UTYPE: *(U16 *)(vp->varaddr) = (U16)fv;	break;
			case FTYPE: *(float *)(vp->varaddr) = fv;		break;
			case STYPE: strncpy(vp->varaddr, cv, STRSIZE);	break;
			default:	return 1;
        }
    }
	return 0;
}

/* Procedure for LIST command.	With no arguments, show a list of what can
** can be listed.  With one, list the items specified by the argument.
*/
int list(int ac, argument *av)
{
	int i, r;
	struct _imgbuf *ip;
    static char *listitems[] = {
		"VARIABLES", "COMMANDS", "BUFFERS", "PRINTERS",
		"FORMATS", "DISPLAYS", NULL };

    if (ac == 1) {
        pl_printf("Listable items:\r\n");
        for (i=0; listitems[i] != NULL; ++i)
          pl_printf("%s\r\n", listitems[i]);
	} else {
		if (ac > 2) pl_warn(1);
		switch (pl_match(av[1].cval, listitems)) {
			case 1: 	if ((r = listvars()) != 0) return r;	break;
			case 2: 	/* COMMANDS */
				for (i=0; cmdtable[i].command != NULL; ++i) {
					if (i > 0 && i % 6 == 0) pl_printf(crlf);
					pl_printf(" %-12.12s", cmdtable[i].command);
				}
				pl_printf(crlf);
				break;
			case 3: 	/* BUFFERS */
				pl_printf("Buffer      Size           Type     Raster\r\n");
				for (i=0; i<NIMAGES; ++i) {
					ip = &images[i];
					pl_printf("%-8.8s    ", ip->bufname);
					if (images[i].planes == 0) pl_printf("<EMPTY>\r\n");
					else pl_printf("%-4d x %-4d    %-9.9s%s\r\n",
					  ip->width, ip->height,
					  ((ip->flags & 2) ? "Mapped" :
						((ip->planes == 1) ? "Mono" : "Color")),
					  ((ip->flags & 1) ? "Bottom-up" : "Top-down"));
				}
				break;
			case 4: 	/* PRINTERS */
				for (i=0; printers[i].name != NULL; ++i) {
					if (i > 0 && i % 6 == 0) pl_printf(crlf);
					pl_printf(" %-12.12s", printers[i].name);
				}
				pl_printf(crlf);
				break;
			case 5: 	/* FORMATS */
				for (i=0; formats[i].name != NULL; ++i) {
					if (i > 0 && i % 6 == 0) pl_printf(crlf);
					pl_printf(" %-12.12s", formats[i].name);
				}
				pl_printf(crlf);
				break;
			case 6: 	/* DISPLAYS */
				for (i=0; displays[i].name != NULL; ++i) {
					pl_printf("%-10.10s%4dx%4d %s\r\n", displays[i].name,
					  displays[i].width, displays[i].height,
					  displays[i].comment);
				}
				break;
			default:	return 9;
		}
    }
	return 0;
}

/* Procedure for PAUSE command.  If no argument, wait for keypress.  If one,
** wait for specified number of seconds.  This is currently implemented only
** for MSDOS (the function below will not work for OS/2 or any other OS).
** This must be reimplemented on each machine, because there is no portable
** way to poll a keyboard in C.
*/

#ifdef MSDOS
#include <conio.h>
#define NOW (*(volatile unsigned long far *)0x46CL)
int pause(int ac, argument *av)
{
	int d;
	unsigned long then;

	if (ac == 1) {
		getch();
	} else {
		d = (int)(av[1].fval * 18.2);
		then = NOW + d;
		while (then > NOW) {
			if (kbhit()) {
				getch();
				break;
			}
		}
	}
	return 0;
}
#else
int pause(int ac, argument *av)
{
}
#endif

/* Procedure for QUIT command.	If any arguments are given, quit immediately.
** Otherwise quit if no point transform is pending.
*/
int quit(int ac, argument *av)
{
	if (ac == 1 && transpend) {
		pl_printf("Point transformation is pending.  Either save with TRANSFORM\r\n");
		pl_printf("command or cancel with UNDO command before exiting.\r\n");
	} else quitflag = 1;
	return 0;
}

/* Procedure for UNDO command.	Takes no arguments.  If there is a pending
** point transformation, clear it; otherwise, exchange the OLD and NEW
** edit buffers.  Some commands (like SAVE and HISTOGRAM) cause no change
** in the edit buffers; therefore, UNDO after one of these commands will
** reverse the last command which did affect the buffers.
*/
int undo(int ac, argument *av)
{
	int i;
    struct _imgbuf *temp;
	char *tc;

	if (ac != 1) pl_warn(1);
	if (transpend) {
		for (i=0; i<256; ++i) lookup[0][i] = lookup[1][i] = lookup[2][i] = (U8)i;
		transpend = 0;
	} else {
		if (old->planes == 0) return 7;
		temp = old; old = new; new = temp;
		tc = old->bufname; old->bufname = new->bufname; new->bufname = tc;
		histvalid = 0;
	}
	return 0;
}
