/*  asmhdr - process assembly language headers

    usage:  asmhdr document_file *.asm

    Borland    C>tcc -mc asmhdr.c wildargs.obj
    Microsoft  C>cl  /AC asmhdr.c setargv.obj /link /noe

    Compile with compact memory model.

    John Otken, Soft Advances
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define LINE_SIZE       256
#define MAX_ROUTINES    1000    /* maximum routines */
#define READ_TEXT       "rt"
#define WRITE_TEXT      "wt"



struct index_tag {
    char *routine_name;
    char *routine_header;
} index[MAX_ROUTINES];


struct public_tag {		/* link list of public procedure names */
    struct public_tag *pub_next;
    char   pub_name[1];
} *public_names;



void process_asm_file(char *fn);
void out_of_memory(void);
int search_public_names(char *);
int stricmp_pt(const struct index_tag *a, const struct index_tag *b);



char all_switch;		/* 0 to output only public procedures */
unsigned index_cnt;		/* number of index entries */
unsigned long coreused;		/* total storage used */



void main(int argc, char *argv[])
{
    int ai, i;
    unsigned u;
    FILE *fo;

    if (argc < 3) {
        printf(
	    "asmhdr [-a] outfile *.asm\n"
	    "Extracts header comments from *.ASM files, sorts them, and\n"
	    "writes them to outfile.  Without -a, only extracts headers\n"
	    "for public procedures.  8-22-91");
        exit(EXIT_SUCCESS);
    }

    ai = 1;
    if (strcmp(argv[ai], "-a") == 0)  {
	all_switch = 1;
	ai++;
    }

    fo = fopen(argv[ai], READ_TEXT);
    if (fo != NULL) {
        printf("Error: %s exists!  Overwriting source file?\n", argv[ai]);
        fclose(fo);
        exit(EXIT_FAILURE);
    }

    for (i = ai+1; i < argc; i++)
        process_asm_file(argv[i]);

    qsort(index, index_cnt, sizeof(index[0]), stricmp_pt);

    fo = fopen(argv[ai], WRITE_TEXT);
    if (fo == NULL) {
        printf("Error opening output file: %s\n", argv[1]);
        exit(EXIT_FAILURE);
    }

    for (u = 0; u < index_cnt; u++)
        fprintf(fo,"%s\n",index[u].routine_header);
    fclose(fo);

    printf("Used %d of %d index entries.\n", index_cnt, MAX_ROUTINES);
    printf("Used %lu bytes.\n", coreused);
#ifdef __TURBOC__
    printf("%lu free bytes.\n", (unsigned long) coreleft());
#endif
}




void process_asm_file(char *fn)
{
    char *p, *q;
    char line[LINE_SIZE];	/* file line buffer */
    char pname[LINE_SIZE];	/* procedure name */
    char *buf;			/* temp buffer */
    unsigned s;
    FILE *fi;
    struct public_tag *x;

    if ((buf = malloc(3000)) == NULL)
	out_of_memory();

    fi = fopen(fn, READ_TEXT);
    if (fi == NULL) {
        printf("Error opening input file: %s\n", fn);
        exit(EXIT_FAILURE);
    }

    while (fgets(line, LINE_SIZE, fi) != NULL)	/* skip title & include lines */
	if (line[0] != '\t')  break;
    while (fgets(line, LINE_SIZE, fi) != NULL)	/* skip blank lines */
	if (line[0] != '\n')  break;

    /* extract public names from beginning of source file */
    while (strncmp(line, "\tpublic\t", 8) == 0)  {

	for (q = p = line+8; isgraph(*q); q++)  ;	// find public name

	s = sizeof(struct public_tag)+(unsigned)(q-p)+1;// allocate new public
	if ((x = calloc(s, 1)) == NULL)			//  structure
	    out_of_memory();

	x->pub_next = public_names;			// link public
	public_names = x;

	s = (unsigned) (q-p);				// copy public name
	q = x->pub_name;
	for (; s>0; s--)  *q++ = *p++;
	
	if (fgets(line, LINE_SIZE, fi) == NULL)  break;
    }

    while (fgets(line, LINE_SIZE, fi) != NULL) {
        if (line[0] == ';'  &&  line[1] == ';'  &&  line[2] == '\t') {

            p = buf;
            do {
                p = strcpy(p, line) + strlen(line);
            } while (fgets(line, LINE_SIZE, fi) != NULL  &&  line[0] == ';');
            if (*(p-1) == '\n'  &&  *(p-2) == ';')  p -= 2;
            *p = '\0';

            p = buf+3;        /* extract routine name from header */
            q = pname;
            while (isprint(*p)) {
                if ((*q++ = *p++) == ' ')  *(q-1) = '_';
            }
            *q = '\0';

	    if (search_public_names(pname))  {
		if ((index[index_cnt].routine_header = strdup(buf)) == NULL)
		    out_of_memory();
		if ((index[index_cnt++].routine_name = strdup(pname)) == NULL)
		    out_of_memory();
		coreused += strlen(buf)+1 + strlen(pname)+1;
	    }
        }
    }

    while ((x = public_names) != NULL)  {
	public_names = x->pub_next;
	free(x);
    }
    free(buf);
    fclose(fi);
}




void out_of_memory(void)
{
    printf("Out of memory");
    exit(EXIT_FAILURE);
}




int search_public_names(char *target)
{
    int i,l;
    char c, d, *p, *q;
    struct public_tag *x;

    if (all_switch)  return 1;
    if ((x = public_names) == NULL)  return 1;	/* return match if no publics */
    
    l = (int) strlen(target);

    while (x != NULL)  {
	p = target;				/* compare names */
	q = x->pub_name;

	for (i=l; i>0; i--) {
	    c = *p;
	    d = *q;
	    if ((c == ' ') && (d == '_'))  d = ' ';	/* space & underscore */
	    if ((c == '_') && (d == ' '))  c = ' ';	/* are equivalent */
	    if (c != d)  break;
	    p++;
	    q++;
	}
	if (i == 0)  return 1;			/* if match found */
	
	x = x->pub_next;
    }
    return 0;					/* if no match found */
}




int stricmp_pt(const struct index_tag *a, const struct index_tag *b)
{
    return stricmp(a->routine_name, b->routine_name);
}
