//************************************************************
// PDXPP.H -- Header file for Paradox Engine Class Library
//
//	Gary B. Weinfurther, 06/27/90
//      Last revision:   06/28/90
//************************************************************

#include <pxengine.h>

/***********************************************************
// PXOBJ class -- Abstract base class for other PX classes.
************************************************************/

class PXOBJ
	{
	protected:
		TABLEHANDLE tblHandle;	// Table handle
		int pxerr;				// Error code

	public:
		TABLEHANDLE TblHandle() {return(tblHandle);}
		int Result()	{return pxerr;}
		char *ErrMsg()	{return PXErrMsg(pxerr);}
		operator void*(){return (void *)(pxerr == PXSUCCESS);}	// 0 if failed
		operator!()		{return (pxerr != PXSUCCESS);}	// non-0 if failed
	};

/******************************************************
// PXTABLE class
*******************************************************/

class PXTABLE : public PXOBJ
	{
	public:
		PXTABLE(char *name, int index = 0, int saveEveryChange = FALSE)
			{pxerr = PXTblOpen(name, &tblHandle, index, saveEveryChange);}

		PXTABLE(char *name, int nFlds, char **fldnames, char **types, int saveEvery = FALSE);

		~PXTABLE() {pxerr = PXTblClose(tblHandle);}

		int Name(int bufSize, char *s)
			{return(pxerr = PXTblName(tblHandle, bufSize, s));}

		RECORDNUMBER NRecs();
		int NFlds();
		int KeyNFlds();
		int RecLocked(void);
		int Changed();

		int Lock(int lockType)
			{return(pxerr = PXNetTblLock(tblHandle, lockType));}
		int Unlock(int lockType)
			{return(pxerr = PXNetTblUnlock(tblHandle, lockType));}

		int Refresh()	{return(pxerr = PXNetTblRefresh(tblHandle));}
		int RecDelete()	{return(pxerr = PXRecDelete(tblHandle));}
		int First()		{return(pxerr = PXRecFirst(tblHandle));}
		int Last()		{return(pxerr = PXRecLast(tblHandle));}
		int Next()		{return(pxerr = PXRecNext(tblHandle));}
		int Prev()		{return(pxerr = PXRecPrev(tblHandle));}

		RECORDNUMBER RecNum();
		int RecGoto(RECORDNUMBER rn)
			{return(pxerr = PXRecGoto(tblHandle, rn));}
	};

/******************************************************
// PXTABLE() -- Connstructor to CREATE a table.
*******************************************************/

inline PXTABLE::PXTABLE(char *name, int nFlds, char **fldnames, char **types,
							int saveEvery)
{
	pxerr = PXTblCreate(name, nFlds, fldnames, types);
	if(pxerr == PXSUCCESS)
		pxerr = PXTblOpen(name, &tblHandle, 0, saveEvery);
}

/******************************************************
// NRecs() -- Returns the number of records in the table.
*******************************************************/

inline RECORDNUMBER PXTABLE::NRecs()
{
	RECORDNUMBER recnum;

	pxerr = PXTblNRecs(tblHandle, &recnum);
	return(recnum);
}

/******************************************************
// NFlds() -- Returns the number of fields in the table.
*******************************************************/

inline int PXTABLE::NFlds()
{
	int n;

	pxerr = PXRecNFlds(tblHandle, &n);
	return(n);
}

/***************************************************************
// KeyNFlds() -- Returns the number of key fields in the table.
****************************************************************/

inline int PXTABLE::KeyNFlds()
{
	int n;

	pxerr = PXKeyNFlds(tblHandle, &n);
	return(n);
}

/********************************************************************
// RecLocked() -- Tests if the current record in the table is locked.
*********************************************************************/

inline int PXTABLE::RecLocked()
{
	int locked;

	pxerr = PXNetRecLocked(tblHandle, &locked);
	return(locked);
}

/***************************************************************
// Changed() -- Tests if a table has changed.
****************************************************************/

inline int PXTABLE::Changed()
{
	int changed;

	pxerr = PXNetTblChanged(tblHandle, &changed);
	return(changed);
}

/******************************************************
// RecNum() -- Returns the current record number.
*******************************************************/

inline RECORDNUMBER PXTABLE::RecNum()
{
	RECORDNUMBER recnum;

	pxerr = PXRecNum(tblHandle, &recnum);
	return(recnum);
}

/******************************************************
// PXFIELD -- Class for Paradox fields.
*******************************************************/

class PXFIELD : public PXOBJ
	{
	protected:
		FIELDHANDLE handle;     	// Field handle

	public:
		PXFIELD(PXTABLE &pxtable, char *name);
		PXFIELD(TABLEHANDLE th, char *name);
		FIELDHANDLE FldHandle() {return(handle);}

		int FldName(int bufSize, char *s)
			{return(pxerr = PXFldName(tblHandle, handle, bufSize, s));}

		int FldBlank();
		int FldType(int bufSize, char *type)
			{return(pxerr = PXFldType(tblHandle, handle, bufSize, type));}
	};

/******************************************************
// PXFIELD() -- Constructor for PXFIELD.
*******************************************************/

inline PXFIELD::PXFIELD(PXTABLE &table, char *fname)
{
	tblHandle = table.TblHandle();
	pxerr = PXFldHandle(tblHandle, fname, &handle);
}

/******************************************************
// PXFIELD() -- Constructor for PXFIELD.
*******************************************************/

inline PXFIELD::PXFIELD(TABLEHANDLE th, char *fname)
{
	tblHandle = th;
	pxerr = PXFldHandle(tblHandle, fname, &handle);
}

/******************************************************
// FldBlank() -- Returns non-zero if field is blank.
*******************************************************/

inline int PXFIELD::FldBlank()
{
	int blank;

	pxerr = PXFldBlank(tblHandle, handle, &blank);
	return(blank);
}

/******************************************************
// PXRECORD class
*******************************************************/

class PXRECORD : public PXOBJ
	{
	protected:
		RECORDHANDLE handle;

	public:
		PXRECORD(PXTABLE &pxtable);
		PXRECORD(TABLEHANDLE th);
		PXRECORD(PXRECORD &sourcerec);
		PXRECORD &operator=(PXRECORD &sourcerec);
		~PXRECORD() {pxerr = PXRecBufClose(handle);}

		RECORDHANDLE RecordHandle() {return(handle);}

		int NFlds(void);

		int Append() {return(pxerr = PXRecAppend(tblHandle,handle));}
		int Update() {return(pxerr = PXRecUpdate(tblHandle,handle));}
		int RecGet() {return(pxerr = PXRecGet(tblHandle,handle));}
		int Insert() {return(pxerr = PXRecInsert(tblHandle,handle));}
		int Empty()  {return(pxerr = PXRecBufEmpty(handle));}
		int Delete() {return(pxerr = PXRecDelete(tblHandle));}

		RECORDNUMBER RecNum(void);
		int Goto(RECORDNUMBER rn) {return(pxerr = PXRecGoto(tblHandle,rn));}

		int First();
		int Last();
		int Next();
		int Prev();

		int SrchFld(PXFIELD &field, int scope = SEARCHFIRST)
			{return SrchFld(field.FldHandle(), scope);}

		int SrchFld(FIELDHANDLE fh, int scope = SEARCHFIRST)
			{return(pxerr = PXSrchFld(tblHandle, handle, fh, scope));}

		int SrchKey(int nFlds, int scope = SEARCHFIRST)
			{return(pxerr = PXSrchKey(tblHandle, handle, nFlds, scope));}

		int Put(FIELDHANDLE fh)
			{return(pxerr = PXPutBlank(handle, fh));}

		int Put(FIELDHANDLE fh, short s)
			{return(pxerr = PXPutShort(handle, fh, s));}

		int Put(FIELDHANDLE fh, long l);
		int Put(FIELDHANDLE fh, int month, int day, int year);

		int Put(FIELDHANDLE fh, double d)
			{return(pxerr = PXPutDoub(handle, fh, d));}

		int Put(FIELDHANDLE fh, char *a)
			{return(pxerr = PXPutAlpha(handle, fh, a));}

		int Put(PXFIELD &field)
			{return Put(field.FldHandle());}

		int Put(PXFIELD &field, short s)
			{return Put(field.FldHandle(), s);}

		int Put(PXFIELD &field, long l)
			{return Put(field.FldHandle(), l);}

		int Put(PXFIELD &field, int month, int day, int year)
			{return Put(field.FldHandle(), month, day, year);}

		int Put(PXFIELD &field, double d)
			{return Put(field.FldHandle(), d);}

		int Put(PXFIELD &field, char *a)
			{return Put(field.FldHandle(), a);}

		int Get(FIELDHANDLE fh, short &s)
			{return (pxerr = PXGetShort(handle, fh, &s));}

		int Get(FIELDHANDLE fh, long &l);

		int Get(FIELDHANDLE fh, double &d)
			{return (pxerr = PXGetDoub(handle, fh, &d));}

		int Get(FIELDHANDLE fh, int &m, int &d, int &y);

		int Get(FIELDHANDLE fh, int bufSize, char *a)
			{return (pxerr = PXGetAlpha(handle, fh, bufSize, a));}

		int Get(PXFIELD &field, short &s)
			{return Get(field.FldHandle(), s);}

		int Get(PXFIELD &field, long &l)
			{return Get(field.FldHandle(), l);}

		int Get(PXFIELD &field, double &d)
			{return Get(field.FldHandle(), d);}

		int Get(PXFIELD &field, int bufSize, char *a)
			{return Get(field.FldHandle(), bufSize, a);}

		int Get(PXFIELD &field, int &m, int &d, int &y)
			{return Get(field.FldHandle(), m, d, y);}

	};


/******************************************************
// PXRECORD() -- constructor for PXRECORD.
*******************************************************/

inline PXRECORD::PXRECORD(PXTABLE &table)
{
	tblHandle = table.TblHandle();
	pxerr = PXRecBufOpen(tblHandle, &handle);
}

/******************************************************
// PXRECORD() -- constructor for PXRECORD.
*******************************************************/

inline PXRECORD::PXRECORD(TABLEHANDLE th)
{
	tblHandle = th;
	pxerr = PXRecBufOpen(tblHandle, &handle);
}

/******************************************************
// PXRECORD() -- constructor for PXRECORD.
*******************************************************/

inline PXRECORD::PXRECORD(PXRECORD &sourcerec)
{
	tblHandle = sourcerec.tblHandle;
	pxerr = PXRecBufOpen(tblHandle, &handle);
	if(pxerr == PXSUCCESS)
		pxerr = PXRecBufCopy(sourcerec.handle, handle);
}

/******************************************************
// PXRECORD() -- constructor for PXRECORD.
*******************************************************/

inline PXRECORD& PXRECORD::operator=(PXRECORD &sourcerec)
{
	pxerr = PXRecBufCopy(sourcerec.handle, handle);
	return(*this);
}

/******************************************************
// NFlds() -- Returns the number of fields in the table.
*******************************************************/

inline int PXRECORD::NFlds()
{
	int n;

	pxerr = PXRecNFlds(tblHandle, &n);
	return(n);
}

/******************************************************
// RecNum() -- Returns the current record number.
*******************************************************/

inline RECORDNUMBER PXRECORD::RecNum()
{
	RECORDNUMBER recnum;

	pxerr = PXRecNum(tblHandle, &recnum);
	return(recnum);
}

/******************************************************
// First() -- Retrieves the first record in the table.
*******************************************************/

inline int PXRECORD::First()
{
	pxerr = PXRecFirst(tblHandle);
	if(pxerr == PXSUCCESS)
		pxerr = PXRecGet(tblHandle, handle);

	return(pxerr);
}

/******************************************************
// Last() -- Retrieves the last record in the table.
*******************************************************/

inline int PXRECORD::Last()
{
	pxerr = PXRecLast(tblHandle);
	if(pxerr == PXSUCCESS)
		pxerr = PXRecGet(tblHandle, handle);

	return(pxerr);
}

/******************************************************
// Next() -- Retrieves the next record in the table.
*******************************************************/

inline int PXRECORD::Next()
{
	pxerr = PXRecNext(tblHandle);
	if(pxerr == PXSUCCESS)
		pxerr = PXRecGet(tblHandle, handle);

	return(pxerr);
}

/******************************************************
// Prev() -- Retrieves the previous record in the table.
*******************************************************/

inline int PXRECORD::Prev()
{
	pxerr = PXRecPrev(tblHandle);
	if(pxerr == PXSUCCESS)
		pxerr = PXRecGet(tblHandle, handle);

	return(pxerr);
}

/******************************************************
// Put() -- Puts a date or long value into a field.
*******************************************************/

inline int PXRECORD::Put(FIELDHANDLE fh, long l)
{
	char type[6];

	pxerr = PXFldType(tblHandle, fh, 6, type);
	if(pxerr == PXSUCCESS)
		pxerr = (type[0] == 'D') ? PXPutDate(handle, fh, l)
								 : PXPutLong(handle, fh, l);
	return(pxerr);
}

/******************************************************
// Put() -- Puts an MDY date into a field.
*******************************************************/

inline int PXRECORD::Put(FIELDHANDLE fh, int month, int day, int year)
{
	long date;

	pxerr = PXDateEncode(month, day, year, &date);
	if(pxerr == PXSUCCESS)
		pxerr = PXPutDate(handle, fh, date);

	return(pxerr);
}

/******************************************************
// Get() -- Gets a date or long value from a field.
*******************************************************/

inline int PXRECORD::Get(FIELDHANDLE fh, long &l)
{
	char type[6];

	pxerr = PXFldType(tblHandle, fh, 6, type);
	if(pxerr == PXSUCCESS)
		pxerr = (type[0] == 'D') ? PXGetDate(handle, fh, &l)
								 : PXGetLong(handle, fh, &l);
	return(pxerr);
}

/******************************************************
// Get() -- Gets an MDY date from a field.
*******************************************************/

inline int PXRECORD::Get(FIELDHANDLE fh, int &month, int &day, int &year)
{
	long date;

	pxerr = PXGetDate(handle, fh, &date);
	if(pxerr == PXSUCCESS)
		pxerr = PXDateDecode(date, &month, &day, &year);

	return(pxerr);
}


/******************************************************
// PXLOCK class
*******************************************************/

class PXLOCK : public PXOBJ
	{
	protected:
		LOCKHANDLE handle;

	public:
		PXLOCK(PXTABLE &pxtable);
		PXLOCK(PXRECORD &pxrecord);
		~PXLOCK()	{Unlock();}
		int Unlock(){return(pxerr = PXNetRecUnlock(tblHandle, handle));}

		LOCKHANDLE LockHandle()	{return(handle);}
		int Locked();

		int Goto()
			{return(pxerr = PXNetRecGotoLock(tblHandle, handle));}
	};

/******************************************************
// PXLOCK() -- Constructor for PXLOCK class
//			-- Locks the current record.
*******************************************************/

inline PXLOCK::PXLOCK(PXTABLE &pxtable)
{
	tblHandle = pxtable.TblHandle();
	pxerr = PXNetRecLock(tblHandle, &handle);
}

/******************************************************
// PXLOCK() -- Constructor for PXLOCK class
//			-- Locks the current record.
*******************************************************/

inline PXLOCK::PXLOCK(PXRECORD &pxrecord)
{
	tblHandle = pxrecord.TblHandle();
	pxerr = PXNetRecLock(tblHandle, &handle);
}