
/*SDOC*************************************************************

PPPPPPPPP  DDDDDDDDD   OOOOOOOO   CCCCCCCC  
PPPPPPPPPP DDDDDDDDDD OOOOOOOOOO CCCCCCCCCC 
PP      PP DD      DD OO      OO CC      CC 
PP      PP DD      DD OO      OO CC         
PPPPPPPPPP DD      DD OO      OO CC         
PPPPPPPPP  DD      DD OO      OO CC         
PP         DD      DD OO      OO CC         
PP         DD      DD OO      OO CC      CC 
PP         DDDDDDDDDD OOOOOOOOOO CCCCCCCCCC 
PP         DDDDDDDDD   OOOOOOOO   CCCCCCCC  

	$Header$

	Module: PDOC formatted code printer

	Author:	Pat Beirne

	Copyright 1990 Corel Systems Corp.

	Description: PDOC will format .c and .h files, using the 
			concepts of Marcus at Berkeley

			it is designed to produce native laserjet and PostScript codes

			all units are 300 dpi

	Compile: cl /AT pdoc.c c:\lib\setargv.obj /link /noe

	Syntax:
			PDOC [-L] | [-P] | [-R] [-n] [-?] sourcefile destfile

			-L uses laserjet codes (default)
			-P uses PostScript codes
			-R uses RTF codes
			-n sets the tab width to n
			-? help

*************************************************************EDOC*/

/*SDOC*************************************************************

  Revision Record

	Rev	Date		Auth	Changes
	===	====		====	=======

	1.5	6/9/90	pb  	release to the company; added c++ codes
	1.6	16/01/91	pb		minor changes to comment parsing to avoid
									picking up quotes
								fine tuned the gray comment box
	1.7	08/02/91	pb		worked on the quotes again
								added RTF output
								added "#ifndef" to the keywords
								modified line comments to skip trailing spaces
								added Laserjet3 commands
							NOTE: to implement RTF properly, we really should
							introduce subroutines for "tab" and "underline"

*************************************************************EDOC*/



/*
**
**	 Includes
**
*/
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <dos.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>


/*::defn
**
**	 Defines
**
*/
#define	TRUE 		1
#define 	FALSE 	0
#define	SPACE_WID 21
#define 	TAB_WID 	4

#define 	VERSION 	"1.7"

#define	CD_RULE 		0
#define 	CD_PFONT		1	// preprocessor keywords
#define	CD_CFONT		2	// C keywords
#define	CD_FONT		3	// code text
#define	CD_FNFONT	4	// function names
#define	CD_CMTFONT	5	// comments
#define	CD_STRFONT	6	// strings
#define	CD_START		7
#define	CD_END		8
#define 	CD_XYMOVE	9
#define	CD_RXMOVE	10
#define	CD_LF			11
#define	CD_CRLF		12
#define	CD_GREYLINE	13
#define	CD_FF			14
#define 	CD_HEADERRULE 15
#define 	CD_XMOVE		16
#define ESC "\x1b"
#define FF '\x0C'

enum	RDMODE {RM_NORMAL, RM_BODY, RM_STRING, RM_STARCOMMENT, RM_LINECOMMENT};
enum	WRMODE {WM_NORMAL, WM_STRING, WM_BODYCOMMENT, WM_LINECOMMENT};
enum	SF	{SF_CODE, SF_CWORD, SF_PWORD, SF_BODYCOMMENT, SF_LINECOMMENT,
					SF_FNNAME, SF_STRING};

enum 	RDMODE	nRdMode = RM_NORMAL;
enum 	WRMODE 	nWrMode = WM_NORMAL;


/*::end*/


/*::decl
**
**	 Global Variables
**
*/
char 	szOpts[] = "RrPpLl23456789?";
char	szBuff[256];						/* line buffer */
char	acRdFileBuff[4096];
char	acWrFileBuff[4096];
char	szWord[80];							/* word buffer */
char	szRdFile[80];					/* the file name */
char	szWrFile[80];
char 	**szCodes;	/* points to either szLJetCodes or szPSCodes */

/*::end*/


/*::fnproto
**
**	 Function Prototypes
**
*/
void	Parse(FILE *);
void EndPrint(void);
void StartPrint(void);
void Header(void);
void PrintHelp(void);


/*::end*/


/*
**
**	 Local Variables
**
*/



	/* get options stuff */
int	getopt(int argc, char *argv[], char *optionS);
int	optind	= 1;	/* index of which argument is next	*/
char   *optarg;		/* pointer to argument of current option */
int	opterr	= 1;	/* allow error message	*/

enum {PT_LASERJET,PT_POSTSCRIPT,PT_RTF};

char	*(szLJetCodes[]) =
			{ESC "&f0S" ESC "*p+22Y" ESC "*p750X" ESC "*c2000a12B" ESC "*c0P" ESC "&f1S",
			ESC "(s8vs2b1p4148T",
			ESC "(s8v1sb1p4148T",
			ESC "(s8vsb1p4148T",
			ESC "(s12vs1b1p4148T",
			ESC "(s8vsb5T",
			ESC "(s7v16.6hsb0p3T",
			ESC "(8Q",
			"\x0c",
			ESC "*p%dx%dY",
			ESC "*p%+dX",
			ESC "*p+37Y",
			ESC "*p%dx+37Y",
			ESC "&f0S" ESC "*p-8x-30Y" ESC "*c2400a42b06G" ESC "*c2P" ESC "&f1S",
			"\x0C",
			ESC "&f0S" ESC "*p+22Y" ESC "*p100X" ESC "*c600a5B" ESC "*c0P"
			ESC "*p+650X" ESC "*c3000a5B" ESC "*c0P" ESC "&f1S",
			ESC "*p%dX"
		};
char	*(szPSCodes[]) =
			{")t cp 22 rym 750 xm 2000 12 0 r m(",
			")t 1 sf(",
			")t 2 sf(",
			")t 3 sf(",
			")t 4 sf(",
			")t 5 sf(",
			")t 6 sf(",
			"\x0D" "%%!PS-Adobe-2.0\n%%%%CreationDate:(%.24s)\n"
				"%%%%Creator:PDOC\n%%%%Title: %s\n"
				"%%%%BoundingBox:0 0 576 756\n%%%%DocumentFonts:Helvetica Helvetica-Oblique\n"
				"%%%%+HelveticaBold Courier Times-Roman\n%%%%EndComments\n%%%%BeginProcSet\n"
				"0 756 translate 72 300 div dup neg scale "
				"/point {300 72 div mul}def\n"
				"/m {moveto} def"
				"/cp {currentpoint} def /rm {rmoveto} def\n"
				"/rym {0 exch rm} def /rxm {0 rm} def\n"
				"/xm {cp exch pop m} def /ym {cp pop exch m} def\n"
				"/sf {fa exch get setfont} def\n"
				"/fa [0 /Helvetica-Bold findfont [8 point 0 0 -8 point 0 0]makefont\n"
				"/Helvetica-Oblique findfont [8 point 0 0 -8 point 0 0]makefont\n"
				"/Helvetica findfont [8 point 0 0 -8 point 0 0]makefont\n"
				"/Helvetica-Bold findfont [12 point 0 0 -12 point 0 0]makefont\n"
				"/Times-Roman findfont [8 point 0 0 -8 point 0 0]makefont\n"
				"/Courier findfont [7 point 0 0 -7 point 0 0]makefont\n"
				"] def %%font array\n"
				"/r {gsave setgray 0 0 rm 1 index 0 rlineto 0 exch rlineto\n"
				"neg 0 rlineto closepath fill grestore} def\n"
				"/t {show} def\n"
				"%%%%EndProcSet\n%%%%EndProlog\n0 0 m (",		/* end of prolog item */
			")t showpage\n%%Trailer",
			")t\n%d %d m (",
			")t %d xm (",
			")t\n37 rym (",
			")t\n%d xm 37 rym (",
			")t cp -8 -30 rm 2400 42 .98 r m (",
			")t gsave showpage grestore 0 0 m (",
			")t cp 22 rym 100 xm 620 6 0 r 650 rxm 3000 5 0 r m (" ,
			")t %d xm ("
			};

char	*(szRTFCodes[]) =
			{"", 					/* rule				*/
			"}{\\b ",			/* preprocess		*/
			"}{\\i ",			/* C keywords		*/
			"}{", 			/* code				*/
			"}{\\fs24\\b ",		/* function names	*/
			"}{\\f0\\cf1 ",		/* comments			*/
			"}{\\f2 ",		/* strings			*/
			"{\\rtf1\\ansi\\margr300\\margl480\\margt300\\margb300\\deftab540\n"
				"{\\fonttbl{\\f0\\froman Tms Rmn;}"
				"{\\f1\\fswiss Helv;}{\\f2\\fmodern Courier;}}\n"
				"{\\colortbl;\\red0\\green0\\blue255;}"
				"{\\stylesheet{\\s0\\f1\\fs16 Code;}{\\s1\\brdrb\\brsp20\\brdrth\\tx3600\\tqr\\tx8640"
				"\\f1\\fs24\\b\\cf0 header;}}\n"
				"{\\header\\s1\\f1\\b\\fs24 %s\\tab %s \\tab page \\chpgn}"
				"\\s0\\fs16\\f1{\n",	/* start	*/
			"}}",			// end
			"",			/* xymove */
			"",			/* rxmove  		*/
			"\\brdrb\\brsp20\\brdrth\\sl40\\par}{",		/* lf; also draws a thick line*/
			"\\par\n",	/* crlf			*/
			"",			/* greyline		*/
			"\\page\n",	/* ff				*/
			"",			/* header rule	*/
			"\\tab "		/* xmove			*/
			};


int 	nBrace;						/* bracket counters */
int	nBracket;
int	nSqBracket;
char	cQuoteMatch;				/* holds the 'endquote' character */
int	fGotFnLine;
int	nNextLineGap = FALSE;
int	fOnlyWhiteSpace;			/* set false when something is on the line */
int	nXGreyStart;
int 	nCurrMargin = 750;
int	nTabSize = TAB_WID;
int	nRightCol1 = 750;
int	nRightCol2 = 2350;

int	nPrinterType;				/* set to 1 for postscript; 0 for laserjet */

int	nXPos, nYPos;						/* position on the output page */
int	nRdCol;
int	nWrCol;				/* the column position of the output */

FILE	*sRdFile;
FILE	*sWrFile;

int main(int argc, char *argv[])
{
	int 	nOption;

	/* first, get the options */
	szCodes = szLJetCodes;
	printf("\nPDOC %s print formatter for C and C++",VERSION);

	while ( (nOption = getopt(argc, argv, szOpts)) != EOF)
		{
		nOption = toupper(nOption);
		if (nOption=='P') {nPrinterType=PT_POSTSCRIPT; szCodes = szPSCodes;}
		if (nOption=='R')	{nPrinterType=PT_RTF; szCodes = szRTFCodes;
								nRightCol1 = nRightCol2 = 3000;}
		if (isdigit(nOption)) nTabSize = nOption-'0';
		if (nOption=='?') PrintHelp();
		}

	if (argc-optind <2)		/* number entered - number left to scan */
		{
		printf("\nusage  PDOC [-L]|[-P]|[-R] [-n] [-?] sourcefile destfile");
		printf("\n   default: Kyocera Lasert   -P PostScript   -R RTF   -H Laserjet3");
		printf("\n   -n set tab width          -? show usage");
		exit(1);
		}

	strcpy(szRdFile, argv[optind]);		/* save the file name */
	strcpy(szWrFile, argv[optind+1]);

	if ((sRdFile = fopen(argv[optind],"r")) == NULL)
		{
		printf("\nerror: cannot open source file %s",szRdFile);
		exit(1);
		}
	if ((sWrFile = fopen(argv[optind+1],"w")) == NULL)
		{
		printf("\nerror: cannot open destination file %s",szWrFile);
		exit(1);
		}

	printf("\n     formatting C code   tabs set to %d\n\tfrom: %s\n\tto: %s  using %s codes",
			nTabSize,
			argv[optind],
			argv[optind+1],
			nPrinterType==PT_RTF ? "RTF Rich Text Format" :
				nPrinterType==PT_POSTSCRIPT ? "PostScript" : "LaserJet");

	setvbuf(sRdFile,acRdFileBuff,_IOFBF,4096);
	setvbuf(sRdFile,acWrFileBuff,_IOFBF,4096);

	StartPrint();
	Parse(sRdFile);
	EndPrint();
	return 0;
}




/*
	getopt.c -- Turbo C

	Copyright (c) 1986,87,88 by Borland International Inc.
	All Rights Reserved.
*/
 
 

static	char   *letP = NULL;	/* remember next option char's location */
static	char	SW = '-';		/* DOS switch character, either '-' or '/' */
static 	char SW1 = '/';

/*
  Parse the command line options, System V style.

  Standard option syntax is:

    option ::= SW [optLetter]* [argLetter space* argument]

  where
    - SW is either '/' or '-', according to the current setting
      of the MSDOS switchar (int 21h function 37h).
    - there is no space before any optLetter or argLetter.
    - opt/arg letters are alphabetic, not punctuation characters.
    - optLetters, if present, must be matched in optionS.
    - argLetters, if present, are found in optionS followed by ':'.
    - argument is any white-space delimited string.  Note that it
      can include the SW character.
    - upper and lower case letters are distinct.
 
  There may be multiple option clusters on a command line, each
  beginning with a SW, but all must appear before any non-option
  arguments (arguments not introduced by SW).  Opt/arg letters may
  be repeated: it is up to the caller to decide if that is an error.

  The character SW appearing alone as the last argument is an error.
  The lead-in sequence SWSW ("--" or "//") causes itself and all the
  rest of the line to be ignored (allowing non-options which begin
  with the switch char).

  The string *optionS allows valid opt/arg letters to be recognized.
  argLetters are followed with ':'.  Getopt () returns the value of
  the option character found, or EOF if no more options are in the
  command line.	 If option is an argLetter then the global optarg is
  set to point to the argument string (having skipped any white-space).

  The global optind is initially 1 and is always left as the index
  of the next argument of argv[] which getopt has not taken.  Note
  that if "--" or "//" are used then optind is stepped to the next
  argument before getopt() returns EOF.

  If an error occurs, that is an SW char precedes an unknown letter,
  then getopt() will return a '?' character and normally prints an
  error message via perror().  If the global variable opterr is set
  to false (zero) before calling getopt() then the error message is
  not printed.

  For example, if the MSDOS switch char is '/' (the MSDOS norm) and

    *optionS == "A:F:PuU:wXZ:"

  then 'P', 'u', 'w', and 'X' are option letters and 'F', 'U', 'Z'
  are followed by arguments.  A valid command line may be:
 
    aCommand  /uPFPi /X /A L someFile

  where:
    - 'u' and 'P' will be returned as isolated option letters.
    - 'F' will return with "Pi" as its argument string.
    - 'X' is an isolated option.
    - 'A' will return with "L" as its argument.
    - "someFile" is not an option, and terminates getOpt.  The
      caller may collect remaining arguments using argv pointers.
*/

int	getopt(int argc, char *argv[], char *optionS)
{
	unsigned char ch;
	char *optP;

	if (argc > optind) {
		if (letP == NULL) {
			if ((letP = argv[optind]) == NULL ||
				(*letP != SW && *letP!=SW1) )  goto gopEOF;
			++letP;
			if (*letP == SW || *letP == SW1) {
				optind++;  goto gopEOF;
			}
		}
		if (0 == (ch = *(letP++))) {
			optind++;  goto gopEOF;
		}
		if (':' == ch  ||  (optP = strchr(optionS, ch)) == NULL)
			goto gopError;
		if (':' == *(++optP)) {
			optind++;
			if (0 == *letP) {
				if (argc <= optind)  goto  gopError;
				letP = argv[optind++];
			}
			optarg = letP;
			letP = NULL;
		} else {
			if (0 == *letP) {
				optind++;
				letP = NULL;
			}
			optarg = NULL;
		}
		return ch;
	}
gopEOF:
	optarg = letP = NULL;
	return EOF;

gopError:
	optarg = NULL;
	errno  = EINVAL;
	if (opterr)
		perror ("get command line option");
	return ('?');
}


/*
	parsing code

*/

void SetFont(enum SF mode);
void PrintWord(char *sz);
void PrintCWord(char *sz);
void PrintPWord(char *sz);
int isPWord(char *szTest);
int isCWord(char *szTest);
int CopyWord(char *dest, char *src);
void Process(char *p);
void PushState(int nNewState);
void PopState(void);
void NewLine(int fEnableGrey);
void JamLeft(void);
void PaintGrey(void);
void NewPage(int flag);
void XMoveTo(int);
void	Expand(char *sz);

void Parse(FILE *sRdFile)
{
	char	*p;

	while (fgets(szBuff,256,sRdFile))
		{
		if (p = strchr(szBuff,'\n')) *p='\0';		/* strip the trailing CR */

			/* check for potential function declaration line */
		if (!strchr(szBuff,'#') && !strchr(szBuff,';') && nRdMode == RM_NORMAL
				&& strchr(szBuff,'(') && strlen(szBuff)>5)
				{
				fGotFnLine = TRUE;
				NewLine(TRUE);
				}
		else
			fGotFnLine = FALSE;

			/* use this boolean to distinguish the two types of comment writing */
		fOnlyWhiteSpace = TRUE;
		for (nRdCol=0, p=szBuff; *p; )
			{
			int nWordLen;
			nWordLen = CopyWord(szWord, p);
			Process(szWord);
			nRdCol += nWordLen;
			p += nWordLen;
			}
		if (nRdMode==RM_LINECOMMENT) PopState();
		NewLine(TRUE);
		if (fOnlyWhiteSpace && nYPos>2800) NewPage(0);	/* try to avoid orphans */
		if (nYPos>3050) NewPage(0);
		}
}


/* process a single word and the special digraphs */
void Process(char *p)
{
	switch(nRdMode)
		{
		case RM_NORMAL:
			switch (*p)
				{
				case '{':
					nRdMode = RM_BODY;
					PrintWord(p);
					nBrace++;
					break;
				case '\"':
				case '\'':
					PrintWord(p);        /* print the word first */
					PushState(RM_STRING);
					cQuoteMatch = *p;
					break;
				case '/':
					if (!strcmp(p,"/*"))
						PushState(RM_STARCOMMENT);
					else if (!strcmp(p,"//"))
						PushState(RM_LINECOMMENT);
					else
						PrintWord(p);
					break;
				default:
					if (isCWord(p))
						PrintCWord(p);
					else if (isPWord(p))
						PrintPWord(p);
					else PrintWord(p);
					if (!isspace(*p)) fOnlyWhiteSpace = FALSE;
					break;
				}
			break;

		case RM_BODY:
			switch (*p)
				{
				case '{': nBrace++;   /* */
					PrintWord(p);
					break;
				case '}': if (--nBrace==0)
					nRdMode = RM_NORMAL;
					PrintWord(p);
					break;
				case '\"':
				case '\'':
					PrintWord(p);	/* print the word first */
					PushState(RM_STRING);
					cQuoteMatch = *p;
					break;
				case '/':
					if (!strcmp(p,"/*"))
						PushState(RM_STARCOMMENT);
					else if (!strcmp(p,"//"))
						PushState(RM_LINECOMMENT);
					else
						PrintWord(p);
					break;
				default:
					if (isCWord(p))
						PrintCWord(p);
					else if (isPWord(p))
						PrintPWord(p);
					else
						PrintWord(p);
					if (!isspace(*p)) fOnlyWhiteSpace = FALSE;
					break;

				}
			break;

		case RM_STRING:
			if (*p==cQuoteMatch)
				PopState();
			PrintWord(p);
			break;

		case RM_LINECOMMENT:
			PrintWord(p);
			if (nXPos > nRightCol1 && nWrMode==WM_LINECOMMENT)	/* check to see if the comment is encroaching */
				NewLine(TRUE);
			break;

		case RM_STARCOMMENT:
			if (!strcmp(p,"*/"))
				PopState();
			else if (isspace(*p) && fOnlyWhiteSpace && nWrMode==WM_LINECOMMENT)	/* drop leading white space in left margin*/
				;
			else
				PrintWord(p);
			if (nXPos > nRightCol1 && nWrMode==WM_LINECOMMENT)	/* check to see if the comment is encroaching */
				{
				NewLine(TRUE);
				}
			if (!isspace(*p)) fOnlyWhiteSpace = FALSE;
			break;
		}
	if (nXPos>nRightCol2) NewLine(TRUE);
}

/* read the input stream, picking out words

	the exceptions are that multiple spaces and space-tab
	are returned as a word, and slash-star, star-slash are returned
	as a word, and all \ escapes are a word */
int CopyWord(char *dest, char *src)
{
	int nRet = 0;

	switch (*src)
		{
		case ' ':
		case '\t':					/* pick off the space combos first */
			while (*src==' '  || *src=='\t')
				{
				*dest++ = *src++;
				nRet++;
				}
			// discard trailing spaces
			if (!*src) dest -= nRet;
			break;
		case '/':					/* now pick off both types of comment start marker */
			*dest++ = *src++;
			nRet++;
			if (*src=='*' || *src=='/')
				{
				*dest++ = *src;
				nRet++;
				}
			break;
		case '*':				/* pick off end-of-comment marker */
			*dest++ = *src++;
			nRet++;
			if (*src=='/')
				{
				*dest++ = *src;
				nRet++;
				}
			break;
		case '\\':				/* pick off in-string escapes */
			*dest++ = *src++;
			nRet++;
			if (*src!='\n')
				{
				*dest++ = *src;
				nRet++;
				}
			break;
		case '\'':	/* take just one */
		case '\"':
		case '(':
		case ')':
		case '{':
		case '}':
			*dest++ = *src;
			nRet++;
			break;
		case FF:
			NewPage(1);
			nRet++;
			break;
		default:
			do
				{
				*dest++ = *src++;
				nRet++;
				}
			while ((isalnum(*src) || strchr("_[]=+-|&.<>",*src)) && *src);
			break;
		}
	*dest = '\0';
	return nRet;
}

#define MAX_CWORD	51
char	szCWord[][10] = {"BOOL",    "BYTE",   "FAR",    "HDC",    "HMEM",
              "HWND",   "LPSTR",  "NEAR",   "PASCAL", "WORD",
              "break",  "case",   "cdecl",  "char",   "class",
				  "const",	"continue","default","delete","do",
				  "double", "else",   "extern", "far",    "float",
				  "for",    "goto",   "if",     
				  "inline", "int",	  "long",   "near",   "new",
					"pascal", "private","protected","public","register",
					"return","short",		"signed", "sizeof", "static",
					"struct", "switch",	"typedef","union",  "unsigned",
					"virtual","void",	  "while"
					};

int isCWord(char *szTest)
{
	return (int) bsearch(szTest,szCWord,MAX_CWORD,10,strcmp);
}

#define MAX_PWORD 8
char szPWord[][10] = {"#define", "#elif","#endif","#if",
			"#ifdef", "#include","#ifndef","#pragma"};

int isPWord(char *szTest)
{
	return (int) bsearch(szTest,szPWord,MAX_PWORD,10,strcmp);
}


/*
	printing code

*/

void PrintPWord(char *sz)
{
	register int len;
	SetFont(SF_PWORD);
	if (nPrinterType>PT_LASERJET) Expand(sz);
	fputs(sz,sWrFile);
	SetFont(SF_CODE);
	nXPos += SPACE_WID*(len=strlen(sz));
	nWrCol += len;
}

void PrintCWord(char *sz)
{
	register int len;
	SetFont(SF_CWORD);
	if (nPrinterType>PT_LASERJET) Expand(sz);
	fputs(sz,sWrFile);
	SetFont(SF_CODE);
	nXPos += SPACE_WID*(len=strlen(sz));
	nWrCol += len;
}



void PrintWord(char *sz)
{
	register int len;
	if (fGotFnLine && nRdMode == RM_NORMAL && *sz!=' ' && *sz!='\t')
		{
		if (sz[0]=='(')
			{
			fGotFnLine = FALSE;
			SetFont(SF_CODE);
			}
		else
			{
				/* print the rule */
			fputs(szCodes[CD_RULE],sWrFile);
			nNextLineGap = TRUE;
			SetFont(SF_FNNAME);
			fGotFnLine++;
			}
		}

		/* work on the space problem */
	if (*sz==' ' || *sz=='\t')
		{
			/* check for one or two consecutive spaces */
		if (*sz==' ' && (sz[1]=='\0' || (sz[1]==' ' && sz[2]=='\0')))
			{
			fputs(sz,sWrFile);
			nXPos += SPACE_WID*(len=strlen(sz));
			nWrCol += len;
			}
			/* longer sequences of spaces and tabs cause column resync */
		else
			{
			while (*sz)
				{
				if (*sz==' ') nWrCol++;		/* for spaces */
				else nWrCol = ((nWrCol+nTabSize)/nTabSize)*nTabSize;	/* for tabs */
				sz++;
				}
			XMoveTo(nCurrMargin + SPACE_WID*nWrCol);
			}
		}	/* end of space/tab case */

	else	/* normal char case */
		{
		if (nPrinterType>PT_LASERJET) Expand(sz);
		fputs(sz,sWrFile);
		nXPos += SPACE_WID * (len=strlen(sz));
		nWrCol += len;
		}
}


void SetFont(enum SF mode)
{
#if 0
	switch(mode)
		{
		case SF_PWORD:
			fputs("..preprocess..",sWrFile);
			break;
		case SF_CWORD:
			fputs("..c word..",sWrFile);
			break;
		case SF_CODE:
			fputs("..body..",sWrFile);
			break;
		case SF_LINECOMMENT:
			fputs("..line comment..",sWrFile);
			break;
		}
#endif
	switch(mode)
		{
		case SF_PWORD:
			fputs(szCodes[CD_PFONT],sWrFile);
			break;
		case SF_CWORD:
			fputs(szCodes[CD_CFONT],sWrFile);
			break;
		case SF_CODE:
			fputs(szCodes[CD_FONT],sWrFile);
			break;
		case SF_FNNAME:
			fputs(szCodes[CD_FNFONT],sWrFile);
			break;
		case SF_LINECOMMENT:
		case SF_BODYCOMMENT:
			fputs(szCodes[CD_CMTFONT],sWrFile);
			break;
		case SF_STRING:
			fputs(szCodes[CD_STRFONT],sWrFile);
			break;
		}
}


/*
	printer support code

*/

void StartPrint(void)
{
	char szTemp[80];
	time_t t;

	strcpy(szTemp,szRdFile);
	Expand(szTemp);
	time(&t);

	fprintf(sWrFile,szCodes[CD_START],ctime(&t),szTemp);

	SetFont(SF_CODE);
	nXPos = 750; nYPos = 0;
	Header();
}

void EndPrint(void)
{
	fputs(szCodes[CD_END],sWrFile);
}


/*
	movement controls
*/
void MoveTo(int x, int y)
{
	if (nPrinterType==PT_RTF)
		fprintf(sWrFile,szCodes[CD_XYMOVE],(int)(24L*x/5));
	else
		fprintf(sWrFile,szCodes[CD_XYMOVE],x,y);
	nXPos = x;
	nYPos = y;
}
void RXMoveTo(int dx)
{
	if (nPrinterType==PT_RTF)
		fprintf(sWrFile,szCodes[CD_RXMOVE],(int)(24L*dx/5));
	else
		fprintf(sWrFile,szCodes[CD_RXMOVE],dx);
	nXPos += dx;
}
void XMoveTo(int x)
{
	if (nPrinterType==PT_RTF)
		{
		int w;
			// a little loop to send out tabs
		for (w = x-nXPos; w>0; w -= SPACE_WID * nTabSize)
			fprintf(sWrFile,szCodes[CD_XMOVE]);	// send out the tabs
		}
	else
		fprintf(sWrFile,szCodes[CD_XMOVE],x);
	nXPos = x;
}


void NewLine(int fEnableGrey)
{
	while (nNextLineGap>0)
		{
		fputs(szCodes[CD_LF],sWrFile);
		nYPos += 37;
		nNextLineGap--;
		}
	nXPos = nCurrMargin;
	if (nPrinterType==PT_RTF)
		fprintf(sWrFile, szCodes[CD_CRLF],(int)(24L*nXPos/5));
	else
		fprintf(sWrFile, szCodes[CD_CRLF],nXPos);
	nYPos += 37;
	if (nWrMode == WM_BODYCOMMENT && fEnableGrey)
		{
		XMoveTo(nXPos = nXGreyStart);
		PaintGrey();
		}
	nWrCol = 0;
}

void JamLeft()
{
	XMoveTo(100);
	nWrCol = 0;
}
void	PaintGrey()
{
		/* push, move up, draw, pop */
	fputs(szCodes[CD_GREYLINE],sWrFile);
}

/*SDOC*************************************************************

	Name: NewPage

	Action: outputs a page in the appropriate codes
		The flag parameter indicates that a page should
		be forced, even if pagination is not being performed

	Returns: void

*************************************************************EDOC*/
void NewPage(int flag)
{
		// don't bother for RTF
	if (flag==0 && nPrinterType==PT_RTF) return;

	fputs(szCodes[CD_FF],sWrFile);
	nYPos = 0;
	nXPos = 750;
		// print the page header
	Header();
}

void Header(void)
{
	time_t t;
	static int nPage=1;
	char szHeader [80];
	char	*p;

	time(&t);	/* get the time */
	sprintf(szHeader,"%s",ctime(&t));	/* get the time */

	if (nPrinterType == PT_RTF)
		return;

	else
		{
		NewLine(FALSE);
		XMoveTo(100);
		SetFont(SF_FNNAME);

		if (p = strchr(szHeader,'\n')) *p='\0';	/* strip off trailing line feed */
		fputs(szHeader,sWrFile);
		XMoveTo(750);
		fputs(szRdFile,sWrFile);		/* and the file name */
		XMoveTo(2000);
		fprintf(sWrFile,"page %d",nPage++);			/* and page number */
			/* draw a thick line under the header */
		fputs(szCodes[CD_HEADERRULE],sWrFile);
		NewLine(FALSE);
		NewLine(TRUE);		/* enable the next line of grey */
		switch(nWrMode)
			{
			case WM_LINECOMMENT:
			case WM_BODYCOMMENT:
				SetFont(SF_LINECOMMENT);
				break;
			case WM_STRING:
				SetFont(SF_STRING);
				break;
			default:
				SetFont(SF_CODE);
			}
		}
}



/*
	state management code

*/

static int StateStack[10];
static int SSIndex = 0;
void PopState(void)
{
	if (SSIndex<=0) SSIndex = 1;
	nRdMode = StateStack[--SSIndex];
	switch (nRdMode)
		{
		case RM_BODY:
		case RM_NORMAL:
			SetFont(SF_CODE);
			nWrMode = WM_NORMAL;
			nCurrMargin = 750;
			break;
		}
}
void PushState(int nNewState)
{
	StateStack[SSIndex++] = nRdMode;
	nRdMode = nNewState;
	switch (nRdMode)
		{
		case RM_STRING:
			SetFont(SF_STRING);
			nWrMode = WM_STRING;
			break;
		case RM_LINECOMMENT:
		case RM_STARCOMMENT:
				/* decide on which of two comment methods to use */
			if (fOnlyWhiteSpace)
				{
				XMoveTo(nXGreyStart = nXPos);
				PaintGrey();
				nWrMode = WM_BODYCOMMENT;
				}
			else
				{
				JamLeft();
				nCurrMargin = nXPos;
				nWrMode = WM_LINECOMMENT;
				}
			SetFont(SF_LINECOMMENT);
			break;
		}
}

void Expand(char *sz)
{
	char 	temp[80];
	char 	*p;
	strcpy(temp,sz);
	if (nPrinterType==PT_POSTSCRIPT)
		for (p=temp; *p; p++)
			{
			switch(*p)
				{
				case '(': case ')':
				case '\\': *sz++ = '\\'; *sz++ = *p; break;
				default: *sz++ = *p; break;
				}
			}
	else
		for (p=temp; *p; p++)
			{
			switch(*p)
				{
				case '{': case '}': 
				case '\\': *sz++ = '\\'; *sz++ = *p; break;
				default: *sz++ = *p; break;
				}
			}

	*sz='\0';
}

/*SDOC*************************************************************

	Name: PrintHelp

	Action: just dumps out a synopsis of the program

	Returns: void

*************************************************************EDOC*/
void PrintHelp(void)
{
	printf("\n\nSyntax: ");
	printf("\n    PDOC [-L] | [-P] | [-R] [-n] [-?] sourcefile destfile");
	printf("\n       -L uses LaserJet (default)");
	printf("\n       -P uses PostScript codes");
	printf("\n       -R uses RTF codes");
	printf("\n       -n sets the tab width to n");
	printf("\n       -? help");
	printf("\n\nPDOC formats the sourcecode into printer commands and puts the");
	printf("\nresult in dest file. Different sections of the source are highlighted");
	printf("\nby different fonts, and by specific positioning:");
	printf("\n  keywords-------->italic");
	printf("\n  preprocessor---->bold");
	printf("\n  inline comment-->times, in left margin");
	printf("\n  bulk comment---->times, in grey box");
	printf("\n  strings--------->courier");
	printf("\n  function defn--->bold and big");
	printf("\n  code------------>helvetica");
	exit(0);
}

#ifdef _MSC_VER
_setenvp(){

	}
#endif
