/*
    pcmouse.h - PC Mouse Class - Freeware for Borland C++
    Copyright 1994, John W. Small, All Rights Reserved
    voice: (703) 759-3838,  email: john.small@wdn.com

    Works consulted:

    "Microsoft Mouse Programmer's Reference. 2nd Edition"
        Bellevue, Washington: Microsoft Press, 1991.

    "Microsoft Mouse Programmer's Reference."
        Bellevue, Washington: Microsoft Press, 1989.

    "Microsoft Mouse Programmer's Reference Guide."
        Bellevue, Washington: Microsoft Press, 1986.

    Refer to above works for mouse function documentation.
*/

#ifndef pcmouse_h
#define pcmouse_h


#include <dos.h>

typedef  void interrupt (*MouseDriver)(...);
typedef  void interrupt (*MouseHandler)(...);

#define MOUSE_INT 0x33

// mouse button designators
#define MBleft            1
#define MBright           2

// mouse events for callMask, altCallMask,
// eventMask, eventFlags
#define MEmoved           1
#define MEleftPressed     2
#define MEleftReleased    4
#define MErightPressed    8
#define MErightReleased  16
#define MEshiftPressed   32
#define MEctrlPressed    64
#define MEaltPressed    128

// mouse types
#define MTunknown         0
#define MTbus             1
#define MTserial          2
#define MTinPort          3
#define MTps2             4
#define MThp              5

// mouse error message languages
#define MLenglish         0
#define MLfrench          1
#define MLdutch           2
#define MLgerman          3
#define MLswedish         4
#define MLfinnish         5
#define MLspanish         6
#define MLportuguese      7
#define MLitalian         8

// mouse driver types
#define MDcom             0
#define MDsys             1
#define MDunknown         2

// mouse cursor types
#define MCsoftText        0
#define MChardText        1
#define MCgraphics        2
#define MCunknown         3

// mouse interupt rates
#define MI0               0
#define MI30              1
#define MI50              2
#define MI100             3
#define MI200             4
#define MIunknown         5


struct MouseAccelerationCurves  {
    unsigned char curveLength[4];
    unsigned char mouseCount[4][32];
    unsigned char scaleFactor[4][32];
    char curName[4][16];
};
struct MouseSwitches  {
    unsigned char PortType;
    unsigned char language;
    unsigned char horzSens;
    unsigned char vertSens;
    unsigned char doubleThres;
    unsigned char ballisticCurve;
    unsigned char intrRate;
    unsigned char cursorMask;
    unsigned char laptopAdj;
    unsigned char memoryType;
    unsigned char SVGA;
    unsigned char rotationAngle;
    unsigned char primaryButton;
    unsigned char secondaryButton;
    unsigned char clickLock;
    MouseAccelerationCurves mac;
};


// Do not instantiate the MicrosoftMouse class!
// Reference MM.??? instead!
// Call MM.open() to initialize the mouse for your program.
// Call MM.close() when your program is finished with the mouse.
// Write appropriate fields in MM and call member function
// or call function and read appropriate fields.
// Use physical screen coordinates for all values not
// the mouse virtual coordinates.  Conversion to/from
// mouse virtual coordinate is automatically handled
// by the class member functions.

class MicrosoftMouse  {

    static unsigned char Xcell[];
    static unsigned char Ycell[];
    static unsigned char LeftTopOfs[];

    // MF0, MF33 : Video mode at reset/softReset
    // Call reset() after all video mode changes!
    unsigned vmode;    // MF40
    // reset() and softReset() call vmodeUpdate()
    void vmodeUpdate();

    // MF21, MF22, MF23 : Used by save and restore
    void * origState, * state;
    unsigned stateSize;

    // MF31, MF32 : Used by off/on
    MouseDriver mouseIntrVector;

public:
    // graphics cursor masks
    static unsigned masks_arrow[];
    static signed char arrow_horzHS;
    static signed char arrow_vertHS;
    static unsigned masks_hand[];
    static signed char hand_horzHS;
    static signed char hand_vertHS;

    // MF0 : Set by reset
    unsigned present;
    unsigned buttons;

    // MF3, MF5, MF6 : Button Status
    unsigned leftPressed, rightPressed;

    // MF3, MF4 : Mouse Positon
    unsigned x, y;
    // X & Y are physical, not virtual coordinates!
    // Text modes upper left corner : 1,1
    // Graphics modes upper left corner : 0,0

    // MF5 : Button Press Information
    unsigned lastLeftPressX, lastLeftPressY, leftPresses;
    unsigned lastRightPressX, lastRightPressY, rightPresses;
    // X & Y are physical, not virtual coordinates!

    // MF5, MF6 : Button Requested
    unsigned buttonRequested;

    // MF6 : Button Release Information
    unsigned lastLeftReleaseX, lastLeftReleaseY, leftReleases;
    unsigned lastRightReleaseX, lastRightReleaseY, rightReleases;
    // X & Y are physical, not virtual coordinates!

    // MF7,8 : trap
    unsigned x1, y1, x2, y2;  // MF49

    // MF11 : Mouse motion counters
    unsigned vertMickeys, horzMickeys;

    // MF16 : condOff coordinates
    unsigned cox1, coy1, cox2, coy2;

    // MF12, MF20 : set/swapInterrupt(s) mask/addr
    MouseHandler handler; // See MMautoEventHandler
    unsigned callMask;
    // Also used by autoEventUpdate.

    // Set by Mouse Event Handler.
    unsigned eventMask, eventFlags, eventCount, eventMoved;
    long eventTime;
    long clickTimeOut;
    long leftClickTime;
    unsigned leftClicks;
    long rightClickTime;
    unsigned rightClicks;

    // MF24, MF25 : set/getAltInterrupt mask/addr
    MouseHandler altHandler; // See MMautoEventHandler
    unsigned altCallMask;
    // MF25 can read those set by MF20!

    // MF26, MF27 : set/getSensitivity
    unsigned horzPercent, vertPercent, doublePercent;

    // MF29, MF30 : set/getCRTpage
    unsigned crtPage;

    // MF34, MF35 : Used by get/setLanguage
    unsigned language;

    // MF36 : Set by getDriverVer
    unsigned driverVersionMajor, driverVersionMinor;
    unsigned IRQ;
    unsigned typeRequired;   // MF42

    // MF37 : Set by getDriverInfo
    unsigned driverType;
    unsigned MDDs;
    unsigned cursorType;     // MF9,10
    unsigned intrRate;       // MF28

    // MF38 : Set by getMaxCoor
    unsigned maxX, maxY, maxVX, maxVY;
    unsigned disabled;

    // MF39 : Set by getMaskKey
    unsigned accumVertMickeys, accumHorzMickeys;
    unsigned scrMask, curMask;    // MF9

    // MF41 : Set by getVmode
    char * far vmodeString;

    // MF42 : Set by getCursor
    signed char horzHS, vertHS;   // MF9
    int hidden;  // MF0, 1, 2, 33 and not 16!

    // MF48 : Set by get/setBallPointInfo
    int ballPointRotationAngle;
    unsigned ballPointButtonState;
    unsigned ballPointButtonMasks;

    void open();
    void close();

    void reset();                                               /* MF0 */

    void show() { _AX = 1; geninterrupt(MOUSE_INT);
        if (hidden) hidden++; }                                 /* MF1 */
    void hide() { _AX = 2; geninterrupt(MOUSE_INT);
        hidden--; }                                             /* MF2 */
    void updateStatusInfo();                                    /* MF3 */
    void gotoxy();                                              /* MF4 */
    void updatePressInfo();                                     /* MF5 */
    void updateReleaseInfo();                                   /* MF6 */
    void trap();                                           /* MF7, MF8 */
    void graphicsCursor(                                        /* MF9 */
        signed char horzHS = MicrosoftMouse::arrow_horzHS,
        signed char vertHS = MicrosoftMouse::arrow_vertHS,
        unsigned far *scrCurMask = MicrosoftMouse::masks_arrow);
    void handGraphicsCursor()
        {
            graphicsCursor(
                MicrosoftMouse::hand_horzHS,
                MicrosoftMouse::hand_vertHS,
                MicrosoftMouse::masks_hand
            );
        }
    void textCursor(unsigned scrMask = 0x77FF,                  /* MF10 */
        unsigned curMask = 0x7700,
        unsigned type = MCsoftText);
    void hardTextCursor()
        { textCursor(0,((vmode == 7)? 13 : 7),
        MChardText); }
    void updateMotionInfo()                                     /* MF11 */
        { _AX = 11; geninterrupt(MOUSE_INT);
        horzMickeys = _CX; vertMickeys = _DX; }
    // MF20 is preferred to MF12
    void setInterrupt()  {                                      /* MF12 */
        _ES = FP_SEG(handler);
        _DX = FP_OFF(handler);
        _CX = callMask;
        _AX = 12;
        geninterrupt(MOUSE_INT);
    }
    void lightPenOn()                                           /* MF13 */
        { _AX = 13; geninterrupt(MOUSE_INT); }
    void lightPenOff()                                          /* MF14 */
        { _AX = 14; geninterrupt(MOUSE_INT); }
    void speed(unsigned horz, unsigned vert)                    /* MF15 */
        { _CX = horz; _DX = vert; _AX = 15;
        geninterrupt(MOUSE_INT); }
    void condOff();                                             /* MF16 */
    void doubleSpeedThreshold(unsigned mickeys)                 /* MF19 */
        { _DX = mickeys; _AX = 19;
        geninterrupt(MOUSE_INT); }
    void swapInterrupts();                                      /* MF20 */
    void autoEventUpdate();                                     /* N/A  */
    void save();                                          /* MF21, MF22 */
    void restore();                                             /* MF23 */
    void setAltInterrupt()                                      /* MF24 */
    {
        _ES = FP_SEG(altHandler);
        _DX = FP_OFF(altHandler);
        _CX = altCallMask;
        _AX = 24;
        geninterrupt(MOUSE_INT);
    }
    void getAltInterrupt()                                      /* MF25 */
    {
        _CX = altCallMask;
        _AX = 25;
        geninterrupt(MOUSE_INT);
        altHandler = (MouseHandler) MK_FP(_BX,_DX);
    }
    void clearAltInterrupts();                                  /* N/A  */
    void setSensitivity()                                       /* MF26 */
        { _DX = doublePercent; _CX = vertPercent;
        _BX = horzPercent; _AX = 26;
        geninterrupt(MOUSE_INT); }
    void getSensitivity()                                       /* MF27 */
        { _AX = 27; geninterrupt(MOUSE_INT);
        horzPercent = _BX; vertPercent = _CX;
        doublePercent = _DX; }
    void setInterruptRate(unsigned rate);                       /* MF28 */
    void setCRTpage()                                           /* MF29 */
        { _BX = crtPage; _AX = 29;
        geninterrupt(MOUSE_INT); }
    void getCRTpage()                                           /* MF30 */
        { _AX = 30; geninterrupt(MOUSE_INT);
        crtPage = _BX; }
    void off();                                                 /* MF31 */
    void on();                                                  /* MF32 */
    void softReset();                                           /* MF33 */
    void setLanguage()                                          /* MF34 */
        { _BX = language; _AX = 34;
        geninterrupt(MOUSE_INT); }
    void getLanguage()                                          /* MF35 */
        { _AX = 35; geninterrupt(MOUSE_INT);
        language = _BX; }
    void getDriverVer();                                        /* MF36 */
// All MF calls > 36 are checked against driver version
    int getDriverInfo();                                        /* MF37 */
    int getMaxCoor();                                           /* MF38 */
    int getMaskKey();                                           /* MF39 */
    int setVmode(unsigned vmode,
        unsigned fontSize = 0);                                 /* MF40 */
    unsigned getVmode(unsigned nextFirst);                      /* MF41 */
    unsigned getFirstVmode() { return getVmode(0); }
    unsigned getNextVmode()  { return getVmode(1); }
    int getCursor();                                            /* MF42 */
    int loadAC(MouseAccelerationCurves far * MAC,
        int curve = -1);                                        /* MF43 */
    //  returns active curve number
    int readAC(MouseAccelerationCurves far ** MAC);             /* MF44 */
    //  returns active curve number
    int setGetAAC(int curve = -1,
        char far ** curveName = (char far **)0);                /* MF45 */
    int hardReset();                                            /* MF47 */
    int setBallPoint();                                         /* MF48 */
    int getBallPoint();                                         /* MF48 */
    int getTrap();                                              /* MF49 */
    unsigned getActAdvFncs();                                   /* MF50 */
    int getSwitches(MouseSwitches far * MS);                    /* MF51 */
    char far * getMouse_ini();                                  /* MF52 */

    unsigned virtualX(unsigned x)  {
        return ((x - LeftTopOfs[vmode])
        * Xcell[vmode]); }
    unsigned virtualY(unsigned y)  {
        return ((y - LeftTopOfs[vmode])
        * Ycell[vmode]); }
    unsigned physicalX(unsigned x)  {
        return (x / Xcell[vmode]
        + LeftTopOfs[vmode]); }
    unsigned physicalY(unsigned y)  {
        return (y / Ycell[vmode]
        + LeftTopOfs[vmode]); }

};  // MicrosoftMouse

extern MicrosoftMouse MM;

#endif
