/*************************************************************************
**           Paradox-Like Picture Field Input Processing
**************************************************************************
**                                                                      **
**                                                                      **
**  Copyright (c) 1992  Flexible Information Systems, Inc.              **
**                                                                      **
**    This code may be freely used by any programmer                    **
**    including royalty free inclusion in any commercial                **
**    application, but any commercial rights to the source              **
**    code or object files of this code is are reserved.                **
**                                                                      **
**    This code is supplied strictly as-is, and FIS, Inc. and the       **
**    author assume no responsibility for the accuracy, use or fitness  **
**    for a particular purpose                                          **
**                                                                      **
**      Author:         Ken Vogel                                       **
**      CIS Id:         74007,564                                       **
**      Filename:       parapict.hpp                                    **
**      Prefix:         PPIC_                                           **
**      Date:           24-Mar-92                                       **
**                                                                      **
**      Description:    A set of recursive C++ classes which process
**                      data entry with Paradox-like picture formats.
**
**                      The classes in this file do not actually perform
**                      any keyboard or display functions, but should be
**                      integrated into a data entry system.  This allows
**                      them to be used in TurboVision, Windows, etc.
**
**                                                                      **
**************************************************************************/


#if 0
    ---->>> Revision History <<<----
    ---->>> Revision History <<<----
#endif


#ifndef PPIC_Included
#define PPIC_Included

// Define the Boolean class if necessary
#if !True
typedef enum { False, True } Boolean;
#endif

/*--------------------------- Module Constants ---------------------------*/
//
// These are the "internal" representations of the formatting characters.
// They are defined this way to distinguish them from literals, as they
// occur together in the picture strings.
//
// Don't change these without also changing the function PPIC_NextChar
//
#define PPIC_cRepetition   1
#define PPIC_cOptionOpen   2
#define PPIC_cOptionClose  3
#define PPIC_cGroupOpen    4
#define PPIC_cGroupClose   5
#define PPIC_cAlternate    6

#define PPIC_cMaxSpecialChar   6  // All chars <= this are special
                                  // Non-formatting indicator characters

#define PPIC_cDigit        7
#define PPIC_cLetter       8
#define PPIC_cUpperLetter  9
#define PPIC_cAny          10
#define PPIC_cUpperAny     11

#define PPIC_cMaxFormatChar 11  // All chars <= this are special characters

/**************************************************************************/


/*--------------------------- Macro Functions ---------------------------*/
/**************************************************************************/


/*--------------------------- Return Constants ---------------------------*/
/**************************************************************************/


/*--------------------------- Module Constants ---------------------------*/
/**************************************************************************/


/*--------------------------- Exported Variables -------------------------*/
/**************************************************************************/


/*--------------------------- Class Definitions -------------------------------*/
class PPIC_ElementClass;

// The top-level picture class.  All interface to this function should
// occur through an object of this class.

class PPIC_PictureClass
  {
    protected:
        PPIC_ElementClass *thePicture;

    public:
        // True after creation if the picture is valid & ready to go
        Boolean valid;

        // The constructor for the picture parser, which accepts the input
        // picture string.  Check valid after constructing to see if it
        // worked.
        PPIC_PictureClass (const char *PictureP);

        ~PPIC_PictureClass();

        // Process a letter (either a keypress, or one that's already
        //                   in the input stream)
        //
        // Input:  Letter is the letter to process
        //
        //         IsKeypress is true if this is a keypress (enables
        //         auto-fill of literal data.  False for reprocessing
        //         already-typed information
        //
        //         DestP is a target for the resultant keys
        //         MaxLength is the max # of chars which may be returned
        //
        // Output:
        //         DestP has a characters copied into it
        // Returns:
        //         The # of characters copied into DestP (0 = invalid key)
        //         *DoneP is True if input is complete (after c/r)
        //
        int ProcessLetter( char    Letter
                         , Boolean *DoneP
                         , Boolean IsKeypress
                         , char    *DestP
                         , int     MaxLength );

        // Reprocess the input string -- call this function after the user
        // changes the string directory (without a backspace, etc.)
        // Returns:  Ptr to the first character in the string which has
        //           a problem (fails the process assignment)
        char *ReprocessString (char *StringP);

        // Reset the input state -- call if user restarts assignment.
        void ResetState( void );
  };


// A simple list of elements.  If you are using TurboVision, you may
// replace this with TNSCollection with the same effect.
//
class PPIC_ElementListClass
  {
    protected:
        PPIC_ElementClass    **dElementsCPP;
        int                  dCount;

    public:
        PPIC_ElementListClass();
        ~PPIC_ElementListClass();
        PPIC_ElementClass *at (int Index);
        void insert (PPIC_ElementClass *ElementP);
        void forEach ( void (*actionFun)(PPIC_ElementClass *));
        int getCount()
            { return dCount; }
  };


// An element of a picture.  The descendants of this class implement
// each of the different kinds of elements that can appear in a picture.
// They can maintain state, which is used during input processing.
class PPIC_ElementClass
  {
    public:
        virtual ~PPIC_ElementClass() {}

        // Reset state to start of input
        virtual void ResetState( void ) =0;

        // Process a letter (either a keypress, or one that's already
        //                   in the input stream)
        //
        // Input:  Letter is the letter to process
        //
        //         IsKeypress is true if this is a keypress (enables
        //         auto-fill of literal data.  False for reprocessing
        //         already-typed information
        //
        //         DestP is a target for the resultant keys
        //         MaxLength is the max # of chars which may be returned
        //
        // Output:
        //         DestP has a characters copied into it
        // Returns:
        //         The # of characters copied into DestP (0 = invalid key)
        //         *DoneP is True if input is complete (field is full)
        virtual int ProcessLetter( char    Letter
                                     , Boolean *DoneP
                                     , Boolean IsKeypress
                                     , char    *DestP
                                     , int     MaxLength ) =0;
  };

// A sequence of other picture elements.
class PPIC_SequenceClass : public PPIC_ElementClass
  {
    protected:
        // A list of other elements which make up the sequence.
        PPIC_ElementListClass *dElementListCP;

        // The cursor to the current element which is "working" on input
        // (This is part of the input state)
        int              dCurrentEl;

    public:
        // Create a sequence from a single other element
        PPIC_SequenceClass ( PPIC_ElementClass  *ElementCP);

        // Add an element to the end of a sequence
        void AddElement (PPIC_ElementClass *ElementCP);

        virtual ~PPIC_SequenceClass ();

        // Reset state to start of input (defined in PPIC_ElementClass)
        virtual void ResetState( void );

        // Process a letter (defined in PPIC_ElementClass)
        virtual int ProcessLetter( char    Letter
                                     , Boolean *DoneP
                                     , Boolean IsKeypress
                                     , char    *DestP
                                     , int     MaxLength );
  }; // PPIC_SequenceClass


// An list of several alternative picture elements
class PPIC_AlternateClass : public PPIC_ElementClass
  {
    protected:
        // A list of other PPIC_ElementClass * which make up the alternatives
        PPIC_ElementListClass *dAlternativeListCP;

        // The cursor to the selected element.  If it is -1, then we have
        // not yet selected an element
        int              dSelectedEl;

    public:
        // Create an alternate from a single other element
        PPIC_AlternateClass (PPIC_ElementClass  *ElementCP);

        // Add an element to the end of an alternate
        void AddElement (PPIC_ElementClass *ElementCP);

        virtual ~PPIC_AlternateClass ();

        // Reset state to start of input (defined in PPIC_ElementClass)
        virtual void ResetState( void );

        // Process a letter (defined in PPIC_ElementClass)
        virtual int ProcessLetter( char    Letter
                                     , Boolean *DoneP
                                     , Boolean IsKeypress
                                     , char    *DestP
                                     , int     MaxLength );
  }; // PPIC_AlternateClass

// An optional element.  Also contains the *next* element, so we can
// decide if we want to pass over ourselves or not.
class PPIC_OptionalClass : public PPIC_ElementClass
  {
    protected:
        PPIC_ElementClass  *dLeftElementCP;
        PPIC_ElementClass  *dRightElementCP; // May be 0 (NULL)

        // The state is whether Left is done with it's input
        enum { PPIC_fStarting
             , PPIC_fLeftActive
             , PPIC_fRightActive }    dState;

    public:
        // Construct an option from an element and the next
        PPIC_OptionalClass ( PPIC_ElementClass *LeftCP
                          , PPIC_ElementClass *RightCP );
        virtual ~PPIC_OptionalClass ();

        // Reset state to start of input (defined in PPIC_ElementClass)
        virtual void ResetState( void );

        // Process a letter (defined in PPIC_ElementClass)
        virtual int ProcessLetter( char    Letter
                                     , Boolean *DoneP
                                     , Boolean IsKeypress
                                     , char    *DestP
                                     , int     MaxLength );
  }; // PPIC_OptionalClass

// A string of any number of formatting characters (literals or pictures).
// The reason we combine these two types of elements is because literals
// behave differently if they occur immediately following formatting
// characters (they must auto-fill).  If the formatting characters are
// enclosed in grouping brackets {}, then they will be a separate element.
class PPIC_FormatClass : public PPIC_ElementClass
  {
    protected:
        // The formatting string.  Note that picture characters are
        // represented by the constants at the beginning of the file.
        char         *dFormatStringOP;

        // A pointer into the above string -- the next formatting character.
        char         *dCursorP;

    public:
        PPIC_FormatClass (const char *FormatStrP);
        virtual ~PPIC_FormatClass ();

        // Reset state to start of input (defined in PPIC_ElementClass)
        virtual void ResetState( void );

        // Process a letter (defined in PPIC_ElementClass)
        // Returns the # of chars stored
        virtual int ProcessLetter( char    Letter
                                 , Boolean *DoneP
                                 , Boolean IsKeypress
                                 , char    *DestP
                                 , int     MaxLength );
  };

// A repetition count of another element.
class PPIC_RepetitionClass : public PPIC_ElementClass
  {
    protected:
        // The number of times to repeat.  If -1, then repeat 0 or more
        int          dRepeatCount;

        // The # of repetitions left (-1 = infinite)
        int          dRepsLeft;

        // The element being repeated
        PPIC_ElementClass *dElementCP;

    public:
        PPIC_RepetitionClass (int Count, PPIC_ElementClass *ElementCP);
        virtual ~PPIC_RepetitionClass ();

        // Reset state to start of input (defined in PPIC_ElementClass)
        virtual void ResetState( void );

        // Process a letter (defined in PPIC_ElementClass)
        virtual int ProcessLetter( char    Letter
                                     , Boolean *DoneP
                                     , Boolean IsKeypress
                                     , char    *DestP
                                     , int     MaxLength );
  }; // PPIC_RepetitionClass

/**************************************************************************/

#endif /* Check for AlreadyIncluded */
