#define OWS_OPEN_CPP    // Include static information
//===================================================================
//  *** Windows ofstream emulation - Print Stream class
//  This class defines a Windows print stream which provides the same
//  functionality as the standard C++ ofstream class on a Windows
//  printer device.
//
//  Author - P. J. Weber
//  Eveready Battery Company, Inc.
//
//  $log$
//
//===================================================================
#include <owstream.h>
#include <mem.h>

//===================================================================
// "Sink" stream buffer constructors and destructors
//===================================================================
//===================================================================
//  Constructor with document information
//
_Cdecl wpFilebuf::wpFilebuf(
    HWND hWnd,
    char *docName,
    long val_1,
    long val_2,
    char *docFile) : streambuf()
{
    winHandle = hWnd;
    inited = Initialize();
    open(docName, val_1, val_2, docFile);
}
//===================================================================
//  Constructor without document information, but with a window handle
//
_Cdecl wpFilebuf::wpFilebuf(HWND hWnd) : streambuf()
{
    winHandle = hWnd;
    inited = Initialize();
}
//===================================================================
//  Constructor without arguments.
//
_Cdecl wpFilebuf::wpFilebuf() : streambuf()
{
    winHandle = NULL;
    inited = Initialize();
}
//===================================================================
//  Common Constructor initialization.
//
BOOL wpFilebuf::Initialize(void)
{
    BOOL ok;

    error = 0;
    opened = FALSE;
// Set the buffer information ...
    ioBuffer = GlobalAlloc(GMEM_MOVEABLE, IOBUFFLEN + 16);
    setbuf((signed char _FAR *)NULL, 0);
    ok = (ioBuffer != NULL);
// Initialize structures
    memset(&pd, 0, sizeof(PRINTDLG));
    pd.lStructSize = sizeof(PRINTDLG);
    pd.hwndOwner = winHandle;
    _fAbort = NULL;
    hAbortWnd = NULL;
    WPFBUF = NULL;
// Initialize printer dialog...
    return ok;
}
//===================================================================
//  Open document to be printed.
//
void wpFilebuf::open(
        char *docName,              // Document Name
        long flagWord,               // Flag word
        long dlgFlag,               // File flag word
        char *outFile)              // Output file name
{
    if (!opened)
    {
        cancelDlg = FALSE;
        if (dlgFlag & OWS_CANCELBOX)
        {
            if (winHandle) cancelDlg = TRUE;
            dlgFlag &= ~OWS_CANCELBOX;
        }
        pd.Flags = PD_RETURNDC | dlgFlag;
        if (inited && PrintDlg(&pd))
        {
// Set page height in pixels
            pageHt = GetDeviceCaps(pd.hDC, VERTRES);
            SetTextAlign(pd.hDC, TA_LEFT|TA_TOP);
            di.cbSize = sizeof(DOCINFO);
            di.lpszDocName = docName;
            di.lpszOutput = outFile;
            opened = TRUE;
// Start the document
            if (StartDoc(pd.hDC, &di) > 0)
            {
// Start the first page
                pageStarted = FALSE;
// Get the text metrics and set charHt.
                TEXTMETRIC tm;
                GetTextMetrics(pd.hDC, &tm);
                charHt = tm.tmHeight + tm.tmExternalLeading;
                xPos = yPos = 0;
                if (cancelDlg)
                {
                    _fAbort = FALSE;
                    WPFBUF = this;
// Create THUNKS for the callback functions
                    HINSTANCE hInstance = GetWindowWord(winHandle, GWW_HINSTANCE);
                    lpAbortDlg = MakeProcInstance((FARPROC) AbortDlg,  hInstance);
                    lpAbortProc = MakeProcInstance((FARPROC)AbortProc, hInstance);
// Set the print manager's abort process;
                    SetAbortProc(pd.hDC, (ABORTPROC) lpAbortProc);
// Display the user cancel dialog box
                    hAbortWnd = CreateDialog(hInstance, "CancelPrint",
                        winHandle, (DLGPROC) lpAbortDlg);
                    if (hAbortWnd)
                    {
                        ShowWindow(hAbortWnd, SW_RESTORE);
                        UpdateWindow(hAbortWnd);
                        EnableWindow(winHandle, FALSE);
                    }
                }
            }
            else    // Message - StartDoc Failed!
            {
                _fAbort = TRUE;
                error = OWS_DOCERROR;
                MessageBox(
                    winHandle,
                    "Could not create document.",
                    "Print Error",
                    MB_OK | MB_ICONINFORMATION);
            }
        }   // End PrintDlg if statement
        else    // Message - Print Dialog Failed!
        {
            opened = FALSE;
            error = OWS_OPENERROR;
            MessageBox(
                winHandle,
                "Could not open Print Device.",
                "Print Error",
                MB_OK | MB_ICONINFORMATION);
        }
    }
}
//===================================================================
//  Get the printer device context.
//
HDC wpFilebuf::GetDC()
{
    if (opened)
        return(pd.hDC);
    else
        return NULL;
}
//===================================================================
// Set the printer font (Using LOGFONT)
//
void wpFilebuf::SetFont(PLOGFONT newFont)
{
    if (opened)
    {
        printFont = CreateFontIndirect(newFont);
        UseFont(printFont);
    }
}
//===================================================================
// Set the printer font (Using LOGFONT)
//
void wpFilebuf::UseFont(HFONT font)
{
    if (opened)
    {
        TEXTMETRIC tm;
        SelectObject(pd.hDC, font);
        GetTextMetrics(pd.hDC, &tm);
        charHt = tm.tmHeight + tm.tmExternalLeading;
    }
}
//===================================================================
//  Close a document
//
void wpFilebuf::close(void)
{
    if (opened)
    {
        overflow(NULL);
        if (!_fAbort)  // Don't perform this procedure if user aborted.
        {
            if (pageStarted) EndPage(pd.hDC);
            EndDoc(pd.hDC);
        }
        if (cancelDlg)  // Was a cancel dialog box present?
        {
            if (hAbortWnd && !_fAbort)  // If the user aborted, this already occured
            {
                DestroyWindow(hAbortWnd);       // Close Dialog, Enable parent
                EnableWindow(winHandle, TRUE);
                SetFocus(winHandle);
                hAbortWnd = NULL;
            }
            FreeProcInstance(lpAbortDlg);   // Free THUNKS
            FreeProcInstance(lpAbortProc);
        }
        if (printFont) DeleteObject(printFont);
        opened = FALSE;
    }
}
//===================================================================
//  "Sink" destructor.
//
_Cdecl wpFilebuf::~wpFilebuf(void)
{
    close();
    if (inited)
    {
        GlobalUnlock(ioBuffer);
        GlobalFree(ioBuffer);
    }
    if (pd.hDC != NULL)         DeleteDC(pd.hDC);
    if (pd.hDevMode != NULL)    GlobalFree(pd.hDevMode);
    if (pd.hDevNames != NULL)   GlobalFree(pd.hDevNames);
}
//===================================================================
// fAbort status function
//
BOOL wpFilebuf::fAbort(void)
{
    return WPFBUF->_fAbort;
}
//===================================================================
// Abort Printing Dialog box procedure
//
BOOL FAR PASCAL _export wpFilebuf::AbortDlg(
    HWND hWnd,
    UINT msg,
    WPARAM wParam,
    LPARAM lParam)
{
    if (msg == WM_COMMAND)
    {
        if (wParam == IDCANCEL)     // User pushed the CANCEL button!
        {
            WPFBUF->_fAbort = TRUE;
            DestroyWindow(hWnd);        // Destroy this window
            EnableWindow(WPFBUF->winHandle, TRUE);  // Enable parent!
            SetFocus(WPFBUF->winHandle);// Get Windows Focus
            WPFBUF->hAbortWnd = NULL;   // Set window pointer to NULL.
            return TRUE;                // Handled message
        }
    }
    else if (msg == WM_INITDIALOG)
    {
        SetFocus(hWnd); // Get Windows Focus
        SetDlgItemText(hWnd, IDC_FILENAME, WPFBUF->di.lpszDocName);
        return TRUE;    // Handled message
    }
    return FALSE;       // Did not handle message
}
//===================================================================
// Abort Printing Check for Abort messages.
//
BOOL CALLBACK _export wpFilebuf::AbortProc(
    HDC hdcPrinter,
    int code)
{
    MSG msg;
    HWND hWnd = WPFBUF->hAbortWnd;
    while (PeekMessage((LPMSG) &msg, NULL, NULL, NULL, PM_REMOVE))
    {
        if (!IsDialogMessage(hWnd, (LPMSG) &msg))
        {
            TranslateMessage((LPMSG) &msg);
            DispatchMessage((LPMSG) &msg);
        }
    }
    return (!WPFBUF->_fAbort);
}
