#ifndef OWSTREAM_H
#define OWSTREAM_H
//===================================================================
//  *** Windows ofstream emulation - Print Stream class
//  This class defines a Windows print stream which provides the same
//  functionality as the standard C++ ofstream class on a Windows
//  printer device.
//
//  Author - P. J. Weber
//  Eveready Battery Company, Inc.
//
/*
$log$
*/
//
//===================================================================
#if defined(_Windows) || defined(_WINDOWS)
#include <windows.h>
#include <commdlg.h>
#include <iostream.h>
#include <winprint.h>

#define IOBUFFLEN   516             // Static buffer space
#define OWS_CANCELBOX   0x00000100L // Display a cancel print dialog box
// Error codes
#define OWS_OPENERROR   100         // Could not open printer device
#define OWS_DOCERROR    101         // Could not create a document

//===================================================================
// "Sink" stream - a stream buffer to handle output to a WINDOWS 
//                 printer device context.
// 
class _CLASSTYPE wpFilebuf : public streambuf
{
  public:
// Attributes
    enum dlgFlags
    {
        defaults    = PD_RETURNDEFAULT, // No Dialog box, return defaults
        nowarning   = PD_NOWARNING,     // No "No Default Printer" warning
        setup       = PD_PRINTSETUP,    // Display only the print setup dialog
        cancelbox   = OWS_CANCELBOX,    // Display a cancel box while printing.
                                        // Note:  To use this flag, make sure
                                        //        you add winprint.rc to your
                                        //        resource script.  This file
                                        //        contains the dialog box for
                                        //        canceling a print job.
    };
// Constructors
    _Cdecl wpFilebuf();             // Default constructor.  hWnd = NULL.
    _Cdecl wpFilebuf(
        HWND hWnd);                 // Handle to a window. MAY BE NULL if
                                    //  no dialog boxes are used.
                                    // User must open the document first
    _Cdecl wpFilebuf(
        HWND hWnd,                  // Handle to a window. MAY BE NULL
                                    //  if no dialog boxes will be used.
        char *docName,              // Document Name
        long flagWord = 0,          // Stream buffer flag word (ios)
        long dlgFlag = 0,           // Dialog flag (See enum dlgFlags)
        char *outFile = NULL);      // Output file name
                                    // Performs document opening services
    virtual _Cdecl ~wpFilebuf(void);

    void _Cdecl open(               // Opens a document to begin printing
        char *docName,              // Document Name
        long flagWord = 0,          // Stream buffer flag word (ios)
        long dlgFlag = 0,           // Dialog flag word
        char *outFile = NULL);      // Output file name

    void _Cdecl close(void);        // Closes a document.
    virtual BOOL fAbort(void);      // Did the user abort printing?
    int wperror(void)               // Was there an error?
    { return error; }
// Windows functions...
    HDC GetDC(void);                // Return the printer device context
    void SetFont(                   // Print using a new font...
        HFONT newFont)              // Font Handle to use in printing.
    { UseFont(newFont); }
    void SetFont(                   // Print using a new font
        PLOGFONT newFont);          // Logical font style to use in printing
    void Parent(HWND hWnd)          // Set reference to parent window.
    { winHandle = hWnd; }
// Virtual functions.
    virtual int _Cdecl do_sputn(    // Implemented sputn.
        const char _FAR *string,    // String to place into buffer.
        int length);                // Size of string.
    virtual streambuf * _Cdecl setbuf( // Set initial buffer positions
        signed char _FAR *buffer,   // buffer pointer
        int length);                // buffer length
    virtual int _Cdecl overflow(    // Flush the buffer! (Do actual printing)
        int cVal = EOF);            // Character to be placed in stream.
    virtual int _Cdecl sync(void);  // Establish consistancy between internal
                                    // structures and external representations

  protected:
// Attributes
    HWND winHandle;                 // Handle to the owner window (may be NULL)
    PRINTDLG pd;                    // Print Dialog structure
    DOCINFO di;                     // Document information structure
    int charHt;                     // Character height
    int xPos;                       // Current x position for Device Context
    int yPos;                       // Current y position for Device Context
    int error;                      // Any error which may occur
    long pageHt;                    // Page height for opened device
    BOOL opened;                    // Device opened flag.
    BOOL inited;                    // Initialize returned TRUE.
    BOOL pageStarted;               // Initialized to FALSE.
    HGLOBAL ioBuffer;               // Handle to global buffer area.
// Services
    BOOL winWrite(                  // Handles writes to the hDC.
        const signed char _FAR *lpsString,  // String to write
        int length);                        // Length of string
    int printBuf(               // Handles newline and form feed CTRL
        const char _FAR *string,// buffer to be printed
        int length);            // Size of string. (0 = Null term)
    void NewLine(void);         // Newline handler
    void NewPage(void);         // Newpage handler
    BOOL Initialize(void);      // Initialize the stream sink
  private:
// Attributes
    BOOL _fAbort;               // What the abort button pushed?
    HWND hAbortWnd;             // Handle to the Abort (cancel) dialog
    BOOL cancelDlg;             // Did we create a cancel dialog?
    HFONT printFont;            // Handle to a printing font
    FARPROC lpAbortDlg;         // THUNK to the AbortDlg
    FARPROC lpAbortProc;        // THUNK to the AbortProc
// Services
    void UseFont(               // Print using a new font.
        HFONT font);            // Font to use in printing.
    static BOOL FAR PASCAL _export AbortDlg( // Dialog box callback function
        HWND hWnd,              // Handle to the dialog box
        UINT msg,               // Message to be processed
        WPARAM wParam,          // Word Parameter
        LPARAM lParam);         // Double-word parameter
    static BOOL CALLBACK _export AbortProc(  // Abort printing routine
        HDC hdcPrinter,         // Handle to printer device context
        int code);              // Code word.
};

#ifdef OWS_OPEN_CPP
//===================================================================
// Static pointer to wpFilebuf instance.  Defined only for OWS_OPEN.CPP.
//  Note:  If more than one instance of wpFilebuf exists and uses the
//         dialog boxes, problems WILL occur.  The callback functions
//         use this pointer to obtain needed information.  Please keep
//         this in mind!  Most likely, though, you'll only have one
//         instance at any given time.
//         If you must have more than one instance, you could try using
//         SetWindowLong and GetWindowLong API's to pass this information
//         to the callbacks.  Ahhh, if only you could take the address of
//         a non-static member function! 
//===================================================================
static wpFilebuf FAR *WPFBUF;
#endif

//===================================================================
// "Sink" stream base - an ios subclass to handle output to a WINDOWS 
//                 printer device context.
// 
class _CLASSTYPE wpStreambase : virtual public ios
{
  public:
// Services
    _Cdecl wpStreambase();  // Default constructor
    _Cdecl wpStreambase(    // Constructor using Window Handle
        HWND hWnd);                 // Handle to a window.
                                    // User must open the document first
    _Cdecl wpStreambase(    // Constructor that opens a printer
        HWND hWnd,                  // Handle to a window. MAY BE NULL
        char *docName,              // Document Name
        int flagWord = 0,           // Stream Flag word (see ios)
        int dlgFlag = 0,            // Dialog Flag word (see wpFilebuf)
        char *outFile = NULL);      // Output file name
                                    // Performs document opening services
    void _Cdecl open(       // Opens a document to begin printing
        char *docName,              // Document Name
        int flagWord = 0,           // Stream Flag word (see ios)
        int dlgFlag = 0,            // Dialog flag word (see wpFilebuf)
        char *outFile = NULL)       // Output file name
    { wpBuf.open(docName, flagWord, dlgFlag, outFile); }

    void _Cdecl close(void) // Closes a document.
    { wpBuf.close(); }

    HDC GetDC(void)         // Return a printer device context to the user.
    { return (wpBuf.GetDC()); }

    void SetFont(HFONT font)    // Print using a new font handle
    { wpBuf.SetFont(font); }

    void SetFont(PLOGFONT font) // Print using a pointer to a logical font
    { wpBuf.SetFont(font); }

    void Parent(HWND hWnd)      // Set reference to parent window
    { wpBuf.Parent(hWnd); }

    wpFilebuf _FAR * _Cdecl rdbuf(void) { return &wpBuf; }

  private:
    wpFilebuf wpBuf;
};

//===================================================================
// output stream - an ostream to handle output to a WINDOWS 
//                 printer device context.
// 
class _CLASSTYPE owstream : public wpStreambase, public ostream
{
  public:
// Attributes
// Services
    _Cdecl owstream():     // Constructor - No document information
                                    // User must open the document first
        wpStreambase() { bp = rdbuf(); }
    _Cdecl owstream(       // Constructor - No document information
        HWND hWnd) :                // Handle to a window.
                                    // User must open the document first
        wpStreambase(hWnd) { bp = rdbuf(); }
    _Cdecl owstream(       // Constructor - Document Information
        HWND hWnd,                  // Handle to a window. MAY BE NULL
        char *docName,              // Document Name
        int flagWord = 0,           // Flag word
        int dlgFlag = 0,            // File flag word
        char *outFile = NULL) :     // Output file name
                                    // Performs document opening services
        wpStreambase(hWnd, docName, flagWord, dlgFlag, outFile)
        { bp = wpStreambase::rdbuf(); }
    _Cdecl owstream(       // Constructor - Document Information (no HWND)
        char *docName,              // Document Name
        int flagWord = 0,           // Flag word
        int dlgFlag = 0,            // File flag word
        char *outFile = NULL) :     // Output file name
                                    // Performs document opening services
        wpStreambase(NULL, docName, flagWord, dlgFlag, outFile)
        { bp = wpStreambase::rdbuf(); }
    // Overload rdbuf()
    wpFilebuf _FAR * _Cdecl rdbuf(void) { return wpStreambase::rdbuf(); }

  protected:
// Attributes
// Services
};
#else   // If Windows is not defined, replace owstream with ofstream
#include <fstream.h>
#define owstream    ofstream
#endif  // Only valid for Windows code - else owstream is an ofstream.
#endif
