////////////////////////////////////////////////////////////////////////////
//       File:  TOOLTIP.H
//    Created:  June 1994
//     Author:  Steve Saxon (Compuserve: 100321,2355)
//
//   Modified:  Don Griffin (DWG)
//       Date:  Sept 1994
//
//   Comments:
//
//              This header file declares the TipControlBar and
//              TipStatusBar class.  You must use both to get balloon
//              hints (aka, tool tips) to work.
//
//              The idea here is to catch the virtual SetHintText() call
//              in TStatusBar and do our some magic at that time.  The
//              call to SetHintText() occurs when fly-by hints are
//              enabled and the mouse moves over a gadget.  It is also
//              called when a menu item is selected.
//
//              TipControlBar distinguishes between these two scenarios
//              by monitoring the EvMouseMove messages.  When the mouse
//              moves over the control bar, it receives WM_MOUSEMOVE
//              messages.  TControlBar passes these onto the gadgets in
//              the tool bar, which call TControlBar::SetHintCommand()
//              when fly-by hints are enabled.  Unfortunately, this is
//              not a virtual function.  SetHintCommand() remembers the
//              command ID, and the text is sent to the status bar via
//              SetHintText().
//
//              All TipControlBar has to do is set a flag before calling
//              TControlBar::EvMouseMove() and clear it afterwards.  This
//              will allow the overridden SetHintText() function to know
//              whether the hint text is coming from fly-by hints or a
//              menu item.  The tool tip text follows the normal status
//              bar hint text and is separated by a '\n' character.  The
//              two strings are separated in SetHintText() and the base
//              string goes to TStatusBar::SetHintText() while the end
//              of the string goes into the balloon window.
//
//              The balloon window's SetCaption() method handles the
//              work of showing and hiding the balloon, based on the
//              control's text.  If an empty string (or NULL ptr) is
//              passed to SetCaption(), the balloon is hidden.  If a
//              non-empty string is passed, the balloon window sets a
//              timer for 400ms.  When the timer fires, the balloon is
//              shown.  That is, unless SetCaption() was called in the
//              meantime with an empty caption.
//
//              The only other trick is that SetHintText() is passed a
//              NULL text ptr when the mouse moves from one gadget to
//              another, followed immediately by a second call to
//              SetHintText() with the new hint text.  The first call
//              has the (desired) effect of hiding the balloon.  However,
//              the second call needs to show the balloon without using
//              the normal timer delay.
//
//              Again, EvMouseMove() comes to the rescue!  Another flag
//              is set if the current gadget (pointed to by 'AtGadget')
//              and the gadget under the cursor are different.  This
//              allows the balloon window's SetCaption() to skip the
//              timer and display the balloon immediately when the
//              cursor moves from one gadget to another.
//
////////
#ifndef __TOOLTIP_H
#define __TOOLTIP_H

#include <owl\controlb.h>
#include <owl\statusba.h>
#include <owl\buttonga.h>

class TipControlBar : public TControlBar
{
    protected:

        void 	        EvMouseMove (UINT modKeys, TPoint& point);
        void 	        EvLButtonDown (UINT modKeys, TPoint& point);

        BOOL            m_bEnabled;

    public:

        TipControlBar (TWindow* parent = 0,
                       TTileDirection direction = Horizontal,
                       TFont* font = new TGadgetWindowFont,
                       TModule* module = 0);

        ~TipControlBar ();

        BOOL            SaveSettingsToIni (const char far *szIniFile,
                                           const char far *szSection) const;

        BOOL            GetSettingsFromIni (const char far *szIniFile,
                                            const char far *szSection);

        enum BalloonStyle { Rounded, Square };

        BalloonStyle    GetBalloonStyle () const;
        void            SetBalloonStyle (BalloonStyle newStyle);

        BOOL            IsBallonHelpEnabled () const
                            { return m_bEnabled; }
        void            EnableBallonHelp (BOOL bEnable)
                            { m_bEnabled = bEnable; }

        TColor          GetBalloonBkGndColor () const;
        TColor          GetBalloonTextColor () const;
        TColor          GetBalloonBorderColor () const;

        void            SetBalloonBkGndColor (const TColor &);
        void            SetBalloonTextColor (const TColor &);
        void            SetBalloonBorderColor (const TColor &);

        DECLARE_RESPONSE_TABLE (TipControlBar);
};

class TipStatusBar : public TStatusBar
{
    protected:

        BYTE    m_bFlags;

    public:

        TipStatusBar (TWindow*                 parent = 0,
                      TGadget::TBorderStyle    borderStyle = TGadget::Recessed,
                      UINT                     modeIndicators = 0,
                      TFont*                   font = new TGadgetWindowFont,
                      TModule*                 module = 0);

        ~TipStatusBar();

        virtual void SetHintText (const char *lpszText);

        BOOL            SaveSettingsToIni (const char far *szIniFile,
                                           const char far *szSection) const;

        BOOL            GetSettingsFromIni (const char far *szIniFile,
                                            const char far *szSection);

        BOOL    AreHintsFullLine () const;
        void    SetFullLineHints (BOOL bOn);
};


#endif   // __TOOLTIP_H

