////////////////////////////////////////////////////////////////////////////
//       File:  TOOLTIP.CPP
//    Created:  June 1994
//     Author:  Steve Saxon (Compuserve: 100321,2355)
//
//   Modified:  Don Griffin (DWG)
//       Date:  Sept 1994
//
//   Comments:
//
//              This source file contains a private class for the tip
//              balloon control (BalloonWindow) and the implementations
//              for TipControlBar and TipStatusBar.
//
////////
#include <owl\owlpch.h>
#pragma hdrstop

#include "tooltip.h"

#define MAX_TIPLEN  25
#define TIP_DELAY	400
#define ID_TIMER	1000


static char *f_szDefault = "#$%$$%%$#$";
static char *f_szYes = "Yes";
static char *f_szNo = "No";

BOOL StrIEql (const char *sz1, const char *sz2)
{
    return stricmp (sz1, sz2) == 0;
}

static BOOL ini_GetString (LPCSTR lpszFile, LPCSTR lpszSection,
                           LPCSTR lpszEntry, char *szValue, UINT bufSize)
{
    char buf [200];

    GetPrivateProfileString (lpszSection, lpszEntry, f_szDefault,
                             buf, sizeof(buf), lpszFile);

    if (strcmp (f_szDefault, buf))
    {
        strncpy (szValue, buf, bufSize);
        szValue[bufSize-1] = '\0';
        return TRUE;
    }

    return FALSE;
}

static BOOL ini_GetBool (LPCSTR lpszFile, LPCSTR lpszSection,
                         LPCSTR lpszEntry, BOOL *pBool)
{
    char buf [30];

    *buf = '\0';

    if (ini_GetString (lpszFile, lpszSection, lpszEntry, buf, sizeof(buf)) &&
        pBool)
    {
        *pBool = (StrIEql (buf, f_szYes) ||
                  StrIEql (buf, "on") ||
                  StrIEql (buf, "true") ||
                  StrIEql (buf, "1"));
    }

    return *buf != '\0';
}

static BOOL ini_GetLong (LPCSTR lpszFile, LPCSTR lpszSection,
                         LPCSTR lpszEntry, long *pLong)
{
    char buf [30];

    *buf = '\0';

    if (ini_GetString (lpszFile, lpszSection, lpszEntry, buf, sizeof(buf)))
    {
        char *end;
        long val;

        val = strtol (buf, &end, 10);

        //
        //  We can terminate the integer by EOL, white space or a ';'...
        //  other stuff makes for an invalid integer...
        //
        if (*end != '\0' && *end != ' ' && *end != ';')
            *buf = '\0';
        else
        {
            if (pLong)
                *pLong = val;
        }
    }

    return *buf != '\0';
}

static BOOL ini_GetShort (LPCSTR lpszFile, LPCSTR lpszSection,
                          LPCSTR lpszEntry, short *pShort)
{
    long val;

    if (ini_GetLong (lpszFile, lpszSection, lpszEntry, &val))
    {
        if (pShort)
            *pShort = (short) val;
        return TRUE;
    }
    return FALSE;
}

//-----------------

static BOOL ini_PutString (LPCSTR lpszFile,  LPCSTR lpszSection,
                           LPCSTR lpszEntry, LPCSTR lpszValue)
{
    return WritePrivateProfileString (lpszSection, lpszEntry, lpszValue,
                                      lpszFile);
}

static BOOL ini_PutBool (LPCSTR lpszFile, LPCSTR lpszSection,
                         LPCSTR lpszEntry, BOOL bBool)
{
    return ini_PutString (lpszFile, lpszSection, lpszEntry,
                          bBool ? f_szYes : f_szNo);
}

static BOOL ini_PutLong (LPCSTR lpszFile, LPCSTR lpszSection,
                         LPCSTR lpszEntry, long lLong)
{
    char buf [20];

    return ini_PutString (lpszFile, lpszSection, lpszEntry,
                          ltoa (lLong, buf, 10));
}

static BOOL ini_PutShort (LPCSTR lpszFile, LPCSTR lpszSection,
                          LPCSTR lpszEntry, short sShort)
{
    return ini_PutLong (lpszFile, lpszSection, lpszEntry, (long)sShort);
}


class BalloonWindow;

//-----------------------------
//  Some necessary variables:
//
static BalloonWindow *  f_pBalloon  = 0;
static BOOL             f_bCanPopup = FALSE;
static BOOL             f_bChangeGadgets = FALSE;

////////////////////////////////////////////////////////////////////////////
//      Class:  BalloonWindow
// Base class:  TWindow
// Derivation:  public
//
//   Comments:
//
//              This class is the balloon control.  It is controlled
//              primarily setting the caption.  When the caption is set
//              to NULL, the balloon will disappear.  If the caption is
//              set to some text, a timer is created that will fire in
//              about 1/5 of a second.  When the timer fires (and if
//              the text is non-empty), the balloon will appear below
//              the mouse cursor.  The balloon will stay on the screen
//              until its caption is set to NULL.
//
////////
class BalloonWindow : public TWindow
{
    protected:

        UINT    m_uiTimer;
        TSize   m_sizeText;
        BYTE    m_bSquare;
        TColor  m_clrBorder;
        TColor  m_clrBkGnd;
        TColor  m_clrText;

        void    KillTimer ();
        void    ShowNow ();
        void    PositionBalloon ();

        void    GetWindowClass (WNDCLASS &wc);
        LPSTR   GetClassName ();

        void    Paint (TDC &dc, BOOL erase, TRect &rc);
        void    EvTimer (UINT uiTimerId);

    public:

        BalloonWindow ();
        ~BalloonWindow ();

        void    SetCaption (const char far* title);
        void    HideBalloon ()  { SetCaption (NULL) ; }
        BOOL    IsBalloonSquare () const { return m_bSquare; }
        void    SetSquareStyle (BOOL bSquare) { m_bSquare = bSquare; }

        TColor  GetBkGndColor () const { return m_clrBkGnd; }
        TColor  GetBorderColor () const { return m_clrBorder; }
        TColor  GetTextColor () const { return m_clrText; }

        void    SetBkGndColor  (const TColor &clr) { m_clrBkGnd = clr; }
        void    SetBorderColor (const TColor &clr) { m_clrBorder = clr; }
        void    SetTextColor   (const TColor &clr) { m_clrText = clr; }

	DECLARE_RESPONSE_TABLE (BalloonWindow);
};

DEFINE_RESPONSE_TABLE1 (BalloonWindow, TWindow)
  EV_WM_TIMER,
END_RESPONSE_TABLE;

////////////////////////////////////////////////////////////////////////////
//     Method:  ctor
//      Class:  BalloonWindow
//
//     Params:
//
//              None
//
//   Comments:
//
//              This ctor has no parameters, and calls TWindow::TWindow
//              passing a NULL parent ptr and empty text.  We setup the
//              style bits to contain WS_POPUP and WS_EX_TOPMOST.  We
//              then create the window via TWindow::Create().
//
////////
BalloonWindow::BalloonWindow ()
    :   TWindow (NULL, ""),
        m_bSquare(FALSE),
        m_clrText (0, 0, 0),
        m_clrBorder (0, 0, 0),
        m_clrBkGnd (255, 255, 0)
{
    Attr.Style = WS_POPUP;

    Attr.ExStyle = WS_EX_TOPMOST;

	m_uiTimer = NULL;

    Create ();
}

////////////////////////////////////////////////////////////////////////////
//     Method:  dtor
//      Class:  BalloonWindow
//
//   Comments:
//
//              The dtor will free the timer if it is active before we
//              go to oblivion.
//
////////
BalloonWindow::~BalloonWindow ()
{
	KillTimer ();
}

////////////////////////////////////////////////////////////////////////////
//     Method:  GetWindowClass
//      Class:  BalloonWindow
//
//     Params:
//
//              wc        - Reference to the WNDCLASS object to fill in
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will fill in the WNDCLASS object passed to
//              us.  We call TWindow::GetWindowClass() to fill in the
//              necessary stuff.  After that we add CS_SAVEBITS for a
//              speed boost and set the background brush.  For rounded
//              balloons, we do NOT paint a background and so set the
//              brush to 0.  Otherwise, we create a BRIGHT YELLOW solid
//              brush (Windows will free it).
//
////////
void BalloonWindow::GetWindowClass (WNDCLASS &wc)
{
	TWindow::GetWindowClass (wc);

    wc.style |= CS_SAVEBITS;

	wc.hbrBackground = 0;
}

////////////////////////////////////////////////////////////////////////////
//     Method:  GetClassName
//      Class:  BalloonWindow
//
//     Params:
//
//              None
//
//    Returns:
//
//              LPSTR     - The control class name (must be different
//                          from other OWL windows because we have our
//                          own class styles).
//
//   Comments:
//
//              This method returns the control class name.
//
////////
LPSTR BalloonWindow::GetClassName ()
{
	return "BallonWindow";
}

////////////////////////////////////////////////////////////////////////////
//     Method:  Paint
//      Class:  BalloonWindow
//
//     Params:
//
//              dc        - The TDC to draw on
//
//              BOOL      - Ignored
//
//              TRect &   - Ignored
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will paint the balloon with a BRIGHT YELLOW
//              background using the ANSI_VAR stock font.
//
////////
void BalloonWindow::Paint (TDC &dc, BOOL, TRect &)
{
	char	szText[MAX_TIPLEN+1];
	TRect	client;
    TBrush  brBkGnd (m_clrBkGnd);
    TPen    penBorder (m_clrBorder);

	GetClientRect (client);

    dc.SelectObject (brBkGnd);
    dc.SelectObject (penBorder);

    if (m_bSquare)
        dc.Rectangle (client);
    else
        dc.RoundRect (0,0, client.right,client.bottom, 15, client.bottom);

	// get the tooltip text
	GetWindowText (szText, sizeof (szText));

	// set up the device context
	dc.SetBkMode (TRANSPARENT);
    dc.SetTextColor (m_clrText);
	dc.SelectStockObject (ANSI_VAR_FONT);

	// draw the text
	dc.DrawText (szText, -1, client, DT_CENTER | DT_SINGLELINE | DT_VCENTER);
}

////////////////////////////////////////////////////////////////////////////
//     Method:  SetCaption
//      Class:  BalloonWindow
//
//     Params:
//
//              title         - The new caption
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will set the balloon text.  If the 'title'
//              ptr is NULL or empty, we hide the window.  Otherwise,
//              we get the text extents, hide the balloon and setup our
//              timer to tell us when to show ourselves.
//
////////
void BalloonWindow::SetCaption (const char far* title)
{
	TWindow::SetCaption (title);

	// if the caption is missing, hide the tip window
	if (title == NULL || !*title)
	{
		KillTimer ();
		Show (SW_HIDE);
	}
	else
	{
		// work out the extent of the text
        TClientDC	dc (HWindow);
        int         strLen;

        strLen = lstrlen (title);
        if (strLen > MAX_TIPLEN)
            strLen = MAX_TIPLEN;

        dc.SelectStockObject (ANSI_VAR_FONT);
        m_sizeText = dc.GetTextExtent (title, strLen);

        m_sizeText.cx	+= (m_bSquare ? 5 : 10);
        m_sizeText.cy	+= 4;

        Show (SW_HIDE);

        if (f_bChangeGadgets)
            ShowNow ();
        else
        {
            // create the timer - this will send a WM_TIMER message
            // after 'TIP_DELAY' milliseconds
		    m_uiTimer = SetTimer (ID_TIMER, TIP_DELAY);
        }
	}
}

////////////////////////////////////////////////////////////////////////////
//     Method:  PositionBalloon
//      Class:  BalloonWindow
//
//     Params:
//
//              None
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will calculate the balloon position.  This
//              done by simply getting the current mouse position for
//              the X,Y and using the text size determined inside of
//              SetCaption() for W,H.  The rect is adjusted as needed
//              to keep it entirely on the screen.
//
//              Once the TRect for the window position has been found,
//              SetWindowPos() is called to move the balloon (with the
//              SWP_NOZORDER and SWP_NOACTIVATE flags so that we stay
//              hidden).
//
////////
void BalloonWindow::PositionBalloon ()
{
	TPoint	ptMouse;
	TSize	scr (GetSystemMetrics (SM_CXSCREEN),
                 GetSystemMetrics (SM_CYSCREEN));

	GetCursorPos (ptMouse);

	ptMouse.x	-= 2;
	ptMouse.y	+= 22;

	TRect	rc (ptMouse, m_sizeText);

	// check x screen position
	if (rc.left < 0)
	{
		rc.Offset (-rc.left + 2, 0);
	}
	else
	{
		if (rc.right > scr.cx)
		{
			rc.Offset (scr.cx - rc.right - 2, 0);
		}
	}

	// check y screen position
	if (rc.bottom > scr.cy)
	{
		rc.Offset (0, -42);
	}

	SetWindowPos (NULL, rc, SWP_NOZORDER | SWP_NOACTIVATE);
}

////////////////////////////////////////////////////////////////////////////
//     Method:  KillTimer
//      Class:  BalloonWindow
//
//     Params:
//
//              None
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will free our Windows timer.
//
////////
void BalloonWindow::KillTimer ()
{
	if (m_uiTimer)
	{
		TWindow::KillTimer (ID_TIMER);
		m_uiTimer = NULL;
	}
}

////////////////////////////////////////////////////////////////////////////
//     Method:  ShowNow
//      Class:  BalloonWindow
//
//     Params:
//
//              None
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will call PositionBalloon() to calculate a
//              position for the balloon and will then call Show() and
//              UpdateWindow() to actually show the balloon.
//
////////
void BalloonWindow::ShowNow ()
{
	PositionBalloon ();

	Show (SW_SHOWNA);
	UpdateWindow ();
}

////////////////////////////////////////////////////////////////////////////
//     Method:  EvTimer
//      Class:  BalloonWindow
//
//     Params:
//
//              UINT          - Ignored
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method is called when our timer ticks.  At this
//              point we should show the balloon.
//
////////
void BalloonWindow::EvTimer (UINT)
{
	if (m_uiTimer)
	{
		KillTimer ();
		ShowNow ();
	}
}

////////////////////////////////////////////////////////////////////////////
//
//      Class:  T i p C o n t r o l B a r
//


//  INI entries:
static char *f_szBallonsEnabled = "Balloon_Help_Enabled";
static char *f_szBallonStyle    = "Balloon_Style";
static char *f_szBallonBkGndR   = "Balloon_BkGnd_Red";
static char *f_szBallonBkGndG   = "Balloon_BkGnd_Green";
static char *f_szBallonBkGndB   = "Balloon_BkGnd_Blue";
static char *f_szBallonTextR    = "Balloon_Text_Red";
static char *f_szBallonTextG    = "Balloon_Text_Green";
static char *f_szBallonTextB    = "Balloon_Text_Blue";
static char *f_szBallonBorderR  = "Balloon_Border_Red";
static char *f_szBallonBorderG  = "Balloon_Border_Green";
static char *f_szBallonBorderB  = "Balloon_Border_Blue";

DEFINE_RESPONSE_TABLE1 (TipControlBar, TControlBar)
  EV_WM_MOUSEMOVE,
  EV_WM_LBUTTONDOWN,
END_RESPONSE_TABLE;

////////////////////////////////////////////////////////////////////////////
//     Method:  ctor
//      Class:  TipControlBar
//
//     Params:
//
//              Same as TControlBar
//
//   Comments:
//
//              This ctor has the same parameters as TControlBar::ctor
//              and passes these values to its base class (like a good
//              OOP citizen should).
//
//              We make sure that the hint mode is fly-by, since we need
//              this mechanism active to use balloon help.
//
//              We also create the balloon window object if it has not
//              been created already.
//
////////
TipControlBar::TipControlBar (TWindow* parent,
                              TTileDirection direction,
                              TFont* font,
                              TModule* module)
	 : 	TControlBar (parent, direction, font, module),
        m_bEnabled(TRUE)
{
	SetHintMode (TGadgetWindow::EnterHints);  // fly-by hints are a must...

    //
    //  Create the balloon control for later use:
    //
    if (! f_pBalloon)
        f_pBalloon = new BalloonWindow;

    SetBalloonStyle (Rounded);
}

////////////////////////////////////////////////////////////////////////////
//     Method:  dtor
//      Class:  TipControlBar
//
//   Comments:
//
//              The dtor just deletes the BalloonWindow if it still
//              exists.
//
////////
TipControlBar::~TipControlBar()
{
    if (f_pBalloon)
    {
        delete f_pBalloon;
        f_pBalloon = 0;
    }
}

////////////////////////////////////////////////////////////////////////////
//     Method:  EvMouseMove
//      Class:  TipControlBar
//
//     Params:
//
//              modKeys / point   - Parameters for WM_MOUSEMOVE.
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method is called whenever the mouse moves over the
//              control bar window.  We destroy the balloon if the
//              cursor is not over any gadget, and call the base class
//              EvMouseMove() method.  Since fly-by-hints are enabled,
//              this may result in TipStatusBar::SetHintText() being
//              called by TControlBar::EvMouseMove().
//
//              Before calling TControlBar::EvMouseMove(), we set the
//              f_bCanPopup BOOL so that we can tell in SetHintText()
//              if we are indeed over the control bar and not browsing
//              through a menu.  We set f_bCanPopup to m_bEnabled so
//              that if EnableBalloonHelp() has been called to disable
//              balloon help, no balloon will appear (SetHintText() will
//              think we are in a menu).
//
////////
void TipControlBar::EvMouseMove (UINT modKeys, TPoint& point)
{
    TGadget *pg;

    pg = GadgetFromPoint (point);

	//
    //  'Capture' points to the gadget that has the mouse capture
    //
    if (! Capture && !pg)
    {
        // hide the tip window if not over a gadget
        f_pBalloon->HideBalloon ();
    }

    f_bCanPopup = m_bEnabled;

    //
    //  We check 'pg' and 'TGadgetWindow::AtMouse' to see if we are
    //  making a transition from one gadget to another.  If so, we need
    //  to remember this...
    //
    f_bChangeGadgets = (AtMouse && pg && AtMouse!=pg);

    TControlBar::EvMouseMove (modKeys, point);

    f_bChangeGadgets = FALSE;
    f_bCanPopup = FALSE;
}

////////////////////////////////////////////////////////////////////////////
//     Method:  EvLButtonDown
//      Class:  TipControlBar
//
//     Params:
//
//              modKeys / point   - Parameters for WM_LBUTTONDOWM
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method is called when the left mouse button is
//              pressed on the control bar.  If the button is pressed
//              over an enabled gadget or the cursor was not over a
//              gadget, we hide the balloon window.
//
//              After this checking, we pass the button event to our
//              base class TControlBar::EvLButtonDown() method.
//
////////
void TipControlBar::EvLButtonDown (UINT modKeys, TPoint& point)
{
    TGadget *pGadget;
    BOOL    bHide;

    pGadget = GadgetFromPoint (point);

    if (! pGadget)
        bHide = TRUE;
    else
        bHide = pGadget->GetEnabled();

    if (bHide)
        f_pBalloon->HideBalloon ();

    TControlBar::EvLButtonDown (modKeys, point);
}

////////////////////////////////////////////////////////////////////////////
//     Method:  SaveSettingsToIni
//      Class:  TipControlBar
//
//     Params:
//
//              szIniFile     - The name of the .INI file (NULL for
//                              WIN.INI).
//
//              szSection     - The name of the section to hold the
//                              state information.
//
//    Returns:
//
//              TRUE if state was saved successfully, FALSE otherwise.
//
//   Comments:
//
//              This method will store the object's option settings to
//              the to designated section of the App's .INI file.  These
//              settings can be retrieved by GetStateFromIni().
//
//              At present, these settings are saved under the given
//              entry names:
//
//                  Enabled         --> Balloon_Help_Enabled
//
//                  Balloon style   --> Balloon_Style
//
//                  BkGnd Color     --> Balloon_BkGnd_Red
//                                      Balloon_BkGnd_Green
//                                      Balloon_BkGnd_Blue
//
//                  Text Color      --> Balloon_Text_Red
//                                      Balloon_Text_Green
//                                      Balloon_Text_Blue
//
//                  Border Color    --> Balloon_Border_Red
//                                      Balloon_Border_Green
//                                      Balloon_Border_Blue
//
////////
BOOL TipControlBar::SaveSettingsToIni (const char far *szIniFile,
                                       const char far *szSection) const
{
    int okCount;
    TColor clr;

    okCount = 0;

    okCount += ini_PutBool  (szIniFile, szSection, f_szBallonsEnabled,
                                        IsBallonHelpEnabled ());

    okCount += ini_PutShort (szIniFile, szSection, f_szBallonStyle,
                                        GetBalloonStyle ());

    clr = GetBalloonBkGndColor ();
    okCount += ini_PutShort (szIniFile, szSection, f_szBallonBkGndR,
                                        clr.Red());
    okCount += ini_PutShort (szIniFile, szSection, f_szBallonBkGndG,
                                        clr.Green());
    okCount += ini_PutShort (szIniFile, szSection, f_szBallonBkGndB,
                                        clr.Blue());

    clr = GetBalloonTextColor ();
    okCount += ini_PutShort (szIniFile, szSection, f_szBallonTextR,
                                        clr.Red());
    okCount += ini_PutShort (szIniFile, szSection, f_szBallonTextG,
                                        clr.Green());
    okCount += ini_PutShort (szIniFile, szSection, f_szBallonTextB,
                                        clr.Blue());

    clr = GetBalloonBorderColor ();
    okCount += ini_PutShort (szIniFile, szSection, f_szBallonBorderR,
                                        clr.Red());
    okCount += ini_PutShort (szIniFile, szSection, f_szBallonBorderG,
                                        clr.Green());
    okCount += ini_PutShort (szIniFile, szSection, f_szBallonBorderB,
                                        clr.Blue());

    return 11 == okCount;
}

////////////////////////////////////////////////////////////////////////////
//     Method:  GetSettingsFromIni
//      Class:  TipControlBar
//
//     Params:
//
//              szIniFile     - The name of the .INI file (NULL for
//                              WIN.INI).
//
//              szSection     - The name of the section holding the
//                              state information.
//
//    Returns:
//
//              TRUE if state was read successfully, FALSE otherwise.
//
//   Comments:
//
//              This method will read the options stored by the
//              SaveSettingsToIni() method.  If any option cannot be
//              read, it will be unchanged.  See SaveSettingsToIni()
//              for a detailed list of the settings that are stored
//              in the .INI file.
//
////////
BOOL TipControlBar::GetSettingsFromIni (const char far *szIniFile,
                                        const char far *szSection)
{
    int     okCount;
    BOOL    bEnable;
    short   sStyle, R, G, B;

    okCount = 0;

    if (ini_GetBool (szIniFile, szSection, f_szBallonsEnabled,
                     &bEnable))
    {
        okCount++;
        EnableBallonHelp (bEnable);
    }

    if (ini_GetShort (szIniFile, szSection, f_szBallonStyle,
                      &sStyle))
    {
        okCount++;
        SetBalloonStyle ( (TipControlBar::BalloonStyle) sStyle);
    }

    if (ini_GetShort (szIniFile, szSection, f_szBallonBkGndR, &R) &&
        ini_GetShort (szIniFile, szSection, f_szBallonBkGndG, &G) &&
        ini_GetShort (szIniFile, szSection, f_szBallonBkGndB, &B))
    {
        okCount += 3;
        SetBalloonBkGndColor (TColor(R,G,B));
    }

    if (ini_GetShort (szIniFile, szSection, f_szBallonTextR, &R) &&
        ini_GetShort (szIniFile, szSection, f_szBallonTextG, &G) &&
        ini_GetShort (szIniFile, szSection, f_szBallonTextB, &B))
    {
        okCount += 3;
        SetBalloonTextColor (TColor(R,G,B));
    }

    if (ini_GetShort (szIniFile, szSection, f_szBallonBorderR, &R) &&
        ini_GetShort (szIniFile, szSection, f_szBallonBorderG, &G) &&
        ini_GetShort (szIniFile, szSection, f_szBallonBorderB, &B))
    {
        okCount += 3;
        SetBalloonBorderColor (TColor(R,G,B));
    }

    return 11 == okCount;
}

////////////////////////////////////////////////////////////////////////////
//     Method:  GetBalloonStyle
//      Class:  TipControlBar
//
//     Params:
//
//              None
//
//    Returns:
//
//              BalloonStyle      - The current balloon style
//
//   Comments:
//
//              This method will return the current balloon style.
//
////////
TipControlBar::BalloonStyle TipControlBar::GetBalloonStyle () const
{
    return f_pBalloon->IsBalloonSquare() ? TipControlBar::Square
                                         : TipControlBar::Rounded;
}

////////////////////////////////////////////////////////////////////////////
//     Method:  SetBalloonStyle
//      Class:  TipControlBar
//
//     Params:
//
//              newStyle      - The new style of the balloon
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will set the balloon style.  To get the
//              current balloon style, call GetBalloonStyle().
//
////////
void TipControlBar::SetBalloonStyle (TipControlBar::BalloonStyle newStyle)
{
    f_pBalloon->SetSquareStyle (TipControlBar::Square == newStyle);
}

////////////////////////////////////////////////////////////////////////////
//     Method:  GetBalloonBkGndColor
//      Class:  TipControlBar
//
//     Params:
//
//              None
//
//    Returns:
//
//              TColor        - The color of the balloon back ground
//
//   Comments:
//
//              This method calls the BalloonWindow method to retrieve
//              the current color setting for the background color.
//
////////
TColor TipControlBar::GetBalloonBkGndColor () const
{
    return f_pBalloon->GetBkGndColor();
}

////////////////////////////////////////////////////////////////////////////
//     Method:  GetBalloonTextColor
//      Class:  TipControlBar
//
//     Params:
//
//              None
//
//    Returns:
//
//              TColor        - The color of the balloon text
//
//   Comments:
//
//              This method calls the BalloonWindow method to retrieve
//              the current color setting for the text color.
//
////////
TColor TipControlBar::GetBalloonTextColor () const
{
    return f_pBalloon->GetTextColor();
}

////////////////////////////////////////////////////////////////////////////
//     Method:  GetBalloonBorderColor
//      Class:  TipControlBar
//
//     Params:
//
//              None
//
//    Returns:
//
//              TColor        - The color of the balloon border
//
//   Comments:
//
//              This method calls the BalloonWindow method to retrieve
//              the current color setting for the border color.
//
////////
TColor TipControlBar::GetBalloonBorderColor () const
{
    return f_pBalloon->GetBorderColor();
}

////////////////////////////////////////////////////////////////////////////
//     Method:  SetBalloonBkGndColor
//      Class:  TipControlBar
//
//     Params:
//
//              clr           - The new balloon background color
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will call the BalloonWindow method to set
//              the color of the balloon's background.
//
////////
void TipControlBar::SetBalloonBkGndColor (const TColor &clr)
{
    f_pBalloon->SetBkGndColor(clr);
}

////////////////////////////////////////////////////////////////////////////
//     Method:  SetBalloonTextColor
//      Class:  TipControlBar
//
//     Params:
//
//              clr           - The new balloon text color
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will call the BalloonWindow method to set
//              the color of the balloon's text.
//
////////
void TipControlBar::SetBalloonTextColor (const TColor &clr)
{
    f_pBalloon->SetTextColor(clr);
}

////////////////////////////////////////////////////////////////////////////
//     Method:  SetBalloonBorderColor
//      Class:  TipControlBar
//
//     Params:
//
//              clr           - The new balloon border color
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will call the BalloonWindow method to set
//              the color of the balloon's border.
//
////////
void TipControlBar::SetBalloonBorderColor (const TColor &clr)
{
    f_pBalloon->SetBorderColor(clr);
}

////////////////////////////////////////////////////////////////////////////
//
//      Class:  T i p S t a t u s B a r
//

static const char *f_szFullLineHints = "Full_Line_Hints";

//-------------------------------------
//  The TipStatusBar flag values:
//
#define  TSB_FULLLINEHINTS      0x01
#define  TSB_LASTHINTFULLLINE   0x02    // remember last hint display type

////////////////////////////////////////////////////////////////////////////
//     Method:  ctor
//      Class:  TipStatusBar
//
//     Params:
//
//              Same as for TStatusBar
//
//   Comments:
//
//              This ctor basically does little other than call the ctor
//              for the TStatusBar base class, passing the parameters it
//              receives (which exactly match the parameters expected by
//              TStatusBar::TStatusBar).
//
//              We take the opportunity to create the BalloonWindow if
//              it has not been created already.
//
////////
TipStatusBar::TipStatusBar (TWindow* parent,
                            TGadget::TBorderStyle borderStyle,
                            UINT modeIndicators,
                            TFont *font,
                            TModule* module)
    :   TStatusBar (parent, borderStyle, modeIndicators, font, module),
        m_bFlags (TSB_FULLLINEHINTS)
{
    //
    //  Create the balloon control for later use:
    //
    if (! f_pBalloon)
        f_pBalloon = new BalloonWindow;
}

////////////////////////////////////////////////////////////////////////////
//     Method:  dtor
//      Class:  TipStatusBar
//
//   Comments:
//
//              The dtor just deletes the BalloonWindow if it still
//              exists.
//
////////
TipStatusBar::~TipStatusBar()
{
    if (f_pBalloon)
    {
        delete f_pBalloon;
        f_pBalloon = 0;
    }
}

////////////////////////////////////////////////////////////////////////////
//     Method:  SaveSettingsToIni
//      Class:  TipStatusBar
//
//     Params:
//
//              szIniFile     - The name of the .INI file (NULL for
//                              WIN.INI).
//
//              szSection     - The name of the section to hold the
//                              state information.
//
//    Returns:
//
//              TRUE if state was saved successfully, FALSE otherwise.
//
//   Comments:
//
//              This method will store the object's option settings to
//              the to designated section of the App's .INI file.  These
//              settings can be retrieved by GetSettingsFromIni().
//
//              At present, these settings are saved under the given
//              entry names:
//
//                  Full line hints     --> Full_Line_Hints
//
////////
BOOL TipStatusBar::SaveSettingsToIni (const char far *szIniFile,
                                      const char far *szSection) const
{
    return ini_PutBool (szIniFile, szSection, f_szFullLineHints,
                                   AreHintsFullLine ());
}

////////////////////////////////////////////////////////////////////////////
//     Method:  GetSettingsFromIni
//      Class:  TipStatusBar
//
//     Params:
//
//              szIniFile     - The name of the .INI file (NULL for
//                              WIN.INI).
//
//              szSection     - The name of the section holding the
//                              state information.
//
//    Returns:
//
//              TRUE if state was read successfully, FALSE otherwise.
//
//   Comments:
//
//              This method will read the options stored by the
//              SaveSettingsToIni() method.  If any option cannot be
//              read, it will be unchanged.  See SaveSettingsToIni()
//              for a detailed list of the settings that are stored
//              in the .INI file.
//
////////
BOOL TipStatusBar::GetSettingsFromIni (const char far *szIniFile,
                                       const char far *szSection)
{
    BOOL bFullLine;

    bFullLine = 2;

    if (ini_GetBool (szIniFile, szSection, f_szFullLineHints, &bFullLine))
        SetFullLineHints (bFullLine);

    return bFullLine < 2;
}

////////////////////////////////////////////////////////////////////////////
//     Method:  AreHintsFullLine
//      Class:  TipStatusBar
//
//     Params:
//
//              None
//
//    Returns:
//
//              BOOL      - The current status of the full line hint
//                          mode flag
//
//   Comments:
//
//              This method will tell the caller if the current hint
//              mode is full line.  See SetFullLineHints() for more
//              discussion on the difference between full line hints
//              and non-full line hints.
//
////////
BOOL TipStatusBar::AreHintsFullLine () const
{
    return !! (m_bFlags & TSB_FULLLINEHINTS);
}

////////////////////////////////////////////////////////////////////////////
//     Method:  SetFullLineHints
//      Class:  TipStatusBar
//
//     Params:
//
//              bOn       - New status of full line hint mode
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will set or clear the full line hint mode
//              flag in 'm_bFlags' based on the value of 'bOn'.  The
//              normal OWL TStatusBar uses full line hints, but this
//              class allows hint text to be placed in the text pane
//              of the status bar, minimizing the changes to the status
//              bar (but, at the same time, limiting the readable area
//              for the hint text).
//
//              The default mode for TipStatusBar is full line hint
//              mode.
//
////////
void TipStatusBar::SetFullLineHints (BOOL bOn)
{
    if (bOn)
        m_bFlags |= TSB_FULLLINEHINTS;
    else
        m_bFlags &= ~TSB_FULLLINEHINTS;
}

////////////////////////////////////////////////////////////////////////////
//     Method:  SetHintText
//      Class:  TipStatusBar
//
//     Params:
//
//              lpszText      - The hint text to display
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method overrides TStatusBar::SetHintText().  We
//              play some games with the text (like separating the tip
//              text from the normal hint text) before we pass on the
//              call to TStatusBar.
//
//              We also allow for the possibility of putting the hint
//              text into pane 0 (i.e., the TTextGadget) of the status
//              bar instead of the normal full line display.  This can
//              be turned on and off by calling SetFullLineHints() and
//              the current status can be determined by calling the
//              AreHintsFullLine() method.
//
////////
void TipStatusBar::SetHintText (const char *lpszText)
{
    char    buf[128];

	if (lpszText != NULL)
	{
        char    *sz;

		lstrcpy (buf, lpszText);

        //
		//  Locate and remove the tip text from the hint text (remembering
        //  where tip text in 'sz'):
        //
        sz = strchr (buf, '\n');
        if (sz)
            *(sz++) = '\0';

        //
		//  Only display hints from gadgets (not menus!) and then only
        //  if tips are enabled.  By setting the caption we enable a
        //  timer.  The timer tick will cause the balloon window to be
        //  displayed (with the tip text in it).
        //
		if (f_bCanPopup)
			f_pBalloon->SetCaption (sz);

        lpszText = buf;
	}
	else
	{
		f_pBalloon->SetCaption (NULL);  // hides the balloon
	}

    //
    //  Now we need to pass on the hint text so that it appears on the
    //  status bar.  We may have modified 'lpszText' to point at 'buf'
    //  if we had to chop off the tip text.  If 'lpszText' was NULL, we
    //  want to remove the hint text.  In this case, 'lpszText' has not
    //  been changed.
    //
    if (AreHintsFullLine ())
    {
        TStatusBar::SetHintText (lpszText); // to the base class...

        //
        //  A NULL ptr in lpszText will erase the hint line, so we should
        //  "forget" that we called TStatusBar::SetHintText()...
        //
        if (lpszText)
            m_bFlags |= TSB_LASTHINTFULLLINE;
        else
            m_bFlags &= ~TSB_LASTHINTFULLLINE;
    }
    else    // put the text into pane 0 of the status line...
    {
        //
        //  We may need to call TStatusBar::SetHintText() to remove the
        //  full line if we used full line hints last time...
        //
        if (m_bFlags & TSB_LASTHINTFULLLINE)
        {
            TStatusBar::SetHintText(0);
            m_bFlags &= ~TSB_LASTHINTFULLLINE;
        }

        SetText (lpszText);
    }
}

