////////////////////////////////////////////////////////////////////////////
//       File:  TIMER.H
//    Created:  Sept 1994
//     Author:  Don Griffin (DWG)
//
//   Comments:
//
//              This header file contains the declaration for the Timer
//              class as well as a simple derived class called the
//              OneTimeMessageTimer.
//
////////
#ifndef __TIMER_H
#define __TIMER_H

#define  TF_ACTIVE      0x0001
#define  TF_ONESHOT     0x0002

////////////////////////////////////////////////////////////////////////////
//      Class:  Timer
//
//   Comments:
//
//              This class encapsulates the Windows timer API.  Any
//              instance of a Timer-derived class will have its virtual
//              Tick() method called when the timer fires.  A derived
//              class takes that opportunity to do its work (but it is
//              best if the work is in little pieces at a time).
//
//              The Windows timer is multiplexed to service all Timer
//              object instances, because Windows has a global limit on
//              the number of timers that can be allocated.  When the 1st
//              instance of a Timer object is instantiated, the Windows
//              timer is allocated.  When the last Timer object goes out
//              of scope (most likely via operator delete), the timer is
//              freed.
//
//              A derived class's Tick() method MUST call the base class
//              Tick() method AFTER its processing is complete (not
//              before, because the base class may delete the object).
//
////////
class Timer
{
    protected:

        static  void            StaticInit ();
        static  void            StaticCleanup ();

        static  void CALLBACK   TheTimerProc (HWND, UINT, UINT, DWORD);
        static  TIMERPROC       s_lpfnTimerProcInstance;
        static  Timer *         s_pFirstTimer;
        static  UINT            s_uTimerCount;
        static  UINT            m_uTimerID;

        Timer *             m_pNextTimer;
        DWORD               m_dwLastTick;
        UINT                m_uTimeOut;
        WORD                m_wFlags;

        BOOL                IsFlagSet (WORD flag) const
                                { return (m_wFlags & flag) == flag; }
        void                SetFlag (WORD flag)
                                { m_wFlags |= flag; }
        void                ResetFlag (WORD flag)
                                { m_wFlags &= ~flag; }

    public:

        Timer (UINT uTimeOut, BOOL oneShot = FALSE);
        ~Timer ();

        UINT    GetTimeOut () const { return m_uTimeOut; }
        DWORD   GetLastTickTime () const { return m_dwLastTick; }
        BOOL    IsActive () const { return IsFlagSet (TF_ACTIVE); }

        void    SetTimeOut (UINT timeOut) { m_uTimeOut = timeOut; }
        void    SetActive (BOOL bActive)
                    {
                        if (bActive) SetFlag   (TF_ACTIVE);
                        else         ResetFlag (TF_ACTIVE);
                    }

        virtual void    Tick (DWORD dwTime);
};

////////////////////////////////////////////////////////////////////////////
//      Class:  OneShotMessageTimer
// Base class:  Timer
// Derivation:  public
//
//   Comments:
//
//              This class (derived from Timer) makes for a simple way
//              to get a message to a window after a specified amount
//              of time has ellapsed.  Once the object has sent the msg
//              to the window it will delete itself.
//
//              Example:
//
//                  {
//                      ...
//
//                      new OneShotMessageTimer (1000, hwnd, msg, 0, 0);
//
//                      ...
//                  }
//
//              This will send the message 'msg' to 'hwnd' after a 1000
//              millisecond timer has ellapsed.  The object will be
//              deleted automatically.
//
////////
class OneShotMessageTimer : public Timer
{
    protected:

        HWND        m_hWnd;
        UINT        m_uMsg;
        WPARAM      m_wp;
        LPARAM      m_lp;

    public:

        OneShotMessageTimer (UINT, HWND, UINT, WPARAM, LPARAM);

        virtual void    Tick (DWORD dwTime);
};

#endif  // __TIMER_H

