////////////////////////////////////////////////////////////////////////////
//       File:  TIMER.CPP
//    Created:  Sept 1994
//     Author:  Don Griffin (DWG)
//
//   Comments:
//
//              This source file contains the implementation of the
//              methods for Timer and OneShotMessageTimer.
//
////////
#include <owl\owlpch.h>
#pragma hdrstop

#include "timer.h"

TIMERPROC   Timer::s_lpfnTimerProcInstance = 0;
Timer *     Timer::s_pFirstTimer = 0;
UINT        Timer::m_uTimerID = 0;
UINT        Timer::s_uTimerCount = 0;

////////////////////////////////////////////////////////////////////////////
//     Method:  TheTimerProc    (protected static)
//      Class:  Timer
//
//     Params:
//
//              HWND          - Ignored
//              UINT          - Ignored
//              UINT          - Ignored
//              dwTime        - The current system tick count (the same
//                              value we would get via GetTickCount).
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method is called periodically by Windows when our
//              timer is active.  We iterate over the instances of Timer
//              objects and see which ones need to tick.
//
////////
void CALLBACK _export Timer::TheTimerProc (HWND, UINT, UINT, DWORD dwTime)
{
    Timer *theTimer, *nextTimer;

    theTimer = s_pFirstTimer;

    while (theTimer)
    {
        // Need to get nextTimer now because Timer objects can delete
        // themselves:
        nextTimer = theTimer->m_pNextTimer;

        if (theTimer->IsActive())
        {
            if (dwTime >= theTimer->m_dwLastTick + theTimer->m_uTimeOut)
            {
                theTimer->Tick (dwTime);
                // at this point, theTimer may be deleted...
            }
        }

        theTimer = nextTimer;
    }
}

////////////////////////////////////////////////////////////////////////////
//     Method:  StaticInit      (protected static)
//      Class:  Timer
//
//     Params:
//
//              None
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will create a timer callback procedure
//              instance and (if that succeeds) will create a timer
//              via the SetTimer() API.
//
//              We request a 1ms callback interval so that we can do
//              timer multiplexing and still respect the timeout of
//              the Timer instances.  On a PC, we will get approx.
//              55ms (1/18.2) intervals, but that's the best we can
//              do...
//
////////
void Timer::StaticInit ()
{
    s_lpfnTimerProcInstance =
        (TIMERPROC) MakeProcInstance ((FARPROC) TheTimerProc,
                                      GetApplicationObject()->GetInstance());

    if (s_lpfnTimerProcInstance)
    {
        m_uTimerID = SetTimer (0, 101, 1, s_lpfnTimerProcInstance);
    }
}

////////////////////////////////////////////////////////////////////////////
//     Method:  StaticCleanup       (protected static)
//      Class:  Timer
//
//     Params:
//
//              None
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method will cleanup the timer and free the timer
//              procedure instance.  This is done when the last Timer
//              object is destroyed.
//
////////
void Timer::StaticCleanup ()
{
    if (s_lpfnTimerProcInstance)
    {
        if (m_uTimerID)
        {
            KillTimer (0, m_uTimerID);
            m_uTimerID = 0;
        }

        FreeProcInstance ((FARPROC) s_lpfnTimerProcInstance);
        s_lpfnTimerProcInstance = 0;
    }
}

////////////////////////////////////////////////////////////////////////////
//     Method:  ctor
//      Class:  Timer
//
//     Params:
//
//              uTimeOut      - The timeout value for (i.e., resolution
//                              of) the timer.
//
//   Comments:
//
//              The ctor remembers the timeout value and links the Timer
//              object into the Timer chain.  We also need to initialize
//              our statics if we are the first Timer instance, and we
//              need to keep an instance count so we know when to clean
//              things up.
//
//              The new Timer object is linked into the Timer chain at
//              the front for best service.
//
////////
Timer::Timer (UINT uTimeOut, BOOL oneShot /* = FALSE*/)
    :   m_dwLastTick(0), m_uTimeOut(uTimeOut), m_wFlags(0)
{
    m_pNextTimer = s_pFirstTimer;
    s_pFirstTimer = this;

    SetFlag (TF_ACTIVE);
    
    if (oneShot)
        SetFlag (TF_ONESHOT);

    if (! s_uTimerCount)
        StaticInit ();

    s_uTimerCount++;
}

////////////////////////////////////////////////////////////////////////////
//     Method:  dtor
//      Class:  Timer
//
//   Comments:
//
//              The dtor unlinks the Timer object from the Timer chain
//              and decrements the instance count.  When the last Timer
//              object is destructed, StaticCleanup() is called to clean
//              things up.
//
////////
Timer::~Timer ()
{
    if (this == s_pFirstTimer)
        s_pFirstTimer = m_pNextTimer;
    else
    {
        Timer *prevTimer;

        prevTimer = s_pFirstTimer;

        while (prevTimer)
        {
            if (prevTimer->m_pNextTimer == this)
            {
                prevTimer->m_pNextTimer = m_pNextTimer;
                break;
            }

            prevTimer = prevTimer->m_pNextTimer;
        }
    }

    s_uTimerCount--;

    if (! s_uTimerCount)
        StaticCleanup ();
}

////////////////////////////////////////////////////////////////////////////
//     Method:  Tick
//      Class:  Timer
//
//     Params:
//
//              dwTime        - The system time of the tick event
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method is called when the Timer fires.  We remember
//              the time in 'm_dwLastTick' and if TF_ONESHOT is set, we
//              delete this.
//
//              A derived class MUST call this method in its override if
//              the Timer object is to maintain its declared event rate
//              and if the TF_ONESHOT flag is to be respected.
//
////////
void Timer::Tick (DWORD dwTime)
{
    m_dwLastTick = dwTime;

    if (IsFlagSet (TF_ONESHOT))
        delete this;
}


////////////////////////////////////////////////////////////////////////////
//     Method:  ctor
//      Class:  OneShotMessageTimer
//
//     Params:
//
//              timeOut   - The time to wait before sending the msg
//              hwnd      - The HWND to send the msg to
//              uMsg      - The message to send
//              wp        - The WPARAM value to send
//              lp        - The LPARAM value t send
//
//   Comments:
//
//              The ctor remembers the HWND/UINT/WPARAM/LPARAM 4-tuple
//              defining the msg and the wnd to send it to.  If 'lp' is
//              a pointer to some structure, the structure must remain
//              valid until the message is sent (which will be after the
//              object is created and the creating function returns).
//
//              For example,
//
//                  void DelayedMessage()
//                  {
//                      //...
//                      new OneShotMessageTimer (1000, hwnd, uMsg, wp, lp);
//                      //...
//                  }
//
//              If the code above the 'new' sets 'lp' to point at
//              something, there will be a problem if that something is
//              on the stack.  This is because the message will not be
//              sent by the object until after DelayedMessage() has
//              returned, making 'lp' an invalid pointer.
//
////////
OneShotMessageTimer::OneShotMessageTimer (UINT timeOut, HWND hwnd,
                                          UINT uMsg, WPARAM wp, LPARAM lp)
    : Timer (timeOut, TRUE), m_hWnd(hwnd), m_uMsg(uMsg), m_wp(wp), m_lp(lp)
{
}

////////////////////////////////////////////////////////////////////////////
//     Method:  Tick
//      Class:  OneShotMessageTimer
//
//     Params:
//
//              dwTime        - The tick event time
//
//    Returns:
//
//              Nothing
//
//   Comments:
//
//              This method sends the message to the window.  It then
//              calls Timer::Tick(), which will delete the object since
//              the TF_ONESHOT flag is set.
//
////////
void OneShotMessageTimer::Tick (DWORD dwTime)
{
    SendMessage (m_hWnd, m_uMsg, m_wp, m_lp);
    Timer::Tick (dwTime);
}

