////////////////////////////////////////////////////////////////////////////
//       File:  CLOCKGAD.CPP
//    Created:  Sept 1994
//     Author:  Don Griffin (DWG)
//
//   Comments:
//
//              This source file contains the implementation of the
//              DClockGadget class and the FormatWinTime() function.
//
//              To use a DClockGadget in a status bar, try the code
//              below:
//
//                  TStatusBar *sb = new TStatusBar (...);
//
//                  sb->Insert (*new TSeparatorGadget (4));
//                  sb->Insert (*new DClockGadget (TGadget::Recessed, TRUE));
//
//              The above code is even more than is necessary.  The
//              constructor for DClockGadget has defaults for both of
//              its arguments.  The first arg is the border style (of
//              type TBorderStyle) and it defaults to Recessed.  The
//              second arg is a BOOL as to whether the seconds should be
//              shown on the clock.
//
//              Therefore, if you want a recessed clock w/o seconds, try
//              this:
//
//                  sb->Insert (*new DClockGadget);
//
//              The TSeparatorGadget makes for a better looking status
//              bar (IMHO), but is not required.
//
////////
#include <owl\owlpch.h>
#pragma hdrstop

#include <owl\statusba.h>
#include <classlib\time.h>

#include "timer.h"
#include "clockgad.h"


#define  FWT_INITIALIZED    0x80
#define  FWT_12HR           0x01
#define  FWT_0PADDED        0x02
////////////////////////////////////////////////////////////////////////////
//   Function:  FormatWinTime
//
//     Params:
//
//              Hr            - The hour of day (0-23)
//              Mn            - The minutes (0-59)
//              Sec           - The seconds (0-59) or < 0 to omit
//              szBuf         - The buffer to hold the result
//              bufSize       - The size of the buffer
//
//    Returns:
//
//              char *        - The value passed as 'szBuf'
//
//   Comments:
//
//              This function will format the given hour(Hr), minute(Mn)
//              and second(Sec) using the WIN.INI [intl] settings setup
//              but Control Panel.  The seconds are optional, and are
//              turned off by a value < 0.
//
//              This function keeps several static variables around to
//              minimize access to WIN.INI for settings.  However, if
//              these values are changed, this function needs to be told
//              about it.  This can be done in response to the message
//              WM_WININICHANGE.  This message is only received by top
//              level windows, and therefore, the App's main window is
//              the proper place to do this.
//
//              To reinitialize this function, simply make a call like
//              this:
//
//                  FormatWinTime (0,0,0,0,0);
//
//              The 0 for 'szBuf' and 'bufSize' indicates that there is
//              not buffer to format the time into, and would normally
//              be an invalid parameter.  FormatWinTime() will detect
//              this situation and reinitialize its state. 
//
////////
char *FormatWinTime (int Hr, int Mn, int Sec, char *szBuf, int bufSize)
{
    static BYTE bFlags = 0;
    static char szTimeSep[3] = { ':', 0, 0 };
    static char szAM [5]     = { 'a', 'm', 0, 0, 0 };
    static char szPM [5]     = { 'p', 'm', 0, 0, 0 };

    if (Hr < 0 || Mn < 0 || !szBuf || bufSize < 1)
        bFlags &= ~FWT_INITIALIZED;

    if (! (bFlags & FWT_INITIALIZED))
    {
        char buf [10];

        bFlags = FWT_INITIALIZED;

        //
        //  Get the 12/24 hr setting:
        //
        GetProfileString ("intl", "iTime", "0", buf, sizeof(buf));
        if (buf[0]=='0')
            bFlags |= FWT_12HR;

        //
        //  Get the 0-padding setting:
        //
        GetProfileString ("intl", "iTLZero", "0", buf, sizeof(buf));
        if (buf[0]=='1')
            bFlags |= FWT_0PADDED;

        //
        //  Get the time separator string:
        //
        GetProfileString ("intl", "sTime", ":", szTimeSep, sizeof(szTimeSep));

        //
        //  Get the time AM and PM strings:
        //
        GetProfileString ("intl", "s1159", "am", szAM, sizeof(szAM));
        GetProfileString ("intl", "s2359", "pm", szPM, sizeof(szPM));
    }

    if (Hr < 0 || Mn < 0 || !szBuf || bufSize < 1)
        return 0;

    BOOL bPM;
    char szTime[60];

    bPM = FALSE;
    if ((bFlags & FWT_12HR) && Hr >= 12)
    {
        Hr -= (Hr>12) * 12;
        bPM = TRUE;
    }

    szTime[0] = szTime[1] = '\0';
    if ((bFlags & FWT_0PADDED) && Hr < 10)
        *szTime = '0';

    wsprintf (szTime+strlen(szTime), "%d%s%02d", Hr, szTimeSep, Mn);

    if (Sec >= 0)
    {
        wsprintf (szTime+strlen(szTime), "%s%02d", szTimeSep, Sec);
    }

    if (bFlags & FWT_12HR)
    {
        strcat (szTime, bPM ? szPM : szAM);
    }

    strncpy (szBuf, szTime, bufSize);
    szBuf[bufSize-1] = '\0';

    return szBuf;
}

////////////////////////////////////////////////////////////////////////////
//      Class:  DClockGadgetTimer
// Base class:  Timer
// Derivation:  public
//
//   Comments:
//
//              This class is used to notify a DClockGadget whenever it
//              needs to update its display.
//
////////
class DClockGadgetTimer : public Timer
{
    protected:

        DClockGadget *      m_pClock;

    public:

        DClockGadgetTimer (DClockGadget *clockGadget, BOOL bSeconds)
            :   Timer (bSeconds ? 900 : 10000), // 0.9 or 10 second timeout
                m_pClock (clockGadget)
        {
        }

        virtual void Tick (DWORD);
};

void DClockGadgetTimer::Tick (DWORD dwTime)
{
    m_pClock->Tick ();
    Timer::Tick (dwTime);
}

//--------------------------------------------------------------------------

const int ClockGadgetId = 32;

DClockGadget::DClockGadget (TBorderStyle borderStyle, BOOL bShowSeconds)
    :   TTextGadget (ClockGadgetId, borderStyle, Center,
                                    bShowSeconds ? 10 : 7)
{
    m_pTimer = new DClockGadgetTimer (this, bShowSeconds);

    Tick();
}

DClockGadget::~DClockGadget()
{
    delete m_pTimer;
}

void DClockGadget::Tick ()
{
    TTime time;
    char buf[20];
    int Sec;

    Sec = (10==NumChars) ? time.Second() : -1;

    SetText (FormatWinTime (time.Hour(), time.Minute(), Sec,
                                         buf, sizeof(buf)));

    Invalidate ();
}

BOOL DClockGadget::IsActive () const
{
    return m_pTimer->IsActive();
}

void DClockGadget::SetActive (BOOL bActive)
{
    m_pTimer->SetActive (bActive);
}

