// *************************************************************** //
// Filename:    subctrl.cpp                                        //
//                                                                 //
// Date:        Mar 5, 1994                                        //
//                                                                 //
// Description:                                                    //
//  Helper functions to add an external control into the tabbing   //
//  sequence of a dialog.                                          //
// *************************************************************** //
#include "MPD.h"

// --------------------------------------------------------------- //
// Global variables                                                //
// --------------------------------------------------------------- //
static HWND AdditionalTabCtrl;
static HWND HDlgFirstCtrl;
static HWND HDlgLastCtrl;
static WNDPROC OldFirstCtrlWndProc;
static WNDPROC OldLastCtrlWndProc;

// --------------------------------------------------------------- //
// For a given window, process all the waiting messages.           //
// Very important when it comes to a window losing focus.          //
// --------------------------------------------------------------- //
void EatWaitingMessages(HWND hwnd)
{
  MSG msg;
  while (PeekMessage(&msg, hwnd, 0, 0, PM_REMOVE))
  {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  }
}

BOOL IsShiftKeyDown()
{
  int shiftState = GetKeyState(VK_SHIFT);
  return (shiftState & 0x8000) == 0x8000;
}

void SetFocusToDlgFirstCtrl()
{
  if (HDlgFirstCtrl == NULL)
    return;
  SetFocus(HDlgFirstCtrl);
}

void SetFocusToDlgLastCtrl()
{
  if (HDlgLastCtrl == NULL)
    return;
  SetFocus(HDlgLastCtrl);
}

// --------------------------------------------------------------- //
// Find first control in the dialog with WS_TABSTOP.               //
// Assumes the control is visible.                                 //
// --------------------------------------------------------------- //
HWND FirstChildWithTabStop(HWND hdlg)
{
  HWND hwnd = GetWindow(hdlg, GW_CHILD);
  while (hwnd && 
    ((GetWindowLong(hwnd, GWL_STYLE) & WS_TABSTOP) == 0))
  {
    hwnd = GetWindow(hwnd, GW_HWNDNEXT);
  }
  return hwnd;
}

// --------------------------------------------------------------- //
// Find last control in the dialog with WS_TABSTOP.                //
// Assumes the control is visible.                                 //
// --------------------------------------------------------------- //
HWND LastChildWithTabStop(HWND hdlg)
{
  HWND hwndFirst = GetWindow(hdlg, GW_CHILD);
  HWND hwnd      = GetWindow(hwndFirst, GW_HWNDLAST);
  while (hwnd && 
    ((GetWindowLong(hwnd, GWL_STYLE) & WS_TABSTOP) == 0))
  {
    hwnd = GetWindow(hwnd, GW_HWNDPREV);
  }
  return hwnd;
}

// --------------------------------------------------------------- //
// New WNDPROC for the first control in the dialog with tabstop.   //
// --------------------------------------------------------------- //
LRESULT CALLBACK _export FirstDialogCtrl(HWND hwnd, UINT message, 
  WPARAM wParam, LPARAM lParam)
{
  switch (message)
  {
    case WM_CHAR:
    {
      if (wParam == VK_TAB)     // ignore the tab key coming in
        return 0;               // produces a beep on an edit ctrl.
      break;
    }
    case WM_KEYDOWN:
    {
      if (wParam == VK_TAB)
      {
        // process the keyup before it loses focus
        EatWaitingMessages(hwnd);
        if (IsShiftKeyDown())
          SetFocus(AdditionalTabCtrl);  // give the listbox focus
        else
          PostMessage(GetParent(hwnd), WM_NEXTDLGCTL, 0, 0);
        return 0;
      }
      break;
    }
    case WM_GETDLGCODE:
    {
      return DLGC_WANTTAB | 
        CallWindowProc(OldFirstCtrlWndProc, hwnd, message, wParam, lParam)  ;
    }
  }
  return CallWindowProc(OldFirstCtrlWndProc, hwnd, message, wParam, lParam);
}

// --------------------------------------------------------------- //
// New WNDPROC for the last control in the dialog with tabstop.    //
// --------------------------------------------------------------- //
LRESULT CALLBACK _export LastDialogCtrl(HWND hwnd, UINT message, 
  WPARAM wParam, LPARAM lParam)
{
  switch (message)
  {
    case WM_CHAR:
    {
      if (wParam == VK_TAB)
        return 0;
      break;
    }
    case WM_KEYDOWN:
    {
      if (wParam == VK_TAB)
      {
        // process the keyup before it loses focus
        EatWaitingMessages(hwnd);
        if (IsShiftKeyDown())
          PostMessage(GetParent(hwnd), WM_NEXTDLGCTL, 1, 0);
        else
          SetFocus(AdditionalTabCtrl);    // give the listbox focus
        return 0;
      }
      break;
    }
    case WM_GETDLGCODE:
    {
      return DLGC_WANTTAB | 
        CallWindowProc(OldLastCtrlWndProc, hwnd, message, wParam, lParam)  ;
    }
  }
  return CallWindowProc(OldLastCtrlWndProc, hwnd, message, wParam, lParam);
}


void SetAdditionalTabCtrl(HWND additionalCtrl)
{
  AdditionalTabCtrl = additionalCtrl;
}

void SubclassFirstCtrl(HWND hwnd)
{
  HDlgFirstCtrl = FirstChildWithTabStop(hwnd);
  OldFirstCtrlWndProc = SubclassWindow(HDlgFirstCtrl, FirstDialogCtrl);
}

void SubclassLastCtrl(HWND hwnd)
{
  HDlgLastCtrl  = LastChildWithTabStop(hwnd);
  OldLastCtrlWndProc = SubclassWindow(HDlgLastCtrl, LastDialogCtrl);
}

