																			  /*
	NewHeap -- Copyright 1994 by Philippe Goutier, all rights reserved.

	FILE		:	newheap.h
	DATE		:	21/9/1994
	CONTENTS
	~~~~~~~~
	NewHeap declarations, to be included in client applications.

----------------------------------------------------------------------------- */

#ifndef NEWHEAP_H
#define NEWHEAP_H

#ifndef __cplusplus
#error	You must use C++ with NewHeap.h
#endif

/* -------------------------------------------------------------------------- */

// If your compiler allows independant overloading of operators new and new[],
// delete and delete[], then #define the macro COMPILER_HAS_NEW_ARRAY.

#ifdef __BORLANDC__
#define	COMPILER_HAS_NEW_ARRAY
#endif

#if _MSC_VER && (_MSC_VER <= 800)
#define	COMPILER_HAS_NO_TEMPLATE
#endif

/* -------------------------------------------------------------------------- */

#include	<string.h>
#include	<limits.h>

/* -------------------------------------------------------------------------- */

struct	NHSlot;
struct	NHSegment;
struct	NHSubSeg;
struct	NHFreeBlock;
struct	NHosBlock;
struct	NHReference;
struct	NHBlockInfo;
struct	NHRefSeg;
struct	NHRefSegPage;
struct	NHBlockRef;

typedef	int	NHBool;
#if INT_MAX == 32767
typedef	signed int		NHWord;
typedef	unsigned int	NHUWord;
#elif SHRT_MAX == 32767
typedef	signed short	NHWord;
typedef	unsigned short	NHUWord;
#else
#error typedef
#endif

#ifdef NH_DLL
#define	NH_EXPORT	_export
#else
#define	NH_EXPORT
#endif

////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//                            Class declarations                              //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

class NH_EXPORT NHeap
{
////////////////////////////////////////////////////////////////////////////////

public:

	///////////////
	// Constants //
	///////////////
	enum	// Used in ctor and Alloc()
	{
		clear 		= 1,
		share 		= 2,
		pageLock	= 4,
		fileMap		= 8,
		noDebug		= 16,
		noAlert		= 32,
		isOS		= 64
	};

	/////////////
	// typedef //
	/////////////
	typedef	void	(*fnEmergency)(NHeap *heap);
	typedef	NHBool	(*fnNewHandler)(unsigned long);
	typedef	NHBool	(*fnIter)(void *block);
	#ifdef _MSC_VER // Uses non-ANSI name
	typedef	int		(*_fnNewHandler)(size_t);
	#else
	typedef	void	(*_fnNewHandler)(void);
	#endif

	//////////////////
	// Constructors //
	//////////////////
	NHeap (const char* const name = "No Name", unsigned long mode = 0, unsigned long baseSize = 4096L);
	virtual NHeap::~NHeap ();

	//////////////////////
	// Public functions //
	//////////////////////
			void*		AllocArray (unsigned long nElement, unsigned long szElement, unsigned long mode = 0);	// Subset of above
	static	NHBool		Reduce (void* &pBlock, unsigned long delta);
	static	NHBool		Reduce (void* &pBlock, unsigned long offset, unsigned long size);
	static	NHBool		Expand (void* &pBlock, unsigned long delta, NHBool clear = 0);
			void*		StoreData (const void *data, unsigned long size, unsigned long mode = 0);
			char*		StoreString (const char *s);
			char*		StoreString (const char *s, size_t len);
	static	NHBool		Resize (void* &pBlock, unsigned long newSize, NHBool clear = 0);
	static	NHBool		Insert (void* &pClient, unsigned long offset, const void* data, unsigned long size);
	static	NHBool		Append (void* &pClient, const void* data, unsigned long size);
	static	NHBool		Prepend (void* &pClient, const void* data,	unsigned long size);
	static	void*		Duplicate (const void *pBlock);
	static	unsigned long	GetSize (const void *pBlock);
	static	NHeap*		GetHeap(const void *p);
			void		Clear();
	static	void		FreePtr (void* &pBlock);
	#ifndef NH_DEBUG	// else alternate declarations further
			void*		Alloc (unsigned long requestedSize, unsigned long mode = 0);
	static	void		Free (void *pBlock);
	#endif
	#ifdef NH_MULTITHREAD
	static	void		ThreadUnregister();
	#endif

	#ifdef NH_DLL
	static	NHeap*		Main();
	#else
	static	NHeap*		Main() { return _pMainHeap; }
	#endif
			void		ForEach(fnIter func);
			void*		FirstThat(fnIter func);
			void		SetNewHandler(fnNewHandler in_fpNewHandler) { fpNewHandler = in_fpNewHandler; }
	static	_fnNewHandler	set_new_handler(_fnNewHandler);

	static	NHBool		SetGlobalReserve(unsigned long size = 0x10000L);
	static	NHBool		GlobalReserveIsOK();
			NHBool		SetEmergencyReserve(unsigned int size = 0xFFFFU);
	static	void		SetEmergencyHook(fnEmergency f);

	const	char*		Name() { return name; }
			unsigned long	PreAlloc(unsigned long size, NHBool alert);
			void		Shrink(unsigned long limit = 0);
			unsigned long	GetFreeSize();
			unsigned long	GetBlockNumber() { return nTotalClientBlock; }
	#ifdef NH_DLL
	static	void		Exit();
	#endif

	//////////////////////
	// Public variables //
	//////////////////////
	static	NHeap*		_pMainHeap;

////////////////////////////////////////////////////////////////////////////////

private:
	/////////////
	// friends //
	/////////////
	friend	struct	NHSegment;
	friend	class	NHGetFreeSize;
	friend	class	NHUnfreeze;
	friend	class	NHHeapTab;
	friend	struct	NHEndWatcher;
	friend	class	NH_EXPORT	NHDebug;
	friend	class	NH_EXPORT	NHBlockWalker;
	friend	class	NH_EXPORT	NHEmergency;

	///////////////
	// Constants //
	///////////////
	enum
	{	// These are implementation dependant! See struct subBLOCK in internal.h
		nSubSeg = 32,
		maxSubBlockSize = 128,
		maxSlot = 26,
		maxSubElement = 256
	};

	///////////////////////
	// Private functions //
	///////////////////////
			NHBool		IsRegistrated();
	static	NHBool		NoMoreBlocks();
			void*		New(size_t size);
			void*		TryOSMalloc (unsigned long size, unsigned long mode);
			void*		TryOSRealloc (void *pBlock, unsigned long oldSize, unsigned long newSize, unsigned long mode);
			void*		TryOSXAlloc (void *pBlock, unsigned long oldSize, unsigned long newSize, unsigned long mode);
			void*		AllocCommonBlock (NHUWord requestedSize, unsigned long mode);
			void 		FreeCommonBlock (NHFreeBlock *block);
	static	void		UnlinkFreeBlock (NHFreeBlock *freeBlock, NHSegment *segment);
	static	void		UnlinkFreeBlock (NHFreeBlock *freeBlock, NHSlot *slot);
	static	void		LinkFreeBlock (NHFreeBlock *freeBlock, NHSegment *segment);
	static	void		LinkFreeBlock(NHFreeBlock *freeBlock, NHSlot* slot);
			NHSegment*	AllocSegment (const unsigned long reqBlockSize, unsigned long mode);
			NHSegment*	TryAllocSegment (const unsigned long reqBlockSize);
			void		FreeSegment (NHSegment *segment);
	static	NHBool		InitNewHeapDynamically();
			int			MaxClientHandlerTry();
			NHBool		GetClientHelp(unsigned long szNeeded, unsigned long mode);
			void		Destroy();
	static	NHBool		CheckBlockFlags(void *pClient, unsigned long mode);
			NHBool		SetNewSegmentPage();
	static	void		ReleaseSegmentPages();
			void		InitSubAllocator();

			NHeap (const NHeap&);
			NHeap& operator = (const NHeap&);

	////////////////////////////////
	// Private instance variables //
	////////////////////////////////
	const	char*		name;
			unsigned long	heapMode;
			NHSegment*	lruSegment;
			unsigned long	segmentBaseSize;
			unsigned long	segmentMoreSize;
			NHeap*		next;
			NHeap*		prev;
			NHosBlock*	pFirstOSBlock;
			unsigned long	nTotalClientBlock;
			void*		hFileMap;
			void*		pFileMap;
			unsigned long	totalCurrSegmentSize;
			unsigned long	szSegmentMinimum;
			NHRefSeg*	pFirstRefSeg;
			fnNewHandler	fpNewHandler;
			unsigned long	magicValue;
			NHSegment*	emergencySegment;
		struct NH_EXPORT NHSubSegSlot
		{
			NHSubSeg*	pFirst;
			long		count;
		} subSeg[nSubSeg];

	/////////////////////////////
	// Private class variables //
	/////////////////////////////
	static	NHBool		_classIsInitialized;
	static	NHBool		_initOnTheFly;

////////////////////////////////////////////////////////////////////////////////

#ifdef NH_DEBUG

	///////////////////////////////////////////////////
	// Public part specific to the debugging version //
	///////////////////////////////////////////////////

public:
	///////////////
	// Constants //
	///////////////
	enum DebugLevel
	{
		low, high
	};
	enum NoMemory
	{
		disabled, once, always
	};
	enum
	{
		terminal = 1,
		messBox = 2,
		callBack = 4,
		allTypes = 7
	};
	enum
	{
		isArray = 1,
		ignoreArray = 2,
		ignoreStats = 4
	};
	enum Mismatch
	{
		noMismatch,
		myMismatch,
		allMismatch
	};

	/////////////
	// typedef //
	/////////////
	typedef	void	(*fnAllocHook)(NHeap *heap, unsigned long requestedSize, unsigned long absCount, const char* fileName, int line);
	typedef	void	(*fnFreeHook)(void *block, const char* fileName, int line);
	typedef	void	(*fnErrorCallBack)(const char* message, void *data);
	typedef	NHBool	(*fnBlockInspect)(NHBlockInfo &info);

	//////////////////////
	// local structures //
	//////////////////////
	struct NH_EXPORT NHstat
	{
		long		nBlockCurrTot;
		long		nBlockAccuTot;
		long		nBlockMaxEverTot;
		long		szCurrTot;
		long		szMaxEverTot;
	};

	//////////////////////////////////////////////
	// Debugging versions of the main functions //
	//////////////////////////////////////////////
			void*		Alloc (unsigned long requestedSize, unsigned long mode = 0, int debugFlag = 0);
	static	void		Free (void *pBlock, int debugFlag = 0);

	///////////////////////////////////////////////
	// Public function for the debugging version //
	///////////////////////////////////////////////
	static	void		GetStatistics (NHeap *heap, NHeap::NHstat &stat);
	static	void		GetDeltaStatistics (NHeap *heap, NHeap::NHstat &difStat, NHeap::NHstat &oldStat);
	static	void		ShowStatistics (NHeap *heap = 0, const char *comment = 0);
	static	void		ShowDeltaStatistics (NHeap *heap, NHeap::NHstat &oldStat, const char *comment = 0);
	static	void		SetDebugLevel(DebugLevel level);
	static	void		SetReportType(int flag);
	static	void		DetectNewDeleteMismatch(Mismatch mode);
	static	void		FreezeFree(NHeap *heap = 0);
	static	void		UnfreezeFree(NHeap *heap = 0);
	static	void		SimulNoMemory (NoMemory mode, unsigned long szRequest = 0xFFFFFFFFL);
	static	void		SetAllocHook(fnAllocHook f, unsigned long count = 0);
	static	void		SetFreeHook(fnFreeHook f, NHBool all = 0);
	static	void		CatchOnFree(const void *pClient);
	static	void		SetErrorCallBack(fnErrorCallBack f);
	static	void		SetBlockGuards (unsigned long low, unsigned long top);
	static	void		ResetDebug();
	static	void		Monitor(NHBool state = 1, const char* file = 0, const char* mess = 0);
	static	unsigned long	GetAllocCounter();
	static	void		Scan(NHeap *heap, fnBlockInspect func);
	static	void		Break(const char* mess, void* p);
	// used internally
			void*		_new (unsigned long requestedSize, unsigned long mode, int debugFlag, int line, const char* fileName);
	static	void		_delete (void* p, int flag);
	static	NHBool		CheckBlock (void *ptr, const char *fileName, int line);
	static	int			PostDebugHeap();
			NHeap*		SetOrigin(const char *fileName, int line);
	static	NHeap*		_SetOrigin(const char *fileName, int line);
	static	void		InlineSetOrigin(const char *fileName,int line);

////////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////////////
	// Private part specific to the debugging version //
	////////////////////////////////////////////////////

private:
	/////////////
	// friends //
	/////////////
	friend	class	NH_EXPORT	NHMess;
	friend	class	NH_EXPORT	NHLeak;
	friend	class	NH_EXPORT	NHAcceptLeak;
	friend	class	NH_EXPORT	NHOverwrite;
	friend	class	NH_EXPORT	NHCheckAddress;
	friend	class	NH_EXPORT	NHContext;
	friend	class	NH_EXPORT	NHWalker;

	///////////////////////
	// Private functions //
	///////////////////////
			void*		_Alloc (unsigned long requestedSize, unsigned long mode);
	static	NHBool		_Resize (void * &pBlock, unsigned long newSize, NHBool clear);
			NHBool		SimulNoMemory(unsigned long mode);
	static	NHBool		RetrieveBlockInfo(const void *pClient0, NHBlockInfo *data, NHBool shortInfo);
	static	NHeap*		FindHisHeap(const void *p);
			NHBool		IsInsideHeapSpace(const void *p);
	static	NHBool		HasDebug(const void *pClient);
	static	NHBool		IsReallyBlock(const void *pClient);
	static	void		CheckClientPtr(const void *pClient, const char *fileName, int line, int callContext);
	static	void		ExamineClientPtr(const void *pClient, const char *fileName, int line, int callContext);
	static	void		FatalPtr(const void *pClient, const char* fileName, int line, int callContext);
	static	void		ShowStatistics (NHeap *heap,  NHeap::NHstat &stat, const char *comment, NHBool deltaMode);
			void		StatResize(long deltaSize);
			NHBool		FullDebug();
	static	NHBool		CheckLowDog (void *pClient);
	static	NHBool		CheckTopDog (void *pClient);
	static	NHReference*	GetReference(const char *fileName, int line);
	static	unsigned long	GetClient0Size(void *pClient0);
	static	NHBool		ReallyReport(NHReference *reference, NHBool checkAll);

	////////////////////////////////
	// Private instance variables //
	////////////////////////////////
			int			acceptLeak;
			NHBool		freezeFree;
			NHstat		stat;

#endif

////////////////////////////////////////////////////////////////////////////////

};	// end of class NHeap

/* -------------------------------------------------------------------------- */

struct NHBlock
{
	enum Type
	{
		common,
		sub,
		os
	};
};

/* -------------------------------------------------------------------------- */

// Small utility class

struct NH_EXPORT NHMem
{
	static	size_t		Count (const void *p, unsigned char chRef, size_t len);
	static	void  		Swap (void *p1, void *p2, size_t len);
	static	NHBool 		IsZero (const void *p, size_t size);
	static	NHBool		IsFilledWith (const void *p, unsigned char chRef, size_t size);
	static	void  		RemoveElement (void *adr, size_t nTotElem, size_t eSize, size_t index);
	static	void  		InsertElement (void *adr, size_t nTotElem, size_t eSize, size_t index, void *pNewData = 0);
	static	void  		Slide (void *dest, void *srce, size_t blockSize);
	static	NHBool		IsHuge(const void *p, unsigned long size)	{ return ((((unsigned long)p & 0xFFFFL) + size ) > 0x10000L); }	// no meaning in 32 bit mode!
};

/* -------------------------------------------------------------------------- */

// Make a small class just to unset automatically _emergencyEnabled when exiting
// from a function or compound.

class NH_EXPORT NHEmergency
{
public:
	NHEmergency();
	~NHEmergency();
};

#define	ENABLE_EMERGENCY	NHEmergency __EnableEmergency

/* -------------------------------------------------------------------------- */

// Declarations for the debugging versions of new and new[].

#ifdef NH_DEBUG
void * operator new(size_t size, int line, const char *fileName);
void * operator new(size_t size, NHeap &heap, int line, const char *fileName);
void * operator new(size_t size, NHeap &heap, unsigned long mode, int line, const char *fileName);

#ifdef COMPILER_HAS_NEW_ARRAY
void * operator new[](size_t size, int line, const char *fileName);
void * operator new[](size_t size, NHeap &heap, int line, const char *fileName);
void * operator new[](size_t size, NHeap &heap, unsigned long mode, int line, const char *fileName);
#endif

#else
void * operator new(size_t size, NHeap &heap);
void * operator new(size_t size, NHeap &heap, unsigned long mode);
#ifdef COMPILER_HAS_NEW_ARRAY
void * operator new[](size_t size, NHeap &heap);
void * operator new[](size_t size, NHeap &heap, unsigned long mode);
#endif
#endif

/* -------------------------------------------------------------------------- */

extern	NHeap __MainHeap;	// name of main heap object.

/* -------------------------------------------------------------------------- */

#define	MainAlloc				NHeap::Main()->Alloc
#define	MainAllocArray	  		NHeap::Main()->AllocArray
#define	MainStoreData			NHeap::Main()->StoreData
#define	MainStoreString			NHeap::Main()->StoreString
#define	MainPreAlloc			NHeap::Main()->PreAlloc
#define	MainShrink				NHeap::Main()->Shrink
#define	MainSetNewHandler		NHeap::Main()->SetNewHandler
#define	MainSetEmergencyReserve	NHeap::Main()->SetEmergencyReserve
#define	MainSimulNoMemory		NHeap::Main()->SimulNoMemory
#define	MainGetFreeSize			NHeap::Main()->GetFreeSize
#define	MainGetBlockNumber		NHeap::Main()->GetBlockNumber

/* -------------------------------------------------------------------------- */

#define	s_AUTO_FREE	"Unknown File (using AUTO_FREE)"

class NH_EXPORT NHAutoFree
{
public:
			NHAutoFree(void *_p) : p(_p) {}
			#ifdef NH_DEBUG
			~NHAutoFree()
			{
				NHeap::_SetOrigin(s_AUTO_FREE, 0);
				NHeap::Free(p, 0);
			}
			#else
			~NHAutoFree() { NHeap::Free(p); }
			#endif
private:
			void*		p;
};

/* -------------------------------------------------------------------------- */

#ifndef COMPILER_HAS_NO_TEMPLATE

#define	s_AUTO_DELETE		"Unknown File (using AUTO_DELETE)"
#define	s_AUTO_DELETE_ARRAY	"Unknown File (using AUTO_DELETE_ARRAY)"

template <class T>
class NHAutoDelete
{
public:
	NHAutoDelete(T *_p) : p(_p) {}
	#ifdef NH_DEBUG
	~NHAutoDelete();
	#else
	~NHAutoDelete() { delete p; }
	#endif
private:
	T *p;
};

#ifdef NH_DEBUG
template <class T>
NHAutoDelete<T>::~NHAutoDelete()	// not inline
{
	NHLocation xxx(s_AUTO_DELETE, 0);
	delete p;
}
#endif

template <class T>
class NHAutoDeleteA
{
public:
	NHAutoDeleteA(T *_p) : p(_p) {}
	#ifdef NH_DEBUG
	~NHAutoDeleteA();
	#else
	~NHAutoDeleteA() { delete[] p; }
	#endif
private:
	T *p;
};

#ifdef NH_DEBUG
template <class T>
NHAutoDeleteA<T>::~NHAutoDeleteA()	// not inline
{
	NHLocation xxx(s_AUTO_DELETE_ARRAY,0);
	delete[] p;
}
#endif

#define	AUTO_DELETE(p,cl)		NHAutoDelete<cl> del##p(p)
#define	AUTO_DELETE_ARRAY(p,cl)	NHAutoDeleteA<cl> del##p(p)

#endif

#define	AUTO_FREE(p)			NHAutoFree __inst_free##p(p)

/* -------------------------------------------------------------------------- */

// Base class to walk from block to block

class NH_EXPORT NHBlockWalker
{
public:
			NHBlockWalker();
	virtual	~NHBlockWalker(){};

			NHBool		ScanAllBlocks(NHeap *heap, int param, NHBool debug);

protected:
			NHBool 		TreatAllHeap() { return targetHeap == 0; }

			NHeap*		targetHeap;
	#ifdef NH_DEBUG
	const	char*		fileName;	// Place of invocation
			int			line;		// " " "
	#endif

private:
	virtual	NHBool		DoBlock(void *, NHBlock::Type, NHBool, int) { return 1; }
	virtual	NHBool		DoBlock2(void *) { return 1; }
	virtual	NHBool		DoFreeBlock(void *, unsigned long, NHBlock::Type, int){ return 1; }
	virtual	NHBool		DoFreeBlock2(unsigned long) { return 1; }
};

/* -------------------------------------------------------------------------- */

class NHBuff
{
public:
#ifdef NH_DEBUG
	NHBuff(size_t size, const char* fileName, int line) { p = NHeap::Main()->NHeap::SetOrigin(fileName,line)->Alloc(size,0,0); }
	NHBuff(const char* s, const char* fileName, int line) { p = NHeap::Main()->NHeap::SetOrigin(fileName,line)->Alloc(strlen(s)+1,0,0); strcpy((char*)p,s); }
	~NHBuff()
	{
		NHeap::_SetOrigin("Unknown File (using NHBuff)", 0);
		NHeap::Free(p,0);
	}
#else
	NHBuff(size_t size) { p = NHeap::Main()->Alloc(size,0); }
	NHBuff(const char* s) { p = NHeap::Main()->Alloc(strlen(s)+1,0); strcpy((char*)p,s); }
	~NHBuff() { NHeap::Free(p); }
#endif
	operator char* () { return (char *)p; }
private:
	NHBuff (const NHBuff&);
	NHBuff& operator = (const NHBuff&);
	void *p;
};

#ifdef NH_DEBUG
#define	NHBuff(size)	NHBuff(size,NH_FILE,__LINE__)
#endif

/* -------------------------------------------------------------------------- */

#ifdef NH_DEBUG

// Leak detection class

class NH_EXPORT NHLeak : public NHBlockWalker
{
public:
	enum { markBlock, unmarkBlock, findLeak };
			NHLeak(NHeap *heap, NHBool checkExternalLibrary, const char *fileName, int line);
	virtual	~NHLeak();

	unsigned long	Check (NHBool exiting, NHBool reset, const char *fileName, int line);
	void			SetModeAnyBlock() { anyBlock = 1; }

private:
	virtual	NHBool	DoBlock(void *p, NHBlock::Type type, NHBool isVFree, int param);
			NHBool	reset;
			NHBool	exiting;
			NHBool	anyBlock;
			NHUWord	mask;
			long	nLeaks;
			NHBool	checkExternalLibrary;
};

/* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  */

class NH_EXPORT NHAcceptLeak
{
public:
			NHAcceptLeak(NHeap *heap);
			~NHAcceptLeak();
private:
			NHeap*	targetHeap;
};

/* -------------------------------------------------------------------------- */

// Class to check if some blocks have been overwritten

class NH_EXPORT NHOverwrite : public NHBlockWalker
{
public:
	enum { markBlock, unmarkBlock, findLeak };

			NHOverwrite(NHeap *heap);
	virtual	~NHOverwrite(){};

			unsigned long	Check(const char *fileName, int line);
private:
	virtual	NHBool		DoBlock(void *p, NHBlock::Type type, NHBool isVFree, int param);
};

/* -------------------------------------------------------------------------- */

// Class to set a string on the stack chain

class NH_EXPORT NHContext
{
public:
	friend	class	NHeap;
	friend	class	NHDebug;

	NHContext(const char * const s, unsigned long passCount);
	~NHContext();
			void		Rename(const char * const s, unsigned long passCount);
	static	void		SetCurrentContext(NHContext *context, int direction);
	static	NHContext*	GetCurrentContext();
	static	const char*	GetCurrentName() { return GetNameFromId(GetCurrentContext()->ID); }

private:
	static const char*	GetNameFromId(NHUWord ID);
			NHBool		Register(const char * const s);
	static	NHUWord		GetIdFromName(const char* s);
			NHUWord		ID;
			NHContext*	last;
			unsigned long passCount;
};

/* -------------------------------------------------------------------------- */

class NH_EXPORT NHLocation
{
public:
	NHLocation(const char* file, int line);
	~NHLocation();
	static	const	char*	_fileName;
	static	int				_line;
	static	NHBool			_deleting;
private:
	const	char*			file;
			int				line;
};

/* -------------------------------------------------------------------------- */

// Structure passed when walking through the heap

struct NHBlockInfo
{
	const	char*		mess;		// message if data is found to be corrupted
	const	char*		heapName;
	const	char*		fileName;	// where block was allocated
			int			line;		// " " "
			unsigned long		localCount;	// local counter value when allocated
	const	char*		context;	// formatted string containing the current context strings
			unsigned long		reqSize;	// requested size when first allocated
			unsigned long		currSize;	// current size
			unsigned long		absCount;	// global allocation counter when allocated
};

#endif

////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//                               Macro section                                //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#undef	new
#undef	delete
#define	BIDS_LIBRARY	"BIDS Library"

#ifdef NH_DEBUG
	#define	NH_NEW(Heap,New,Delete,array)	\
		void* operator New(size_t size, int line, const char *fileName)	\
				{ return (Heap)._new(size,0,array,line,fileName); }	\
		void operator Delete(void *p) { NHeap::_delete(p,array); }
	#define	NH_NEW_EXT(New,Delete,array)	\
		void* operator New(size_t size, int line, const char *fileName)	\
				{ return NHeap::Main()->_new(size,0,array,line,fileName); }	\
		void* operator New(size_t size, NHeap& heap, int line, const char *fileName)	\
				{ return (heap)._new(size,0,array,line,fileName); }	\
		void* operator New(size_t size, NHeap& heap, unsigned long mode, int line, const char *fileName)	\
				{ return (heap)._new(size,(mode),array,line,fileName); }	\
		void operator Delete(void *p) { NHeap::_delete(p,array); }
	#define	NH_NEW_BIDS(New,Delete,array,klass,heap)	\
		friend	void* operator New(size_t size, const klass &)	\
				{ return (heap)._new(size,0,array,0,BIDS_LIBRARY); }	\
		void operator Delete(void *p) { NHeap::_delete(p,array); }
#else
	#define	NH_NEW(Heap,New,Delete,array)	\
		void* operator New(size_t size)	{ return (Heap).Alloc(size); }	\
		void operator Delete(void *p) { NHeap::Free(p); }
	#define	NH_NEW_EXT(New,Delete,array)	\
		void* operator New(size_t size) { return NHeap::Main()->Alloc(size,0); }	\
		void* operator New(size_t size, NHeap& heap) { return (heap).Alloc(size,0); }	\
		void* operator New(size_t size, NHeap& heap, unsigned long mode) { return (heap).Alloc(size,(mode)); }	\
		void operator Delete(void *p) { NHeap::Free(p); }
	#define	NH_NEW_BIDS(New,Delete,array,klass,heap)	\
		friend	void* operator New(size_t size, const klass &) { return (heap).Alloc(size,0); }	\
		void operator Delete(void *p) { NHeap::Free(p); }
#endif

#ifdef COMPILER_HAS_NEW_ARRAY
	#define	ASSIGN_HEAP(Heap)	NH_NEW(Heap,new,delete,0) NH_NEW(Heap,new[],delete[],1)
	#define	OVERLOAD_NEW		NH_NEW_EXT(new,delete,0) NH_NEW_EXT(new[],delete[],1)
	#define	OVERLOAD_NEW_BIDS(klass,heap)	NH_NEW_BIDS(new,delete,0,klass,heap) NH_NEW_BIDS(new[],delete[],1,klass,heap)
	#define	NH_BIDS(klass,heap)	struct klass { OVERLOAD_NEW_BIDS(klass,heap); };
#else
	#define	ASSIGN_HEAP(Heap)	NH_NEW(Heap,new,delete,0)
	#define	OVERLOAD_NEW		NH_NEW_EXT(new,delete,0)
#endif


/* -------------------------------------------------------------------------- */

#endif

