#include "stdafx.h"

#include "template.h"

#include "minmax.h"


#pragma pack(1)
struct dialog_template_struct
	{
	long dtStyle;
	BYTE dtItemCount;
	WORD dtX, dtY;
	WORD dtCX, dtCY;
	};
#pragma pack()

struct dialog_buf_item_struct
	{
	int dtilX, dtilY;
	int dtilCX, dtilCY;
	unsigned int dtilID;
	long dtilStyle;
	};

long CDialogTemplate::m_lDialogUnits = 0L;

#ifdef _DEBUG
	// Define this to get debugging trace output
	// #define DEBUG
	#undef THIS_FILE
	static char BASED_CODE THIS_FILE[] = __FILE__;
	#define new DEBUG_NEW
#else
	// Define this to get debugging trace output in a release version
	// #define RELEASE_DEBUG
#endif

	
CDialogItem::CDialogItem(CDialogTemplate *pTemplate, LPCSTR szClass, long lStyle)
{
	m_rect.SetRect(-1, -1, -1, -1);
	m_uID = 0;
	m_lStyle = lStyle | WS_CHILD | WS_VISIBLE;
	m_szClass = szClass;
	m_szText.Empty();
	m_pTemplate = pTemplate;
}


CDialogItem::CDialogItem(CDialogTemplate *pTemplate, unsigned char cClass, long lStyle)
{
	m_rect.SetRect(-1, -1, -1, -1);
	m_uID = 0;
	m_szClass = (char)cClass;
	m_szText.Empty();
	m_pTemplate = pTemplate;

	if (lStyle != LONG_MIN)
		m_lStyle = lStyle | WS_CHILD | WS_VISIBLE;
	else switch (cClass)
		{
		case CLASS_BUTTON :
			m_lStyle = BS_PUSHBUTTON | WS_TABSTOP | WS_CHILD | WS_VISIBLE;
			break;
		case CLASS_EDIT :
			m_lStyle = ES_LEFT | WS_TABSTOP | WS_CHILD | WS_VISIBLE;
			break;
		case CLASS_STATIC :
			m_lStyle = SS_LEFT | SS_NOPREFIX | WS_CHILD | WS_VISIBLE;
			break;
		case CLASS_LISTBOX :
			m_lStyle = WS_TABSTOP | WS_CHILD | WS_VISIBLE;
			break;
		case CLASS_COMBOBOX :
			m_lStyle = WS_TABSTOP | CBS_DROPDOWN | WS_CHILD | WS_VISIBLE;
			break;
		case CLASS_SCROLLBAR :
		default :
			m_lStyle = WS_CHILD | WS_VISIBLE;
			break;
		}
}


CDialogItem::CDialogItem(CDialogTemplate *pTemplate, const char *pData, int *piBytesUsed)
{
	int iUsed, iLen, nData;
	const struct dialog_buf_item_struct *pItem;

#ifdef DEBUG
	TRACE("CDialogItem::CDialogItem(pTemplate, pData, piBytesUsed) entered [this=%04X:%04X]\n",
		SELECTOROF(this), OFFSETOF(this));
#endif

	m_pTemplate = pTemplate;
	
	pItem = (const struct dialog_buf_item_struct *)pData;
	pData += sizeof(struct dialog_buf_item_struct);
	iUsed = sizeof(struct dialog_buf_item_struct);

	m_uID = pItem->dtilID;
	m_lStyle = pItem->dtilStyle;
#ifdef DEBUG
	TRACE("   iId = %04X; lStyle = %08lX\n", m_uID, m_lStyle);
#endif

	m_rect.left = pItem->dtilX;
	m_rect.top = pItem->dtilY;
	m_rect.right = pItem->dtilX + pItem->dtilCX;
	m_rect.bottom = pItem->dtilY + pItem->dtilCY;

	// pData points to the class right now!
	if (pData[0] & 0x80)
		{
		m_szClass = pData[0];
		pData++;
		iUsed++;
		}
	else
		{
		m_szClass = pData;
		iLen = lstrlen(pData);
		pData += iLen + 1;
		iUsed += iLen + 1;
		}

	m_szText = pData;
	iLen = lstrlen(pData) + 1;
	pData += iLen;
	iUsed += iLen;
#ifdef DEBUG
	TRACE("   szText = \"%s\"\n", (LPCSTR)m_szText);
#endif

    // Check for extra data but toss it for now ...
	nData = pData[0];
	pData++;
	iUsed++;

	if (nData)
		iUsed += nData;

	*piBytesUsed = iUsed;
}

		
CDialogItem::CDialogItem(CDialogTemplate *pTemplate, const CDialogItem *pSrcItem)
{
#ifdef DEBUG
	TRACE("CDialogItem::CDialogItem(pTemplate=%04X:%04X, pSrcItem=%04X:%04X) entered [this=%04X:%04X]\n",
		SELECTOROF(pTemplate), OFFSETOF(pTemplate),
		SELECTOROF(pSrcItem), OFFSETOF(pSrcItem),
		SELECTOROF(this), OFFSETOF(this));
#endif

	ASSERT(pTemplate != NULL);
	ASSERT(pSrcItem != NULL);

	m_rect = pSrcItem->m_rect;
	m_uID = pSrcItem->m_uID;
	m_lStyle = pSrcItem->m_lStyle;
	m_iFocusPriority = pSrcItem->m_iFocusPriority;
	m_szClass = pSrcItem->m_szClass;
	m_szText = pSrcItem->m_szText;
	m_pTemplate = pTemplate;
}


CPoint CDialogItem::get_position(void)
{
	CPoint pt;

	pt.x = m_rect.left;
	pt.y = m_rect.top;

	return pt;
}


CDialogItem *CDialogItem::set_size(int iWidth, int iHeight)
{
	m_rect.right = m_rect.left + iWidth;
	m_rect.bottom = m_rect.top + iHeight;

	return this;
}


CDialogItem *CDialogItem::set_text(LPCSTR szText)
{
	CSize size;

	m_szText = szText;

	if ( (m_rect.bottom == -1) && (m_rect.right == -1) && (m_szClass[0] & 0x80) )
		{
		switch (m_szClass[0])
			{
			case CLASS_BUTTON :
				{
				int iType = (int)(m_lStyle & 0x0F);

				size = m_pTemplate->get_dc()->GetTextExtent(szText, lstrlen(szText));
				switch (iType)
					{
					case BS_PUSHBUTTON :
					case BS_DEFPUSHBUTTON :
						size.cx = max(size.cx + 12, 72);
						size.cy = max(size.cy + 12, 28);
						set_size(m_pTemplate->pixel_to_dialog(size));
						break;
					case BS_RADIOBUTTON :
					case BS_AUTORADIOBUTTON :
					case BS_CHECKBOX :
					case BS_AUTOCHECKBOX :
						size.cx = size.cx + 20;
						size.cy = max(size.cy + 4, 20);
						set_size(m_pTemplate->pixel_to_dialog(size));
						break;
					default :
						break;
					}
				}
				break;
			case CLASS_EDIT :
				size = m_pTemplate->pixel_to_dialog(m_pTemplate->get_dc()->GetTextExtent(szText, lstrlen(szText)));
				set_size(size.cx + 4, size.cy + 4);
				break;
			case CLASS_STATIC :
				size = m_pTemplate->pixel_to_dialog(m_pTemplate->get_dc()->GetTextExtent(szText, lstrlen(szText)));
				set_size(size.cx, size.cy);
				break;
			case CLASS_LISTBOX :
				size = m_pTemplate->pixel_to_dialog(m_pTemplate->get_dc()->GetTextExtent(szText, lstrlen(szText)));
				set_size(size.cx, size.cy);
				break;
			case CLASS_COMBOBOX :
				// Height = 12 + 8 * nVisibleItems
				size = m_pTemplate->pixel_to_dialog(m_pTemplate->get_dc()->GetTextExtent(szText, lstrlen(szText)));
				set_size(size.cx, size.cy);
				break;
			case CLASS_SCROLLBAR :
			default :
				break;
			}
		}
		
	return this;
}


// If bClosed is FALSE and the item is a dropdown (or dropdown list)
// combobox, the height returned is the open height.  If bClosed is TRUE
// (the default) or if the item is not a dropdown (or dropdown list)
// combobox, the open (or actual) height is returned.
int CDialogItem::height(BOOL bClosed) const
{
	CSize size;
	CDC *pDC;

#ifdef DEBUG
	TRACE("CDialogItem::height(%d) entered\n", bClosed);
#endif

	ASSERT(m_pTemplate != NULL);

	if ( (bClosed) && (m_szClass[0] == CLASS_COMBOBOX) )
		{
		if ( ((m_lStyle & 0x0003) == CBS_DROPDOWN)
		  || ((m_lStyle & 0x0003) == CBS_DROPDOWNLIST) )
			{
			pDC = m_pTemplate->get_dc();
			ASSERT(pDC != NULL);
			size = pDC->GetTextExtent("X", 1);
#ifdef DEBUG
			TRACE("   Returning %d + 8 pixels, converted to dialog units\n", size.cy);
#endif
			return m_pTemplate->pixel_y_to_dialog(size.cy + 8);
			}
		}

	return m_rect.Height();
}


unsigned int CDialogItem::bytes_to_generate(void) const
{
	unsigned int uLen;

	uLen = sizeof(struct dialog_buf_item_struct);

	if (m_szClass[0] & 0x80)
		uLen += 1;
	else
		uLen += m_szClass.GetLength() + 1;

	if (m_szText.IsEmpty())
		uLen += 1;
	else
		uLen += m_szText.GetLength() + 1;

	uLen += 1;

	return uLen;
}


unsigned int CDialogItem::generate(char *pBuffer) const
{
	struct dialog_buf_item_struct sHeader;
	int iLen;
	unsigned int uLen;
#ifdef RELEASE_DEBUG
	char szDebug[100];
#endif

#ifdef DEBUG
	TRACE("CDialogItem::generate(%08lX) entered [this=%08lX]\n",
		pBuffer, this);
#endif
#ifdef RELEASE_DEBUG
	sprintf(szDebug, "CDialogItem::generate(%08lX) entered [this=%08lX]\n",
		pBuffer, this);
	OutputDebugString(szDebug);
#endif

	// I used to check for m_rect .left<0 and .top<0 but now I figure,
	// if the user wants that, who am I to stop him/her?
	if (m_rect.IsRectEmpty())
		{
#ifdef DEBUG
		TRACE0("   Empty rectangle!\n");
#endif
		return 0;
		}
	if (m_szClass.IsEmpty())
		{
#ifdef DEBUG
		TRACE0("   Class not defined!\n");
#endif
		return 0;
		}
	if (!(m_lStyle & WS_CHILD))
		{
#ifdef DEBUG
		TRACE0("   WS_CHILD not defined!\n");
#endif
		return 0;
		}

	sHeader.dtilX = m_rect.left;
	sHeader.dtilY = m_rect.top;
	sHeader.dtilCX = m_rect.Width();
	sHeader.dtilCY = m_rect.Height();
	sHeader.dtilID = m_uID;
	sHeader.dtilStyle = m_lStyle;

#ifdef DEBUG
	TRACE("   Rect=(%d,%d),(%d,%d); ID=%04X; style=%08lX written to %08lX\n",
		sHeader.dtilX,
		sHeader.dtilY,
		sHeader.dtilX + sHeader.dtilCX,
		sHeader.dtilY + sHeader.dtilCY,
		sHeader.dtilID, sHeader.dtilStyle,
		pBuffer);
#endif
#ifdef RELEASE_DEBUG
	sprintf(szDebug, "   Rect=(%d,%d),(%d,%d); ID=%04X; style=%08lX written to %08lX\n",
		sHeader.dtilX,
		sHeader.dtilY,
		sHeader.dtilX + sHeader.dtilCX,
		sHeader.dtilY + sHeader.dtilCY,
		sHeader.dtilID, sHeader.dtilStyle,
		pBuffer);
	OutputDebugString(szDebug);
#endif

	memcpy(pBuffer, &sHeader, sizeof(sHeader));
	uLen = sizeof(sHeader);

	// Write class info
	if (m_szClass[0] & 0x80)
		{
		iLen = 1;
#ifdef DEBUG
		TRACE("   Class 0x%02X (1 byte) written to %08lX\n",
			(int)m_szClass[0] & 0x00FF, pBuffer + uLen);
#endif
		}
	else
		{
		iLen = m_szClass.GetLength() + 1;
#ifdef DEBUG
		TRACE("   Class \"%s\" (%d bytes) written to %08lX\n",
			(LPCSTR)m_szClass, iLen, pBuffer + uLen);
#endif
		}
	memcpy(pBuffer + uLen, (LPCSTR)m_szClass, iLen);
	uLen += iLen;

	if (m_szText.IsEmpty())
		{
		pBuffer[uLen++] = '\0';
#ifdef DEBUG
		TRACE0("   Empty text \'\\0\' (1 byte) written\n");
#endif
		}
	else
		{
		iLen = m_szText.GetLength() + 1;
#ifdef DEBUG
		TRACE("   Text \"%s\" (%d bytes) written\n",
			(LPCSTR)m_szText, iLen);
#endif
		memcpy(pBuffer + uLen, (LPCSTR)m_szText, iLen);
		uLen += iLen;
		}

	// No extra data
	pBuffer[uLen++] = '\0';

	return uLen;
}
	

HWND CDialogItem::Create(HWND hWndParent, HINSTANCE hInstance)
{
	LPCSTR szClass;
	CRect rect;

	// Get class name
	if (m_szClass.IsEmpty())
		return (HWND)0;
	switch (m_szClass[0])
		{
		case CLASS_BUTTON :
			szClass = "BUTTON";
			break;
		case CLASS_EDIT :
			szClass = "EDIT";
			break;
		case CLASS_STATIC :
			szClass = "STATIC";
			break;
		case CLASS_LISTBOX :
			szClass = "LISTBOX";
			break;
		case CLASS_SCROLLBAR :
			szClass = "SCROLLBAR";
			break;
		case CLASS_COMBOBOX :
			szClass = "COMBOBOX";
			break;
		default :
			szClass = (LPCSTR)m_szClass;
			break;
		}

	// Get rectangle (in pixels, not dialog units)
	rect = m_pTemplate->dialog_to_pixel(m_rect);

	return CreateWindow(
		szClass,						// Address of registered class name
		(LPCSTR)m_szText,				// Address of window text
		m_lStyle,						// Window style
		rect.left,						// Horizontal position of window
		rect.top,						// Vertical position of window
		rect.Width(),					// Window width
		rect.Height(),					// Window height
		hWndParent,						// Handle of parent window
		(HMENU)m_uID,					// Child-window identifier
		hInstance,						// Handle of application instance
		NULL);							// Address of window-creation data
}


CDialogTemplate::CDialogTemplate(CWnd *pParent)
{
	CWinApp *pApp;

#ifdef DEBUG
	TRACE("CDialogTemplate::CDialogTemplate(%04X:%04X) entered [this=%04X:%04X]\n",
		SELECTOROF(pParent), OFFSETOF(pParent), SELECTOROF(this), OFFSETOF(this));
#endif

	m_lStyle = WS_BORDER | WS_DLGFRAME | WS_POPUP | WS_VISIBLE | WS_CLIPSIBLINGS
		| DS_MODALFRAME | WS_SYSMENU;
	m_rect.SetRectEmpty();
	m_pParent = pParent;

	m_pDC = NULL;
	if (m_pParent)
		{
		m_pDC = new CClientDC(m_pParent);
		}
	else
		{
		pApp = AfxGetApp();
		ASSERT(pApp != NULL);
		ASSERT(pApp->m_pMainWnd != NULL);
		m_pDC = new CClientDC(pApp->m_pMainWnd);
		}
	ASSERT_VALID(m_pDC);		// Also checks for m_pDC != NULL

	// At this point, the DC should contain the system font since we haven't
	// put anything different into it.
	if (!m_lDialogUnits)
		m_lDialogUnits = GetDialogBaseUnits();
	m_lXFontConvert = (long)LOWORD(m_lDialogUnits) * 1000L / 4L;
	m_lYFontConvert = (long)HIWORD(m_lDialogUnits) * 1000L / 8L;
#ifdef DEBUG
	TRACE("   Font conversion factors initialized to (%ld,%ld)\n",
		m_lXFontConvert, m_lYFontConvert);
#endif
}
	

CDialogTemplate::~CDialogTemplate()
{
	CDialogItem *pItem;

#ifdef DEBUG
	TRACE0("CDialogTemplate::~CDialogTemplate() entered\n");
#endif

	while (m_aItems.GetSize())
		{
		pItem = (CDialogItem *)m_aItems.GetAt(0);
		delete pItem;
		m_aItems.RemoveAt(0);
		}

	if (m_pDC)
		delete m_pDC;
}


BOOL CDialogTemplate::load(LPCSTR szTemplateName)
{
	CResource hResource;

#ifdef DEBUG
	if (SELECTOROF(szTemplateName))
		TRACE("CDialogTemplate::load(\"%s\") entered\n", szTemplateName);
	else
		TRACE("CDialogTemplate::load(%04X) entered\n", OFFSETOF(szTemplateName));
#endif

	if (!hResource.find(szTemplateName, RT_DIALOG))
		{
#ifdef DEBUG
		TRACE0("   hResource.find() failed!\n");
#endif
		return FALSE;
		}

	return load(hResource);
}


BOOL CDialogTemplate::load(int iTemplateID)
{
	CResource hResource;
	
#ifdef DEBUG
	TRACE("CDialogTemplate::load(%04X) entered\n", iTemplateID);
#endif

	if (!hResource.find(iTemplateID, RT_DIALOG))
		{
#ifdef DEBUG
		TRACE0("   hResource.find() failed!\n");
#endif
		return FALSE;
		}

	return load(hResource);
}


BOOL CDialogTemplate::load(CResource &hResource)
{
	char *pData;
	int i, iUsed;
	struct dialog_template_struct *pHeader;
	WORD wPointSize;
	CString szFaceName;
	CDialogItem *pItem;

#ifdef DEBUG
	TRACE("CDialogTemplate::load(hResource) entered at %08ld\n", GetTickCount());
#endif

	VERIFY(hResource.load());

	pData = (char *)hResource.lock();
	ASSERT(pData);

	// Extract header
	pHeader = (struct dialog_template_struct *)pData;
	pData += sizeof(struct dialog_template_struct);
	m_lStyle = pHeader->dtStyle;
	m_rect.left = pHeader->dtX;
	m_rect.top = pHeader->dtY;
	m_rect.right = m_rect.left + pHeader->dtCX;
	m_rect.bottom = m_rect.top + pHeader->dtCY;

	// Get menu name
	m_szMenuName = pData;
	pData += lstrlen(pData) + 1;
#ifdef DEBUG
	TRACE("   Menu name \"%s\"\n", (LPCSTR)m_szMenuName);
#endif

	// Get class name
	m_szClassName = pData;
	pData += lstrlen(pData) + 1;
#ifdef DEBUG
	TRACE("   Class name \"%s\"\n", (LPCSTR)m_szClassName);
#endif

	// Get caption text
	m_szCaption = pData;
	pData += lstrlen(pData) + 1;
#ifdef DEBUG
	TRACE("   Caption text \"%s\"\n", (LPCSTR)m_szCaption);
#endif

    // Get font, if specified
    if (m_lStyle & DS_SETFONT)
    	{
    	wPointSize = *((WORD *)pData);
    	pData += sizeof(WORD);
    	szFaceName = pData;
    	pData += lstrlen(pData) + 1;
    	set_font(szFaceName, wPointSize);
    	}
    else
    	{
    	m_wPointSize = 0;
    	m_szFaceName.Empty();
    	}
	
	// Load items
	for (i = 0 ; i < (int)pHeader->dtItemCount ; i++)
		{
		pItem = new CDialogItem(this, pData, &iUsed);
		m_aItems.Add(pItem);
		pData += iUsed;
		}

	// Done with the resource - unlock it
	hResource.unlock();

#ifdef DEBUG
	TRACE("CDialogTemplate::load() complete at %08ld\n", GetTickCount());
#endif

	return TRUE;
}


CDialogTemplate & CDialogTemplate::set_size(int iWidth, int iHeight)
{
	m_rect.right = m_rect.left + iWidth;
	m_rect.bottom = m_rect.top + iHeight;
	
	return *this;
}


CSize CDialogTemplate::get_size(void)
{
	CSize size;

	size.cx = m_rect.Width();
	size.cy = m_rect.Height();

	return size;
}


CDialogItem *CDialogTemplate::add_item(LPCSTR szClass, long lStyle)
{
	CDialogItem *pItem;
	
	pItem = new CDialogItem(this, szClass, lStyle);
	
	m_aItems.Add(pItem);
	
	return pItem;
}


CDialogItem *CDialogTemplate::add_item(unsigned char cClass, long lStyle)
{
	CDialogItem *pItem;
	
	pItem = new CDialogItem(this, cClass, lStyle);
	
	m_aItems.Add(pItem);
	
	return pItem;
}


CDialogItem *CDialogTemplate::add_item(const CDialogItem *pItemToCopy, int iIndex)
{
	CDialogItem *pItem;

	pItem = new CDialogItem(this, pItemToCopy);

	if (iIndex < 0)
		m_aItems.Add(pItem);
	else
		m_aItems.InsertAt(iIndex, pItem);

	return pItem;
}


// This function can be called by the application but is also called during
// load() if a font is specified.
BOOL CDialogTemplate::set_font(LPCSTR szFaceName, WORD wPointSize)
{
	LOGFONT logFont;
	CFont setFont, *pOldFont;
	CSize sysSize, setSize;
	static const char szAlphabet[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

#ifdef DEBUG
	TRACE("CDialogTemplate::set_font(\"%s\", %d) entered\n", szFaceName, wPointSize);
#endif

	m_wPointSize = wPointSize;
	m_szFaceName = szFaceName;

	ASSERT(m_pDC != NULL);

	memset(&logFont, 0, sizeof(LOGFONT));
	logFont.lfHeight = -MulDiv(wPointSize, m_pDC->GetDeviceCaps(LOGPIXELSY), 72);
	strcpy(logFont.lfFaceName, szFaceName);
	logFont.lfWeight = FW_BOLD;	// It appears Windows does this when selecting
								// a font for a dialog box

	if (!setFont.CreateFontIndirect(&logFont))
		{
#ifdef DEBUG
		TRACE("   Unable to create font %d pixels high\n", logFont.lfHeight);
#endif
		return FALSE;
		}

#ifdef DEBUG
	TRACE("   Font created, %d pixels high\n", logFont.lfHeight);
#endif

	// Get sizes in system and new font
	sysSize = m_pDC->GetTextExtent(szAlphabet, sizeof(szAlphabet));
	pOldFont = m_pDC->SelectObject(&setFont);
	setSize = m_pDC->GetTextExtent(szAlphabet, sizeof(szAlphabet));
	m_pDC->SelectObject(pOldFont);
#ifdef DEBUG
	TRACE("   Text size : system (%d,%d) new (%d,%d)\n",
		sysSize.cx, sysSize.cy, setSize.cx, setSize.cy);
#endif

	if (!m_lDialogUnits)
		m_lDialogUnits = GetDialogBaseUnits();
	m_lXFontConvert = (long)LOWORD(m_lDialogUnits) * 1000L * setSize.cx / sysSize.cx / 4L;
	m_lYFontConvert = (long)HIWORD(m_lDialogUnits) * 1000L * setSize.cy / sysSize.cy / 8L;
#ifdef DEBUG
	TRACE("   Font conversion factors set to (%ld,%ld)\n",
		m_lXFontConvert, m_lYFontConvert);
#endif

	return TRUE;
}


HGLOBAL CDialogTemplate::generate(void)
{
	int i;
	CDialogItem *pItem;
	struct dialog_template_struct sHeader;
	HGLOBAL hData;
	unsigned int uSize, uLen, uItemLen;
	char *pData;
#ifdef RELEASE_DEBUG
	char szDebug[80];
#endif

#ifdef DEBUG
	TRACE0("CDialogTemplate::generate() entered\n");
#endif

	if (m_rect.IsRectEmpty())
		{
#ifdef DEBUG
		TRACE0("   m_rect.IsRectEmpty()!\n");
#endif
		return 0;
		}

	// How big will the template be?
	uSize = sizeof(sHeader);
	uSize += m_szMenuName.GetLength() + 1;
	uSize += m_szClassName.GetLength() + 1;
	uSize += m_szCaption.GetLength() + 1;
	if (m_lStyle & DS_SETFONT)
		{
		uSize += sizeof(WORD);
		uSize += m_szFaceName.GetLength() + 1;
		}
	for (i = 0 ; i < num_items() ; i++)
		{
		pItem = get_item(i);
		ASSERT(pItem != NULL);
		uSize += pItem->bytes_to_generate();
		}

	// Allocate data for the template
	hData = GlobalAlloc(GMEM_MOVEABLE | GMEM_DDESHARE, uSize);
	if (!hData)
		return 0;
	pData = (char *)GlobalLock(hData);
	if (!pData)
		{
		GlobalFree(hData);
		return 0;
		}
#ifdef DEBUG
	TRACE("   %d bytes allocated (handle %04X @ %08lX)\n", uSize, (WORD)hData, pData);
#endif

	sHeader.dtStyle = m_lStyle;
	sHeader.dtItemCount = m_aItems.GetSize();
	sHeader.dtX = m_rect.left;
	sHeader.dtY = m_rect.top;
	sHeader.dtCX = m_rect.Width();
	sHeader.dtCY = m_rect.Height();
#ifdef RELEASE_DEBUG
	sprintf(szDebug, "Generating %d items at (%d,%d) for (%d,%d), style %08lX\n",
		sHeader.dtItemCount, sHeader.dtX, sHeader.dtY, sHeader.dtCX, sHeader.dtCY, sHeader.dtStyle);
	OutputDebugString(szDebug);
#endif

	memcpy(pData, &sHeader, sizeof(sHeader));
	uLen = sizeof(sHeader);

	// MFC is very nice and returns a pointer to '\0' when you cast an empty
	// CString to LPCSTR!
	memcpy(pData + uLen, (LPCSTR)m_szMenuName, m_szMenuName.GetLength() + 1);
	uLen += m_szMenuName.GetLength() + 1;
	memcpy(pData + uLen, (LPCSTR)m_szClassName, m_szClassName.GetLength() + 1);
	uLen += m_szClassName.GetLength() + 1;
	memcpy(pData + uLen, (LPCSTR)m_szCaption, m_szCaption.GetLength() + 1);
	uLen += m_szCaption.GetLength() + 1;
	
	if (m_lStyle & DS_SETFONT)
		{
		memcpy(pData + uLen, &m_wPointSize, sizeof(WORD));
		uLen += sizeof(WORD);
		memcpy(pData + uLen, (LPCSTR)m_szFaceName, m_szFaceName.GetLength() + 1);
		uLen += m_szFaceName.GetLength() + 1;
		}

	for (i = 0 ; i < num_items() ; i++)
		{
		pItem = get_item(i);
		ASSERT(pItem != NULL);
		uItemLen = pItem->generate(pData + uLen);
		if (!uItemLen)
			{
#ifdef DEBUG
			TRACE("   Item [%d] cannot be generated!\n", i);
#endif
			return NULL;
			}
		uLen += uItemLen;
		}

	ASSERT(uLen == uSize);

	GlobalUnlock(hData);

	return hData;
}
	

int CDialogTemplate::find_item_index(UINT uID) const
{
	int i;
	const CDialogItem *pItem;

	for (i = 0 ; i < num_items() ; i++)
		{
		pItem = get_item(i);
		if (pItem->get_id() == uID)
			return i;
		}

	return -1;
}


CDialogItem *CDialogTemplate::find_item(UINT uID)
{
	int iIndex = find_item_index(uID);

	if (iIndex >= 0)
		return get_item(iIndex);
	else
		return NULL;
}


BOOL CDialogTemplate::remove_item(UINT uID)
{
	int i;
	CDialogItem *pItem;

	for (i = 0 ; i < num_items() ; i++)
		{
		pItem = get_item(i);
		if (pItem->get_id() == uID)
			{
			m_aItems.RemoveAt(i);
			delete pItem;
			return TRUE;
			}
		}

	return FALSE;
}


UINT CDialogTemplate::unused_id(void) const
{
	int i;
	UINT uID;
	CDialogItem *pItem;

	for (uID = 0x1000 ; uID < 0x7FFF ; uID++)
		{
		for (i = 0 ; i < m_aItems.GetSize() ; i++)
			{
			pItem = (CDialogItem *)m_aItems[i];
			ASSERT(pItem != NULL);
			if (pItem->get_id() == uID)
				break;
			}
		if (i >= m_aItems.GetSize())
			return uID;
		}

	// Hopefully we never get here
	ASSERT(0);
	return 0;
}


CSize CDialogTemplate::GetTextExtent(LPCSTR szText)
{
	ASSERT(m_pDC != NULL);

	return m_pDC->GetTextExtent(szText, lstrlen(szText));
}


int CDialogTemplate::pixel_x_to_dialog(int iPixelX)
{
	int iDialogX;

	// Here's the deal : if you want to convert pixels to dialog units, I assume
	// you are doing this because you measured some text (in pixels) and want to
	// know its size in dialog units.  To do this, we need to divide by the
	// dialog unit factor for the font in which you measured the text in.  The
	// GetDialogBaseUnits() is only applicable when the text was measured in
	// the default system font.  That's why the m_pDC for this template is always
	// set to the default system font.

	if (!m_lDialogUnits)
		m_lDialogUnits = GetDialogBaseUnits();

	if (iPixelX < 0)
		iDialogX = -1;
	else
		iDialogX = (iPixelX * 4 + LOWORD(m_lDialogUnits) / 2) / LOWORD(m_lDialogUnits);

#ifdef DEBUG
	TRACE("pixel_x_to_dialog() : %d -> %d\n", iPixelX, iDialogX);
#endif

	return iDialogX;
}


int CDialogTemplate::pixel_y_to_dialog(int iPixelY)
{
  int iDialogY;

	if (!m_lDialogUnits)
		m_lDialogUnits = GetDialogBaseUnits();

  if (iPixelY < 0)
    iDialogY = -1;
  else
    iDialogY = (iPixelY * 8 + HIWORD(m_lDialogUnits) / 2) / HIWORD(m_lDialogUnits);

#ifdef DEBUG
	TRACE("pixel_y_to_dialog() : %d -> %d\n", iPixelY, iDialogY);
#endif

	return iDialogY;
}


CSize CDialogTemplate::pixel_to_dialog(const CSize &size)
{
	CSize newsize;
	
	newsize.cx = pixel_x_to_dialog(size.cx);
	newsize.cy = pixel_y_to_dialog(size.cy);

	return newsize;
}


CRect CDialogTemplate::pixel_to_dialog(const CRect &rect)
{
	CRect newRect;
	
	newRect.left = pixel_x_to_dialog(rect.left);
	newRect.top = pixel_y_to_dialog(rect.top);
	newRect.right = pixel_x_to_dialog(rect.right);
	newRect.bottom = pixel_y_to_dialog(rect.bottom);

	return newRect;
}


int CDialogTemplate::dialog_x_to_pixel(int x)
{
	int iPixelX;

	// Here's the deal : if you want to convert dialog units to pixels, I assume
	// you are doing this because you want to know how big something will be when
	// it's actually created.  Windows does this using the font for this dialog;
	// that means I have to use the dialog conversion factors I calculated
	// previously for the current dialog font.

	if (x < 0)
		iPixelX = -1;
	else
		iPixelX = (int)((long)x * m_lXFontConvert / 1000L);

#ifdef DEBUG
	TRACE("dialog_x_to_pixel() : %d via %ld -> %d\n", x, m_lXFontConvert, iPixelX);
#endif

	return iPixelX;
}


int CDialogTemplate::dialog_y_to_pixel(int y)
{
	int iPixelY;

	if (y < 0)
		iPixelY = -1;
	else
		iPixelY = (int)((long)y * m_lYFontConvert / 1000L);

#ifdef DEBUG
	TRACE("dialog_y_to_pixel() : %d via %ld -> %d\n", y, m_lYFontConvert, iPixelY);
#endif

	return iPixelY;
}


CSize CDialogTemplate::dialog_to_pixel(const CSize &size)
{
	CSize newsize;
	
	newsize.cx = dialog_x_to_pixel(size.cx);
	newsize.cy = dialog_y_to_pixel(size.cy);
	
	return newsize;
}


CRect CDialogTemplate::dialog_to_pixel(const CRect &rect)
{
	CRect newRect;
	
	newRect.left = dialog_x_to_pixel(rect.left);
	newRect.top = dialog_y_to_pixel(rect.top);
	newRect.right = dialog_x_to_pixel(rect.right);
	newRect.bottom = dialog_y_to_pixel(rect.bottom);

	return newRect;
}

