// ------------------------------------------------------------------------
//
//   Create a class that handles multiple screen dialog boxes
//
//   Parent dialog box must contain a groupbox (or other control) which is
//   the same size and position as the area to be used for subdialogs.  This
//   groupbox (or other control) will not be included in the final dialog
//   box.  The parent dialog box must also contain a listbox to which the
//   subdialog names are added using the AddSelection() member function.
//   The data associated with each item in the listbox is set by
//   CMultiDialog and should not be altered.  Subdialog names should not be
//   duplicated in the list.  Do not call ShowWindow() on controls in
//   subdialogs.
//
// ------------------------------------------------------------------------

#ifndef __MULTIDLG_H

#include "map_pl.h"
#include "template.h"

// Every time the CMultiDialog object changes which subtemplate is selected, it
// sends this message to the dialog with :
//		wParam			= # of subtemplate now selected (first subtemplate = 0)
//		LOWORD(lParam)	= Reason for change
//							MDN_USER_SETSEL = user called SetSelection()
//							MDN_UPDATEDATA_FAIL = UpdateData() failed and set
//								focus to an item in an inactive subtemplate
//		HIWORD(lParam)	= 0
#define MDN_SELECTCHANGE		(WM_USER + 0x6322)
#define MDN_USER_SETSEL			0x6800
#define MDN_UPDATEDATA_FAIL		0x6801
#define MDN_MNEMONIC_SWITCH		0x6802		// User pressed Alt+key on an inactive
											// subdlg
#define MDN_TAB_ACTIVATE		0x6803		// MDN_SEQUENTIAL_SUBS was specified and
											// the user tabbed into a new subdialog.

// Do not use this ID value for any controls
#define IDC_MULTIDLG_SEPARATOR	0xCDA0

// Flag values for the wFlags parameter to the constructor.
#define MDN_GROUP_KEEP		0x0001	// If specified, the group box is not
									// destroyed before displaying the
									// dialog box.
#define MDN_GROUP_AUTO		0x0003	// If specified, MDN_GROUP_KEEP is
									// implied and each call to
									// SetSelection() automatically sets
									// the group box title to the description of
									// the active subtemplate.
#define MDN_SEQUENTIAL_SUBS	0x0004	// If specified, pressing the TAB key while the
									// focus is on the last control of one subdialog
									// will cause the next subdialog to be displayed
									// and focus set to the first control of that
									// subdialog.  If not specified, focus will be
									// set to the first control in the main dialog
									// after all subdialogs.
#define MDN_CENTER_SUBS		0x0008	// If specified, subdialogs which are
									// smaller than the group box are centered
									// in the group box area.  Default is to
									// align them at the upper left.

LRESULT CALLBACK __export CMultiDialogControlWndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam);

class CSubTemplateMap;


class CMultiDialog : public CDialog
	{
	DECLARE_DYNAMIC(CMultiDialog)

public:
	// Construct a CMultiDialog from various places.  If a problem is
	// encountered, a message box is displayed describing the error.
	__export CMultiDialog(LPCSTR szBaseTemplate, UINT uIDSizeBox, WORD wFlags = 0, CWnd *pParent = NULL);
	__export CMultiDialog(UINT nIDBaseTemplate, UINT uIDSizeBox, WORD wFlags = 0, CWnd *pParent = NULL);

	__export ~CMultiDialog(void);

	// If szText or szDescription is NULL, uses the caption from the dialog
	// box template.
	// szText is used for FillListBox() and for SetSelection(LPCSTR, BOOL).
	// szDescription is used for auto group box labelling.
	// If a problem occurs, a message box describing the error is displayed
	// and FALSE is returned.
	BOOL __export AddSelection(LPCSTR szSubTemplate, LPCSTR szText = NULL,
		LPCSTR szDescription = NULL);
	BOOL AddSelection(UINT nIDSubTemplate, LPCSTR szText = NULL,
		LPCSTR szDescription = NULL)
		{ return AddSelection(MAKEINTRESOURCE(nIDSubTemplate), szText, szDescription); }

	// Delete a selection (and all controls, if the dialog is executing).  Cannot be
	// used to delete the currently active subdialog.
	BOOL __export DeleteSelection(LPCSTR szSubTemplate);
	BOOL DeleteSelection(UINT nIDSubTemplate)
		{ return DeleteSelection(MAKEINTRESOURCE(nIDSubTemplate)); }

	// If a problem is encountered constructing the dialog box, a message
	// box is displayed describing the problem and -1 is returned.
	int __export DoModal();

	// Call this to fill a list box with the subtemplate item names (as
	// specified in during AddSelection()).
	BOOL __export FillListBox(CListBox *pListBox, int iSelect = -1);

	// The selection can only be set when the dialog is active (during
	// DoModal()).  This function must be called (at least) once from
	// your OnInitDialog().  The first form of this function matches the
	// text passed in the szText parameter to the template names specified
	// in calls to the AddSelection() function.
	BOOL __export SetSelection(LPCSTR szText);
	// 0 = first AddSelection() item
	BOOL __export SetSelection(int iWhich, int iReason = MDN_USER_SETSEL);

	// _NEVER_ call ShowWindow directly on a control in a CMultiDialog!
	BOOL __export ShowDlgItem(int nID, int nCmdShow = SW_SHOW);

	// If you create a window manually which should be made visible & invisible
	// along with a subtemplate, call this function.
	BOOL __export AddDlgItem(HWND hWnd, int iWhichSubTemplate);

	// Make sure to call this instead of CDialog's version
	BOOL __export UpdateData(BOOL bSaveAndValidate = TRUE);

	// After changing the focus to a dialog control, call this function to make
	// sure the proper subdialog is displayed.
	BOOL __export VerifyFocus(void);

	// Make sure to call this instead of the default DDX_Radio.
	void __export DDX_Radio(CDataExchange *pDX, int nIDC, int &value);

	// Returns an ID which is not used in the template
	UINT UnusedID(void) const
		{ return m_sTemplate.unused_id(); }

	// Return the offset of the currently active subdialog.
	int GetSelection(void) const
		{ return m_iActiveSub; }
	// Get text or description of the given subdialog.  If iWhich is -1 (the
	// default) returns information for the currently active subdialog.
	LPCSTR __export GetSelectionText(int iWhich) const;
	LPCSTR __export GetSelectionDescription(int iWhich) const;

	// Returns TRUE if the given window has a mnemonic character, and that key is
	// currently being pressed.
	static BOOL __export IsMnemonicPressed(HWND hWnd);

	BOOL __export ExecuteDlgInit(void);

protected:
	BOOL __export init(LPCSTR szBaseTemplate, UINT uIDGroupBox, WORD wFlags, CWnd *pParent);

	// If a problem is encountered, a message box is displayed describing
	// the problem and FALSE is returned.
	BOOL __export ConstructTemplate(void);

	int __export MapIndex(int iItemIndex);	// Which map does the item
											// exist in?

	BOOL IsInMainDialog(int iIndex) const
		{
		ASSERT(iIndex >= 0);
		ASSERT(iIndex < m_sTemplate.num_items());
		return (BOOL)((iIndex < m_iFirstSubItem) || (iIndex >= m_iInsertIndex));
		}
	BOOL IsInActiveSubDialog(int iIndex) const
		{
		ASSERT(iIndex >= 0);
		ASSERT(iIndex < m_sTemplate.num_items());
		return (BOOL)((iIndex >= m_iFirstEnabledItem) && (iIndex < m_iFirstEnabledItem + m_nEnabledItems));
		}

	// Is the control corresponding to the given template item index a good
	// one to tab to - that is, is it visible, enabled, and WS_TABSTOP?
	HWND IsItemTabbable(int iItemIndex);
	// Find a window in the given subdialog that is tabbable.  If bFirst is
	// TRUE, the first such window is returned; if FALSE, the last.  If
	// iStartIndex is specified, the search begins at the given index instead
	// of the first (or last) control in the subdialog.
	HWND FindTabbableWindow(int iMapIndex, BOOL bFirst, int iStartIndex = -1);
	// Find a window in the main dialog, before any subdialogs, that is tabbable.
	HWND FindTabbablePreWindow(BOOL bFirst);
	// Find a window in the main dialog, after all subdialogs, that is tabbable.
	HWND FindTabbablePostWindow(BOOL bFirst);

	// Let the special control window procedure have protected access
	friend LRESULT CALLBACK __export CMultiDialogControlWndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam);

	LRESULT SpecialControlMessage(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam);

	// Returns the index into m_sTemplate of a control in a subdialog, given the
	// control's HWND.  Returns -1 if can't find it.
	int HWndToItemIndex(HWND hWnd);

	// Make sure that the hWnd is still correct.  Only use for items in
	// subtemplates (not in the main template).
	BOOL VerifyWindow(CDialogItem *pItem);

	// Check if there is initialization data for a subdialog
	void AppendInitData(LPCSTR szSubTemplate, CMemFile &memFile);
	BOOL ExecuteInitData(CMemFile &memFile);

	// Add special separator control
	void AddSeparator(void);

	// Copy items from subtemplate to main template
	BOOL CopyToMainTemplate(CDialogTemplate &subTemplate);

	// If someone calls AddSelection while the dialog is running, this
	// function creates the controls
	BOOL CreateSelection(CSubTemplateMap *pMap);

	// Make sure a VBX control DLL is available
	BOOL LoadVBX(CDialogItem *pItem);

	// Generated message map functions
	virtual BOOL __export OnInitDialog();
	// If the actual dialog box (derived from this class) overrides OnOK, the
	// user is required to call CMultiDialog::UpdateData() instead of CDialog::
	// so everything works.  However, if the user doesn't override OnOK, the
	// default CDialog::OnOK() calls CDialog::UpdateData() which is no good.
	// Provide our own override of OnOK() that calls our UpdateData() to
	// protect against this.
	virtual void __export OnOK();
	//{{AFX_MSG(CMultiDialog)
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()

	WORD m_wFlags;
	unsigned int m_uGroupID;

	CDialogTemplate m_sTemplate;
	CRect m_rectSub;			// In dialog units
	CObArray m_aMaps;			// Array of mappings between text and subtemplate controls

	CMemFile m_memInit;			// Initialization data for main dialog
	CStringArray m_libList;		// List of loaded VBX control DLLs

	WNDPROC m_pOldControlWndProc;

	int m_nMainTemplateItems;	// Total number of items in main dialog
	int m_iInsertIndex;			// Index in m_sTemplate of first control
								// after subtemplates (may be after end of
								// array if no controls come after
								// subtemplates)
	int m_iFirstSubItem;		// Index in m_sTemplate of first control
								// in first subtemplate.  Negative if no
								// subtemplates have been added.

	int m_iFirstEnabledItem;	// Index in m_sTemplate of first control
								// in currently active subdialog
	int m_nEnabledItems;		// Number of controls in currently active
								// subdialog
	int m_iActiveSub;			// Index of currently active subdialog

	CMapPtrToLong m_aStatus;	// Array of control status values.  High word of status
								// is either WS_VISIBLE or 0; low word contains handle
								// of window

	//{{AFX_DATA(CMultiDialog)
		// NOTE: the ClassWizard will add data members here
	//}}AFX_DATA
	};


#define __MULTIDLG_H
#endif


