/*
     Use Runge-Kutta method (Algorithm 5.7) to solve systems of differential 
     equations:
     
     Exercise Set 5.9, p.289, #1(b)
     
     u1' = -4*u1 - 2*u2 + cos(t) + 4*sin(t), 0 <= t <= 2, u1(0) = 0;
     u2' = 3*u1 + u2 - 3*sin(t), 0 <= t <= 2, u2(0) = -1, h = 0.1
     
     note: true solution => u1(t) = 2*exp(-t) - 2*exp(-2*t) + sin(t)
                            u2(t) = -3*exp(-t) + 2*exp(-2*t)

*/                        

#include <stdio.h>
#include <math.h>

#define EXIT(s)  { printf("%s\n", s); exit(0); }
#define abs(x) (x < 0. ? -x : x)

//fptype1, fptype2 are pointers to functions returning double
typedef double (*fptype1)(double, double *, double *, double);
typedef double (*fptype2)(double);

//declare function prototypes
double f1(double, double *, double *, double),   //function f1()
       f2(double, double *, double *, double),   //function f2()
       u1(double), u2(double);                   //u1() and u2()
void output(double, double *, fptype2 *);        //output values

const int m = 2;    //the number of equations

class UserEquation {
  protected:             //declare protected members for inheritance
     double h, a, b;     //steps and starting and stop value
     long N;             //number of steps
     double alpha[m];    //alpha is the initial values
     fptype1 f[m];       //f will point to f1() and f2()
     fptype2 g[m];       //g will point to u1() and u2()
  public:
     UserEquation();     //the constructor will initial the members
};

//object: Systems of Differential Equations
class SDE : public UserEquation {  //all the members of SDE are inherited
  public:
     SDE();    //constructor does everything for this program
};

main()
{
     SDE sde;  //This is all we need in the main function
}

// Constructor for UserEquation
UserEquation::UserEquation()
{
     a = 0.;
     b = 2.;
     h = 0.1;
     N = (long) (b - a) / h;
     alpha[0] = 0.;
     alpha[1] = -1.;

     f[0] = &f1;    //f[0] points to f1()
     f[1] = &f2;    //f[1] points to f2()
     g[0] = &u1;    //g[0] points to u1()
     g[1] = &u2;    //g[1] points to u2()
}

//Constructor SDE has all the logic for solving any system of equations
SDE::SDE()
{
     double dummy[m] = {0., 0.}; //set to zero to make it usable
     double w[m], k1[m], k2[m], k3[m], k4[m];

     if((h == 0) && (N != 0))                     //Step 1.
          h = (b - a) / N;
     double t = a;

     for(int j = 0; j < m; j++)                   //Step 2.
          w[j] = alpha[j];

     output(t, w, g);                             //Step 3.

     for(int i = 0; i <= N; i++) {                    //Step 4.

          for(j = 0; j < m; j++)                      //Step 5.
               k1[j] = h * f[j](t, w, dummy, 0.);

          for(j = 0; j < m; j++)                       //Step 6.
               k2[j] = h * f[j](t + h/2, w, k1, 0.5);

          for(j = 0; j < m; j++)                       //Step 7.
               k3[j] = h * f[j](t + h/2, w, k2, 0.5);

          for(j = 0; j < m; j++)                       //Step 8.
               k4[j] = h * f[j](t + h, w, k3, 1.);

          for(j = 0; j < m; j++)                       //Step 9.
               w[j] += (k1[j] + 2*k2[j] + 2*k3[j] + k4[j]) / 6.;

          t = a + h * (i + 1);                         //Step 10.
          output(t, w, g);                             //Step 11.
     }
     return;
}

//Output the t, w, real values and %error
void output(double t, double *w, fptype2 *g)
{
     double x;

     printf("\n   t = %9lf\n", t);
     for(int j = 0; j < m; j++) {
          x = g[j](t);   //x = u1(t) or u2(t)
          printf("w[%d] = %9lf, u[%d] = %9lf, %%error = %9lf\n", j, w[j],
               j, x, (abs(x) < 1.e-9 ? 0. : abs((x - w[j]) * 100./ x)));
     }
}

//f1 = u1'(t)
double f1(double t0, double *w, double *k, double weight)
{
     //the weight and k[] are passed as argument for generic use
     return -4. * (weight * k[0] + w[0]) - 2 * (weight * k[1] + w[1])
                + cos(t0) + 4. * sin(t0);
}

//f2 = u2'(t)
double f2(double t0, double *w, double *k, double weight)
{
     double d = 3. * (weight * k[0] + w[0]) + (weight * k[1] + w[1])
                - 3. * sin(t0);
     return d;
}

//u1 is the genuine solution for u1(t)
double u1(double t)
{
     return 2.* exp(-t) - 2.* exp(-2. * t) + sin(t);
}

//u2 is the genuine solution for u2(t)
double u2(double t)
{
     return -3.* exp(-t) + 2.* exp(-2. * t);
}
