/*
	Use Poisson Equation Finite-Difference Algorithm (Algorithm 12.1, p.615)
	
	to solve Exercise Set 12.2, p.619, #3(c).
	
	True solution:  u(x, y) = exp(x*y)
*/

#include <stdio.h>      //Standard C i/o
#include <stdlib.h>     //Standard C library
#include <math.h>       //Standard C math routines for exp(x)


#define u(x, y) exp((x) * (y))  //solution is u(x) = exp(x*y)

typedef double (*funptr)(double, double);   //function pointer

double G(double, double),       //g(x, y)
      F(double, double);        //f(x, y)

const size = 48;    //size of the array used

class UserEquation {
    protected:
        double h, k;
        double a, b, c, d;       
        long   m, n;
        double TOL;              //tolerance
        long N;             //maximum number of iterations
        funptr f, g;            //function pointers
    public:
        UserEquation();     //constructor for UserEquation
};

//Use object inheritance
class PoissonEquationFiniteDifference : public UserEquation {
    public:
        PoissonEquationFiniteDifference();
        void print_initial_values(void);
};

main()
{
    PoissonEquationFiniteDifference pefd;   //This is all we need.
    pefd.print_initial_values();
}

//The constructor initializes its member variables.
UserEquation::UserEquation()
{
    a = 0.0;
    b = 2.0;
    c = 0.0;
    d = 1.0;

    h = 0.2;                                    //Step 1.
    k = 0.1;

    n = (long)(b - a) / h + 1;  //steps for x
    m = (long)(d - c) / k + 1;  //steps for y
    if((n > size) || (m > size)) {
        printf("Size of array should be increased.\n");
        exit(0);
    }

    TOL = 0.00001;  //tolerance = 1.e-5
    N = 10000;  //maximum 10000 iterations

    f = F;      //f(x, y) = F(x, y)
    g = G;      //g(x, y) = G(x, y)
}

//The constructor function does it all.
PoissonEquationFiniteDifference::PoissonEquationFiniteDifference()
{
    double w[size][size], x[size], y[size], z;
    double NORM, soln;

    for(int i = 1; i < n; i++)              //Step 2.
        x[i] = a + h * i;

    for(int j = 1; j < m; j++)                      //Step 3.
        y[j] = c + k * j;

    for(i = 1; i < n; i++)                  //Step 4.
        for(j = 1; j < m; j++)
            w[i][j] = 0;

    double namda = (h * h) / (k * k);            //Step 5.
    double nu = 2.0 * (1.0 + namda);
    long l = 1;

    while(l < N) {                          //Step 6.
        z = (- h * h * f(x[1], y[m-1]) + g(a, y[m-1]) + //Step 7.
            namda * g(x[1], d) + namda * w[1][m-2] + w[2][m-1]) / nu;
        NORM = fabs(z - w[1][m-1]);
        w[1][m-1] = z;

        for(i = 2; i < n - 1; i++) {                    //Step 8.
            z = (- h * h * f(x[i], y[m-1]) + namda * g(x[i], d) +
                w[i-1][m-1] + w[i+1][m-1] + namda * w[i][m-2]) / nu;
            if(fabs(w[i][m-1] - z) > NORM)
                NORM = fabs(w[i][m-1] - z);
            w[i][m-1] = z;
        }
                                                //Step 9.
        z = (- h * h * f(x[n-1], y[m-1]) + g(b, y[m-1]) + namda *
            g(x[n-1], d) + w[n-2][m-1] + namda * w[n-1][m-2]) / nu;
        if(fabs(w[n-1][m-1] - z) > NORM)
            NORM = fabs(w[n-1][m-1] - z);
        w[n-1][m-1] = z;

        for(j = m - 2; j > 1; j--) {                    //Step 10.
                                                    //Step 11.
            z = (- h * h * f(x[1], y[j]) + g(a, y[j]) + namda *
                w[1][j+1] + namda * w[1][j-1] + w[2][j]) / nu;
            if(fabs(w[1][j] - z) > NORM)
                NORM = fabs(w[1][j] - z);
            w[1][j] = z;

            for(i = 2; i < n - 1; i++) {                    //Step 12.
                z = (- h * h * f(x[i], y[j]) + w[i-1][j] + namda *
                    w[i][j+1] + w[i+1][j] + namda * w[i][j-1]) / nu;
                if(fabs(w[i][j] - z) > NORM)
                    NORM = fabs(w[i][j] - z);
                w[i][j] = z;
            }
                                                    //Step 13.
            z = (- h * h * f(x[n-1], y[j]) + g(b, y[j]) + w[n-2][j] +
                + namda * w[n-1][j+1] + namda * w[n-1][j-1]) / nu;
            if(fabs(w[n-1][j] - z) > NORM)
                NORM = fabs(w[n-1][j] - z);
            w[n-1][j] = z;
        }
                                                //Step 14.
        z = (- h * h * f(x[1], y[1]) + g(a, y[1]) + namda * g(x[1], c)
            + namda * w[1][2] + w[2][1]) / nu;
        if(fabs(w[1][1] - z) > NORM)
            NORM = fabs(w[1][1] - z);
        w[1][1] = z;

        for(i = 2; i < n - 1; i++) {                    //Step 15.
            z = (- h * h * f(x[i], y[1]) + namda * g(x[i], c) +
                w[i-1][1] + namda * w[i][2] + w[i+1][1]) / nu;
            if(fabs(w[i][1] - z) > NORM)
                NORM = fabs(w[i][1] - z);
            w[i][1] = z;
        }
                                                //Step 16.
        z = (- h * h * f(x[n-1], y[1]) + g(b, y[1]) + namda * g(x[n-1], c)
            + w[n-2][1] + namda * w[n-1][2]) / nu;
        if(fabs(w[n-1][1] - z) > NORM)
            NORM = fabs(w[n-1][1] - z);
        w[n-1][1] = z;

        if(NORM <= TOL) {                           //Step 17
            printf("After %ld iterations:\n\n", l);
            for(i = 1; i < n; i++)                      //Step 18.
                for(j = 1; j < m; j++) {
                    printf("x[%d] = %lf, y[%d] = %lf, ", i, x[i], j, y[j]);
                    if(i && j && (i != n + 1) && (j != m + 1)) {
                        printf("w[%d][%d] = %lf\n", i, j, w[i][j]);
                        soln = u(x[i], y[j]);
                        printf("True solution is %lf, %%error = %lf%%\n\n",
                            soln, fabs((soln - w[i][j])/soln*100));
                    }
                }
            return;                                 //Step 19.
        }
        l = l + 1;                                  //Step 20.
    }
    printf("Maximum number of iterations exceeded!\n");//Step 21.
    exit(0);
}

//F(x, y) = f(x, y) = (x**2 + y**2) * exp(x * y)
double F(double x, double y)
{
    return (x * x + y * y) * exp(x * y);
}

//G(x, y) = g(x, y)
double G(double x, double y)
{
    double tolerance = 1.e-10;

    if(fabs(x) < tolerance)
        return 1.;          //u(0, y) = 1
    if(fabs(x - 2.0) < tolerance)
        return exp(y * 2.);     //u(2, y) = exp(2y)
    if(fabs(y) < tolerance)
        return 1.;          //u(x, 0) = 1
    if(fabs(y - 1.0) < tolerance)
        return exp(x);          //u(x, 1) = exp(x)

    printf("Strange!\n");
    exit(0);
}

//Print the initial values
void PoissonEquationFiniteDifference::print_initial_values()
{
    double x = 0., y = - 0.1;

    printf("\n\nThe initial values are:\n\n");
    for(int i = 0; i < m + 1; i++)
        printf("x[%d] = %lf, y[%d] = %lf\n\n", 0, 0., i, y += 0.1);
    for(i = 1; i < n; i++)
        printf("x[%d] = %lf, y[%d] = %lf\n\n", i, x += 0.2, 10, 1.0);
    y = - 0.1;
    for(i = 0; i < m + 1; i++)
        printf("x[%d] = %lf, y[%d] = %lf\n\n", 10, 2.0, i, y += 0.1);
}
