/*
     Use Linear Finite-Difference method (Algorithm 11.3, p.576) with 
     
     Thomas Algorithm to solve Exercise Set 11.3, p.579, #2(c):
     

     y" = -3*y' + 2*y + 2*x + 3, 0 <= x <= 1, y(0) = 2, y(1) = 1, h = 0.1
     
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

typedef double (*funtype)(double);

double y(double), P(double), Q(double), R(double);

const int n = 250;            //Maximum subintervals

class UserEquation {
     protected:
          double    a, b, h;
          double    alpha, beta;
          funtype   p, q, r;  //functions
          long      N;
     public:
          UserEquation();
};

class LinearFiniteDifferenceAlgorithm : public UserEquation {
     public:
          LinearFiniteDifferenceAlgorithm(); //only a constructor
};

main()
{
     LinearFiniteDifferenceAlgorithm lfda;   //This is all we need.
}

/* constructor initializes the initializes the initial values */
UserEquation::UserEquation()
{
     a = 0.0;
     b = 1.0;
     h = 0.1;
     N = (b - a) / h;
     if(n < N - 1) {
          printf("Number of intervals is too big for the program!\n");
          exit(0);
     }
     alpha = 2.0;
     beta = 1.0;
     p = P;    //p(x)
     q = Q;    //q(x)
     r = R;    //r(x)
}

//Linear finite difference algorithm
LinearFiniteDifferenceAlgorithm::LinearFiniteDifferenceAlgorithm()
{
     double a0 = a, b0 = b; //do this before we redeclares a and b as array
     double x, a[n], b[n], c[n], d[n];

     x = a0 + h;                                       //Step 1.
     b[1] = 2.0 + h * h * q(x);
     c[1] = -1.0 + (h / 2) * p(x);
     d[1] = - h * h * r(x) + (1.0 + (h/2) * p(x)) * alpha;

     for(int i = 2; i < N; i++) {                      //Step 2.
          x = a0 + h * i;

          a[i] = -1.0 - (h / 2) * p(x);
          b[i] = 2.0 + h * h * q(x);
          c[i] = -1.0 + (h / 2) * p(x);
          d[i] = - h * h * r(x);
     }

     x = b0 - h;                                       //Step 3.
     a[N] = -1.0 - (h / 2) * p(x);
     b[N] = 2.0 + h * h * q(x);
     d[N] = - h * h * r(x) + (1.0 - (h / 2) * p(x)) * beta;

     /***** use Thomas Algorithm ******/

     double beta[n], gamma[n];                         //Step 4.
     double w[n];   //approximated solution

     beta[1] = b[1];          //initialization
     gamma[1] = d[1] / b[1];

     for(i = 2; i < N + 1; i++) {  //set up
          beta[i] = b[i] - a[i] * c[i-1] / beta[i-1];
          gamma[i] = (d[i] - a[i] * gamma[i-1]) / beta[i];
     }

     w[N] = gamma[N];
     for(i = N - 1; i > 0; i--)
          w[i] = gamma[i] - c[i] * w[i+1] / beta[i];   //find w

     for(i = 1; i < N + 1; i++) {                      //output
          double x = a0 + h * i;
          double soln = y(x);
          double percent_error = fabs((soln - w[i]) / soln * 100);

          printf("\nx = %lf, w[%d] = %lf\n", x, i, w[i]);
          printf("y(x) = %lf, %%error = %lf%%\n", soln, percent_error);
     }
}

//P(x) == p(x)
double P(double x)
{
     return -3.0;
}

//Q(x) == q(x)
double Q(double x)
{
     return 2.0;
}

//R(x) == r(x)
double R(double x)
{
     return 2.0 * x + 3.0;
}

//returns the true solution y(x) for comparison
double y(double x)
{
     double sqrt17 = sqrt(17.0);
     double c2 = 5. * (exp((-3. + sqrt17) / 2.) - 1.) /
                 (exp((-3.+ sqrt17) / 2.) - exp((-3.-sqrt17) / 2.));
     double c1 = 5. - c2;

     return      c1 * exp((-3. + sqrt17) / 2. * x)
               + c2 * exp((-3. - sqrt17) / 2. * x) - x - 3;
}

