//*************************************************************
//  File name: METRICS.H
//
//  Description:        
//       Header file for CMetrics and derived classes.
//
//  History:    Date       Author     Comment
//              2/26/94    FJB        Created
//
// Written by Microsoft Product Support Services, Windows Developer Support
// Copyright (c) 1994 Microsoft Corporation. All rights reserved.
//*************************************************************


//*************************************************************
//
//  Class
//      CMetrics
//
//  Member Function:
//    Construction:
//       CMetrics        - Initializes map mode, 
//
//    Attributes:
//       IsValid         - Indicates whether current computed metrics
//                         are valid
//
//    Operations:
//       Compute         - Recomputes metrics
//       CreatePrinterDC - Creates a printer DC using current default
//                         printer.  
//       SetMapMode      - Sets map mode for all CMetrics objects
//       Invalidate      - Invalidates current metrics.
//  
//  Description: 
//    This is the base class for several helper classes used by CTextView
//    to compute and store metrics required for text output.  Some examples
//    of these metrics are:
//         - height and width of a line on the display
//         - height and width of a character on the display
//         - height and width of a line on the current printer
//         - height and width of a character on the current printer
//         - current user defined margins
//         - number of lines per page, on the current printer.
//    To reduce the complexity of the CTextView class, these calculations
//    were removed from the view and divided among three CMetrics derived
//    classes:  CViewMetrics, CPageMetrics, and CMargins.
//
//
//  History:    Date       Author     Comment
//              2/25/94    FJB        Created
//
//************************************************************* 

class CMetrics
{  
// construction
public:
   // initial state for a CMetrics object is invalid (computation required)
   CMetrics() {m_fValid = FALSE; m_nMapMode = 0;}   

// attributes
public:                                                                   
   // This is a static function, since the TEXT sample application uses a
   // single mapping mode.  There is no need for each CMetrics derived 
   // object to track the mapping mode independantly.
   static void SetMapMode(int nMapMode);
          BOOL IsValid() {return m_fValid;}
// overrideables
public:
   virtual void Invalidate() {m_fValid = FALSE;}
   // Override this function in derived classes and perform the necessary
   // computations
   virtual void Compute() {m_fValid = TRUE;}
   
// implementation  
protected:
   // Several calculations require a printer CDC*
   CDC* CreatePrinterDC(); 
   
   BOOL       m_fValid;  
   static int m_nMapMode;
};

//*************************************************************
//
//  Class
//      CViewMetrics
//
//  Member Function:
//    Construction:
//       CMetrics -  Initializes CTextDoc pointer, 
//
//    Attributes:
//       GetLineSize  - Returns the width and height of the longest line in 
//                      document.  Printer information unavailable.
//       GetCharSize  - Returns the width and height of the average char.
//       GetLogInch   - Returns the size (in pixels) of a logical inch.
//       GetExtLeading - Returns the tmExternalLeading value for the current
//                       font.
//
//    Operations:
//       SetFont -  Specifies the font to be used as the basis for calcu-
//                  lations.  Invalidates metrics. 
//       Compute -  Recomputes metrics.
// 
//    Implementation
//       ComputeScreenMetrics - Compute all screen metrics
//       ComputePrintMetrics  - Compute all printer metrics
//
//  Description: 
//       CViewMetrics provides font metric information, for both printer
//       and screen.
//
//  History:    Date       Author     Comment
//              2/25/94    FJB        Created
//
//*************************************************************


class CViewMetrics : public CMetrics
{ 
// construction
public:
   CViewMetrics(CTextDoc* m_pDoc);

// attributes
public:
   virtual CSize GetLineSize(BOOL fWantPrint = FALSE);
   virtual CSize GetCharSize(BOOL fWantPrint = FALSE);  
           CSize GetLogInch();
           int   GetExtLeading();
              
// operations
public:   
   virtual void  Compute();  
           void  SetFont(CFont* pFnt);

// implementation 
protected:
   void ComputeScreenMetrics();
   void ComputePrintMetrics();

protected:   
   CSize     m_sizeCharPrt,     // printer character metrics
             m_sizeCharScn,     // screen character metrics
             m_sizeLine,        // screen  line metrics only 
             m_sizeLogInch;     // size of a logical inch, in pixels
         
   CTextDoc* m_pDoc;           
         
   int       m_cyExtLeading;    // current font
   CFont*    m_pFnt;
}; 


//*************************************************************
//
//  Class
//      CMargins
//
//  Member Function:
//    Construction:
//       CMargins -  Initializes default user margins to .5"
//
//    Attributes:
//       GetHardMargins - Returns a CRect that contains the hardware margins
//                        in MM_LOENGLISH units for the current printer.
//                        All margins are taken as positive offsets.
//       
//       GetUserMargins - Returns a CRect that contains the current user
//                        define marigins.  See GetHardMargins for unit
//                        information.
//
//       GetPrintableRect -   Returns a CRect that describes the printable
//                            region, in MM_LOENGLISH units.  The printable 
//                            region is the usable portion of the page.
//       
//       GetPhysSizeDev - Returns a CSize that contains the physical page
//                        size, in device units.
//
//       GetAdjustedRect - Returns a CRect that describes the page specified
//                         by the user defined margins.
//
//       IsInvalid - Attempts to validate the specified user defined margins.
//                   Does two checks:
//                   1. Ensures the user defined margins are >= the hardware
//                      margins for the current printer. 
//                   2. Checks to see if the user defined margins overlap.
//                                     
//    Operations:
//       SetMargins - Specifies the user defined margins
//       Compute - Recomputes metrics 
//
//  Description: 
//       CMargins calculates and maintains hardware margins, user defined
//       margins.
//
//  History:    Date       Author     Comment
//              2/25/94    FJB        Created
//
//*************************************************************

class CMargins : public CMetrics
{  
public:
   static enum enMargins {LEFT=1, TOP, RIGHT, BOTTOM}; 
   
// construction
public:   
   CMargins();   
   
// attributes
public:                              
   // all results are in logical units, unless function ends in 
   // 'Dev'
   
   CRect     GetHardMargins();   // hardware margins for current printer
                                 // as positive offsets      
   CRect     GetUserMargins();   // user defined margins, as positive offsets
   CRect     GetAdjustedMargins(); // user margins, adjusted by hard margins
       
   CSize     GetPhysSizeDev();   // physical page size, in device units
   CRect     GetPrintableRect(); // describes usable region of page, 
                                 // logical units
   CRect     GetAdjustedRect();  // describes portion of page specified by
                                 // user margins

   enMargins IsInvalid(CRect rectMargins);  // Checks user defined margins.
                                            // If incorrect, returns index of
                                            // first incorrect offset
   void      SetMargins(CRect rectMargins);  // sets user defined margins
   
// operations
public:
   virtual void Compute();

// implementation      
protected:   
   CRect m_rectUserMargins,// user defined margins
         m_rectHardMargins,// hardware margins  
       
         m_rectPrintable,  // physical page excluding hardware margins
                           // coordinates  
         m_rectAdjusted;   // rect defined by user margins
         
   CSize m_sizePhysDev;    // physical page size
         
   int   m_cyPageDev;      // y printable size in dev units
};
   
//*************************************************************
//
//  Class
//      CPageMetrics
//
//  Member Function:
//    Construction:
//       CPageMetrics -  Initializes CTextDoc, CViewMetrics, and CMargins
//                       pointers.  CPageMetrics requires these object to do
//                       its calculations 
//
//    Attributes:
//       GetCPages -     Returns the number of pages in the document. 
//       GetLinesPerPage - Returns the number of lines per page
//       IsPageBreak - Returns true if the indicated line ends a page.
//   
//    Operations:
//       Compute - Recomputes metrics  
//
//  Description: 
//       CPageMetrics calculates and provides page information about the
//       document.
//
//  History:    Date       Author     Comment
//              2/25/94    FJB        Created
//
//*************************************************************

class CPageMetrics : public CMetrics
{
// construction
public:
   CPageMetrics(CTextDoc* pDoc, CMargins* pMar, CViewMetrics* pVMet);    

// attributes
public:
   virtual int  GetCPages();            // Returns page count
           int  GetLinesPerPage();      // Returns lines per page
   virtual BOOL IsPageBreak(int nLine); // Returns true if metrics are valid
                                        // and the specified line ends a page
   
// operations:
   virtual void Compute();  
   
// implementation
protected:  
   int m_nLinesPerPage,
       m_cPages; 
       
   CTextDoc*     m_pDocument;
   CViewMetrics* m_pVMetrics;
   CMargins*     m_pMargins;
};

      

   
   
   
   
   
   
   