//*************************************************************
//  File name: TEXTVIEW.CPP
//
//  Description: 
//      Implementation of the CTextView class
// 
//
//  History:    Date       Author     Comment
//              2/24/94    FJB        Created, via AppWizard
//              2/24/94    FJB        Modified to work with CTextDoc
//
// Written by Microsoft Product Support Services, Windows Developer Support
// Copyright (c) 1992 Microsoft Corporation. All rights reserved.
//*************************************************************

#include "stdafx.h"
#include "text.h"

#include "textdoc.h" 
#include "metrics.h"
#include "textview.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

//*************************************************************
//
//  Class
//      CTextView
//
//  Member Function:  
//    Construction/Destruction:
//      CTextView -            c-tor
//     ~CTextView -            d-tor
//      
//    Drawing:
//      OnIntialUpdate -       Called when view first displayed.                             
//      OnUpdate -             Called each time view is updated.
//
//      ComputeVisibleLines -  computes range of visible lines
//                             for display
//      ComputePrintableLines  computes range of visible lines
//                             on a page during printing
//
//      OnDraw -               1st OnDraw override, displays document
//      OnDraw -               2cnd OnDraw override, renders a given
//                             range of lines
//      DrawPageBreak -        Draws a page break on the display
//
//    Printing:
//      OnPreparePrinting -    See documentation for descriptions of
//      OnBeginPrinting -      Currently unimplemented
//      OnPrint                Called to print or preview a page of the
//                             document.
//      OnEndPrinting -        Currently unimplemented
//
//  Message Map:
//      OnCreate -             Called during view creation 
//      OnWinIniChange() -     Called when WIN.INI is changed. 
//
//
//  History:    Date       Author     Comment
//              2/24/94    FJB        Created
//
//*************************************************************

IMPLEMENT_DYNCREATE(CTextView, CScrollView)

BEGIN_MESSAGE_MAP(CTextView, CScrollView)
   //{{AFX_MSG_MAP(CTextView)
   ON_WM_CREATE()
   ON_WM_WININICHANGE()
   //}}AFX_MSG_MAP
   // Standard printing commands
   ON_COMMAND(ID_FILE_PRINT, CView::OnFilePrint)
   ON_COMMAND(ID_FILE_PRINT_PREVIEW, CView::OnFilePrintPreview)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CTextView construction/destruction
  
CTextView::CTextView()
{
   // specify initial font for this view
   memset(&m_lf,0,sizeof(LOGFONT));
   m_lf.lfCharSet = ANSI_CHARSET;  
   // Specify a 12 point font, in MM_LOENGLISH units
   m_lf.lfHeight = -MulDiv(12,100,72);
   m_lf.lfPitchAndFamily = FIXED_PITCH | FF_MODERN;
   lstrcpy(m_lf.lfFaceName,"Courier New");
}


CTextView::~CTextView()
{                                       
   // delete the CMetric derivatives allocated in OnCreate
   delete m_pVMetrics;
   delete m_pPMetrics; 
   delete m_pMargins;
}

/////////////////////////////////////////////////////////////////////////////
// CTextView drawing  

void CTextView::OnInitialUpdate()
{
    // Create font for this view
    m_fnt.CreateFontIndirect(&m_lf);  
    
    // Specify this font to the CViewMetrics object
    // no need to call CMetrics::Invalidate; CViewMetrics::SetFont
    // does this internally
    m_pVMetrics->SetFont(&m_fnt);
    
    // specify the mapping mode for all CMetrics objects
    CMetrics::SetMapMode(MM_LOENGLISH);  
    // NOTE: not necessary to call Invalidate now because metrics haven't
    // yet been computed.
    
    CScrollView::OnInitialUpdate(); 
}

void CTextView::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint)
{
    // compute total document length:
    // doc length = # lines * line height
    CSize sizeTotal = m_pVMetrics->GetLineSize();
    sizeTotal.cy *= GetDocument()->GetCLines();
    
    // specify the current size of the view using CScrollView::SetScrollSizes
    SetScrollSizes(MM_LOENGLISH, sizeTotal,sizeDefault,
                                 m_pVMetrics->GetCharSize());   
    InvalidateRect(NULL);
}

//*************************************************************
//
//  Member Function:
//      CTextView::ComputePrintableLines
//
//  Purpose:
//      Computes the range of lines in the document that will fit
//      on the current page, without clipping.     
//      
//      The current page is identified by the m_nCurPage member of the
//      CPrintInfo object.
//
//  Parameters:
//      pDC - the CDC object to be rendered upon.
//      pPI - the CPrintInfo object for this page.  m_nCurPage indicates
//            which page to render.
//      nFirst - A reference to the first line index to render
//      nLast  - A reference to the last line index to render
// 
//
//  History:    Date       Author     Comment
//              2/25/94    FJB        Created
//
//*************************************************************
     
void CTextView::ComputePrintableLines( CDC*        pDC, 
                                       CPrintInfo* pPI,
                                       int&        nFirst,
                                       int&        nLast)
{                                        
   CTextDoc* pDoc = GetDocument();
   
   nFirst = m_pPMetrics->GetLinesPerPage() * (pPI->m_nCurPage-1);
   nLast  = min( nFirst + m_pPMetrics->GetLinesPerPage() - 1, 
                 pDoc->GetCLines() - 1);
}

//*************************************************************
//
//  Member Function:
//      CTextView::ComputeVisibleLines
//
//  Purpose:
//      Computes the number of lines necessary to refresh the display.   
//
//  Parameters:
//      pDC - The CDC object to be rendered upon.
//      nFirst - A reference to the first line index to render.
//      nLast - A reference to the last line index to render.
//
//
//  History:    Date       Author     Comment
//              2/25/94    FJB        Created
//
//*************************************************************
     
void CTextView::ComputeVisibleLines(CDC const * const pDC, 
                                    int& nFirst, 
                                    int& nLast)
{   
    // Compute the range of visible lines by offseting the clipping region by
    // the viewport origin.
    
    // Get the viewport origin, convert to logical coordinates
    CPoint pt = pDC->GetViewportOrg();
    pDC->DPtoLP(&pt,1);    
    
    // Get the clipping region, in logical coordinates
    CRect rc;   
    pDC->GetClipBox(&rc);       
    
    // Get the logical line height
    int cyChar = m_pVMetrics->GetCharSize(pDC->IsPrinting()).cy;    
    
    // Compute the first visible line, using the following method:
    //   1. offset the top of the clipping region by the viewport origin
    //   2. divide the result by the logical line height
    //   3. make sure the result doesn't exceed the document length
    nFirst= min(abs((rc.top - pt.y)/cyChar),GetDocument()->GetCLines()-1);
    
    // compute the last visible line, using the following method:
    //   1. divide the height of the clipping region by the logical line
    //      height.
    //   2. add nFirst to the result.  Add 1 to cover partial lines
    //   3. make sure the result doesn't exceed the document length    
    nLast = min(abs(rc.Height())/cyChar + nFirst + 1,
                GetDocument()->GetCLines()-1); 
    
    // make sure something terrible didn't happen
    ASSERT(nLast >= nFirst);
}

void CTextView::OnDraw(CDC* pDC)
{
   int nFirstLn, nLastLn, yPos;
   ComputeVisibleLines(pDC,nFirstLn,nLastLn);
   yPos = - nFirstLn * m_pVMetrics->GetCharSize(pDC->IsPrinting()).cy; 
   OnDraw(pDC,yPos,nFirstLn,nLastLn);
}   

void CTextView::OnDraw(CDC* pDC, int nyPos, int nFirstLn, int nLastLn)
{
    CTextDoc* pDoc = GetDocument(); 
    
    CFont* pfnt = pDC->SelectObject(&m_fnt);
    // Set TRANSPARENT mode so the text won't overwrite the page break 
    // markers 
    pDC->SetBkMode(TRANSPARENT);
    
    CString str;
    
    while (nFirstLn <= nLastLn)
    {
        str = pDoc->m_ary.GetAt(nFirstLn); 
        pDC->TabbedTextOut(0,nyPos,str,str.GetLength(),0,NULL,0);
        nyPos -= m_pVMetrics->GetCharSize(pDC->IsPrinting()).cy;
        
        if (!pDC->IsPrinting() && m_pPMetrics->IsPageBreak(nFirstLn))
            DrawPageBreak(pDC,nyPos); 
            
        nFirstLn++;
    }              

    pDC->SelectObject(pfnt);
} 


void CTextView::DrawPageBreak(CDC* pDC, int yPosition)
{                             
   CPen pen;
   pen.CreatePen(PS_DOT,1,RGB(128,128,128));
   CPen* ppen = pDC->SelectObject(&pen);
   
   pDC->MoveTo(0,yPosition);   
   // that should be far enough
   pDC->LineTo(32000,yPosition);
   
   pDC->SelectObject(ppen);
}


/////////////////////////////////////////////////////////////////////////////
// CTextView printing


void CTextView::OnPrint(CDC* pDC, CPrintInfo* pPI)
{ 
   ASSERT_VALID(pDC);
   
   CRect rectMargins = m_pMargins->GetAdjustedRect();
  
   // Offset viewport Origin by top and left margins
   pDC->LPtoDP(&rectMargins);
   pDC->SetViewportOrg(rectMargins.left, rectMargins.top); 
   
   
   // The top and left margins are handled by shifting the viewport origin.
   // To handle the right and bottom margins, we would normally repaginate
   // the document, changing the position of the line breaks.  However,
   // this sample isn't that sophisticated, so we will visually indicate
   // the right and bottom margins by using a clipping region.
       
   // If print preview mode, convert device units back to logical units,
   // then convert back to device units using the actual screen dc
   // this is necessary because clipping regions are specified using 
   // device units, and aren't scaled by print preview                  
   if (pPI->m_bPreview)
   {
     pDC->DPtoLP(&rectMargins);
     CDC::FromHandle(pDC->m_hDC)->LPtoDP(&rectMargins);
   }   
   
   // Create and select the clipping region.
   CRgn rgn;
   rgn.CreateRectRgnIndirect(&rectMargins);   
   pDC->SelectClipRgn(&rgn);   
   
   int nFirst, nLast;
   ComputePrintableLines(pDC,pPI,nFirst,nLast);
   OnDraw(pDC,0,nFirst,nLast);
}
   

BOOL CTextView::OnPreparePrinting(CPrintInfo* pInfo)
{
   pInfo->SetMaxPage(m_pPMetrics->GetCPages()); 
   pInfo->m_nNumPreviewPages = 1;

   return DoPreparePrinting(pInfo);
}

void CTextView::OnBeginPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
   // nothing here yet...
}

void CTextView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
   // nothing here yet...
}

/////////////////////////////////////////////////////////////////////////////
// CTextView diagnostics

#ifdef _DEBUG
void CTextView::AssertValid() const
{
   CView::AssertValid();
}

void CTextView::Dump(CDumpContext& dc) const
{
   CView::Dump(dc);
}

CTextDoc* CTextView::GetDocument() // non-debug version is inline
{
   ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CTextDoc)));
   return (CTextDoc*)m_pDocument;
}
#endif //_DEBUG
    
    
/////////////////////////////////////////////////////////////////////////////
// CTextView message handlers

//*************************************************************
//
//  Member Function:
//      CTextView::OnCreate(LPCSTR lpszSection)
//
//  Purpose:
//      Called during CTextView creation.  Allocate the 
//      necessary CMetric objects.
//
//  Parameters:
//      lpCreateStruct - see CWnd::OnCreate documentation
//
//
//  History:    Date       Author     Comment
//              2/25/94    FJB        Created
//
//*************************************************************
    
    
int CTextView::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
   if (CScrollView::OnCreate(lpCreateStruct) == -1)
      return -1;
   
   m_pVMetrics = new CViewMetrics(GetDocument());   
   m_pMargins  = new CMargins();
   m_pPMetrics = new CPageMetrics(GetDocument(), m_pMargins, m_pVMetrics);
   
   ASSERT(m_pVMetrics);
   ASSERT(m_pPMetrics);
   ASSERT(m_pMargins);
   
   return 0;
}  

//*************************************************************
//
//  Member Function:
//      CTextView::OnWinIniChange(LPCSTR lpszSection)
//
//  Purpose:
//      Called when the WIN.INI file changes.  If the [windows]
//      section changes, it is possible the default printer has changed.
//      To be safe, invalidate all printer related metrics.     
//
//  Parameters:
//      lpszSection - the WIN.INI section that changed
//
//  Comments:
//      A better implementation would be to check if the hDevMode
//      or hDevNames for the current printer changed.
//
//  History:    Date       Author     Comment
//              2/25/94    FJB        Created
//
//*************************************************************

void CTextView::OnWinIniChange(LPCSTR lpszSection)
{
   CScrollView::OnWinIniChange(lpszSection);
   
   if (lstrcmpi(lpszSection,"windows"))
   {  
      TRACE("CTextView: [windows] section of WIN.INI changed.\n");
      // the user might have changed the default printer
      // invalidate all metrics to be safe
      m_pVMetrics->Invalidate();
      m_pMargins->Invalidate();
      m_pPMetrics->Invalidate();      
   }
}
