//==================================
// NUKEDLL.C - Matt Pietrek 1994
//==================================
#include <windows.h>
#include <toolhelp.h>
#include "nukedll.h"

// A private message posted by the NotifyRegister callback routine when
// module load/unload activity starts.
#define WM_UPDATE_DLL_LIST	(WM_USER+0x200)

BOOL DllListNeedsRefresh = FALSE;	// Set to TRUE when module activity starts
HWND HWndDlg;						// The HWND of the main dialog

// Make a variable in a far data segment, thereby preventing a 2nd
// instance of the program from starting up.  We have a relocation
// to the DGROUP in the NotifyRegisterCallback() routine.  If we
// wanted to be fancy, we could look up the HWND of the previous
// instance and activate it.
int far ForceMultipleDataSegs;

// Returns non-zero if the passed parameter is an HMODULE, 0 otherwise
BOOL IsHModule(HMODULE hModule)
{
	MODULEENTRY me;

	// Use TOOLHELP to tell us if the passed HMODULE is valid.  If we
	// were worried about speed, we could treat the HMODULE as a selector
	// and read the first two bytes of the segment.  If they're 'NE', then
	// it's an HMODULE.
	me.dwSize = sizeof(me);
	return ModuleFindHandle(&me, hModule);
}

// Given a dialog box handle and and a listbox ID, return the item data
// value of the currently selected listbox entry.
DWORD GetSelectedItemData(HWND hWndDlg, int listboxID)
{
	int index;
	
	index = (int)SendDlgItemMessage(hWndDlg, listboxID, LB_GETCURSEL, 0, 0);
	if ( index == - 1 )
		return 0xFFFFFFFF;
	
	return SendDlgItemMessage(hWndDlg, listboxID,
								LB_GETITEMDATA, index, 0);
}

// Update the left listbox to reflect the current list of loaded DLLs
void UpdateDllList(HWND hWndDlg)
{
	MODULEENTRY me;
	BOOL moreToGo;
	char szBuffer[256];
	HWND hWndListbox;
	LPWORD lpModuleFlags;

	// Get the listbox's HWND.  We'll be sending numerous messages to it,
	// so it's more efficient than using SendDlgItemMessage().
	hWndListbox = GetDlgItem(hWndDlg, IDL_DLLS);

	// Clear the listbox's contents
	SendMessage(hWndListbox, LB_RESETCONTENT, 0, 0);

	// Use TOOLHELP to enumerate through all the DLLs in the system
	me.dwSize = sizeof(me);
	moreToGo = ModuleFirst(&me);
	while ( moreToGo )
	{
		// Get a pointer to the flags WORD in the module database
		lpModuleFlags = MAKELP(me.hModule, 0xC);
		
		// Is it a DLL module?
		if ( *lpModuleFlags & 0x8000 )
		{
			// Format a string with the module's name and reference count
			wsprintf(szBuffer, "%s (%u)", (LPSTR)me.szModule, me.wcUsage);

			// Add the string to the beginning of the listbox, and then
			// store the HMODULE in the new entry's item data.
			SendMessage(hWndListbox, LB_INSERTSTRING, 0,
						(LPARAM)(LPSTR)szBuffer);
			SendMessage(hWndListbox, LB_SETITEMDATA, 0, me.hModule);
		}
		
		moreToGo = ModuleNext(&me);		// Go on to next module
	}

	// After listing all the modules, set the selection to the first one
	SendMessage(hWndListbox, LB_SETCURSEL, 0, 0);
}

// update the righthand listbox to show all the modules that reference
// the selected module in the left listbox.
void UpdateReferencingDllList(HWND hWndDlg, HMODULE hModule)
{
	MODULEENTRY me;
	BOOL moreToGo;
	char szBuffer[256];
	HWND hWndListbox;
	BOOL fHasReferencingDll = FALSE;
	
	// Get the listbox's HWND.  We'll be sending numerous messages to it,
	// so it's more efficient than using SendDlgItemMessage().
	hWndListbox = GetDlgItem(hWndDlg, IDL_USED_BY);
	
	// Clear the listbox's contents
	SendMessage(hWndListbox, LB_RESETCONTENT, 0, 0);

	// If we were somehow passed an invalid HMODULE, bail out now
	if ( IsHModule(hModule) == FALSE )
		return;
	
	// Use TOOLHELP to enumerate through all the DLLs in the system
	me.dwSize = sizeof(me);
	moreToGo = ModuleFirst(&me);
	while ( moreToGo )
	{
		// Make a pointer to the size of the module reference table
		// of the current module in the enumeration
		LPWORD lpModRefCount = MAKELP(me.hModule, 0x1E);
		HMODULE far * lpModRefTable;
		WORD i, npModRefTable;

		// Make a far pointer to the module reference table of the
		// current module in the enumeration
		npModRefTable = *(LPWORD)MAKELP(me.hModule, 0x28);
		lpModRefTable = MAKELP(me.hModule, npModRefTable);

		// Iterate through all the entries in the module reference table
		for ( i = 0; i < *lpModRefCount; i++ )
		{
			// Did we find an HMODULE that matches the one selected in
			// the left listbox?
			if ( *lpModRefTable == hModule )
			{
				// At least one module references the selected DLL
				fHasReferencingDll = TRUE;
				
				// Add the module name to the right listbox
				SendMessage(hWndListbox, LB_ADDSTRING, 0,
							(LPARAM)(LPSTR)me.szModule);

				break;	// No need to keep looking!
			}

			// Advance to next HMODULE in the module reference table
			lpModRefTable++;
		}

		// Go on to the next module in the system
		moreToGo = ModuleNext(&me);
	}

	// If no module directly references the selected module, the module
	// must have been loaded via LoadLibrary.
	if ( fHasReferencingDll == FALSE )
	{
		wsprintf(szBuffer, "%d LoadLibrary loads", GetModuleUsage(hModule));
		SendMessage(hWndListbox, LB_ADDSTRING, 0, (LPARAM)(LPSTR)szBuffer);
	}
}

// This routine is similar to the UpdateReferencingDllList above.  Instead
// of filling in a listbox though, it simply returns true if at least one
// module is currently implicitly linked to the passed HMODULE.
BOOL HasReferencingDll(HMODULE hModule)
{
	MODULEENTRY me;
	BOOL moreToGo;

	if ( IsHModule(hModule) == FALSE )
		return FALSE;

	me.dwSize = sizeof(me);
	moreToGo = ModuleFirst(&me);
	while ( moreToGo )
	{
		LPWORD lpModRefCount = MAKELP(me.hModule, 0x1E);
		HMODULE far * lpModRefTable;
		WORD i, npModRefTable;
		
		npModRefTable = *(LPWORD)MAKELP(me.hModule, 0x28);
		lpModRefTable = MAKELP(me.hModule, npModRefTable);
		
		for ( i = 0; i < *lpModRefCount; i++ )
		{
			if ( *lpModRefTable == hModule )
				return TRUE;	// Bail out!  At least one referencer found.
			
			lpModRefTable++;	// Advance to next HMODULE
		}
		moreToGo = ModuleNext(&me);
	}
	return FALSE;
}

// The TOOLHELP notification handler routine.  Looks for for module loads
// and unloads, and tells the main window when to update the DLL list.
BOOL CALLBACK __loadds NotifyRegisterCallback(WORD wID, DWORD dwData)
{
	// BC++ 3.1 has a bug where it ignores __loadds.  Therefore
	// we have to load DS by hand.
	#ifdef __BORLANDC__
	__asm {
			mov		ax, seg HWndDlg
			mov		ds, ax
	}
	#endif
		
	if ( (wID == NFY_DELMODULE) || (wID == NFY_STARTDLL) )
	{
		// Only do this for the first DLL in the group of loads/unloads
		if ( DllListNeedsRefresh == FALSE )
		{
			DllListNeedsRefresh = TRUE;
			// Tell dialog to redraw the DLL list and then select
			// the first DLL in the list
			PostMessage(HWndDlg, WM_UPDATE_DLL_LIST, 0, 0);
			PostMessage(HWndDlg, WM_COMMAND, IDL_DLLS,
				MAKELPARAM(0, LBN_SELCHANGE));
		}
	}
	
	return FALSE;	// Tell TOOLHELP that we didn't handle the notification
}

// Handles all WM_COMMAND messages for the dialog
void Handle_WM_COMMAND(HWND hWndDlg, WPARAM wParam, LPARAM lParam)
{
	if ( wParam == IDL_DLLS )	// Left listbox
	{
		if ( HIWORD(lParam) == LBN_SELCHANGE )	// User has selected something
		{
			HMODULE hModule;
		
			hModule = (HMODULE)GetSelectedItemData(hWndDlg, IDL_DLLS);
			UpdateReferencingDllList(hWndDlg, hModule);
		}
	}
	else if ( wParam == IDB_NUKE_DLLS )	// "Nuke Highlighted DLL" button
	{
		HMODULE hModule;
		int cUsage;

		// Get the HMODULE of the selected DLL out of the item data
		hModule = (HMODULE)GetSelectedItemData(hWndDlg, IDL_DLLS);
		if ( IsHModule(hModule) == FALSE )
			return;

		// Check to see if anybody is directly using the DLL.  If so,
		// prompt the user to make sure they want to procede
		if ( HasReferencingDll(hModule) == TRUE )
		{
			if ( MessageBox(hWndDlg,
				"The selected DLL has modules that reference it.  Are you "
				"sure you want to do this?", "Hold on...",
				MB_YESNO | MB_ICONHAND) != IDYES )
				return;
		}

		// Get the reference count of the DLL and call
		// FreeLibrary that many times
		cUsage = GetModuleUsage(hModule);
		while ( cUsage )
		{
			FreeLibrary( hModule );
			cUsage--;
		}

		// We don't need to bother to update the DLL list here.  The
		// last FreeLibrary call above should have caused an
		// NFY_DELMODULE notification for the DLL.  There should be a
		// WM_UPDATE_DLL_LIST message in our message queue, just
		// waiting for the chance to redraw the DLL list.
	}
	else if ( wParam == IDB_EXIT )	// The "Exit" button
	{
		EndDialog(HWndDlg, 0);
	}
}

// Handles the initial setup of the dialog
void Handle_WM_INITDIALOG(HWND hWndDlg, WPARAM wParam, LPARAM lParam)
{
	HWndDlg = hWndDlg;

	// Draw the DLL list in the left listbox
	UpdateDllList(hWndDlg);

	// Fake a user selection of the first DLL in the list
	PostMessage(hWndDlg, WM_COMMAND, IDL_DLLS, MAKELPARAM(0, LBN_SELCHANGE));

	// Set up our TOOLHELP notification callback
	NotifyRegister(0, (LPFNNOTIFYCALLBACK) NotifyRegisterCallback, NF_NORMAL);
}

// Dialog procedure for the NukeDll dialog
BOOL CALLBACK _export NukeDllDlgProc(HWND hWndDlg, UINT msg,
                                     WPARAM wParam, LPARAM lParam)
{
    switch ( msg )
    {
        case WM_COMMAND:
			Handle_WM_COMMAND(hWndDlg, wParam, lParam);
			return TRUE;
        case WM_INITDIALOG:
			Handle_WM_INITDIALOG(hWndDlg, wParam, lParam);
			return TRUE;
		case WM_UPDATE_DLL_LIST:
			UpdateDllList(hWndDlg);
			DllListNeedsRefresh = FALSE;
			break;
        case WM_CLOSE:
            EndDialog(hWndDlg, 0);
			return FALSE;
    }
    return FALSE;
}

int PASCAL WinMain( HANDLE hInstance, HANDLE hPrevInstance,
                    LPSTR lpszCmdLine, int nCmdShow )
{
	// Bring up the NukeDll dialog
    DialogBox(hInstance, "NukeDllDlg", 0, (DLGPROC)NukeDllDlgProc);

	// Shut down the TOOLHELP notifications set up in Handle_WM_INITDIALOG()
	NotifyUnRegister(0);
    return 0;
}
