/*************************************************************
Module name: InjLib.CPP
Notices: Copyright (c) 1994 Jeffrey Richter
*************************************************************/


#include <Windows.H>

#include "ProcMem.H"
#include "InjLib.h"


//////////////////////////////////////////////////////////////


#define ORD_LoadLibraryA	((LPCSTR) MAKEINTRESOURCE(0x134))
#define ORD_LoadLibraryW	((LPCSTR) MAKEINTRESOURCE(0x137))
#define ORD_FreeLibrary		((LPCSTR) MAKEINTRESOURCE(0x07C))

typedef HINSTANCE (WINAPI *PROCLOADLIBRARY)(LPBYTE);
typedef BOOL (WINAPI *PROCFREELIBRARY)(HINSTANCE);

typedef struct {
	PROCLOADLIBRARY fnLoadLibrary;
	PROCFREELIBRARY fnFreeLibrary;
	BYTE pbLibFile[MAX_PATH * sizeof(WCHAR)];
} INJLIBINFO, *PINJLIBINFO;


//////////////////////////////////////////////////////////////


// Calls to the stack-checking routine must be disabled.
#pragma check_stack (off)

static DWORD WINAPI ThreadFunc (PINJLIBINFO pInjLibInfo) {
	// There must be less than a page-worth of local
	// variables used in this function.
	HINSTANCE hinstLib;


	// Call LoadLibrary(A/W) to load the DLL.
	hinstLib = pInjLibInfo->fnLoadLibrary(pInjLibInfo->pbLibFile);

	// Calling LoadLibrary causes the system to map the DLL
	// into the remote process's address space and call
	// the DLL's DllMain with a reason of DLL_PROCESS_ATTACH.
	// The DLL can do whatever it wants to during this 
	// processing.  When DllMain returns, the system returns
	// the HINSTANCE of the DLL back from our call to 
	// LoadLibrary.  At this point, we call FreeLibrary 
	// passing the library's HINSTANCE in order to free it.

	// If the DLL could not be loaded, or the library's
	// DllMain (DLL_PROCESS_ATTACH) returns FALSE, hinstLib
	// will come back as NULL.

	// If the library initialized successfully, free it.
	if (hinstLib != NULL)  {
		// Calling FreeLibrary causes the system to call the
		// DLL's DllMain with a reason of DLL_PROCESS_DETACH.
		// The DLL can perform what ever cleanup is necessary.
		pInjLibInfo->fnFreeLibrary(hinstLib);
	}

	// The thread's exit code is the handle of the DLL.
	return((DWORD) hinstLib);
}


//////////////////////////////////////////////////////////////


// This function marks the memory address after ThreadFunc.
// ThreadFuncCodeSizeInBytes = (PBYTE) AfterThreadFunc - (PBYTE) ThreadFunc.
static void AfterThreadFunc (void) {
}
#pragma check_stack 


//////////////////////////////////////////////////////////////


static BOOL InjectLibWorA (HANDLE hProcess,
	const BYTE * const pbLibFile, BOOL fUnicode) {

	// Kernel32.DLL's HINSTANCE is used to get the
	// address of LoadLibraryA/W and FreeLibrary.
	HINSTANCE hinstKrnl = GetModuleHandle(__TEXT("Kernel32"));

	// Initialize the InjLibInfo structure here and then copy
	// it to memory in the remote process.
	INJLIBINFO InjLibInfo = {
		(PROCLOADLIBRARY) GetProcAddress(hinstKrnl,
			(fUnicode ? ORD_LoadLibraryW : ORD_LoadLibraryA)),

		(PROCFREELIBRARY) GetProcAddress(hinstKrnl, ORD_FreeLibrary),
		0	// The pbLibFile member is initialized later.
	};


	// The address where code will be copied to in the remote process.
	PDWORD pdwCodeRemote = NULL;

	// Calculate the number of bytes in the ThreadFunc function.
	const int cbCodeSize = ((LPBYTE) AfterThreadFunc - (LPBYTE) ThreadFunc);

	// The address where InjLibInfo will be copied to in the remote process.
	PINJLIBINFO pInjLibInfoRemote = NULL;

	// The number of bytes written to the remote process.
	DWORD dwNumBytesXferred = 0;

	// The handle and Id of the thread executing the remote copy of ThreadFunc.
	DWORD dwThreadId = 0;
	const DWORD cbMemSize = cbCodeSize + sizeof(InjLibInfo) + 3;
	HANDLE hThread = NULL;
	HINSTANCE hinstDLLRemote = NULL;

	BOOL fOk = FALSE;
	DWORD dwOldProtect;

	__try {
		// Finish initializing the InjLibInfo structure by copying the
		// desired DLL's pathname.
		if (fUnicode)
			wcscpy((LPWSTR) InjLibInfo.pbLibFile, (LPCWSTR) pbLibFile);
		else
			strcpy((LPSTR) InjLibInfo.pbLibFile, (LPCSTR) pbLibFile);

		// Allocate memory in the remote process's address space large 
		// enough to hold our ThreadFunc function and a InjLibInfo structure.
		pdwCodeRemote = (PDWORD) AllocProcessMemory(hProcess, 
			cbMemSize);

		if (pdwCodeRemote == NULL)
			__leave;

		// Change the page protection of the allocated memory
		// to executable, read, and write.
		fOk = VirtualProtectEx(hProcess, pdwCodeRemote, cbMemSize,
				PAGE_EXECUTE_READWRITE, &dwOldProtect);
		if (!fOk)
			__leave;

		// Write a copy of ThreadFunc to the remote process.
		fOk = WriteProcessMemory(hProcess, pdwCodeRemote,
			(LPVOID) ThreadFunc, cbCodeSize, &dwNumBytesXferred);
		if (!fOk)
			__leave;


		// Write a copy of InjLibInfo to the remote process
		// (the structure MUST start on an even 32-bit bourdary).
		pInjLibInfoRemote = (PINJLIBINFO) 
			(pdwCodeRemote + ((cbCodeSize + 4) & ~3));

		// Put InjLibInfo in remote thread's memory block.
		fOk = WriteProcessMemory(hProcess, pInjLibInfoRemote,
			&InjLibInfo, sizeof(InjLibInfo), &dwNumBytesXferred);
		if (!fOk)
			__leave;


		hThread = CreateRemoteThread(hProcess, NULL, 0, 
			(LPTHREAD_START_ROUTINE) pdwCodeRemote,
			pInjLibInfoRemote, 0, &dwThreadId);
		if (hThread == NULL)
			__leave;

		WaitForSingleObject(hThread, INFINITE);
	}	// __try
	__finally {
		if (hThread != NULL) {
			GetExitCodeThread(hThread, (PDWORD) &hinstDLLRemote);
			CloseHandle(hThread);
		}

		// Let the remote thread start executing the remote 
		// ThreadFunc function using our modified stack that now
		// contains an initialized InjLibInfo structure.
		FreeProcessMemory(hProcess, pdwCodeRemote);
	}	//__finally

	// Return TRUE if the DLL loaded successfully.
	return(hinstDLLRemote != NULL);
}


//////////////////////////////////////////////////////////////


BOOL InjectLibA (HANDLE hProcess, LPCSTR lpszLibFile) {

	return(InjectLibWorA(hProcess, (LPBYTE) lpszLibFile, FALSE));
}


//////////////////////////////////////////////////////////////


BOOL InjectLibW (HANDLE hProcess, LPCWSTR lpszLibFile) {

	return(InjectLibWorA(hProcess, (LPBYTE) lpszLibFile, TRUE));
}


//////////////////////// End Of File /////////////////////////
