/*

	MIDI.H  -- Midi function interface

*/

#ifndef MIDI_H

#define MIDI_H

#ifndef STD_H
#include "std.h"
#endif

#ifndef MPU_H
#include "mpu.h"
#endif

#ifdef JUNK

// The following defines allow scheduled SYSEX messages to be reclaimed.
// The problem is that the SYSEX data block can't be freed from an 
// interrupt service routine. Once the block is sent, the 
// sysex data block is placed in a temporary free chain which is then 
// freed properly by ReclaimSysexMemory().

// Since SYSEX messages can't currently be scheduled, this code is 
// disabled (for now).

// Warning: Nasty hack follows....
// ANSI C specifically allows (encourages?) this kind of nastiness,
// although it has a reputation for being rather difficult to implement
// properly.

// See the PDOC's for ReclaimSysexMemory which explains what this
// nasty little peice of goods is doing.

#define calloc(x,y) (ReclaimSysexMemory(),calloc(x,y))
#define malloc(x)	(ReclaimSysexMemory(),malloc(x))
#endif

// Midi error codes
#define LOWEST_MIDI_ERROR 0x100

typedef enum MidiErrorT {
	MIDI_ENOMEM = LOWEST_MIDI_ERROR, // Insufficient memory.
} MidiErrorT;

#ifdef MIDI_PRIVATES  // Accessable through MidiErrorString
PRIVATE char * MidiErrorStrings[] = {
	"Insufficient memory"
};
#endif

// Message IDs
#define NOTE_ON_MSG 0x90
#define NOTE_OFF_MSG 0x80
#define CONTROL_MSG 0xB0
#define PROGRAM_MSG 0xC0
#define AFTERTOUCH_MSG 0xD0
#define BENDER_MSG 0xE0
#define COMMON_MSG 0xF0
#define SONG_POSITION_MSG 0xF2
#define SONG_SELECT_MSG 0xF3
#define EOX_MSG 0xF7
#define TIMING_CLOCK_MSG 0xF8
#define START_MSG 0xFA
#define CONTINUE_MSG 0xFB
#define STOP_MSG 0xFC
#define ACTIVE_SENSE_MSG 0xFE
#define SYSEX_MSG 0xF0

typedef long MidiTimeT;

// Midi Message Structure
typedef struct MidiMessageT {
	struct MidiMessageT *next;
	MidiTimeT time;
	BYTE midiCommand;
	BYTE midiData[2];
	unsigned int sysexLength;
	BYTE *sysexData;
} MidiMessageT;

// Midi channel control block

typedef struct MidiChannelT {
	MpuPortT *mpu_port;	/* Warning: MpuPort must be first elelement of MidiChannelT -rd */
	MidiErrorT errno;
	int midiMessagesPending; /* Number of scheduled midi messages pending */
	MidiMessageT *sendQ[MAX_MPU_TRACK];
	MidiMessageT *sendQTail[MAX_MPU_TRACK];
	MidiTimeT qTime[MAX_MPU_TRACK];
	
} MidiChannelT;


// Function prototypes.

void ReclaimSysexMemory(void);
MidiMessageT *AllocMidiMessage(void);
MidiMessageT *ReceiveMidiMessage(MidiChannelT *channel);

void ScheduleMidiMessage(
	MidiChannelT *channel,
	int track,
	MidiTimeT time,
	MidiMessageT *msg
);

BOOL SendMidiMessage(MidiChannelT *channel,MidiMessageT *msg);

BOOL SetMidiMessage(
	MidiMessageT *msg,
	UCHAR midi_command, 
	UCHAR data1, 
	UCHAR data2,
	unsigned int sysex_length,	// Must be zero for non-SYSEX commands!
	UCHAR *sysex_data			// May be NULL
);

void GetMidiMessageData(
	MidiMessageT *msg,
	UCHAR *midi_cmd, // Receives midi command
	UCHAR *data1,	 // Receives data1 (if not NULL)
	UCHAR *data2	 // Receives data2 (if not NULL)
);

void GetMidiMessageSysexData(
	MidiMessageT *msg,
	int *sysexLength,
	UCHAR **sysexData	// Filled if not null
);

MidiTimeT GetMidiMessageReceiveTime(
	MidiMessageT *msg
);
	

MidiChannelT *CreateMidiChannel(
	int mpu_base_address, // Mpu base address (default 0x330)
	int mpu_interrupt,	  // Mpu interrupt number (default 2)
	int rx_buffersize,	  // Size of receive buffer (default 1024)
	enum MpuOperatingModeT operating_mode, // RECORD_MODE,PLAY_MODE,RECORDPLAY_MODE,STOP_MODE
	enum MpuClockT mpu_clock_source, // MPU_INTERNAL_CLOCK, MPU_MIDI_CLOCK
									 // or MPU_FSK_CLOCK
	int tempo,			  // Beats per minute
	enum MpuTimebaseT timebase,	 // Ticks per beat (see MpuTimebaseT in MPU.H)
	int metronome_measure_length, // beats per measure, 0 -> metronome off
	int mode,			  // See description 
	int midi_channel_mask, // bit n controls midi channel n+1
						   // bit = 0 -> pass trough without host intervention
						   // bit = 1 -> record/filter this midi channel	
	int tracks,			   // number of tracks (0 to 7) for scheduled midi messages
	int *result			   // retcode placed here
);

void SetMidiOperatingMode(
	MidiChannelT *channel,
	MpuOperatingModeT operating_mode
);

int DestroyMidiChannel(MidiChannelT *channel);

int MidiStatus(MidiChannelT *channel);

char *MidiErrorString(int error_code);
int MidiMessagesPending(MidiChannelT *channel);
void FreeMidiMessage(MidiMessageT *msg);


#endif //def MIDI_H







