/*--------------------------------------------------------------
 *  File:           GWDWDEMO.C
 *  Description:    Program to demonstrate the window functions
 *                  described in chapter 3 of the MPLUS Graphic
 *                  Interface Library.
 *
 *  This demo developed for the MPLUS Graphic Interface Library.
 *-------------------------------------------------------------*/
#include <math.h>
#include <graph.h>
#include <stdio.h>
#include "gscreen.h"
#include "gplus.h"

/*--------------------------------------------------------------
 *  Function Prototypes
 *--------------------------------------------------------------*/
void setaxis( GWDW *gwptr );
void sinplot( double range1, double range2, short xpixels, int ysf );
void cosplot( double range1, double range2, short xpixels, int ysf );

/*--------------------------------------------------------------
 *  Function:       main
 *  Description:    opens 3 windows, plots some sine and cosine
 *                  waves, and demonstrates most of the MPLUS
 *                  graphic window functions.
 *  Return value:   0 returned to parent process.
 *--------------------------------------------------------------*/
main()
{
    int i;
    char buffer[80];
    short x1, y1, x2, y2;
    word xo,yo;
    double pi;
    GWDW *groot;
    GWDW *gwptr1;
    GWDW *gwptr2;

    if( setvideomode( _ERESCOLOR ) == 0 )
    {
        printf("\nProgram requires EGA mode\n");
        exit(1);
    }
    mpwordwrap(1);

    x1 = 50;
    y1 = 20;
    x2 = 550;
    y2 = 340;
    pi = 3.141592654;

    /*      Since window is large, conserve memory by using
     *      grootopen().
     */
    groot = grootopen( x1, y1, x2, y2, _GBORDER, BLUE, WHITE );

    /*      Draw axis in black.  Set origin to center of window.
     */
    _setcolor( BLACK );
    setaxis( groot );

    /*      Draw sine wave in red.
     */
    _setcolor( RED );
    sinplot( -2*pi, 2*pi, x2-x1, 100 );
    _settextposition( 20, 1 );
    outtext( "y = sin(x)", RED, -1 );

    /*      Draw cosine wave in blue.
     */
    _setcolor( BLUE );
    cosplot( -2*pi, 2*pi, x2-x1, 100 );
    _settextposition( 21, 1 );
    outtext( "y = cos(x)", BLUE, -1 );

    getch();

    gwptr1 = gwdwopen( 250, 100, 600, 300, _GBORDER, GREEN, BLACK );
    _setcolor( LIGHTYELLOW );
    setaxis( gwptr1 );

    gwptr2 = gwdwtopen( 5, 1, 15, 31, _GBORDER, MAGENTA, LIGHTYELLOW );
    outtext( "Press a key to draw y=sin(x)\n", MAGENTA, -1 );
    getch();

    gwdwsetactv( gwptr1 );
    _setcolor( RED );
    sinplot( -2*pi, 2*pi, 600-250, 50 );
    _settextposition( 1, 1 );
    outtext( "y = sin(x)", RED, -1 );

    gwdwsetactv( gwptr2 );
    outtext( "Press a key to draw y=cos(x)\n", MAGENTA, -1 );
    getch();

    gwdwsetactv( gwptr1 );
    _setcolor( BLUE );
    cosplot( -2*pi, 2*pi, 600-250, 50 );
    _settextposition( 2, 1 );
    outtext( "y = cos(x)", BLUE, -1 );

    gwdwsetactv( gwptr2 );
    for( i=0; i<20; ++i )
    {
        sprintf(buffer, "\nThis is line %d", i );
        outtext( buffer, MAGENTA, -1 );
    }
    outtext( "\nPress a key to clear this window...", MAGENTA, -1 );
    getch();
    gwdwclr( gwptr2 );
    outtext( "Press a key to close window at right", MAGENTA, -1 );
    getch();
    gwdwclose( gwptr1 );
    gwdwclr( gwptr2 );
    outtext( "Press a key to close this window.", MAGENTA, -1 );
    getch();
    gwdwclose( gwptr2 );
    getch();
    grootclose( groot );
    setvideomode( _DEFAULTMODE );
    return 0;
}
/*--------------------------------------------------------------
 *  Function:       sinplot
 *  Description:    plot a sine wave
 *  Return value:   none
 *--------------------------------------------------------------*/
void sinplot( range1, range2, xpixels, ysf )
double range1;
double range2;
short xpixels;                  /* x pixels available */
int ysf;                        /* y scale factor */
{
    int i;
    double numperxpix;
    double xpixpernum;
    double x, y;

    numperxpix = (fabs(range2-range1))/(double)xpixels;
    xpixpernum = 1/numperxpix;

    /*      Calculate first point.  Position cursor with _moveto.
     *      Adjust sign for y axis.
     */
    x = range1;
    y = sin( x );
    _moveto( (short)(range1/numperxpix), (short)(-( y*ysf )) );

    for( i=1; i<xpixels; ++i)
    {
        x += numperxpix;
        y = sin( x );
        _lineto( (short)(x * xpixpernum), (short)(-(y*ysf)) );
    }
}
/*--------------------------------------------------------------
 *  Function:       cosplot
 *  Description:    plot a cosine wave
 *  Return value:   none
 *--------------------------------------------------------------*/
void cosplot( range1, range2, xpixels, ysf )
double range1;
double range2;
short xpixels;                  /* x pixels available */
int ysf;                        /* y scale factor */
{
    int i;
    double numperxpix;
    double xpixpernum;
    double x, y;

    numperxpix = (fabs(range2-range1))/(double)xpixels;
    xpixpernum = 1/numperxpix;

    /*      Calculate first point.  Position cursor with _moveto.
     *      Adjust sign for y axis.
     */
    x = range1;
    y = cos( x );
    _moveto( (short)(range1/numperxpix), (short)(-(y*ysf)) );

    for( i=1; i<xpixels; ++i)
    {
        x += numperxpix;
        y = cos( x );
        _lineto( (short)(x*xpixpernum), (short)(-(y*ysf)) );
    }
}
/*--------------------------------------------------------------
 *  Function:       setaxis
 *  Description:    draw axis and set logical origin to center
 *                  of screen.
 *  Return value:   none
 *--------------------------------------------------------------*/
void setaxis( gwptr )
GWDW *gwptr;
{
    word x,y;
    short xctr, yctr;

    xctr = (gwptr->x2-gwptr->x1)/2;
    yctr = (gwptr->y2-gwptr->y1)/2;

    _moveto( xctr, 0 );
    _lineto( xctr, gwptr->y2-gwptr->y1 );
    _moveto( 0, yctr );
    _lineto( gwptr->x2-gwptr->x1, yctr );

    gwdwsetorg( gwptr, xctr, yctr );
}
/*-------------------------------------------------------------*
 *                  End of GWDWDEMO.C                          *
 *-------------------------------------------------------------*/


