

/*************************************************************************

            
					  mouCeLib demostration program
							   for moucLib6
						  Ran Cohen,    Ron Loewy


   This program uses the mouse in order to manipulate a rectangle on
  an X window.
  

  The options are:

   - Translation: if one holds the rectangle in one of its edges,
     dragging the mouse will result in moveing the rectangle according to
     the movement of the mouse.

   - Rotation: Holding the rectangle in the middle and dragging the mouse
     will result in rotation of the rectangle around the small filled
     circle, which also, is movable.

   - Scaling: Holding one of the corners of the rectangle and "pulling"
     it to any direction will scale the rect, where the oposite corner
     stays in place. There is a minimum for both width and height of the
     rectangle, defined by MIN_WIDTH ( = 3 pixels).


   Also, one can, as mentioned before, drag the filled circle around the
  window. The center of the circle is the center of the rotation of the
  rectangle.

   All options are activated by pressing the left mouse button.

   The middle button can be used, at any time, to return to the initial
  state of screen. One might want to use it when loosing the rectangle
  outside the window, scaling it out of range (too big or too small)
  and so. The right button is used for terminating the program.


   All the linear transformations are achieved by matrix multiplications,
  where the rectangle itself is represented by a matrix (CTM - Current
  Ttransformation Matrix), that activating it on a 1x1 rect. sitting
  in the 0,0, will result in the current rectangle in the right place.



*************************************************************************/
















#include "mat.h"                 /* Global matrix library */
#include "ex1.h"
#include "gsx.h"
#include <graphics.h>
#include <stdio.h>
#include <stdlib.h>


/* ------------- Function prototypes -------------------- */
void init(GEvent e)                                      ;
void mainLoop(void)                                      ;
void DrawRect()                                          ;
void DrawCircle (float, float)                           ;
int TranslateLoc (float x, float y)                      ;
void Translate (float x, float y)                        ;
void Rotate (float alpha, Vec2f *Center)                 ;
void Scale (float x, float y)                            ;
float GetRectHeight (void)                               ;
float GetRectWidth (void)                                ;
float GetRectXLoc (void)                                 ;
float GetRectYLoc (void)                                 ;
void FirstClick (GEvent e)                               ;
void ResetEvent (GEvent e)                               ;
void DragMouse (GEvent e)                                ;
void EndProgram (GEvent e)                               ;
int MouseOnRightSide(Vec2f *Loc)                         ;


/* -------------  Global variables  ------------------- */
static Matrix *CTMat             ;  /* Current Transformation Matrix */
static state Last_state = INIT_STATE                ;
static Matrix *Vec00, *Vec01, *Vec10, *Vec11        ;
static float Unity33[]    =  {1,0,0, 0,1,0, 0,0,1}  ;

static float Base00[] = {0, 0, 1},
             Base01[] = {0, 1, 1},
			 Base10[] = {1, 0, 1},
             Base11[] = {1, 1, 1};

static Vec2f V00 = {0,0}, V01 = {0,1}, V10 = {1,0}, V11 = {1,1};

extern unsigned int oMaxY;

/************************************************************************/
void main (void)
{
   GEvent junk;      /* Used for the first initialization of the screen */
   int gdriver = DETECT, gmode, errorcode;

	/* initialize graphics mode */
	initgraph(&gdriver, &gmode, "");

	/* read result of initialization */
	errorcode = graphresult();

	if (errorcode != grOk)  /* an error occurred */
	{
		printf("Graphics error: %s\n", grapherrormsg(errorcode));
		printf("Press any key to halt:");
		getch();
		exit(1);             /* return with error code */
	 }



  GWindow(0, 0, WINDOW_SIZE, WINDOW_SIZE, "  EX1  -  Computer graphics");
  init(junk)  ;

  GFuncSet(FirstClick, GButton1Press)     ;
  GFuncSet(init,       GButton2Press)     ;
  GFuncSet(ResetEvent, GButton1Release)   ;
  GFuncSet(DragMouse,  GDrag)             ;
  GFuncSet(EndProgram, GButton3Press)     ;

  GMainLoop()  ;
  EndProgram (junk) ;
}


/************************************************************************
 Initialize the screen - draw all objects, and prepare some base vectors.
 This routine is activated by pressing the middle button of the mouse,
 and also at execution time (1st thing after creating the window).
*************************************************************************/
void init (GEvent e)
{

  GClear()                          ;
  Vec00 = MatCreate(1, 3, Base00)   ;
  Vec10 = MatCreate(1, 3, Base10)   ;
  Vec01 = MatCreate(1, 3, Base01)   ;
  Vec11 = MatCreate(1, 3, Base11)   ;

  CTMat = MatCreate(3, 3, Unity33)                         ;
  Scale (RectStartW, RectStartH)                           ;
  Translate (RectStartX, oMaxY - RectStartY - RectStartH)   ;

  Vec2fSetX(&Last_state.FilledCircle, CircleStartX)        ;
  Vec2fSetY(&Last_state.FilledCircle, oMaxY - CircleStartY) ;
  DrawRect()                                               ;
  DrawCircle(Vec2fGetX((&Last_state.FilledCircle)),
		 Vec2fGetY((&Last_state.FilledCircle)))        ;
}


/************************************************************************
  Drawing the rectangle: Uses the CTM on the 4 base vectors in order
 to find the 4 corners of the rectangle
*************************************************************************/
void DrawRect()
{
  Matrix *VecFrom = MatNew (1, 3);
  Matrix *VecTo   = MatNew (1, 3);

  VecFrom = MatMul (Vec00, CTMat, VecFrom)   ;
  VecTo   = MatMul (Vec10, CTMat, VecTo)     ;
  GLine ((int)MatGet(VecFrom, 0, 0), oMaxY - (int)MatGet(VecFrom, 0, 1),
	(int)MatGet(VecTo, 0, 0),   oMaxY - (int)MatGet(VecTo, 0, 1))   ;

  MatCopy(VecFrom, VecTo)                    ;
  VecTo = MatMul (Vec11, CTMat, VecTo)       ;
  GLine ((int)MatGet(VecFrom, 0, 0), oMaxY - (int)MatGet(VecFrom, 0, 1),
	 (int)MatGet(VecTo, 0, 0),   oMaxY - (int)MatGet(VecTo, 0, 1))  ;

  MatCopy(VecFrom, VecTo)                    ;
  VecTo = MatMul (Vec01, CTMat, VecTo)       ;
  GLine ((int)MatGet(VecFrom, 0, 0), oMaxY - (int)MatGet(VecFrom, 0, 1),
	 (int)MatGet(VecTo, 0, 0),   oMaxY - (int)MatGet(VecTo, 0, 1))  ;

  MatCopy(VecFrom, VecTo)                    ;
  VecTo = MatMul (Vec00, CTMat, VecTo)       ;
  GLine ((int)MatGet(VecFrom, 0, 0), oMaxY - (int)MatGet(VecFrom, 0, 1),
	 (int)MatGet(VecTo, 0, 0),   oMaxY - (int)MatGet(VecTo, 0, 1))   ;

  MatFree(VecFrom)   ;
  MatFree(VecTo)     ;
}


/*************************************************************************
  Change the CTM - - multiply it by a rotation matrix:
  Rotate CTMat around the center point - Center, by alpha deg.
**************************************************************************/
void Rotate (float alpha, Vec2f *Center)
{
  Matrix *Rmat = MatCreate(3,3, Unity33)                          ;
  float CenterX = Vec2fGetX (Center), CenterY = Vec2fGetY(Center) ;

  Translate (-CenterX, -CenterY)    ;
  MatSet (Rmat, 0, 0,  cos(alpha))  ;
  MatSet (Rmat, 0, 1,  sin(alpha))  ;
  MatSet (Rmat, 1, 0, -sin(alpha))  ;
  MatSet (Rmat, 1, 1,  cos(alpha))  ;

  MatMul(CTMat, Rmat, CTMat)        ;
  Translate (CenterX, CenterY)      ;

  MatFree(Rmat)                     ;
}


/*************************************************************************
  Change the CTM - multiply it by a translation matrix:
  Move the rectangle by x pixels in the X axis, and y pixel in the Y axis.
**************************************************************************/
void Translate (float x, float y)
{
  Matrix *Tmat = MatCreate(3,3, Unity33);

  MatSet (Tmat, 2, 0, x);
  MatSet (Tmat, 2, 1, y);

  MatMul(CTMat, Tmat, CTMat);
  MatFree (Tmat);
}


/************************************************************************
  Change the CTM - multiply it by a scaling matrix:
  Scale the rectangle by x pixels in the X axis, and y pixel in the Y axis.
  The object is assumed to be located where its center is at the center
  of origin (0,0).
**************************************************************************/
void Scale (float x, float y)
{
  Matrix *Smat = MatCreate(3,3, Unity33);

  MatSet (Smat, 0, 0,  x);
  MatSet (Smat, 1, 1,  y);

  MatMul(CTMat, Smat, CTMat);
  MatFree (Smat);
}


/************************************************************************
  Drawing the circle - the radius, as implemented in GCircleFill is
 actually 1/2 of the real radius - so it is shifted 1 bit to the left
 (Multiplied by 2).
*************************************************************************/
void DrawCircle (float x, float y)
{
  GCircleFill ((int)x, oMaxY - (int)y, (CircleRadius<<1));
}


/************************************************************************
  Calculates the mouse location as if the screen was a 1 x 1 window,
 starting at 0,0. Returns the number of the sensitive area, as defined
 in the include file (ex1.h).
*************************************************************************/
int TranslateLoc (float x, float y)
{
  float x1, y1, len                       ;
  float tempV[3] = {0,0,1}                ;
  int result                              ;
  Matrix *Vec13, *NewVec13, *Inv          ;   
  Vec2f *NewV, *Vec, *SubR = Vec2fNew()   ;

  tempV[0] = x; tempV[1] = y  ;

  Vec = Vec2fCreate (x, y)    ;

  /* Check if in circle: */
  if (Vec2fLen (Vec2fSub (&Last_state.FilledCircle, Vec, SubR)) < CircleRadius) {
    free(Vec);  free (SubR)   ;  
    return (InCircle)         ;
  }

  Vec13    = MatCreate (1, 3, tempV)   ;
  NewVec13 = MatNew (1,3)              ;
  Inv      = MatNew (3,3)              ;
  MatInv (CTMat, Inv)                  ;

  NewVec13 = MatMul (Vec13, Inv, NewVec13); /* This vector is in 1 x 1 square */

  x1 = MatGet (NewVec13, 0, 0)         ;
  y1 = MatGet (NewVec13, 0, 1)         ;
  NewV = Vec2fCreate(x1, y1)           ;  

  if (x1 > 1 + SenseArea || x1 < - SenseArea ||
      y1 > 1 + SenseArea || y1 < - SenseArea) {
    result  = Outside                  ;
  }

  else {
    /* Corners checking: */
    if (Vec2fLen (NewV) < SenseArea)
      result = InCorner00  ;
    else if (Vec2fLen (Vec2fSub (NewV, &V01, SubR)) < SenseArea)
      result = InCorner01  ;
    else if (Vec2fLen (Vec2fSub (NewV, &V11, SubR)) < SenseArea)
      result = InCorner11  ;
    else if (Vec2fLen (Vec2fSub (NewV, &V10, SubR)) < SenseArea)
      result = InCorner10  ;

    /* Check inside: */
	else if (x1 > SenseArea && x1 < 1 - SenseArea &&
             y1 > SenseArea && y1 < 1 - SenseArea) {
      result = InMiddle    ;
    }

    /* Else */
    else result = InEdge   ;
  }

  free(NewV);   free(Vec);   free (SubR);
  MatFree(Vec13); MatFree(NewVec13); MatFree(Inv);

  return (result);
}

/************************************************************************
  Return the current height of the rectangle.
*************************************************************************/

float GetRectHeight (void)
{
  Matrix *TempM = MatCreate (1, 3, Base01), *Result = MatNew(1,3) ;
  Vec2f  *TempV = Vec2fNew ()                                     ;
  float len                                                       ;
  float CurX = GetRectXLoc(), CurY = GetRectYLoc()                ;

  Translate (-CurX, -CurY)          ;
  MatMul(TempM, CTMat, Result)      ;

  Vec2fSetX (TempV, MatGet (Result, 0, 0))   ;
  Vec2fSetY (TempV, MatGet (Result, 0, 1))   ;
  len = Vec2fLen(TempV)                      ;

  MatFree(TempM);  MatFree (Result);  free (TempV) ;
  Translate (CurX, CurY)                           ;

  return (len) ;
}

/************************************************************************
  Return the current width of the rectangle.
*************************************************************************/
float GetRectWidth (void)
{
  Matrix *TempM = MatCreate (1, 3, Base10), *Result = MatNew(1,3) ;
  Vec2f  *TempV = Vec2fNew ()                                     ;
  float len                                                       ;
  float CurX = GetRectXLoc(), CurY = GetRectYLoc()                ;

  Translate (-CurX, -CurY)                   ;
  MatMul(TempM, CTMat, Result)               ;

  Vec2fSetX (TempV, MatGet (Result, 0, 0))   ;
  Vec2fSetY (TempV, MatGet (Result, 0, 1))   ;
  len = Vec2fLen(TempV)                      ;

  MatFree(TempM);  MatFree (Result);   free (TempV)   ;

  Translate (CurX, CurY)   ;

  return (len)             ;
}


/***********************************************************************
 Returns the X coord of the leftmost lower corner of the rectangle.
************************************************************************/
float GetRectXLoc (void)
{
  return (MatGet(CTMat, 2,0)) ;
}


/************************************************************************
 Returns the Y coord of the leftmost lower corner of the rectangle.
*************************************************************************/
float GetRectYLoc (void)
{
  return (MatGet(CTMat, 2,1)) ;
}


/************************************************************************
 Calculate the angle between vectors v1 and v2, where Center, is their
 intersection point.
*************************************************************************/
float Angel (Vec2f *v1, Vec2f *v2, Vec2f *Center)
{
  float sign, as, result                                                ;
  Vec2f *RealV1 = Vec2fNew(), *RealV2 = Vec2fNew(), *TempV = Vec2fNew() ;
  

  Vec2fSub (v1,Center,TempV)              ;
  RealV1 = Vec2fNormal (TempV, RealV1)    ;

  Vec2fSub (v2,Center,TempV)              ;
  RealV2 = Vec2fNormal (TempV, RealV2)    ;
  
  as = Vec2fMul (RealV1, RealV2)          ;
  
  sign = ((Vec2fGetX (RealV1) * Vec2fGetY(RealV2) - 
          (Vec2fGetY (RealV1) * Vec2fGetX(RealV2)) > 0) ? 1.0 : -1.0)   ;

  result = sign * acos (as)      ;
  free (RealV1);    free (RealV2);   free (TempV);
  return (result)                ;

}


/************************************************************************
 Handle of the 1st left-button-click: Check which sensitive area the
 mouse cursor is on (if any) - and record the information.
 If the mouse holds a corner of the rect, the scaling center (SCenter)
 is saved in order to do the right scaling, later (in the DragMouse
 handler).
*************************************************************************/
void FirstClick (GEvent e)
{
  e.y                = oMaxY - e.y               ;
  Last_state.coord.x = e.x                      ;
  Last_state.coord.y = e.y                      ;
  Last_state.Area    = TranslateLoc (e.x, e.y)  ;

  switch (Last_state.Area) {
  case InMiddle:
    Last_state.activity = ROTATE;
    break;
  case InCircle:
    Last_state.activity = DRAG_CIRCLE;
	break;
  case InEdge:
    Last_state.activity = TRAN;
    break;
  case InCorner00:
    Last_state.activity = SCALE;
    Last_state.SCenter.x = -1;
    Last_state.SCenter.y = -1;
    break;
  case InCorner01:
    Last_state.activity = SCALE;
    Last_state.SCenter.x = -1;
    Last_state.SCenter.y =  0;
	break;
  case InCorner10:
    Last_state.activity = SCALE;
    Last_state.SCenter.x =  0;
    Last_state.SCenter.y = -1;
    break;
  case InCorner11:
    Last_state.activity = SCALE;
    Last_state.SCenter.x = 0;
    Last_state.SCenter.y = 0;
    break;
  case Outside:
    Last_state.activity = NONE;
	break;
  }
}


/***********************************************************************
 Release left mouse button handler - set the flag that indicated the
 current activity to NONE.
************************************************************************/
void ResetEvent (GEvent e)
{
  Last_state.activity = NONE;
}


/************************************************************************
  Main routine: What to do when dragging the mouse. This is the GDrag
  handler. It checks what the current activity is, and acts accordinaly -
  if DRAG_CIRCLE - moves the circle to the new location of the mouse,
  after calculating the moveing vector (using the information about
  the last mouse cursor location, stored in struct Last_state.
  If TRAN - move the rectangle to the nre location, and so on - according
  to the current activity (also stored in struct Last_state).
************************************************************************/
void DragMouse (GEvent e)
{
  Vec2f  Curr, *MoveV = Vec2fNew()                                ;
  Matrix *Move = MatCreate (1, 3, Base00), *Inv = MatNew(3,3)     ;
  Matrix *SaveCTMat = MatNew(3,3)                                 ;
  float ScaleByX, ScaleByY                                        ;
  float W = GetRectWidth(), H = GetRectHeight()                   ;

  e.y = oMaxY - e.y         ;
  Vec2fSetX (&Curr, e.x)   ;  
  Vec2fSetY (&Curr, e.y)   ;
  
  Vec2fSub (&Curr, &Last_state.coord, MoveV) ; /* Calculate real move */
  MatSet (Move, 0, 0, Vec2fGetX((&Curr)))    ; /* Store it in matrix-wise vector */
  MatSet (Move, 0, 1, Vec2fGetY((&Curr)))    ;

  switch (Last_state.activity) {

  case DRAG_CIRCLE:
    GClear()                  ;
    Vec2fSetX (&Last_state.FilledCircle,
	       Vec2fGetX ((&Last_state.FilledCircle)) +
	       Vec2fGetX (MoveV))  ;
    Vec2fSetY (&Last_state.FilledCircle,
	       Vec2fGetY ((&Last_state.FilledCircle)) +
	       Vec2fGetY (MoveV))  ;
    break                     ;

  case TRAN:
    GSetForeground(INVERTED)  ;
    DrawRect()                ;
    Translate (Vec2fGetX (MoveV), Vec2fGetY (MoveV));
    break                     ;

  case SCALE:
    MatCopy (SaveCTMat, CTMat)   ;
    GSetForeground(INVERTED)     ;
    DrawRect()                   ;
    MatInv(CTMat, Inv)           ;
    MatMul(CTMat, Inv, CTMat)    ;
    Translate (Vec2fGetX ((&Last_state.SCenter)),
		   Vec2fGetY ((&Last_state.SCenter))) ;
    MatMul (Move, Inv, Move)                 ;
    MatMul (Move, CTMat, Move)               ;
    Vec2fSetX (MoveV, MatGet (Move, 0,0))    ;
    Vec2fSetY (MoveV, MatGet (Move, 0,1))    ;

    ScaleByX = fabs(Vec2fGetX (MoveV))       ;
    ScaleByY = fabs(Vec2fGetY (MoveV))       ;
    if ((W * ScaleByX >= MIN_WIDTH) && (H * ScaleByY >= MIN_WIDTH) &&
      MouseOnRightSide(MoveV)) {
      Scale (ScaleByX, ScaleByY)             ;
    }
    Translate (-Vec2fGetX ((&Last_state.SCenter)),
			   -Vec2fGetY ((&Last_state.SCenter))) ;
    MatMul (CTMat, SaveCTMat, CTMat)               ;
    break                                          ;

  case ROTATE:
    GSetForeground(INVERTED)        ;
    DrawRect()                      ;
    Rotate (Angel (&Last_state.coord, &Curr,&Last_state.FilledCircle),
	    &Last_state.FilledCircle)    ;
    break                           ;

  }    
 
  GSetForeground(NORMAL)            ;
  DrawCircle(Vec2fGetX((&Last_state.FilledCircle)),
             Vec2fGetY((&Last_state.FilledCircle)))      ;

  DrawRect()                                             ;
  Vec2fSetX (&Last_state.coord, e.x)                     ;
  Vec2fSetY (&Last_state.coord, e.y)                     ;

  MatFree (Move);  MatFree (Inv);  MatFree (SaveCTMat)   ;
  free(MoveV)                                            ;   
}


/************************************************************************
   Exit - on right button click
*************************************************************************/
void EndProgram (GEvent e)
{
  free (Vec00), free (Vec01), free (Vec10), free (Vec11);
  MatFree (CTMat) ;
  GcloseWindow () ;
  exit(0)         ;
}

/************************************************************************
 Checks if the mouse has not passed to the other side of the rectangle
 (where scaling is impossible). Returns TRUE if the mouse is on the
 right side, FALSE else.
*************************************************************************/
int MouseOnRightSide(Vec2f *Loc)
{
   float x = Vec2fGetX (Loc), y = Vec2fGetY(Loc);

   switch (Last_state.Area) {
      case  InCorner00:
         if (x > 0 || y > 0) return (FALSE)  ;
         break                               ;
      case  InCorner11:
         if (x < 0 || y < 0) return (FALSE)  ;
		 break                               ;
      case  InCorner10: 
         if (x < 0 || y > 0) return (FALSE)  ;
         break                               ;
      case  InCorner01: 
         if (x > 0 || y < 0) return (FALSE)  ;
         break                               ;
      default:
         return (TRUE)                       ;
         break                               ;
      }
  /*   Else   */
   return (TRUE)                             ;
}


/* -------------------------------------------------------------------- */
