/****************************************************************************
*
*						MegaGraph Graphics Library
*
*                   Copyright (C) 1993 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: mgraph.h $
* Version:		$Revision: 1.1 $
*
* Language:		ANSI C
* Environment:	IBM PC (MS DOS)
*
* Description:	Header file for the MegaGraph graphics library.
*
* $Id: mgraph.h 1.1 1993/12/30 16:09:16 kjb release $
*
****************************************************************************/

#ifndef	__MGRAPH_H
#define	__MGRAPH_H

#ifndef	__DEBUG_H
#include "debug.h"
#endif

#ifndef	__MOUSE_H
#include "mouse.h"
#endif

/*---------------------- Macros and type definitions ----------------------*/

/* Define the graphics subsystems available	*/

typedef enum {
	grDETECT,				/* Auto detect the graphics subsystem		*/
	grNONE,					/* No graphics hardware detected			*/
	grEGA,					/* Standard EGA with 256k RAM				*/
	grVGA,					/* Standard VGA								*/
	grSVGA,					/* VESA VBE compliant Super VGA				*/
	grSVGA_S3,				/* S3 accelerated SuperVGA					*/
	grUSER,					/* User drivers start here					*/
	} MGL_driver;

/* Graphics modes supported	- the only video modes supported by this
 * graphics library are those that support at least 16 colors per pixel.
 */

typedef enum {
	/* 16 color EGA video modes */

	grEGA_320x200x16,			/* 320x200 16 color 8 page				*/
	grEGA_640x200x16,			/* 640x200 16 color 4 page				*/
	grEGA_640x350x16,			/* 640x350 16 color 2 page				*/

	/* 16 color VGA video modes */

	grVGA_640x400x16,			/* 640x400 16 color 2 page				*/
	grVGA_640x480x16,			/* 640x480 16 color 1 page				*/

	/* 256 color VGA video modes */

	grVGA_320x200x256,			/* 320x200 256 color 1 page				*/

	/* 256 color VGA ModeX video modes */

	grVGAX_320x200x256,			/* 320x200 256 color 4 page				*/
	grVGAX_320x240x256,			/* 320x240 256 color 3 page				*/
	grVGAX_320x400x256,			/* 320x400 256 color 2 page				*/
	grVGAX_320x480x256,			/* 320x480 256 color 1 page				*/
	grVGAX_360x200x256,			/* 360x200 256 color 3 page				*/
	grVGAX_360x240x256,			/* 360x240 256 color 3 page				*/
	grVGAX_360x400x256,			/* 360x400 256 color 1 page				*/
	grVGAX_360x480x256,			/* 360x480 256 color 1 page				*/

	/* 16 color Super VGA video modes */

	grSVGA_800x600x16,			/* 800x600 16 color 1 page				*/

	/* 256 color Super VGA video modes */

	grSVGA_640x350x256,			/* 640x250 256 color 1 page				*/
	grSVGA_640x400x256,			/* 640x400 256 color 1 page				*/
	grSVGA_640x480x256,			/* 640x480 256 color 1 page				*/
	grSVGA_800x600x256,			/* 800x600 256 color 1 page				*/
	grSVGA_1024x768x256,		/* 1024x768 256 color 1 page			*/
	grSVGA_1280x1024x256,		/* 1280x1024 256 color 1 page			*/

	/* 32,768 color Super VGA video modes */

	grSVGA_320x200x32k,			/* 320x200 32,768 color 1 page			*/
	grSVGA_640x350x32k,			/* 640x350 32,768 color 1 page			*/
	grSVGA_640x400x32k,			/* 640x400 32,768 color 1 page			*/
	grSVGA_640x480x32k,			/* 640x480 32,768 color 1 page			*/
	grSVGA_800x600x32k,			/* 800x600 32,768 color 1 page			*/
	grSVGA_1024x768x32k,		/* 1024x768 32,768 color 1 page			*/
	grSVGA_1280x1024x32k,		/* 1280x1024 32,768 color 1 page		*/

	/* 65,536 color Super VGA video modes */

	grSVGA_320x200x64k,			/* 320x200 65,536 color 1 page			*/
	grSVGA_640x350x64k,			/* 640x350 65,536 color 1 page			*/
	grSVGA_640x400x64k,			/* 640x400 65,536 color 1 page			*/
	grSVGA_640x480x64k,			/* 640x480 65,536 color 1 page			*/
	grSVGA_800x600x64k,			/* 800x600 65,536 color 1 page			*/
	grSVGA_1024x768x64k,		/* 1024x768 65,536 color 1 page			*/
	grSVGA_1280x1024x64k,		/* 1280x1024 65,536 color 1 page		*/

	/* 16 million color Super VGA video modes */

	grSVGA_320x200x16m,			/* 320x200 16 million color 1 page		*/
	grSVGA_640x350x16m,			/* 640x350 16 million color 1 page		*/
	grSVGA_640x400x16m,			/* 640x400 16 million color 1 page		*/
	grSVGA_640x480x16m,			/* 640x480 16 million color 1 page		*/
	grSVGA_800x600x16m,			/* 800x600 16 million color 1 page		*/
	grSVGA_1024x768x16m,		/* 1024x768 16 million color 1 page		*/
	grSVGA_1280x1024x16m,		/* 1280x1024 16 million color 1 page	*/

	grMAXMODE,					/* Maximum mode number					*/
	} MGL_mode;

/* MGL_result() error codes	*/

typedef enum {
	grOK			= 0,	/* No error									*/
	grNoInit		= -1,	/* Graphics driver has not been installed	*/
	grNotDetected	= -2,	/* Graphics hardware was not detected		*/
	grDriverNotFound= -3,	/* Graphics driver file was not found		*/
	grBadDriver		= -4,	/* File loaded was not a graphics driver	*/
	grLoadMem		= -5,	/* Not enough memory to load graphics driver*/
	grInvalidMode	= -6,	/* Invalid graphics mode for selected driver*/
	grInvalidDriver = -7,	/* Driver number is invalid					*/
	grError			= -8,	/* General graphics error					*/
	grInvalidName	= -9,	/* Invalid driver name						*/
	grNoMem			= -10,	/* Not enough memory to perform operation	*/
	grNoModeSupport	= -11,	/* Select video mode not supported by hard.	*/
	grInvalidFont	= -12,	/* Invalid font data						*/
	grBadFontFile	= -13,	/* File loaded was not a font file			*/
	grFontNotFound	= -14,	/* Font file was not found					*/

	grLastError		= -14,	/* Last error number in list				*/
	} MGL_error;

#define	CLIPON		true
#define	CLIPOFF		false

/* All colors are represented as longs by the library. This allows
 * code to work correctly with up to 24 bit color device drivers. The
 * device drivers themselves expect the color to be a color index if in
 * a color mapped mode, or a 15/16/24 bit RGB tuple in a hicolor or truecolor
 * mode. You can use the appropriate routines to pack and unpack
 * colors into the color_t format.
 */

typedef	long		color_t;

/* Standard colors - these colors are also defined in conio.h, so we only
 * define them if they have not been defined before. This is the standard
 * set of colors for the IBM PC. The default palette will have been
 * programmed to contain these values when graphics modes is started. If
 * the palette has been changed, they will not correspond to the actual
 * colors on the screen.
 */

#ifndef __COLORS
#define	__COLORS

enum COLORS {
	BLACK,							/* dark colors	*/
	BLUE,
	GREEN,
	CYAN,
	RED,
	MAGENTA,
	BROWN,
	LIGHTGRAY,
	DARKGRAY,						/* light colors	*/
	LIGHTBLUE,
	LIGHTGREEN,
	LIGHTCYAN,
	LIGHTRED,
	LIGHTMAGENTA,
	YELLOW,
	WHITE,
	};

#endif

/* EGA color components. These values can be used to program the palette
 * with colors for the EGA. These are defined purely for convenience.
 */

typedef enum {
	RED0			= 0x00,
	RED1			= 0x54,
	RED2			= 0xA8,
	RED3			= 0xFC,
	GREEN0			= 0x00,
	GREEN1			= 0x54,
	GREEN2			= 0xA8,
	GREEN3			= 0xFC,
	BLUE0			= 0x00,
	BLUE1			= 0x54,
	BLUE2			= 0xA8,
	BLUE3			= 0xFC,
	} ega_colors;

typedef enum {
	MARKER_SQUARE,
	MARKER_CIRCLE,
	MARKER_X,
	} marker_style;

typedef enum {						/* Write mode operators				*/
	REPLACE_MODE,					/* Replace mode						*/
	AND_MODE,						/* AND mode							*/
	OR_MODE,						/* OR mode							*/
	XOR_MODE,						/* XOR mode							*/
	} write_mode;

typedef enum {
	SOLID_PATTERN,
	BITMAP_PATTERN_OPAQUE,
	BITMAP_PATTERN_TRANSPARENT,
	PIXMAP_PATTERN,
	} fill_style;

typedef enum {
	CONVEX_POLYGON,					/* Monotone vertical polygon		*/
	COMPLEX_POLYGON,				/* Non-Simple polygons				*/
	AUTO_POLYGON,					/* Auto detect the polygon type		*/
	} polygon_types;

/* Text manipulation defines */

typedef enum {
	LEFT_TEXT		=	0,			/* Justify from left				*/
	TOP_TEXT		=	0,			/* Justify from top					*/
	CENTER_TEXT		=	1,			/* Center the text					*/
	RIGHT_TEXT		=	2,			/* Justify from right				*/
	BOTTOM_TEXT		=	2,			/* Justify from bottom				*/
	BASELINE_TEXT	=	3,			/* Justify from the baseline		*/
	} text_just;

typedef enum {
	LEFT_DIR		=	0,			/* Text goes to left				*/
	UP_DIR			=	1,			/* Text goes up						*/
	RIGHT_DIR		=	2,			/* Text goes right					*/
	DOWN_DIR		=	3,			/* Text goes down					*/
	} text_dir;

/* Standard driver names	*/

#define	EGANAME		"EGA.DRV"
#define	VGANAME		"VGA.DRV"
#define	VGA256NAME	"VGA256.DRV"
#define VGAXNAME	"VGAX.DRV"
#define	SVGA256NAME	"SVGA256.DRV"
#define	SVGA32KNAME	"SVGA32K.DRV"
#define	SVGA64KNAME	"SVGA64K.DRV"
#define	SVGA16MNAME	"SVGA16M.DRV"
#define	SVGAS3NAME	"SVGAS3.DRV"

/* Standard bitmap names	*/

#define	EMPTY_FILL	__empty_pat
#define	GRAY_FILL	__gray_pat
#define	SOLID_FILL	__solid_pat

/* Here we define the structures used to represent points and rectangles.
 * If we are compiling in C++ mode and we have included the "point.hpp"
 * and "rect.hpp" file respectively, we can simply provide a typedef
 * to allow us to pass C++ style points and rectangles directly to the
 * MGL.
 */

#if	defined(__cplusplus) && defined(__POINT_HPP)

typedef	Point	point;

#else

typedef struct {
	int	x,y;
	} point;

#endif

#if	defined(__cplusplus) && defined(__RECT_HPP)

typedef	Rect	rect;

#else

typedef	struct {
	int	left;
	int	top;
	int	right;
	int	bottom;
	} rect;

#endif

/* Pixel format structure */

typedef struct {
	char	redMask,blueMask;		/* Mask values for pixels			*/
	char	greenMask,rsvdMask;
	int		redPos,redAdjust;		/* Blue position and adjustment		*/
	int		greenPos,greenAdjust;	/* Green position and adjustment	*/
	int		bluePos,blueAdjust;		/* Blue position and adjustment		*/
	int		rsvdPos,rsvdAdjust;		/* Reserved position and adjustment */
	} pixel_format;

/* Structure to hold arc coordinate information */

typedef struct {
	int		x,y;					/* Centre point of the arc			*/
	int		startX,startY;			/* Starting point on arc			*/
	int		endX,endY;				/* Ending point on arc				*/
	} arc_coords;

/* Font types */

#define	FONT8X8		0				/* Built in 8x8 bitmap font			*/
#define	VECTORFONT	1				/* Vector font						*/
#define	FIXEDFONT	2				/* Fixed width bitmap font			*/
#define	PROPFONT	3				/* Proportional width bitmap font	*/

/* Default font name */

#define	DEFAULT_FONT	&__def_font	/* Built in 8x8 bitmap font			*/

#define	_FNAMESIZE		58

/* Generic Font structure */

typedef struct {
	char			name[_FNAMESIZE];/* Name of the font				*/
	short			fontType;		/* Type of font						*/
	short			maxWidth;		/* Maximum character width			*/
	short			maxKern;		/* Maximum character kern			*/
	short			fontWidth;		/* Font width						*/
	short			fontHeight;		/* Font height						*/
	short			ascent;			/* Font ascent value				*/
	short			descent;		/* Font descent value				*/
	short			leading;		/* Font leading value				*/
	} font;

/* Character and font metrics structure */

typedef struct {
	int				width;			/* Width of character or font		*/
	int				fontWidth;		/* Character width (tightest fit)	*/
	int				fontHeight;		/* Height of the font				*/
	int				ascent;			/* Ascent value						*/
	int				descent;		/* Descent value					*/
	int				leading;		/* Leading value					*/
	int				kern;			/* Kern value						*/
	} metrics;

/* Text settings structure */

typedef struct {
	int				horiz_just;		/* Horizontal justfication			*/
	int				vert_just;		/* Vertical justification			*/
	int				dir;			/* Text drawing direction			*/
	int 			sz_numerx;		/* Text x size numerator			*/
	int 			sz_numery;		/* Text y size numerator			*/
	int				sz_denomx;		/* Text x size denominator			*/
	int				sz_denomy;		/* Text y size denominator			*/
	int				space_extra;	/* Space extra term					*/
	font 			*fnt;			/* Currently selected font			*/
	} text_settings;

/* Macros to access the (left,top) and (right,bottom) points of a
 * rectangle.
 */

#define	leftTop(r)		(((point *) &(r))[0])
#define	rightBottom(r)	(((point *) &(r))[1])

typedef uchar	pattern[8];
typedef color_t	pixpattern[8][8];

/* Attributes structure */

typedef struct {
	color_t			color;			/* Foreground color					*/
	color_t			backColor;		/* Background color					*/
	int				markerSize;		/* Size of markers in pixels		*/
	marker_style	markerStyle;	/* Style of markers					*/
	color_t			markerColor;	/* Color to draw markers in			*/
	color_t			bdr_bright;		/* Border bright color				*/
	color_t			bdr_dark;		/* Border dark color				*/
	point			CP;				/* Graphics pen position			*/
	write_mode		writeMode;		/* Scan conversion write mode op.	*/
	fill_style		penStyle;		/* Pen style						*/
	int				penHeight;		/* Size of pen						*/
	int				penWidth;
	pattern			penPat;			/* Pattern for pen					*/
	pixpattern		penPixPat;		/* Pixmap pattern for pen			*/
	rect			viewport;		/* Viewport dimensions				*/
	rect			clipRect;		/* Clipping rectangle dimensions	*/
	bool			clip;			/* Is clipping on?					*/
	int				poly_type;		/* Polygon drawing type				*/
	text_settings	tsettings;		/* Text drawing attributes			*/
	} attributes;

/* Palette entry structure	*/

typedef struct {
	uchar	red;					/* Blue component of color			*/
	uchar	green;					/* Green component of color			*/
	uchar	blue;					/* Blue component of color			*/
	} palette;

/* Maximum value for each palette entry component */

#define	PALMAX		255				/* Max value for palette components	*/

/* Palette rotation directions */

#define	PAL_ROTATE_UP	0			/* Rotate the palette values up		*/
#define	PAL_ROTATE_DOWN	1			/* Rotate the palette values down	*/

/* Border drawing routine styles */

#define	BDR_INSET	0				/* Interior is inset into screen	*/
#define	BDR_OUTSET	1				/* Interior is outset from screen	*/
#define	BDR_OUTLINE	2				/* Border is 3d outline				*/

/* Bit Block Image header structure. The bit mask field is not used by
 * the 256 color and above routines.
 */

typedef struct {
	ushort	pixelRows;				/* Number of rows in image			*/
	ushort	pixelColumns;			/* Number of columns in image		*/
	ushort	bytesPerRow;			/* Width of image in bytes			*/
	uchar	filler;					/* Filler to keep word aligned		*/
	uchar	lastByteMask;			/* Last byte mask (16 color modes)	*/
	} image_header;

/* Icon structures. Each icon type is mode dependant, so be careful
 * when using the structures that you are in the correct video mode.
 * The mask is first used to punch a hole in the background, and the
 * icon image data is then OR'ed onto the display.
 *
 * The format of the header for each icon type should be:
 *
 *		icon16		= { 32, 32, 16, 255 }
 *		icon256		= { 32, 32, 32, 255 }
 *		icon32k		= { 32, 32, 64, 255 }
 *		icon16m		= { 32, 32, 96, 255 }
 */

typedef struct {					/* Basic icon structure				*/
	uchar			mask[128];		/* 32x32 monochrome image mask		*/
	image_header	image;			/* Image data header info			*/
	} icon_header;

typedef struct {
	icon_header     icon;			/* Icon header information			*/
	uchar			data[512];		/* 32x32 16 color data				*/
	} icon16;

typedef struct {
	icon_header     icon;			/* Icon header information			*/
	uchar			data[1024];		/* 32x32 256 color data				*/
	} icon256;

typedef struct {
	icon_header     icon;			/* Icon header information			*/
	uchar			data[2048];		/* 32x32 32k color data				*/
	} icon32k;

typedef struct {
	icon_header     icon;			/* Icon header information			*/
	uchar			data[2048];		/* 32x32 64k color data				*/
	} icon64k;

typedef struct {
	icon_header     icon;			/* Icon header information			*/
	uchar			data[3072];		/* 32x32 16m color data				*/
	} icon16m;

/*---------------------- Inline functions as Macros -----------------------*/

#define	MGL_equalPoint(p1,p2)	((p1).x == (p2).x && (p1).y == (p2).y)

#define	MGL_equalRect(r1,r2)	((r1).left == (r2).left &&			\
								 (r1).top == (r2).top &&			\
								 (r1).right == (r2).right &&		\
								 (r1).bottom == (r2).bottom)

#define	MGL_emptyRect(r)		((r).bottom <= (r).top || 			\
								 (r).right <= (r).left)

#define	MGL_disjointRect(r1,r2)	((r1).right <= (r2).left ||			\
								 (r1).left >= (r2).right ||			\
								 (r1).bottom <= (r2).top ||			\
								 (r1).top >= (r2).bottom)

#define	MGL_offsetRect(r,dx,dy)										\
	{	(r).left += dx; (r).right += dx;							\
		(r).top += dy; (r).bottom += dy; }

#define	MGL_insetRect(r,dx,dy)										\
	{	(r).left += dx; (r).right -= dx;							\
		(r).top += dy; (r).bottom -= dy;							\
		if (MGL_emptyRect(r))										\
			(r).left = (r).right = (r).top = (r).bottom = 0; }

#define	MGL_ptInRect(p,r)		((p).x >= (r).left &&				\
								 (p).x < (r).right &&				\
								 (p).y >= (r).top &&				\
								 (p).y < (r).bottom)

#define	MGL_ptInRectCoord(x,y,r)	((x) >= (r).left &&				\
									 (x) < (r).right &&				\
									 (y) >= (r).top &&				\
									 (y) < (r).bottom)

#define	MGL_pixel(p)		MGL_pixelCoord((p).x,(p).y)
#define	MGL_getPixel(p)		MGL_getPixelCoord((p).x,(p).y)
#define	MGL_moveTo(p)		MGL_moveToCoord((p).x,(p).y)
#define MGL_moveRel(p)		MGL_moveRelCoord((p).x,(p).y)
#define	MGL_line(p1,p2)		MGL_lineCoord((p1).x,(p1).y,(p2).x,(p2).y)
#define	MGL_lineFast(p1,p2)	MGL_lineCoordFast((p1).x,(p1).y,(p2).x,(p2).y)
#define	MGL_lineTo(p)		MGL_lineToCoord((p).x,(p).y)
#define	MGL_lineRel(p)		MGL_lineRelCoord((p).x,(p).y);
#define	MGL_rectPt(lt,rb)	MGL_rectCoord((lt).x,(lt).y,(rb).x,(rb).y)
#define	MGL_rect(r)			MGL_rectCoord((r).left,(r).top,			\
									(r).right,(r).bottom)
#define	MGL_drawBorder(r,s,t)	MGL_drawBorderCoord((r).left,(r).top,	\
									(r).right,(r).bottom,(s),(t))
#define	MGL_fillRectPt(lt,rb)	MGL_fillRectCoord((lt).x,(lt).y,	\
									(rb).x,(rb).y)
#define	MGL_fillRect(r)			MGL_fillRectCoord((r).left,(r).top,	\
									(r).right,(r).bottom)
#define	MGL_getImage(r,image)	MGL_getImageCoord((r).left,(r).top,	\
									(r).right,(r).bottom,image)
#define	MGL_putImage(r,image,op)	MGL_putImageCoord((r).left,		\
									(r).top,(r).right,(r).bottom,	\
									image,op)
#define	MGL_imageSize(r)		MGL_imageSizeCoord((r).left,(r).top,\
									(r).right,(r).bottom)
#define	MGL_copyImage(r,dl,dt,sp,dp) \
								MGL_copyImageCoord((r).left,(r).top,\
									(r).right,(r).bottom,dl,dt,sp,dp);
#define	MGL_getDivot(r,divot)	MGL_getDivotCoord((r).left,(r).top,	\
									(r).right,(r).bottom,divot)
#define	MGL_divotSize(r)		MGL_divotSizeCoord((r).left,(r).top,\
									(r).right,(r).bottom)

#ifdef	__cplusplus

// Undefine a number of troublesome macros that work with rectangles. We
// will define them again later on as inline functions. This is a hack,
// but it works, assuming you have left(), top(), right() and bottom()
// member functions for your rectangle class.

#undef	MGL_rect
#undef	MGL_drawBorder
#undef	MGL_fillRect
#undef	MGL_getImage
#undef	MGL_putImage
#undef	MGL_copyImage
#undef	MGL_imageSize
#undef	MGL_getDivot
#undef	MGL_divotSize
#endif

/*--------------------------- Global Variables ----------------------------*/

extern	pattern	__empty_pat;
extern	pattern	__gray_pat;
extern	pattern __solid_pat;
extern	bool	_installMouse;
extern	cursor	DEF_CURSOR;
extern	font	__def_font;

/*------------------------- Function Prototypes ---------------------------*/

#ifdef	__cplusplus
extern "C" {			/* Use "C" linkage when in C++ mode	*/
#endif

/* In mgraph.c */

int 	MGL_result(void);
void	MGL_setResult(int result);
char 	* MGL_errorMsg(int err);
char	* MGL_modeName(int mode);
char	* MGL_driverName(int driver);
color_t	MGL_maxColor(void);
int		MGL_maxPage(void);
int 	MGL_sizex(void);
int 	MGL_sizey(void);
int		MGL_getActivePage(void);
int		MGL_getVisualPage(void);
int		MGL_getDriver(void);
int		MGL_getMode(void);
bool	MGL_doubleBuffer(void);
void	MGL_singleBuffer(void);
void	MGL_swapBuffers(void);
int 	MGL_getAspectRatio(void);
void 	MGL_setAspectRatio(int aspectRatio);
void 	MGL_setGraphMode(void);
void 	MGL_restoreCRTMode(void);
int 	MGL_registerDriver(const char *name,void *driver);
void 	MGL_detectGraph(int far *graphdriver,int far *graphmode);
void 	MGL_init(int *graphdriver,int *graphmode,const char *pathtodriver);
void 	MGL_exit(void);
void 	MGL_setBufSize(unsigned size);

/* In modes.c */

int     * MGL_availableModes(int driver);
int		MGL_availablePages(int driver,int mode);

/* In pixel.c */

void	MGL_beginPixel(void);
void 	MGL_pixelCoord(int x,int y);
color_t	MGL_getPixelCoord(int x,int y);
void	MGL_endPixel(void);

/* In line.c */

void 	MGL_moveToCoord(int x,int y);
void 	MGL_moveRelCoord(int dx,int dy);
int 	MGL_getX(void);
int 	MGL_getY(void);
void	MGL_getCP(point* CP);
void 	MGL_lineCoord(int x1,int y1,int x2,int y2);
void	MGL_lineCoordFast(int x1,int y1,int x2,int y2);
void 	MGL_lineToCoord(int x,int y);
void 	MGL_lineRelCoord(int dx,int dy);
bool	MGL_clipLine(point *p1,point *p2,rect r);
void	MGL_lineEngine(int x1,int y1,int x2,int y2,
		void (*plotPoint)(int x,int y));

/* In polyline.c */

void 	MGL_marker(point p);
void	MGL_polyPoint(int count,point *vArray);
void 	MGL_polyMarker(int count,point *vArray);
void 	MGL_polyLine(int count,point *vArray);

/* In polygon.c */

bool 	MGL_fillPolygon(int count,point *vArray,
		int xOffset,int yOffset);
bool	MGL_fillPolygonFast(int count,point *vArray,
		int xOffset,int yOffset);
bool 	MGL_fillGouraudPolygon(int count,point *vArray,
		color_t *cArray,int xOffset,int yOffset);

/* Note: The above fast polygon routine and gouraud shaded polygon
 * 		 routines only work with convex polygons
 */

/* In rectangl.c */

void 	MGL_rectCoord(int left,int top,int right,int bottom);
void 	MGL_fillRectCoord(int left,int top,int right,int bottom);

/* In border.c */

void	MGL_drawBorderCoord(int left,int top,int right,int bottom,
		int style,int thickness);
void 	MGL_drawHDivider(int y,int x1,int x2);
void 	MGL_drawVDivider(int x,int y1,int y2);

/* In ellipse.c */

bool 	MGL_ellipseArc(rect extentRect,int startAngle,int endAngle);
bool 	MGL_ellipseArcCoord(int x,int y,int xradius,int yradius,
		int startAngle,int endAngle);
void	MGL_getArcCoords(arc_coords *coords);
bool 	MGL_ellipse(rect extentRect);
bool 	MGL_ellipseCoord(int x,int y,int xradius,int yradius);
bool 	MGL_fillEllipseArc(rect extentRect,int startAngle,int endAngle);
bool 	MGL_fillEllipseArcCoord(int x,int y,int xradius,int yradius,
		int startAngle,int endAngle);
bool 	MGL_fillEllipse(rect extentRect);
bool 	MGL_fillEllipseCoord(int x,int y,int xradius,int yradius);
void	MGL_ellipseEngine(rect extentRect,
		void (*setup)(int topY,int botY,int left,int right),
		void (*set4pixels)(bool inc_x,bool inc_y,bool region1),
		void (*finished)(void));

/* In attr.c */

color_t	MGL_packColor(uchar R,uchar G,uchar B);
void	MGL_unpackColor(color_t color,uchar *R,uchar *G,uchar *B);
void	MGL_setMarkerSize(int size);
int		MGL_getMarkerSize(void);
void	MGL_setMarkerStyle(int style);
int		MGL_getMarkerStyle(void);
void	MGL_setMarkerColor(color_t color);
color_t	MGL_getMarkerColor(void);
void	MGL_setBorderColors(color_t bright,color_t dark);
void	MGL_getBorderColors(color_t *bright,color_t *dark);
color_t	MGL_defaultColor(void);
void 	MGL_setWriteMode(int mode);
int 	MGL_getWriteMode(void);
void 	MGL_setPenStyle(int style);
int 	MGL_getPenStyle(void);
void 	MGL_setPenBitmapPattern(pattern *pat);
void 	MGL_getPenBitmapPattern(pattern *pat);
void 	MGL_setPenPixmapPattern(pixpattern *pat);
void 	MGL_getPenPixmapPattern(pixpattern *pat);
void	MGL_setPenSize(int height,int width);
void 	MGL_getPenSize(int *height,int *width);
void	MGL_setPolygonType(int type);
int		MGL_getPolygonType(void);
void 	MGL_getAttributes(attributes *attr);
void 	MGL_restoreAttributes(attributes *attr);
void 	MGL_defaultAttributes(void);

/* In palette.c */

color_t	MGL_realColor(int color);
void	MGL_setPaletteSnowLevel(int level);
int		MGL_getPaletteSnowLevel(void);
void 	MGL_setPaletteEntry(int entry,uchar red,uchar green,uchar blue);
void 	MGL_getPaletteEntry(int entry,uchar *red,uchar *green,
		uchar *blue);
void 	MGL_setPalette(palette *pal,int numColors,int startIndex);
void 	MGL_getPalette(palette *pal,int numColors,int startIndex);
void	MGL_rotatePalette(palette *pal,int numColors,int direction);
bool	MGL_fadePalette(palette *dest,palette *src,
		int numColors,uchar intensity);
int		MGL_getPaletteSize(void);
void	MGL_getDefaultPalette(palette *pal);
void 	MGL_setDefaultPalette(void);

/* In viewport.c */

void 	MGL_setViewport(rect view);
void	MGL_setRelViewport(rect view);
void 	MGL_getViewport(rect *view);
void	MGL_globalToLocal(point *p);
void	MGL_localToGlobal(point *p);
int 	MGL_maxx(void);
int 	MGL_maxy(void);
void 	MGL_clearViewport(void);
void 	MGL_setClipRect(rect clip);
void 	MGL_getClipRect(rect *clip);
void 	MGL_setClipMode(bool mode);
bool 	MGL_getClipMode(void);
void	MGL_pushViewport(void);
void	MGL_popViewport(void);

/* In copypixe.c */

void 	MGL_getImageCoord(int left,int top,int right,int bottom,
		void *image);
void	MGL_putImageCoord(int left,int top,int right,int bottom,
		void *image,int op);
long	MGL_imageSizeCoord(int left,int top,int right,int bottom);
void	MGL_putMonoImage(int x,int y,int byteWidth,int height,
		void *image);
void 	MGL_getDivotCoord(int left,int top,int right,int bottom,
		void *divot);
void 	MGL_putDivot(void *divot);
long 	MGL_divotSizeCoord(int left,int top,int right,int bottom);
void	MGL_copyImageCoord(int left,int top,int right,int bottom,
		int dstLeft,int dstTop,int srcPage,int dstPage);
void	MGL_putIcon(int x,int y,icon_header *icon);

/* In text.c */

void 	MGL_setTextJustify(int horiz,int vert);
void 	MGL_getTextJustify(int *horiz,int *vert);
void 	MGL_setTextDirection(int direction);
int     MGL_getTextDirection(void);
void	MGL_setTextSize(int numerx,int denomx,int numery,int denomy);
void	MGL_getTextSize(int *numerx,int *denomx,
		int *numery,int *denomy);
void	MGL_setSpaceExtra(int extra);
int		MGL_getSpaceExtra(void);
void	MGL_setTextSettings(text_settings *settings);
void	MGL_getTextSettings(text_settings *settings);
int 	MGL_textHeight(void);
int 	MGL_textWidth(const char *str);
int		MGL_charWidth(char ch);
void	MGL_getFontMetrics(metrics *metrics);
void	MGL_getCharMetrics(char ch,metrics *metrics);
int		MGL_maxCharWidth(void);
void 	MGL_drawStr(const char *str);
void	MGL_drawStrXY(int x,int y,const char *str);
void	MGL_underScoreLocation(int *x,int *y,
		const char *str);
font 	* MGL_loadFont(const char *fontname);
bool	MGL_availableFont(const char *fontname);
bool	MGL_unloadFont(font *font);
bool	MGL_useFont(font *font);
font 	* MGL_getFont(void);
bool	MGL_vecFontEngine(int x,int y,const char *str,
		void (*move)(int x,int y),void (*draw)(int x,int y));

/* In mglmouse.c */

void	MS_obscure(void);
void	MS_unobscure(void);

/* Globals to access routines within loaded driver */

extern void (*MGL_beginDrawing)(void);
extern void (*MGL_endDrawing)(void);
extern void (*MGL_setColor)(color_t color);
extern color_t (*MGL_getColor)(void);
extern void (*MGL_setBackColor)(color_t color);
extern color_t (*MGL_getBackColor)(void);
extern void (*MGL_setCursorColor)(color_t color);
extern color_t (*MGL_getCursorColor)(void);
extern void	(*MGL_clearDevice)(void);
extern void	(*MGL_scanLine)(int y,int x1,int x2);
extern void (*MGL_beginGouraud)(void);
extern void	(*MGL_gouraudScanLine)(int y,int x1,int x2,color_t c1,
	color_t c2);
extern void (*MGL_endGouraud)(void);
extern void (*MGL_setActivePage)(int page);
extern void (*MGL_setVisualPage)(int page);
extern int (*MGL_scanRightForColor)(int x,int y,color_t color);
extern int (*MGL_scanLeftForColor)(int x,int y,color_t color);
extern int (*MGL_scanRightWhileColor)(int x,int y,color_t color);
extern int (*MGL_scanLeftWhileColor)(int x,int y,color_t color);
extern void (*MGL_vSync)(void);

/* In order to allow the driver files to be linked in, we need to provide
 * the following declarations so that the linker can perform the appropriate
 * fixup's at link time.
 */

extern void EGA_driver(void);
extern void VGA_driver(void);
extern void VGA256_driver(void);
extern void VGAX_driver(void);
extern void SVGA256_driver(void);
extern void SVGA32K_driver(void);
extern void SVGA64K_driver(void);
extern void SVGA16M_driver(void);
extern void SVGAS3_driver(void);

#ifdef	__cplusplus
}						/* End of "C" linkage for C++	*/

#ifdef	__RECT_HPP

// Redefine a number of troublesome macros as inline functions for C++

inline void MGL_rect(Rect r)
{
	MGL_rectCoord(r.left(),r.top(),r.right(),r.bottom());
}

inline void MGL_drawBorder(Rect r,int style,int width)
{
	MGL_drawBorderCoord(r.left(),r.top(),r.right(),r.bottom(),style,width);
}

inline void MGL_fillRect(Rect r)
{
	MGL_fillRectCoord(r.left(),r.top(),r.right(),r.bottom());
}

inline void MGL_getImage(Rect r,void *image)
{
	MGL_getImageCoord(r.left(),r.top(),r.right(),r.bottom(),image);
}

inline void MGL_putImage(Rect r,void *image,int op)
{
	MGL_putImageCoord(r.left(),r.top(),r.right(),r.bottom(),image,op);
}

inline void	MGL_copyImage(rect r,int dstLeft,int dstTop,int srcPage,int dstPage)
{
	MGL_copyImageCoord(r.left(),r.top(),r.right(),r.bottom(),
		dstLeft,dstTop,srcPage,dstPage);
}

inline long MGL_imageSize(Rect r)
{
	return MGL_imageSizeCoord(r.left(),r.top(),r.right(),r.bottom());
}

inline void MGL_getDivot(Rect r,void *divot)
{
	MGL_getDivotCoord(r.left(),r.top(),r.right(),r.bottom(),divot);
}

inline long MGL_divotSize(Rect r)
{
	return MGL_divotSizeCoord(r.left(),r.top(),r.right(),r.bottom());
}
#endif	/* __RECT_HPP */

#endif	/* __cplusplus */

#endif	/* __MGRAPH_H */
