/********************************************************************/
/*             tbmdidlg.cpp - MDI Dialog Window Class               */
/*                 Written by Benjamin Cooley                       */
/********************************************************************/

// This class allows an MDI window to be initialized with a
// dialog resource.  This particular class performs extra initialization
// needed to support Borland's custom controls.
//
// Because this class merges a dialog with an MDI window, there are
// several special considerations that must be taken into account when
// using it.  First and formost, the window created IS an MDI child window
// and is a direct decendant of TWindow, and NOT TDialog.  This means that
// some of the member functions found in the TDialog class are NOT supported
// in this class.  See the TWindow class documentation in the OWL manuals
// for details on the TWindow class (remember, not the TDialog class).  Note
// that the contructors for the TBMDIDialog class are identical to the
// dialog constructors though, but be sure to use MakeWindow() instead of
// ExecDialog() when displaying the window.
//
// There are two quirks in the class presently that aren't serious, but I
// find rather annoying.  First, the first edit control to get the focus
// ends up having a caret that is a little bit too wide.  There should be
// a solution to this.  If someone finds it before I post a new version of
// this class, please send me a message.
//
// So.. to use this class, simply add the two Borland background brushes to
// your resource (you'll find them in BWCC.DLL), then include the files
// TBMDIDLG.H and TBMDIDLG.CPP in your project, then create the windows
// and display them using MakeWindow().  Also, if you wan't quiet
// initialization and the styles you selected in the Resource Workshop,
// try the MDIS_ALLCHILDSTYLES (0x0001) style in the MDI client window
// (subclass TMDIClient, or the style in the new class contructor, then
// redefine TMDIFrames InitClientWindow() function).  Just rembere the
// quirks I outlined above when using this new 3.1 style bit though.
//
// Hopefully I've caught all of Window's standard control classes.  Window's
// uses byte sized codes above hex 0x80 for standard controls instead of
// ASCII like the book said.  I think I caught them all.  If one of your
// controls doesn't show up.. try debugging the code in Create to see if
// the class code is being interpreted correctly.
//
// Note: if your borland controls do not match the controls in your dialog
// NASTY things will happen.  Make sure all borland controls have IDs that
// match at least something in the dialog resource.
//
// Also note: keyboard handling has been enabled in this window, as has
// the transfer mechanism.  Ok and Cancel buttons are also handled by
// default.

#include <owl.h>
#include <bwcc.h>
#include <string.h>
#include <mem.h>
#include <control.h>
#pragma hdrstop

#include "tbmdidlg.h"

#define BMP_VGABackground 998 // Put these bitmaps in your resource
#define BMP_EGABackground 999 // or you won't get a patterned background

#ifndef MDIS_ALLCHILDSTYLES
#define MDIS_ALLCHILDSTYLES    0x0001
#endif

//	*****************************
//	* Static Class Data Members *
//	*****************************

int TBMDIDialog::users = 0;
HBITMAP TBMDIDialog::bgbitmap = NULL;
HBRUSH TBMDIDialog::bgbrush = NULL;
HBRUSH TBMDIDialog::graybrush = NULL;

//	******************************
//	* Dialog Resource Structures *
//	******************************

// These structures are taken directly from the book Windows 3 : A
// Developer's Guide.

typedef struct {
	long dtStyle;
	BYTE dtItemCount;
	int dtX;
	int dtY;
	int dtCX;
	int dtCY;
	// char dtMenuName;
	// char dtClassName;
	// char dtCaptionText;
} DLGTEMPLATE;

typedef struct {
	short int PointSize;
	// char szTypeFace;
} FONTINFO;

typedef struct {
	int dtilX;
	int dtilY;
	int dtilCX;
	int dtilCY;
	int dtilID;
	long dtilStyle;
	// char dtilClass[];
	// char dtilText[];
	// BYTE dtilInfo;
	// char dtilData;
} DLGITEMTEMPLATE;

/********************************************************************/
/*      TBMDIDialog - Borland Custom Control MDI Dialog Class       */
/********************************************************************/

static Pchar GetString(Pchar &ptr)
{
	Pchar retptr = ptr;
	ptr += strlen(ptr) + 1;
	return retptr;
}

int TBMDIDialog::DialogX(int x)
{
	return (x * fontwidth)/4;
}

int TBMDIDialog::DialogY(int y)
{
	return (y * fontheight)/8;
}

void TBMDIDialog::LoadWindowAttr()
{
	HANDLE hdlg;

  /* Initialize dialog data members */
	font = NULL;
	fontwidth = LOWORD(GetDialogBaseUnits());
	fontheight = HIWORD(GetDialogBaseUnits());

  /* Get background brushes */
	if (users == 0) {
		bgbitmap = LoadBitmap(GetModule()->hInstance, MAKEINTRESOURCE(BMP_VGABackground));
		// EGA brush load should be here.. I didn't bother.. sorry
		if (bgbitmap) bgbrush = CreatePatternBrush(bgbitmap);
		  else bgbrush = CreateSolidBrush(RGB(192,192,192));
		graybrush = CreateSolidBrush(RGB(192,192,192));
	}
	users++;

  /* Load resource template (if not supplied by contstructor) */
	if (!templatebfr) {
		if (Name)
		  hdlg = FindResource(GetModule()->hInstance, Name, RT_DIALOG);
		else
		  hdlg = FindResource(GetModule()->hInstance, MAKEINTRESOURCE(Id), RT_DIALOG);
		if (!hdlg) return;
		HANDLE hres = LoadResource(GetModule()->hInstance, hdlg);
		if (!hres) return;
		int size = (int)SizeofResource(GetModule()->hInstance, hdlg);
		LPSTR rdata = LockResource(hres);
		if (!rdata) return;
		templatebfr = new char[size];
		ownstemplate = TRUE;
		_fmemcpy(templatebfr, rdata, size);	// Put into temporary buffer until Create()
		UnlockResource(hres);
	}

	if (templatebfr)
	{
		DLGTEMPLATE *dlgtemp;
		FONTINFO *fontinfo;

		Pchar ptr = templatebfr;

	  /* Set Attr structure with dlgtemp data (Can then change in constructor) */
		dlgtemp = (DLGTEMPLATE *)ptr;
		ptr += sizeof(DLGTEMPLATE);
		GetString(ptr); // Don't care about menu name
		GetString(ptr); // Don't care about class name (using TBWindow)
		SetCaption(GetString(ptr));

	  /* Set font information for window */
		fontinfo = (FONTINFO *)ptr;
		if (fontinfo->PointSize > 0)
		{
			ptr += sizeof(FONTINFO); // Skip font info
			Pchar fontname = GetString(ptr);
			if (!*fontname) fontname = NULL;
			font =  CreateFont(fontinfo->PointSize, 0, 0, 0,
				FW_BOLD, 0, 0, 0, 0, 0, 0, 0, 0, fontname);  // Use boldface

			HDC hdc = GetDC(Parent->HWindow);
			HFONT oldfont = SelectObject(hdc, font);
			TEXTMETRIC textmetric;
			GetTextMetrics(hdc, &textmetric);
			SelectObject(hdc, oldfont);
			ReleaseDC(Parent->HWindow, hdc);

			fontwidth = textmetric.tmAveCharWidth + 1;
			fontheight = textmetric.tmHeight;
		}

	  /* Set window attribute structure */
		Attr.Style = dlgtemp->dtStyle;
		Attr.X = DialogX(dlgtemp->dtX);
		Attr.Y = DialogY(dlgtemp->dtY);
		int framex, framey;
		if (Attr.Style & (WS_THICKFRAME + DS_MODALFRAME))
		{
			framex = GetSystemMetrics(SM_CXFRAME);
			framey = GetSystemMetrics(SM_CYFRAME);
		}
		else
		{
			framex = GetSystemMetrics(SM_CXBORDER);
			framey = GetSystemMetrics(SM_CYBORDER);
		}
		Attr.W = DialogX(dlgtemp->dtCX) + (framex * 2);
		Attr.H = DialogY(dlgtemp->dtCY) + GetSystemMetrics(SM_CYCAPTION) +
			(framey * 2);

   }

}

BOOL TBMDIDialog::Create()
{

  /* Check template buffer */
	if (!templatebfr) return FALSE;

  /* Turn of visible style till all created */
	visible = FALSE;

  /* Create controls */
	DLGTEMPLATE *dlgtemp;
	DLGITEMTEMPLATE *itemtemplate;
	Pchar ptr = templatebfr;
	Pchar items;
	int c;

  /* Set Attr structure with dlgtemp data */
	dlgtemp = (DLGTEMPLATE *)ptr;
	ptr += sizeof(DLGTEMPLATE); // Skip DLGTEMPLATE
	GetString(ptr); // Don't care about menu name
	GetString(ptr); // Don't care about class name (using TBWindow)
	GetString(ptr); // Don't care about dlg caption

  /* Font information for window */
	ptr += sizeof(FONTINFO); // Skip font info
	GetString(ptr); 		 // Don't care about font name right now

  /* Initialize OWL child controls */
	items = ptr;
	for (c = 1; c <= dlgtemp->dtItemCount; c++)
	{

	  /* Create dialog controls */
		itemtemplate = (DLGITEMTEMPLATE *)ptr;
		ptr += sizeof(DLGITEMTEMPLATE);
		if (*(BYTE *)ptr >= 0x80) ptr += 1; // Ignore class name here
			else GetString(ptr);
		Pchar text = GetString(ptr);
		int infolen = *(((BYTE *)ptr)++);
		Pchar info;
		if (infolen == 0) info = NULL;
			else info = ptr;
		ptr += infolen;

	  /* Turn FROMRESOURCE controls into normal controls */
		PTControl control = (PTControl)ChildWithId(itemtemplate->dtilID);
		if (control && (itemtemplate->dtilID > 0))
		{
			control->SetCaption(text);
			control->Attr.Style = itemtemplate->dtilStyle;
			control->Attr.X = DialogX(itemtemplate->dtilX);
			control->Attr.Y = DialogY(itemtemplate->dtilY);
			control->Attr.W = DialogX(itemtemplate->dtilCX);
			control->Attr.H = DialogY(itemtemplate->dtilCY);
			control->Attr.Param = info;
			control->SetFlags(WB_FROMRESOURCE, FALSE);
		}

	}

  /* Create window and OWL controls */
	if (!TBWindow::Create()) return FALSE;

  /* Create any child controls left in template */
	FocusChildHandle = NULL;	// Reset control focus ourselves
	ptr = items;                // Rewind template pointer
	for (c = 1; c <= dlgtemp->dtItemCount; c++)
	{

	  /* Create dialog controls */
		itemtemplate = (DLGITEMTEMPLATE *)ptr;
		ptr += sizeof(DLGITEMTEMPLATE);

		Pchar classname;
		if (*(BYTE *)ptr == 0x80) {
			classname = "BUTTON";
			ptr += 1;
		} else if (*(BYTE *)ptr == 0x81) {
			classname = "EDIT";
			ptr += 1;
		} else if (*(BYTE *)ptr == 0x82) {
			classname = "STATIC";
			ptr += 1;
		} else if (*(BYTE *)ptr == 0x83) {
			classname = "LISTBOX";
			ptr += 1;
		} else if (*(BYTE *)ptr == 0x84) {
			classname = "SCROLLBAR";
			ptr += 1;
		} else if (*(BYTE *)ptr == 0x85) {
			classname = "COMBOBOX";
			ptr += 1;
		} else if (*(BYTE *)ptr >= 0x80) {
			classname = "STATIC"; // Use for other unkown controls
			ptr += 1;
		} else {
			classname = GetString(ptr);
		}

		Pchar text = GetString(ptr);

		int infolen = *(((BYTE *)ptr)++);
		Pchar info;
		if (infolen == 0) info = NULL;
			else info = ptr;
		ptr += infolen;

		int X = DialogX(itemtemplate->dtilX);
		int Y = DialogY(itemtemplate->dtilY);
		int W = DialogX(itemtemplate->dtilCX);
		int H = DialogY(itemtemplate->dtilCY);

		HWND hcontrol;
		if (itemtemplate->dtilID <= 0) hcontrol = NULL;
		  else hcontrol = GetDlgItem(HWindow, itemtemplate->dtilID);
		if (!hcontrol)
		{
		  // *** Create control and set font ***
		  hcontrol = CreateWindow(classname, text, itemtemplate->dtilStyle,
			X, Y, W, H, this->HWindow, itemtemplate->dtilID,
			GetModule()->hInstance, info);
		   SendMessage(hcontrol, WM_SETFONT, (WORD)font, (DWORD)TRUE);
		}

		if (!hcontrol) return FALSE;

	}

  /* Free temporary template buffer */
	if (ownstemplate)
	{
		delete templatebfr;
		templatebfr = NULL;
	}

  /* Now that everythings hunky dory, show window */
	visible = TRUE;
	ShowWindow(HWindow, SW_SHOW);

  /* Success! */
	return TRUE;
}

static void SetChildFont(void *P, void *font)
{
	SendMessage(((PTWindowsObject)P)->HWindow,
		WM_SETFONT, (WORD)(*((HFONT *)font)), (DWORD)TRUE);
}

static BOOL HasStyle(void *P, void *Style)
{
  return ((PTWindowsObject)P)->HWindow &&
	(GetWindowLong(((PTWindowsObject)P)->HWindow, GWL_STYLE) &
	   *(long *)Style) == *(long *)Style;
}

/* Initializes ("sets up") the TWindow.  Called following a
   successful association between an MS-Windows interface element
   and a TWindow.  Calls TWindowsObject::SetupWindow to create windows
   in child list. If keyboard handling has been requested and
   FocusChildHandle has not been set, sets it to the first child.
   Sets the focus to TWindows created as MDI children.  If the
   TWindow has a TScroller object, calls the TScroller's
   SetSBarRange to set the range of the TWindow's window scrollbars.
   Can be redefined in descendant classes to perform additional
   initialization. BEN'S NOTE: same as in WINDOW.CPP in OWL source,
   but sets font between creating children and setting focus for
   kbhandler window */

void TBMDIDialog::SetupWindow()
{
  TWindowsObject::SetupWindow(); // BEN'S NOTE: skips TBWindow's SetupWindow()

  ForEach(SetChildFont, &font);	 // Set's child window's fonts

  if ( IsFlagSet(WB_KBHANDLER) && !FocusChildHandle )
  {
	long Style = WS_TABSTOP;
	PTWindowsObject TmpWO;
	TmpWO = FirstThat(HasStyle, &Style);
	if ( !TmpWO ) // no child has WS_TABSTOP
	{
	  Style = WS_CHILD;
	  TmpWO = FirstThat(HasStyle, &Style);
	}
	if ( TmpWO )
	  FocusChildHandle = TmpWO->HWindow;
  }

  if ( IsFlagSet(WB_MDICHILD) )
	SetFocus(HWindow);
  if ( Scroller )
	Scroller->SetSBarRange();
}

TBMDIDialog::TBMDIDialog(PTWindowsObject AParent, int ResourceId, PTModule AModule) :
	TBWindow(AParent, NULL, AModule)
{
	Id = ResourceId;    // Set Resource Id for window
	Name = NULL;		// NULL Means name not used

	templatebfr = NULL;
	ownstemplate = FALSE;

	visible = FALSE;	// Windows are drawn invisible, then shown

	LoadWindowAttr();

  /* Enable transfer and keyboard */
	EnableTransfer();
	EnableKBHandler();
}

TBMDIDialog::TBMDIDialog(PTWindowsObject AParent, Pchar AName, PTModule AModule) :
	TBWindow(AParent, NULL, AModule)
{
	Id = -1;     		// Not Used
	Name = strcpy(new char[strlen(AName + 1)], AName);
						// Set to resource name (non NULL means name used)

	templatebfr = NULL;
	ownstemplate = FALSE;

	visible = FALSE;	// Windows are drawn invisible, then shown

	LoadWindowAttr();

  /* Enable transfer and keyboard */
	EnableTransfer();
	EnableKBHandler();
}

TBMDIDialog::TBMDIDialog(PTWindowsObject AParent, Pvoid ATemplate, PTModule AModule) :
	TBWindow(AParent, NULL, AModule)
{
	Id = -1;    		// Not used
	Name = NULL;		// NULL Means name not used

	templatebfr = (Pchar)ATemplate;
	ownstemplate = FALSE;

	LoadWindowAttr();

  /* Enable transfer and keyboard */
	EnableTransfer();
	EnableKBHandler();
}

void TBMDIDialog::GetWindowClass(WNDCLASS& AWndClass)
{
	  TBWindow::GetWindowClass(AWndClass);

	/* Get pattererned background */
	  AWndClass.hbrBackground = bgbrush;
}

void TBMDIDialog::ControlColor(RTMessage Msg)
{

  /* Change background color for controls that need it */
	if (Msg.LP.Hi == CTLCOLOR_LISTBOX)
	{
		char buf[10];
		::GetClassName(Msg.LP.Lo, buf, 10);
		if (buf[0] != 'L') return;
	}
	else if ((Msg.LP.Hi != CTLCOLOR_LISTBOX) &&
		(Msg.LP.Hi != CTLCOLOR_STATIC) &&
		(Msg.LP.Hi != CTLCOLOR_BTN))
	  return;

	SetBkColor(Msg.WParam, RGB(192, 192, 192));
	Msg.Result = (DWORD)graybrush;

}

void TBMDIDialog::CloseWindow()
{
	CloseWindow(IDCANCEL);
}

void TBMDIDialog::CloseWindow(int ARetValue)
{
	HWND control = GetDlgItem(HWindow, ARetValue);
	if (control)
		SendMessage(HWindow, WM_COMMAND, ARetValue, MAKELONG(control, BN_CLICKED));
		else TBWindow::CloseWindow();
}

TBMDIDialog::~TBMDIDialog()
{
  /* If name of dialog stored.. delete it */
	if (Name) delete Name;

  /* Delete font object (if used) */
	if (font) DeleteObject(font);

  /* Delete brush objects when no longer needed */
	users--;
	if (users == 0) {
		DeleteObject(graybrush);
		DeleteObject(bgbrush);
		DeleteObject(bgbitmap);
	}

}

#pragma argsused
void TBMDIDialog::WMClose(TMessage& Msg)
{
	CloseWindow();
}

#define STYLESET ((DWORD)(WS_MAXIMIZEBOX | WS_MINIMIZEBOX | WS_CAPTION | WS_THICKFRAME | WS_SYSMENU))

void TBMDIDialog::WMNCPaint(RTMessage Msg)
{
	DWORD styleset = (GetWindowLong(HWindow, GWL_STYLE) & ~STYLESET) | (Attr.Style & STYLESET);
	if (visible) styleset = styleset | WS_VISIBLE;
		else styleset = styleset & ~((DWORD)WS_VISIBLE);
	SetWindowLong(HWindow, GWL_STYLE, styleset);
	DefWndProc(Msg);
}

void TBMDIDialog::WMNCCalcSize(RTMessage Msg)
{
	DWORD styleset = (GetWindowLong(HWindow, GWL_STYLE) & ~STYLESET) | (Attr.Style & STYLESET);
	if (visible) styleset = styleset | WS_VISIBLE;
		else styleset = styleset & ~((DWORD)WS_VISIBLE);
	SetWindowLong(HWindow, GWL_STYLE, styleset);
	DefWndProc(Msg);
}

void TBMDIDialog::WMCommand(RTMessage Msg)
{
	if (Msg.LP.Hi == BN_CLICKED)
	{
		if (Msg.WParam == IDOK)
		{
		  /* Transfer data and close window */
			if (CanClose()) {
				TransferData(TF_GETDATA);
				ShutDownWindow();
			}
			return;
		}
		else if (Msg.WParam == IDCANCEL)
		{
			TBWindow::CloseWindow();
			return;
		}
	}
	TBWindow::WMCommand(Msg);
}
