/*----------------------------------------------------------------------------*\
 *  MciMan.cpp: MCI class objects test application.
 *  
 *     $Author:   John Musser  $
 *       $Date:   24 Feb 1993 19:37:02  $
 *  $Copyright:   Personal Media International Inc., 1993 $
 *
 * Description:  Provides a simple menu interface to allow the user to 
 *               select either the Wave or AVI device, open a file, and
 *               play, stop, pause or resume. One file of each type may
 *               be open simultaneously.
 *               Designed to show how to use the MCIDevice classes.
 *
\*----------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "MCI.h"
#include "Resource.h"
#include <commdlg.h>

//-----------------------------------------------------------------------------
// Defines
//-----------------------------------------------------------------------------
#define  BUFLEN      128                              // char array size

#define  WAVEFILTER  "WAVE Files (*.WAV)\0*.WAV\0"    // open dialog filters
#define  AVIFILTER   "AVI Files (*.AVI)\0*.AVI\0"

//-----------------------------------------------------------------------------
// Globals
//-----------------------------------------------------------------------------
AVI      avi;         // One mci object for each mci type we're using
Wave     wave;

UINT            uCurDevice;      // current selection from the device menu
LPSTR           lpszCurFile;     // current file name
CompoundDevice  *mciDevice;      // currently selected device 

char chWaveFile[BUFLEN];         // current wave filename 
char chAviFile[BUFLEN];          // current avi filename 

HMENU      hMenu        = NULL;
HWND       hWndApp      = NULL;
HINSTANCE  hAppInst     = NULL;
char       szAppName[]  = "MciMan";

//-----------------------------------------------------------------------------
// Function Prototypes
//-----------------------------------------------------------------------------
BOOL NEAR PASCAL              InitApplication( HINSTANCE );
BOOL NEAR PASCAL              InitInstance( HINSTANCE, int );
LRESULT FAR PASCAL _export    MainWndProc( HWND, UINT, WPARAM, LPARAM);
BOOL FAR PASCAL _export       AboutProc( HWND, UINT, WPARAM, LPARAM);
void                          UpdateCaption();
void                          UpdateMenu( UINT );

//-----------------------------------------------------------------------------
// WinMain
//-----------------------------------------------------------------------------
int PASCAL WinMain(  HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR     lpszCmdLine,		  
                     int       nCmdShow )			  
{
   MSG msg;
   
   hAppInst = hInstance;

   if (!hPrevInstance)	
	   if (!InitApplication( hInstance )) 
	       return FALSE;	  

   if (!InitInstance( hInstance, nCmdShow ))
        return FALSE;

   while (GetMessage( &msg, NULL, NULL, NULL ))	 
  	{
      TranslateMessage( &msg );
     	DispatchMessage( &msg );	  
   }

   return msg.wParam;
}

//-----------------------------------------------------------------------------
// InitApplication
//-----------------------------------------------------------------------------
BOOL NEAR PASCAL InitApplication( HINSTANCE hInstance )
{
    WNDCLASS  wc;

    wc.style         = NULL;      
    wc.lpfnWndProc   = MainWndProc;
    wc.cbClsExtra    = 0;             
    wc.cbWndExtra    = 0;             
    wc.hInstance     = hInstance;      
    wc.hIcon         = LoadIcon( hInstance, szAppName );
    wc.hCursor       = LoadCursor( NULL, IDC_ARROW );
    wc.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
    wc.lpszMenuName  = szAppName;
    wc.lpszClassName = szAppName;

    return (RegisterClass(&wc));
}

//-----------------------------------------------------------------------------
// InitInstance
//-----------------------------------------------------------------------------
BOOL NEAR PASCAL InitInstance( HINSTANCE hInstance, int nCmdShow )
{
   HWND           hWnd;      

   hWnd = CreateWindow( szAppName,
                        szAppName,   
                        WS_OVERLAPPEDWINDOW,  
                        CW_USEDEFAULT, CW_USEDEFAULT,
                        500, 200,
                        NULL,                 
                        NULL,                 
                        hInstance,            
                        NULL                  
                     );

   if (!hWnd)
       return FALSE;
   else {
      hWndApp = hWnd;
      hMenu   = GetMenu(hWnd);
   }

   ShowWindow( hWnd, nCmdShow );
   UpdateWindow( hWnd );

   return TRUE;
}

//-----------------------------------------------------------------------------
// MainWndProc
//-----------------------------------------------------------------------------
LRESULT FAR PASCAL _export
MainWndProc( HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam )			  
{
   static OPENFILENAME ofn;        // used for all opens
   static char buf[BUFLEN];        // scratch buffer

   switch (wMsg)
   {
      case WM_CREATE:
         memset( &ofn, 0, sizeof(ofn));                // Setup openfile struct
         ofn.lStructSize   = sizeof(OPENFILENAME);
         ofn.hwndOwner     = hWnd;
         ofn.nFilterIndex  = 1;
         ofn.lpstrFile     = (LPSTR)buf;
         ofn.nMaxFile      = BUFLEN;

         PostMessage( hWnd, WM_COMMAND, IDM_WAVE, 0L);   // Start w/WAVE active 

         break;

	   case WM_COMMAND:
         switch (wParam)
         {
            case IDM_AVI:
              mciDevice       = &avi;          // Set object, file, open filter
              ofn.lpstrFile   = (LPSTR)chAviFile;
              ofn.lpstrFilter = (LPSTR)AVIFILTER;
              UpdateMenu(wParam);
              break;

            case IDM_WAVE:
              mciDevice       = &wave; 
              ofn.lpstrFile   = (LPSTR)chWaveFile;
              ofn.lpstrFilter = (LPSTR)WAVEFILTER;
              UpdateMenu(wParam);
              break;

            case IDM_OPEN:
               if (GetOpenFileName(&ofn))                 // Use open dialog
               {
                  if (mciDevice->Open(ofn.lpstrFile))     // Open element
                     mciDevice->ErrorMessageBox();         
               }
               break;

            case IDM_CLOSE:
               if (mciDevice->Close())
                  mciDevice->ErrorMessageBox();
               break;

            case IDM_PLAY:
               if (mciDevice->Play(1))            // play from the beginning
                  mciDevice->ErrorMessageBox();
               break;

            case IDM_STOP:
               if (mciDevice->Stop())
                  mciDevice->ErrorMessageBox();
               break;

            case IDM_PAUSE:
               if (mciDevice->Pause())
                  mciDevice->ErrorMessageBox();
               break;

            case IDM_RESUME:
               if (mciDevice->Resume())
                  mciDevice->ErrorMessageBox();
               break;

            case IDM_EXIT:
               SendMessage( hWnd, WM_CLOSE, 0, 0L);
               return 0;

            case IDM_NOTIFY:
               if (mciDevice->GetParent())
                  mciDevice->SetParent(NULL);         
               else
                  mciDevice->SetParent(hWnd);         
               break;

            case IDM_ABOUT:
               {
                  FARPROC lpfnAbout ;
                  lpfnAbout = MakeProcInstance( (FARPROC)AboutProc, hAppInst );
         		   DialogBox( hAppInst, "ABOUTBOX", hWnd, (DLGPROC)lpfnAbout );
                  FreeProcInstance( lpfnAbout );
               }
         }
         UpdateMenu(NULL);
         break;

      case MM_MCINOTIFY:
         wsprintf( buf, "Notified by device: %lu", lParam);
         MessageBox( NULL, buf, "MM_MCINOTIFY", MB_OK);
         break;

	   case WM_DESTROY:
	      PostQuitMessage(0);
	      break;

	   default:		
	      return DefWindowProc( hWnd, wMsg, wParam, lParam );
   }
   return 0L ;
}

//-----------------------------------------------------------------------------
// AboutProc
//-----------------------------------------------------------------------------
BOOL FAR PASCAL _export
AboutProc( HWND hDlg, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
   switch (wMsg) {
      case WM_COMMAND:
         switch (wParam) {
            case IDOK:
            case IDCANCEL:
   	   	   EndDialog( hDlg, TRUE );
   	   	   return TRUE;
	      }
   }
   return FALSE;
}

//-----------------------------------------------------------------------------
// UpdateCaption
//-----------------------------------------------------------------------------
void
UpdateCaption()
{
   static char chTitle[BUFLEN];           // main window title 

   if (mciDevice->IsOpen())
      wsprintf( chTitle, "%s: %s", (LPSTR)szAppName, 
                  (LPSTR)(mciDevice->InfoFile()));
   else
      wsprintf( chTitle, "%s: [Current Device Closed]", (LPSTR)szAppName);

   SetWindowText( hWndApp, chTitle);
}

//-----------------------------------------------------------------------------
// UpdateMenu
//-----------------------------------------------------------------------------
void
UpdateMenu(UINT uNewID)
{
   UpdateCaption();

   if (uNewID) {
      CheckMenuItem( hMenu, uCurDevice, MF_BYCOMMAND | MF_UNCHECKED);
      uCurDevice = uNewID;
      CheckMenuItem( hMenu, uNewID, MF_BYCOMMAND | MF_CHECKED);
   }

   CheckMenuItem( hMenu, IDM_NOTIFY, MF_BYCOMMAND | 
         mciDevice->GetParent() ? MF_CHECKED : MF_UNCHECKED);

   BOOL bOpen = mciDevice->IsOpen();
   EnableMenuItem( hMenu, IDM_PLAY, bOpen ? MF_ENABLED : MF_GRAYED);
   EnableMenuItem( hMenu, IDM_STOP, bOpen ? MF_ENABLED : MF_GRAYED);
   EnableMenuItem( hMenu, IDM_PAUSE, bOpen ? MF_ENABLED : MF_GRAYED);
   EnableMenuItem( hMenu, IDM_RESUME, bOpen ? MF_ENABLED : MF_GRAYED);
   EnableMenuItem( hMenu, IDM_NOTIFY, bOpen ? MF_ENABLED : MF_GRAYED);
   EnableMenuItem( hMenu, IDM_OPEN, bOpen ? MF_GRAYED : MF_ENABLED);
   EnableMenuItem( hMenu, IDM_CLOSE, bOpen ? MF_ENABLED : MF_GRAYED);
}

