/*----------------------------------------------------------------------------*\
 *     Mci.cpp:  C++ class hierarchy for Windows multimedia objects using MCI.
 *  
 *     $Author:   John Musser  $
 *       $Date:   24 Feb 1993 19:37:02  $
 *  $Copyright:   Personal Media International Inc., 1993 $
 *
 * Description:  A set of classes to encapsulate the command-message interface
 *               to MCI. Currently implemented: CDAudio, Waveform audio, AVI.
 *               Two base classes MCIDevice and CompoundDevice provide most of
 *               the basic functionality needed for derived MCI types.
 *
\*----------------------------------------------------------------------------*/

#include "MCI.h"
#include <memory.h>
#include <string.h>

#define   MCI_BUFSIZE  128           // used for char array sizing


/*----------------------------------------------------------------------------*\
 * MCIDevice functions
\*----------------------------------------------------------------------------*/

MCIDevice::MCIDevice(LPSTR lpszDevice)
{
   wDeviceID      = NULL;
   hWndParent     = NULL;
   dwErrState     = NULL;
   lpszDeviceType = NULL;

   if (lpszDevice)
      SetDeviceType(lpszDevice);
}

MCIDevice::~MCIDevice()
{
   if (wDeviceID)
      Close();
   if (lpszDeviceType)
      delete [] lpszDeviceType;
}

LPSTR
MCIDevice::Info(DWORD dwFlags)
{
   MCI_INFO_PARMS mciInfoParms;
   static char cBuf[MCI_BUFSIZE];

   if (!wDeviceID)
      return (LPSTR)NULL;

   mciInfoParms.lpstrReturn = cBuf;
   mciInfoParms.dwRetSize   = MCI_BUFSIZE;
   dwErrState = mciSendCommand(wDeviceID, MCI_INFO, dwFlags,
                       (DWORD)(LPMCI_INFO_PARMS) &mciInfoParms);

   return mciInfoParms.lpstrReturn;
}

DWORD
MCIDevice::Set(DWORD dwFlags, DWORD dwExtra)
{
   MCI_SET_PARMS mciSetParms;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   if (dwFlags & MCI_SET_TIME_FORMAT)
      mciSetParms.dwTimeFormat = dwExtra;             // dwExtra is format

   dwErrState = mciSendCommand(wDeviceID, MCI_SET, dwFlags,
                       (DWORD)(LPMCI_SET_PARMS) &mciSetParms);

   return dwErrState;
}

DWORD
MCIDevice::Status(DWORD dwFlags, DWORD dwItem, DWORD dwExtra)
{
   MCI_STATUS_PARMS mciStatusParms;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   mciStatusParms.dwItem = dwItem;

   // Determine which extra struct fields to set based on flags.
   //
   if (dwFlags & MCI_TRACK)
      mciStatusParms.dwTrack = dwExtra; 

   dwErrState = mciSendCommand(wDeviceID, MCI_STATUS, dwFlags,
                       (DWORD)(LPMCI_STATUS_PARMS) &mciStatusParms);

   return mciStatusParms.dwReturn;
}

DWORD
MCIDevice::GetDevCaps(DWORD dwItem)
{
   MCI_GETDEVCAPS_PARMS mciGetDevCapsParms;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   mciGetDevCapsParms.dwItem = dwItem;
   dwErrState = mciSendCommand(wDeviceID, MCI_GETDEVCAPS, MCI_GETDEVCAPS_ITEM,
                       (DWORD)(LPMCI_GETDEVCAPS_PARMS) &mciGetDevCapsParms);

   return mciGetDevCapsParms.dwReturn;
}

void
MCIDevice::SetDeviceType(LPSTR lpszDevice)
{
   if (lpszDeviceType)
      delete [] lpszDeviceType;

   lpszDeviceType = new __far char[lstrlen(lpszDevice)+1];
   lstrcpy( lpszDeviceType, lpszDevice);
}

void
MCIDevice::ErrorMessageBox()
{
   char szErrorBuf[MAXERRORLENGTH];

   if (mciGetErrorString(dwErrState, (LPSTR)szErrorBuf, MAXERRORLENGTH))
      MessageBox(hWndParent, szErrorBuf, "MCI Error", MB_ICONEXCLAMATION);
   else
      MessageBox(hWndParent, "Unknown MCI Error", "MCI Error",
                 MB_ICONEXCLAMATION);
}

DWORD
MCIDevice::Open(LPSTR lpszDevice /* = NULL */)
{
   MCI_OPEN_PARMS  mciOpenParms;

   if (wDeviceID)                        // Already open don't do it again
      return MCI_WARN_ALREADY_OPEN;

   if (lpszDevice)                       // Type given, save it in private data
      SetDeviceType(lpszDevice);
   else
      if (!lpszDeviceType)               // Make sure we have a type to open,
         return MCI_ERR_NO_DEVICENAME;   // had to given here or in ctor.

   mciOpenParms.lpstrDeviceType = lpszDeviceType;

   dwErrState = mciSendCommand(NULL, MCI_OPEN, MCI_OPEN_TYPE,
                       (DWORD)(LPMCI_OPEN_PARMS) &mciOpenParms);

   if (!dwErrState)
      wDeviceID = mciOpenParms.wDeviceID;

   return dwErrState;
}

DWORD
MCIDevice::Close()
{
   MCI_GENERIC_PARMS mciGenericParms;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   if (!(dwErrState = mciSendCommand(wDeviceID, MCI_CLOSE, MCI_WAIT,
                             (DWORD)(LPMCI_GENERIC_PARMS) &mciGenericParms)))
      wDeviceID = NULL;

   return dwErrState;
}

DWORD
MCIDevice::Stop()
{
   MCI_GENERIC_PARMS mciGenericParms;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   return dwErrState = mciSendCommand(wDeviceID, MCI_STOP, MCI_WAIT, 
                              (DWORD)(LPMCI_GENERIC_PARMS) &mciGenericParms);
}

DWORD
MCIDevice::Pause()
{
   MCI_GENERIC_PARMS mciGenericParms;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   return dwErrState = mciSendCommand(wDeviceID, MCI_PAUSE, MCI_WAIT,
                              (DWORD)(LPMCI_GENERIC_PARMS) &mciGenericParms);
}

DWORD
MCIDevice::Resume()
{
   MCI_GENERIC_PARMS mciGenericParms;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   return dwErrState = mciSendCommand(wDeviceID, MCI_RESUME, MCI_WAIT,
                              (DWORD)(LPMCI_GENERIC_PARMS) &mciGenericParms);
}

DWORD
MCIDevice::Play(LONG lFrom, LONG lTo)
{
   MCI_PLAY_PARMS mciPlayParms;
   DWORD dwFlags = 0L;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   if (hWndParent) {
      mciPlayParms.dwCallback = (DWORD)(LPVOID) hWndParent;
      dwFlags |= MCI_NOTIFY;
   }
   if (lFrom != current) {
      mciPlayParms.dwFrom = lFrom;
      dwFlags |= MCI_FROM;
   }
   if (lTo != end ) {
      mciPlayParms.dwTo = lTo;
      dwFlags |= MCI_TO;
   }

   dwErrState = mciSendCommand(wDeviceID, MCI_PLAY, dwFlags,
                               (DWORD)(LPMCI_PLAY_PARMS) &mciPlayParms);

   return(dwErrState);
}

DWORD
MCIDevice::Seek(LONG lTo)
{
   MCI_SEEK_PARMS mciSeekParms;
   DWORD dwFlags = 0L;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   switch(lTo) {
      case start:
         dwFlags = MCI_SEEK_TO_START;
         break;
      case end:
         dwFlags = MCI_SEEK_TO_END;
         break;
      default:
         mciSeekParms.dwTo = (DWORD)lTo;
         dwFlags = MCI_TO;
   }

   dwErrState = mciSendCommand(wDeviceID, MCI_SEEK, dwFlags,
                         (LONG)(LPMCI_SEEK_PARMS) &mciSeekParms);

   return dwErrState;
}

DWORD
MCIDevice::StopAll()
{
   return mciSendCommand(MCI_ALL_DEVICE_ID, MCI_STOP, 0, NULL);
}

/*----------------------------------------------------------------------------*\
 * CompoundDevice functions
\*----------------------------------------------------------------------------*/

CompoundDevice::CompoundDevice(LPSTR lpszFile, LPSTR lpszDevice)
   : MCIDevice(lpszDevice)
{
   lpszFileName = NULL;

   if (lpszFile)
      Open(lpszFile);
}

CompoundDevice::~CompoundDevice()
{
   if (wDeviceID)
      Close();
   if (lpszFileName)
      delete [] lpszFileName;
}

DWORD
CompoundDevice::Open(LPSTR lpszFile)
{
   MCI_OPEN_PARMS mciOpenParms;
   DWORD dwFlags = 0L;

   if (wDeviceID)                   // If we're already open don't do it again
      return MCI_WARN_ALREADY_OPEN;

   lpszFileName = new __far char[lstrlen(lpszFile)+1];
   lstrcpy( lpszFileName, lpszFile);

   mciOpenParms.lpstrElementName = lpszFileName;
   mciOpenParms.lpstrDeviceType = lpszDeviceType;
   dwFlags = MCI_OPEN_ELEMENT | MCI_OPEN_TYPE;

   dwErrState = mciSendCommand(NULL, MCI_OPEN, dwFlags,
                         (DWORD)(LPMCI_OPEN_PARMS) &mciOpenParms);

   if (!dwErrState)
      wDeviceID = mciOpenParms.wDeviceID;

   return dwErrState;
}

DWORD
CompoundDevice::Close()
{ 
   MCIDevice::Close();           

   if (lpszFileName) {
      delete [] lpszFileName;
      lpszFileName = NULL;
   }

   return dwErrState;
}

/*----------------------------------------------------------------------------*\
 * AVI functions
\*----------------------------------------------------------------------------*/

DWORD
AVI::Update(HDC hdc)
{
   MCI_DGV_UPDATE_PARMS mciUpdateParms;
   DWORD dwFlags;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   mciUpdateParms.hDC = hdc;
   dwFlags = MCI_DGV_UPDATE_HDC | MCI_DGV_UPDATE_PAINT;

   dwErrState = mciSendCommand(wDeviceID, MCI_UPDATE, dwFlags,
                         (LONG)(LPMCI_DGV_UPDATE_PARMS) &mciUpdateParms);

   return dwErrState;
}

DWORD
AVI::PutDestination(RECT &rect)
{
   MCI_DGV_PUT_PARMS mciPutParms;
   DWORD dwFlags;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   mciPutParms.rc = rect;
   dwFlags = MCI_DGV_PUT_DESTINATION | MCI_DGV_RECT;

   dwErrState = mciSendCommand(wDeviceID, MCI_PUT, dwFlags,
                         (LONG)(LPMCI_DGV_PUT_PARMS) &mciPutParms);

   return dwErrState;
}

DWORD
AVI::Signal(DWORD dwPosition)
{
   MCI_DGV_SIGNAL_PARMS mciSignalParms;
   DWORD dwFlags;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   if (!hWndParent)                  // we need a parent to get the message
      return MCI_ERR_NO_PARENT;

   mciSignalParms.dwCallback =(DWORD)(LPVOID)hWndParent; // MCI_SIGNAL recipient
   mciSignalParms.dwPosition =dwPosition;                // when to send
   mciSignalParms.dwUserParm =(DWORD)(LPVOID)this;       // self will be lParam
   dwFlags = MCI_DGV_SIGNAL_AT | MCI_DGV_SIGNAL_USERVAL;

   dwErrState = mciSendCommand(wDeviceID, MCI_SIGNAL, dwFlags,
                         (LONG)(LPMCI_DGV_SIGNAL_PARMS) &mciSignalParms);

   return dwErrState;
}

DWORD
AVI::Configure()
{
   MCI_GENERIC_PARMS mciGenericParms;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   return dwErrState = mciSendCommand(wDeviceID, MCI_CONFIGURE, 0,
                          (DWORD)(LPMCI_GENERIC_PARMS) &mciGenericParms);
}

DWORD
AVI::Cue(DWORD dwTo)
{
   MCI_DGV_CUE_PARMS mciCueParms;
   DWORD dwFlags;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   mciCueParms.dwTo = dwTo;
   dwFlags = MCI_DGV_CUE_OUTPUT | MCI_TO;
   dwErrState = mciSendCommand(wDeviceID, MCI_CUE, dwFlags,
                         (LONG)(LPMCI_DGV_CUE_PARMS) &mciCueParms);

   return dwErrState;
}

DWORD
AVI::Step(DWORD dwFrames, BOOL bReverse)
{
   MCI_DGV_STEP_PARMS mciStepParms;
   DWORD dwFlags;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   mciStepParms.dwFrames = dwFrames;
   dwFlags = MCI_DGV_STEP_FRAMES;

   if (bReverse)
      dwFlags |= MCI_DGV_STEP_REVERSE;

   dwErrState = mciSendCommand(wDeviceID, MCI_STEP, dwFlags,
                         (LONG)(LPMCI_DGV_STEP_PARMS) &mciStepParms);

   return dwErrState;
}

DWORD
AVI::SetAudioVolume(DWORD dwVolume)
{
   MCI_DGV_SETAUDIO_PARMS mciSetAudioParms;
   DWORD dwFlags;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   mciSetAudioParms.dwValue = dwVolume;
   mciSetAudioParms.dwItem  = MCI_DGV_SETAUDIO_VOLUME;
   dwFlags = MCI_DGV_SETAUDIO_ITEM | MCI_DGV_SETAUDIO_VALUE;

   dwErrState = mciSendCommand(wDeviceID, MCI_SETAUDIO, dwFlags,
                         (LONG)(LPMCI_DGV_SETAUDIO_PARMS) &mciSetAudioParms);

   return dwErrState;
}

DWORD
AVI::SetSpeed(DWORD dwSpeed)
{
   MCI_DGV_SET_PARMS mciSetParms;
   DWORD dwFlags;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   mciSetParms.dwSpeed = dwSpeed;
   dwFlags  = MCI_DGV_SET_SPEED;

   dwErrState = mciSendCommand(wDeviceID, MCI_SET, dwFlags,
                         (LONG)(LPMCI_DGV_SET_PARMS) &mciSetParms);

   return dwErrState;
}


DWORD
AVI::Window(HWND hWnd)
{
   MCI_DGV_WINDOW_PARMS  mciWindowParms;
   DWORD dwFlags;

   if (!wDeviceID)
      return MCI_ERR_NOT_OPEN;

   mciWindowParms.hWnd = hWnd;
   dwFlags = MCI_DGV_WINDOW_HWND;
   dwErrState = mciSendCommand(wDeviceID, MCI_WINDOW, dwFlags,
                         (LONG)(LPMCI_DGV_WINDOW_PARMS) &mciWindowParms);

   return dwErrState;
}

