/*
 * MAP.C - Utility to display current redirections and server logins
 *         for Artisoft's LANtastic Network Operating System
 *
 * MAP was written in Turbo C++ Version 1.0 but does not use any of the
 * C++ extensions.  It should compile under any version of Turbo C.  I
 * don't know about Microsoft et al; but it should not be difficult to
 * modifiy if it doesn't.
 *
 * The information required to develop this simple program was obtained
 * from the file TECH.DOC which is available from Aritsoft.
 *
 * MAP will tell you the version of the NOS you are running under, display
 * the redirections which are in effect for your machine and also show you
 * all of the servers that you are logged into.
 *
 * Author:	Ken Cline
 *		    4755 Meadow Wood Lane, Suite 202-N
 *			Chantilly, VA  22021-2808
 *
 * Date:	December 3, 1990
 *
*/

#include <conio.h>
#include <dos.h>

typedef unsigned char	byte;
typedef unsigned int	word;
typedef unsigned long	dword;
typedef word			Date;
typedef word			Time;

//
// Queue entry record structure
//

typedef struct QE
	{
	byte	QE_status;				// Status of queue entry
	dword	QE_size;				// size of spooled file
	byte	QE_type;				// Type of queue entry
	byte	QE_output_control;		// Control of despooled file
	word	QE_copies;				// Number of copies
	dword	QE_sequence;			// Sequence number of queue entry
	char	QE_spooled_file[48];	// Pathname of spooled file
	char	QE_user[16];			// Username of spooled file owner
	char	QE_machine[16];			// Name of machine of origin
	Date	QE_date;				// Date file was spooled
	Time	QE_time;				// Time file was spooled
	char	QE_destination[17];		// Asciiz device name / username of Target
	char	QE_comment[48];			// Comment field
	} QE;

// QE_status values

#define QE_status_free			0	// Entry is available
#define QE_status_update        1   // Entry being updated
#define QE_status_hold          2   // Entry is held
#define QE_status_wait          3   // Entry is waiting to be despooled
#define QE_status_active        4   // Entry is being despooled
#define QE_status_cancel        5   // Entry has been cancelled
#define QE_status_file_error    6   // Spooled file could not be accessed
#define QE_status_spool_error   7   // Destination file could not be accessed
#define QE_status_rush          8   // This is a 'rush' job

// QE_type values

#define QE_type_print			0	// Spooled printer file
#define QE_type_message			1   // Spooled message (mail)
#define QE_type_local_file      2   // Spooled local file
#define QE_type_remote_file     3   // Spooled remote file
#define QE_type_modem           4   // Spooled to remote modem
#define QE_type_batch           5   // Spooled batch processor file

// QE_output_control values

#define QE_OC_keep		 	0x40	// Keep after despooling (don't delete)
#define QE_OC_voice			0x20	// For mail - file contains voice mail

//
// User Account record structure
//

typedef struct UA
	{
	char	UA_name[16];		// Blank padded user name
	byte	UA_reserved[16];	// Reserved
	char	UA_description[32];	// Full user description
	byte	UA_privilege;		// Privilige bits
	byte	UA_concurrent;		// Max concurrent logins
	byte	UA_allowed_times[42];  // 1 bit for each half hour
								   //       seven days a week
								   //		begins on Sunday
								   //		0 means allowed
	word	UA_reserved0[2]; 	// Reserved
	Time	UA_last_login;		// Time last logged in
	Date	UA_account_expire;	// Account expiration date
	Date	UA_passwrd_expire;	// Password expiration date
								//		0 = no expiration date
	byte	UA_passwrd_extend;	// 1-31 number of days to extend extend
								// 		password after change
	byte	UA_reserved1[5];	// Reserved
	} UA;

#define UA_priv_superACL	0x80	// Bypass ACL's
#define UA_priv_superQueue	0x40	// Bypass queue protection
#define UA_priv_peer		0x20	// Treat as local process
#define UA_priv_superMail	0x10	// Bypass mail protection
#define UA_priv_audit		0x08	// Allow creation of audit entry
#define UA_priv_system		0x04	// System manager

#define FALSE	0
#define TRUE	!FALSE

#define IsRedirector	0x08	// Redir is installed

int		NetBIOSExists ()
{
	byte	IllegalNcb[64];
	void	interrupt (*int5C) ();
	word	int5Cseg;
	union	REGS	regs;
	int		retc = FALSE;

	int5C = getvect (0x5C);
	int5Cseg = FP_SEG (int5C);

	if (int5Cseg != 0)
		{
		if (int5Cseg != 0xF000)
			{
			memset (&IllegalNcb, '\0', sizeof (IllegalNcb));
			IllegalNcb [0] = 0x7F;

			regs.x.bx = FP_OFF (IllegalNcb);
			regs.h.al = 0;

			int86 (0x5C, &regs, &regs);

			if (regs.h.al != 0)
				{
				retc = TRUE;
				}
			}
		}
	return (retc);
}

byte InstalledStatus ()
{
	union REGS	regs;

	regs.x.ax = 0xB800;

	int86 (0x2F, &regs, &regs);

	if (regs.h.al == 0)
		{
		regs.h.bl = 0;
		}

	return (regs.h.bl);
}

int	LANOSVersion (int *Major, int *Minor)
{
	union REGS	regs;

	regs.x.ax = 0xB809;

	int86 (0x2F, &regs, &regs);

	*Major = regs.h.ah;
	*Minor = regs.h.al;

	return (regs.h.al);
}

int	GetMachineName (char *MachineName, int *NetBIOSNum)
{
	union 	REGS	regs;
	struct SREGS	sregs;
	int				retc = TRUE;
	char			NetBIOSName[16];

	regs.x.ax = 0x5E00;

	sregs.ds = FP_SEG (NetBIOSName);
	regs.x.dx = FP_OFF (NetBIOSName);

	intdosx (&regs, &regs, &sregs);

	if (regs.h.ch == 0)
		{
		retc = FALSE;
		}
	else
		{
		strncpy (MachineName, NetBIOSName, 16);
		*NetBIOSNum = regs.h.cl;
		}

	return (retc);
}

int		GetAccountInfo (char *ServerName, UA *UA_Info)
{
	union 	REGS	regs;
	struct SREGS 	sregs;
	int				retc = TRUE;

	regs.x.ax = 0x5F87;

	regs.x.di = FP_OFF (ServerName);
	sregs.es  = FP_SEG (ServerName);

	regs.x.si = FP_OFF (UA_Info);
	sregs.ds  = FP_SEG (UA_Info);

	intdosx (&regs, &regs, &sregs);

	if (regs.x.cflag != 0)
		{
		retc = FALSE;
		}

	return (retc);
}

int	GetRedirection (word *RedirIndex, byte *DevType, byte *DevStatus,
					char *LocalName,  char *RemoteName)
{
	union 	REGS	regs;
	struct SREGS 	sregs;
	int				retc = TRUE;

	regs.x.ax = 0x5F02;
	regs.x.bx = *RedirIndex;

	regs.x.si = FP_OFF (LocalName);
	regs.x.di = FP_OFF (RemoteName);

	sregs.ds = FP_SEG (LocalName);
	sregs.es = FP_SEG (RemoteName);

	intdosx (&regs, &regs, &sregs);

	if (regs.x.cflag != 0)
		{
		retc = FALSE;
		*DevType = 0;
		*DevStatus = 1;
		}
	else
		{
		*DevType = regs.h.bl;
		*DevStatus = regs.h.bh & 1;
		}

	return (retc);
}

int		GetLoginEntry (word *ServerIndex, char *ServerName)
{
	union 	REGS	regs;
	struct SREGS 	sregs;
	int				retc = TRUE;

	regs.x.ax = 0x5F80;
	regs.x.bx = *ServerIndex;

	regs.x.di = FP_OFF (ServerName);
	sregs.es  = FP_SEG (ServerName);

	intdosx (&regs, &regs, &sregs);

	if (regs.x.cflag != 0)
		{
		retc = FALSE;
		}

	return (retc);
}

int	GetQueueEntry (word *EntryNum, char *ServerName, QE *QueueEntry)
{
	union 	REGS	regs;
	struct SREGS 	sregs;
	int				retc = TRUE;

	regs.x.ax = 0x5FA0;

	regs.x.di = FP_OFF (ServerName);
	sregs.es  = FP_SEG (ServerName);

	regs.x.si = FP_OFF (QueueEntry);
	sregs.ds  = FP_SEG (QueueEntry);

	regs.x.bx = *EntryNum;

	intdosx (&regs, &regs, &sregs);

	if (regs.x.cflag)
		{
		retc = FALSE;
		}
	else
		{
		*EntryNum = regs.x.bx;
		}

	return (retc);


}

void main ()
{
	byte	NetResources = 0;
	char	MachineName[17];
	int		NetBIOSNum;
	int		Major, Minor;
	int		SpoolMode;
	word	RedirIndex = 0, ServerIndex= 0;
	char	LocalName[17];
	char	ServerName[20];
	char	RemoteName[128];
	byte	DevType, DevStatus;
	word	EntryNum;
	int		MailCount;
	QE		QueueEntry;
	UA		UserAcct;

	ServerName [0] = '\\';
	ServerName [1] = '\\';

	clrscr();
	if (!NetBIOSExists ())
		{
		cputs ("NetBIOS not installed, processing aborted!!");
		exit (1);
		}

	NetResources = InstalledStatus ();
	if (!(NetResources & IsRedirector))
		{
		cputs ("REDIR.EXE not installed, processing aborted!");
		exit (2);
		}

	LANOSVersion (&Major, &Minor);
	cprintf ("\n\r%15sLANtastic Network Operating System - Version %d.%02d",
			" ", Major, Minor);

	memset (MachineName, '\0', sizeof (MachineName));
	GetMachineName (&MachineName[0], &NetBIOSNum);

	cprintf ("\n\n\rMachine Name: %s\n", MachineName);

    cprintf ("\n\r%16s  %s\n", "Local Name", "Remote Device");
	while (GetRedirection (&RedirIndex, &DevType, &DevStatus,
						   &LocalName[0], &RemoteName[0]))
		{
        cprintf ("\n\r%16s  %s", LocalName, RemoteName);
		RedirIndex++;
		}

	cprintf ("\n\n\rLogged in to the following servers:\n");

	while (GetLoginEntry (&ServerIndex, &ServerName[2]))
		{
		EntryNum = 0;
		MailCount = 0;
		GetAccountInfo (ServerName, &UserAcct);
		while (GetQueueEntry (&EntryNum, ServerName, &QueueEntry))
			{
			if (QueueEntry.QE_type == QE_type_message)
				{
				if (stricmp (QueueEntry.QE_destination, UserAcct.UA_name) == 0)
					{
					MailCount++;
					}
				}
			}
		cprintf ("\n\r%3s%-20s  AS %-20s With %3d Mail Messages",
				 " ", ServerName, UserAcct.UA_name, MailCount);
		ServerIndex++;
		}
}
