//---------------------------------------------------------------
//  LINE_IMAGE:  A C++ class for handling 'printf' type output
//               Garry J. Vass  [72307,3311]
//
//  This file contains a class declaration and useful code for directing
//  formatted output into one of the C++ predefined stream
//  objects, or virtually any processing defined.  A test case is
//  included.
//
//  This code will compile without errors or warnings under
//  the Turbo C++ or Zortech compilers.  Note the compiler
//  specific flags discussed below.
//
//
//--------------------------------------------------------------
//  Programmer's notes:
//  To declare an instance of the LINE_IMAGE class called 'x':
//      LINE_IMAGE x;
//  To sprint and print from 'x':
//      x.xprintf( char *fmt, ... );
//      example:  x.xprintf( "The price of %s is %6.2f\n", item, price );
//                x.xprintf( "Hello out there\n" );
//                x.xprintf( "Name: %*s", width, name );
//  To examine or manipulate the asciiz contents of the 
//  member buffer of 'x':
//      strlen( x.contents );
//      strcpy( x.contents, "Hello world\n" );
//  To set whether the output goes to stdout, stderr, stdprn,
//  to stdaux, to a file, or anywhere else:
//      x.set_process( function );
//        where:  'function' is any legally prototyped function
//                taking a character pointer and returning void.
//--------------------------------------------------------------
//  These are the standard run time headers for any C compiler.
#include  <stdio.h>
#include <stdlib.h>
#include <string.h>
//-----------------------------------------------------------------
//
//
//
//
//-----------------------------------------------------------------
//  These are the headers required in a DOS environment.
#include <stdarg.h>
//-----------------------------------------------------------------
//
//
//
//
//-----------------------------------------------------------------
//  Remove this define if you are working with Turbo!
#define ZORTECH
//  These are compiler specific header files.  Note that the manifest
//  constant, __TURBOC__ is automatically defined by TC.  For Zortech,
//  an explicit definition must be made here, e.g., #define ZORTECH.
//  Under TC, I use -ml -P
//  In ZWB, I use -ml -cpp -f -p -v
#ifdef __TURBOC__
#include <iostream.h>
#endif
//-----------------------------------------------------------------
//
//
//
//
//-----------------------------------------------------------------
#ifdef ZORTECH
#include <stream.hpp>
#endif
//-----------------------------------------------------------------
//
//
//
//
//-----------------------------------------------------------------
//  This is a manifest constant used to define how much memory is
//  allocated for each string.
#ifndef MAX_STRING_LENGTH
#define MAX_STRING_LENGTH 255
#endif
//-----------------------------------------------------------------
//
//
//
//
//-----------------------------------------------------------------
//  This is used in constructing an object of the LINE_IMAGE class.
//  See its usage below for more information.
#ifndef DEFAULT_OUT_FUNCTION
#define DEFAULT_OUT_FUNCTION str_cout
#endif
//-----------------------------------------------------------------
//
//
//
//
//-----------------------------------------------------------------
//  This is used in constructing an object of the LINE_IMAGE class.
//  See its usage below for more information.
#ifndef DEFAULT_NO_MEMORY
#define DEFAULT_NO_MEMORY  quit
#endif
//-----------------------------------------------------------------
//
//
//
//
//-----------------------------------------------------------------
//  These are prototypes for functions used in the example.
//  Note that the str_cout function is also the so-called
//  default when an object of the LINE_IMAGE class is
//  constructed.
void str_cout   ( char * );
void str_cerr   ( char * );
void any_output ( char * );
void quit( void );
//-----------------------------------------------------------------
//
//
//
//
//-----------------------------------------------------------------
//  This is the declaration of the LINE_IMAGE class.
class LINE_IMAGE {
private:

    char  *str;  //  dynamically allocated buffer
    int result;  //  for functions that return integers

protected:

    void (*process)( char * ); // a pointer to a function (that you write)
                               // that handles the actual output, or
                               // virtually any sort of post-printf type
                               // processing.  In this file, the 
                               // constructor uses a default function.

    void (*no_memory)();       // a pointer to a function that handles
                               // the condition where 'new' returns
                               // NULL (out of memory).

    //  This protected function is used by the constructors.  It
    //  takes an argument to facilitate access by any derived
    //  classes.
    unsigned setup( unsigned size ) {
    
        //  The two function pointers are assigned to
        //  the default values.  Note that the 'process'
        //  function can be changed by the member function,
        //  'set_process'.
        no_memory = DEFAULT_NO_MEMORY;
        process = DEFAULT_OUT_FUNCTION;
 
        //  Memory is allocated here.  If the request fails,
        //  the member error handler is invoked.
        str = new char[ size ];
        if( !str ) 
            no_memory();
        return( size );
    }

public:

    //  The only constructor for this class.  It simply allocates
    //  the heap space for a string, sets the output function
    //  to the default stream 'cout', and sets an error handler 
    //  function.
    LINE_IMAGE() {
        setup( MAX_STRING_LENGTH );
    }

    //  This function provides a public mechanism for examining
    //  the contents of the string.  It can be used as an argument
    //  to the standard run-time functions, eg., strlen( X.contents() );
    //  and so on.
    char *contents() {
        return( str );
    }

    //  This function resets the output processing to a different
    //  function.  To use this feature, simply prepare a void function
    //  taking a pointer to a character string.  Then invoke this
    //  function.
    void set_process( void (*newfunc)( char *) ) {
        process = newfunc;
    }
    
    //  This function resets the error handler to a different 
    //  function.  
    void set_error( void (*newfunc)() ) {
        no_memory = newfunc;
    }

    //  This is the engine of the LINE_IMAGE class.  It takes a variable
    //  argument list, sprints into the class contents, and then invokes
    //  the output function.
    int xprintf( char *format, ... ) {
        int result;
        va_list arg_ptr;
        va_start( arg_ptr, format );
        result = vsprintf( str, format, arg_ptr );
        va_end( arg_ptr );
        process( str );
        return( result );
    }
};// End of Class LINE_IMAGE
//-----------------------------------------------------------------
//
//
//
//
//-----------------------------------------------------------------
int main() //  This is a test case that exercises the LINE_IMAGE class.
{
double    d = 0.0;
long      l =  0L;

//  Here, an object of the LINE_IMAGE class is declared.
LINE_IMAGE s;

    for( int i = 0; i < 20; i++ ) {
         //  if i is even, output will go to the stderr device.
         //  if i is odd, output will go to the stdout device.
         if( i % 2 )
             s.set_process( str_cout );
         else
             s.set_process( str_cerr );

         //  here, the actual processing takes place...
         s.xprintf( "%d %lf %ld\n", i, d, l );
         ++d;
         ++l;
    }

    //  here, the function is redefined again!
    s.set_process( any_output );
    s.xprintf( "Hello world!!!" );
    return( 0 );
}
//-----------------------------------------------------------------
//
//
//
//
//-----------------------------------------------------------------
//  These are the programmer defined functions for handling output.
//  Note that any function can be used as long as it conforms to the
//  prototype given in the LINE_IMAGE class declaration.
void str_cout( char *str )
{
     cout << "This is your basic stdout  " << str;
}
void str_cerr( char *str )
{
    cerr << "!!!!!This is stderr!!!!!  " << str;
}

void any_output( char *str )
{
    cout << "This is something different\n";
    cout << "this line is " << strlen( str ) << " characters " << str;
}
void quit()
{
    exit( 0 );
}
//--------------------------- end  --------------------------------
//
//
//
//
//-----------------------------------------------------------------
                                                            