//------------------------------------------------------------------------ 
//                                                                         
//	FILENAME	:	LINEAR.C                                                   
//                                                                         
//	PURPOSE		:	Linear regression on real time accumulator information     
//				to smooth calculated flowrates                             
//                                                                         
//	DATE		:	December 25, 1992                                          
//                                                                         
//  	DESIGNER	:	Edward B. Toupin                                           
//				Tory S. Toupin                                             
//                                                                         
//	PROGRAMMER	:	Edward B. Toupin                                           
//                                                                         
//------------------------------------------------------------------------ 
//
//	PURPOSE
//	-------
//		This application has been designed to test Linear Regression techniques
//		on real time flowrate calculations.
//
//		The system first goes through a MAX_ARRAY initialization before actual
//		processing can begin.  The system then takes a few seconds to learn
//		the actions of the point being monitored.  Once learned, the system
//		begins to perform LR on the data in the MAX_ARRAY element array.
//
//		The system takes the number of barrels from a remote device and
//		the current time stamp of the barrel count.
//
//------------------------------------------------------------------------ 


//	Includes	 

#include	<stdio.h>
#include	<stdlib.h>
#include	<math.h>
#include	<string.h>
#include	<time.h>


//	Definitions	 

#define	MAX_ARRAY	20


//	Type declarations	 

struct ARRAY
{
	char		InUse;			//1 = InUse, 0 = Ignore 
	long		dT;			//Time change since last element     
	double		dV;			//Change in volume since last element  
	int			Ref;		//LR Reference                       
	double		LR;                     
	char		RollOver;		//1 = Rollover, 2 = Consistant 
};						//X = Ref, Y = dV/dT                 

//	Variable declarations	 

struct ARRAY	DATA_ARRAY[MAX_ARRAY];		//LR array structure 

char		FirstPass=1;			//If first pass (i.e. first element in array) 
long 		CurrTime;			//Global storage for previous and current  
long		PrevTime;			//	values	 
double  	CurrVol=0;
double		PrevVol=0;

double		PrevFrate=0;			//Calculate flowrate per second calculations  
double		CurrFrate=0;

int		ElementCount=0;

//======================================================================== 
//                                                                         
//	double	Slope()                                                        
//                                                                         
//	Calculate the slope of the global array and return the value           
//                                                                         
//	m = (Count * SumOfXY) - (SumOfX * SumOfY)                              
//	    -------------------------------------                              
//	    (Count * SumOfXX) - (SumOfX * SumOfX)                              
//                                                                         
//======================================================================== 
double Slope()
{
	int		Offset=0;
	int		Count=0;
	double		SumOfX=0;
	double		SumOfY=0;
	double		SumOfXX=0;
	double		SumOfXY=0;
	double		SlopeCalc=0;

  	if(FirstPass) return(0);

	for(Offset=0;Offset<MAX_ARRAY;Offset++)
	{
		if(DATA_ARRAY[Offset].InUse)
		{
			Count++;
			SumOfX 		+= 	(double)DATA_ARRAY[Offset].Ref;
			SumOfY 		+= 	(DATA_ARRAY[Offset].dV/(double)DATA_ARRAY[Offset].dT);
			SumOfXX		+=	(double)DATA_ARRAY[Offset].Ref * (double)DATA_ARRAY[Offset].Ref;
			SumOfXY		+=	(double)DATA_ARRAY[Offset].Ref *
							(DATA_ARRAY[Offset].dV/(double)DATA_ARRAY[Offset].dT);
		}
	}

	SlopeCalc = ((Count * SumOfXY) - (SumOfX * SumOfY)) / ((Count * SumOfXX) - (SumOfX * SumOfX));

	return(SlopeCalc);
}


//======================================================================== 
//                                                                         
//	double	Intercept()                                                    
//                                                                         
//	Calculate the intercept of the global array and return the value       
//                                                                         
//	b = YAvg - m * XAvg                                                    
//                                                                         
//======================================================================== 
double Intercept(double m)
{
	int		Offset=0;
	int		Count=0;
	double		SumOfX=0;
	double		SumOfY=0;
	double		InterceptCalc=0;
	double		XAvg=0;
	double		YAvg=0;

	if(FirstPass) return(0);

	for(Offset=0;Offset<MAX_ARRAY;Offset++)
	{
		if(DATA_ARRAY[Offset].InUse)
		{
			Count++;
			SumOfX 		+= 	(double)DATA_ARRAY[Offset].Ref;
			SumOfY 		+= 	(DATA_ARRAY[Offset].dV/(double)DATA_ARRAY[Offset].dT);
		}
	}

	XAvg = SumOfX / (double)Count;
	YAvg = SumOfY / (double)Count;

	InterceptCalc = YAvg - (m * XAvg);

	return(InterceptCalc);
}


//======================================================================== 
//                                                                         
//	int ConvertDateTime(char* Date, char* Time)                            
//                                                                         
//	Convert text date and time to integer values                           
//                                                                         
//======================================================================== 
long ConvertDateTime(char* Date, char* Time)
{
	char    ltime[20];
  	char	ldate[20];
	int     prev_date = -1;
	short	hour,min,sec;
	short	day,year,month;

 
	//Convert the text time to an integer time stamp 
  	hour	=(short)(strtol(&Time[0],(char**)0,10));
	min		=(short)(strtol(&Time[3],(char**)0,10));
  	sec		=(short)(strtol(&Time[6],(char**)0,10));
 
  	//Convert the text date to a numeric date stamp 
  	day =(short)(strtol(&Date[0],(char**)0,10));
	year=(short)(strtol(&Date[7],(char**)0,10));
 
  	//Compare months for appropriate month number 
  	if(strstr(&Date[3],"JAN"))	month=1;
	if(strstr(&Date[3],"FEB"))	month=2;
	if(strstr(&Date[3],"MAR"))	month=3;
	if(strstr(&Date[3],"APR"))	month=4;
	if(strstr(&Date[3],"MAY"))	month=5;
	if(strstr(&Date[3],"JUN"))	month=6;
	if(strstr(&Date[3],"JUL"))	month=7;
	if(strstr(&Date[3],"AUG"))	month=8;
	if(strstr(&Date[3],"SEP"))	month=9;
	if(strstr(&Date[3],"OCT"))	month=10;
	if(strstr(&Date[3],"NOV"))	month=11;
	if(strstr(&Date[3],"DEC"))	month=12;

	return((long)hour*3600 + min*60 + sec);
}


//======================================================================== 
//                                                                         
//	int AddElement(char* Date, char* Time, double Vol)                     
//                                                                         
//	Convert text date and time to integer values                           
//                                                                         
//======================================================================== 
int AddElement(char* Date, char* Time, char* Vol)
{
	int		Offset=0;

	//Roll the current and previous global values 
	PrevTime = CurrTime;
	PrevVol  = CurrVol;
	CurrVol  = atof(Vol);
	CurrTime = ConvertDateTime(Date,Time);

	if(PrevVol==0)
	{
		PrevTime = CurrTime;
		PrevVol = CurrVol;
	}

  	//Roll each element down in the global LR array 
	for(Offset=0;Offset<MAX_ARRAY-1;Offset++)
	{
		DATA_ARRAY[Offset].InUse = DATA_ARRAY[Offset+1].InUse;
		DATA_ARRAY[Offset].dV = DATA_ARRAY[Offset+1].dV;
		DATA_ARRAY[Offset].dT = DATA_ARRAY[Offset+1].dT;
	}

	DATA_ARRAY[MAX_ARRAY-1].InUse = 1;
	DATA_ARRAY[MAX_ARRAY-1].dT = (CurrTime - PrevTime);
	DATA_ARRAY[MAX_ARRAY-1].dV = (CurrVol - PrevVol);

	if(ElementCount<MAX_ARRAY)
 	{
		ElementCount++;
	}
 	else
	{
		FirstPass=0;
	}

	return(1);
}

//======================================================================== 
//                                                                         
//	main()                                                                 
//                                                                         
//======================================================================== 
void main()
{
	int			Offset=0,Offset2=0;
	double		SlopeCalc=0;
	double		InterceptCalc=0;
  	double		TestVal1=0,TestVal2=0;
	char		TBuff[12], DBuff[12], VBuff[40];
	double		LRResult=0;
	double		Diff1=0, Diff2=0;
	int		DiffOffset=0;
	FILE		*RTData= fopen("d:\\compiler\\msvcnt\\work\\math\\frate\\rtdata.dat","r");
	FILE		*RTResult= fopen("d:\\compiler\\msvcnt\\work\\math\\frate\\rtdata.out","w+");


	//Initialize the global LR array 
	for(Offset=0;Offset<MAX_ARRAY;Offset++)
	{
		DATA_ARRAY[Offset].Ref = Offset * 10;
		DATA_ARRAY[Offset].InUse = 0;
		DATA_ARRAY[Offset].dV = 0;
		DATA_ARRAY[Offset].dT = 0;
		DATA_ARRAY[Offset].RollOver = 0;
	}

	//Read the data 
	while(!feof(RTData))
	{
		fscanf(RTData,"%s %s     %s",DBuff,TBuff,VBuff);
		if(!feof(RTData))
		{
			AddElement(DBuff,TBuff,VBuff);
		}

		//Calculate the slope and intercept of the data 
		SlopeCalc=Slope();
		InterceptCalc=Intercept(SlopeCalc);

		for(Offset=0;Offset<MAX_ARRAY;Offset++)
		{
			DATA_ARRAY[Offset].LR = SlopeCalc*DATA_ARRAY[Offset].Ref + InterceptCalc;
		}

		//Locate the closest calculated LR to the current flowrate 
		if(CurrFrate==0)
		{
			PrevFrate=DATA_ARRAY[MAX_ARRAY-1].LR;
			CurrFrate=DATA_ARRAY[MAX_ARRAY-1].LR;
		}
		else
		{
			Diff1=0;
			Diff2=0;
			DiffOffset=0;
			for(Offset=0;Offset<MAX_ARRAY;Offset++)
			{
				Diff1 = (double)(DATA_ARRAY[Offset].LR - CurrFrate);
				if(Diff2==0) Diff2=Diff1;
				if(fabs(Diff1)<=fabs(Diff2))
				{
					Diff2=Diff1;
					DiffOffset=Offset;
				}
			}
		}

		PrevFrate = CurrFrate;
		CurrFrate = DATA_ARRAY[DiffOffset].LR;

		//Print LR result of first and last elements 
		printf("LR: %f     %f          ",DATA_ARRAY[DiffOffset].LR,DATA_ARRAY[DiffOffset].LR*3600);
		fprintf(RTResult,"LR: %f     %f          ",DATA_ARRAY[DiffOffset].LR,DATA_ARRAY[DiffOffset].LR*3600);

		TestVal1=0;
		if(!FirstPass)
		{
			for(Offset2=0;Offset2<MAX_ARRAY;Offset2++)
			{
				TestVal1+= DATA_ARRAY[Offset2].dV/(double)DATA_ARRAY[Offset2].dT;
			}
		}

		TestVal2 = TestVal1 / MAX_ARRAY;
		printf("MA: %f     %f\n",TestVal2,TestVal2*3600);
		fprintf(RTResult,"MA: %f     %f\n",TestVal2,TestVal2*3600);
	}

	fclose(RTData);
  	fclose(RTResult);
}
