// list.cpp -- List class

#include <stddef.h>
#include "list.h"

/* -- List constructor. Initializes an empty list when the list
object comes into being (i.e. is allocated storage). Note: because
a list is a descendant of an item, the item class constructor also
runs before the list constructor. */

list::list()
{
   anchor = cip = NULL;    // No listed or current items
}

/* -- List destructor. Like a snake eating itself by the tail, the
destructor disposes of all items (if any) on the list, and then
disposes of itself. */

list::~list()
{
   if (anchor != NULL)
      disposeList();
}

/* -- Insert a new item addressed by ip into a list object. The new
item is linked in front of (to the left of) the current item. To
link an item after another, find that item and call nextItem before
calling insertItem. Does nothing if argument is NULL. If list is
empty, then a new list is created with the single item at ip. Returns
address of inserted item or NULL.*/

item *list::insertItem(item *ip)
{
   if (ip == NULL)               // Ignore request to insert
      return NULL;               //  a NULL item.
   if (anchor == NULL)           // If list is empty...
      return anchor = cip = ip;  //  start a new list
   return ip->link(cip);         // Else, link item into list
}

/* -- Remove the item addressed by ip from the list object. Also
adjust the anchor and cip pointers to make sure they do not address
the unlinked item. If the addressed item is the only one in the list,
then this function empties the list. Does NOT dispose the unlinked
item or call its destructor. After calling removeItem, the item
addressed by ip points to itself, and it can be used to begin a new
list, or it can be used as a free-floating object. Returns NULL or
the address of the removed item. */

item *list::removeItem(item *ip)
{
   if (ip == NULL)               // Ignore request to remove
      return NULL;               //  a NULL item.
   if (ip->right == ip)          // If list has only one item...
      anchor = cip = NULL;       //  then empty the list
   else {
      if (ip == anchor)          // Else adjust anchor and
         anchor = anchor->right; //  cip pointers to ensure
      if (cip == ip)             //  they do not address the 
         cip = cip->right;       //  unlinked item.
   }
   return ip->unlink();          // Unlink item from list
}

/* -- Return a pointer to the previous item, the one to the "left" of
the current item. Also sets the current item pointer to that item.
Returns NULL if list is empty. */

item *list::prevItem(void)
{
   if (cip != NULL)        // If list is not empty
      cip = cip->left;     //  set cip to item at left.
   return cip;             // Return current item pointer.
}

/* -- Return a pointer to the next item, the one to the "right" of
the current item. Also sets the current item pointer to that item.
Returns NULL if list is empty. */

item *list::nextItem(void)
{
   if (cip != NULL)        // If list is not empty
      cip = cip->right;    //  set cip to item at right.
   return cip;             // Return current item pointer.
}

/* -- Remove and delete all items (if any) in the list object. If
items have destructors, they are called for each item. Items are not
necessarily disposed in the order they were inserted. */

void list::disposeList(void)
{
   item *ip;

   while (!listEmpty())
      delete removeItem(currentItem());
}


// Copyright (c) 1990 by Tom Swan. All rights reserved
// Revision 1.00    Date: 09/03/1990   Time: 10:45 am


