// selector.cpp -- Selector class

#include "key.h"
#include "selector.h"

#define KEY_ENTER 13    // <Enter> key value
#define KEY_ESC   27    // <Esc> key value
#define KEY_UP    -184  // <Cursor Up> key value
#define KEY_DOWN  -176  // <Cursor Down> key value
#define KEY_PGUP  -183  // <Page Up> key value
#define KEY_PGDN  -175  // <Page Down> key value

/* -- Constructor. Passes arguments to the window class's alternate
constructor, and initializes its own data fields. */

selector::selector(winStruct &ws, const char *title, int popup)
   : window(ws, title)
{
   showHide = popup;       // Select pop-up or stationary style
   selector::row = -1;     // i.e. not initialized
}

/* -- Display current item at window row, col == 0, using the current
attribute. */

void selector::showItem(int row)
{
   gotorc(row, 0);
   window::puts(((strItem *)currentItem())->getString());
   eeol();
}

/* -- Display current item at window row, col == 0 using the
highlight attribute. This procedure displays the "selector bar." */

void selector::highlight(int row)
{
   reverseVideo();
   showItem(row);
}

/* -- Move selector bar up to previous item. Scrolls window contents
down if necessary. */

void selector::moveUp(int &row)
{
   if (!listEmpty() && !atHeadOfList()) {
      normalVideo();
      showItem(row);
      if (row > 0) row--; else scrollDown(1);
      prevItem();
      highlight(row);
   }
}

/* -- Move selector bar down to next item. Scrolls window contents up
if necessary. */

void selector::moveDown(int &row)
{
   if (!listEmpty() && !atEndOfList()) {
      normalVideo();
      showItem(row);
      if (row < ws.height - 3) row++; else scrollUp(1);
      nextItem();
      highlight(row);
   }
}

/* -- Display selector window, list items, and let operator move
selector bar to any item, scrolling up and down as necessary if there
are more items than can fit in the window. Returns a pointer to the
selected item if operator presses <Enter>. Returns NULL if operator
presses <Esc>. */

strItem *selector::getSelection(void)
{
   int key;    // Keypress value
   int i;      // for-loop control variable

   if (listEmpty()) return NULL;
   if (!isOpen) row = -1;  // Reset newly opened windows
   showWindow();           // Make sure window is visible
   normalVideo();          // Select normal text attribute
   if (row < 0) {          // Initialize first time
      row = 0;
      resetList();
      do {
         showItem(row);
         nextItem();
      } while (!atHeadOfList() && (++row <= ws.height - 3));
      row = 0;
      resetList();
   }
   highlight(row);

/* -- Get keypress and move up or down. Return NULL for <Esc>, or
pointer to selected item for <Enter>. */

   for (;;) {
      switch (key = getKey()) {
         case KEY_ESC:
         case KEY_ENTER:
            if (showHide) {
               hideWindow();
               row = -1;
            }
            if (key == KEY_ESC)
               return NULL;
            else
               return (strItem *)currentItem();
         case KEY_UP:
            moveUp(row);
            break;
         case KEY_DOWN:
            moveDown(row);
            break;
         case KEY_PGUP:
            for (i = 0; i < (ws.height - 2) / 2; i++)
               moveUp(row);
            break;
         case KEY_PGDN:
            for (i = 0; i < (ws.height - 2) / 2; i++)
               moveDown(row);
            break;
      }
   }
}


// Copyright (c) 1990 by Tom Swan. All rights reserved
// Revision 1.00    Date: 09/23/1990   Time: 05:39 pm


