// item.cpp -- Item class

#include <stddef.h>
#include "item.h"

/* -- An item object's constructor. Automatically called when space
is allocated for a new item object. */

item::item()
{
   left = right = this;       // Point item to itself.
}

/* -- Item's destructor. Ensures that all items are unlinked from any
others before the item's memory is returned to the heap. */

item::~item()
{
   if (left != this)          // If item is linked to another,
      unlink();               //  unlink it.
}

/* -- Attach one item to another. Assuming A and B are pointers to 
items, to attach a new item A to an existing item B, pass B to A's
link function. B may be attached to another item, or it may be
solitary. The item at A will then be linked to the "left" of B. If A
is already attached to another item, it will be unlinked from that
item before being attached to B. */

item *item::link(item *ip)
{
   if (ip ==  NULL)           // Ignore request to link to
      return NULL;            //  a NULL item.
   if (left != this)          // If item is linked to another
      unlink();               //  unlink it.
   right = ip;                // Adjust item's right and left
   left = ip->left;           //  pointers to link to the item
   ip->left->right = this;    //  addressed by ip.
   ip->left = this;
   return ip;                 // Return address of linked item.
}

/* -- Unattach this item from another. After calling unlink for any
item, you can be sure that the item is not an element of any list.
It's okay to call unlink whether or not the item is currently joined
to another in a list. Return address of unlinked item, allowing
unlink() to be passed to another item* function parameter. */

item *item::unlink(void)
{
   left->right = right;    // Unlink the item by adjusting
   right->left = left;     //  the attached item's pointers.
   left = right = this;    // Point this item to itself.
   return left;            // Return item's address.
}


// Copyright (c) 1990 by Tom Swan. All rights reserved
// Revision 1.00    Date: 09/02/1990   Time: 11:21 am
