// asc.cpp -- Interactive ASCII chart

#include <stdio.h>
#include <disp.h>

#define ESC 27
#define TOPROW  3
#define TOPCOL  7
#define BOTROW 21
#define BOTCOL 73

void showChart(void);
void showASCII(unsigned c, int attrib);
void movecursor(unsigned c);
void showvalues(unsigned c);
int rowasc(unsigned c);
int colasc(unsigned c);

int oldc;      // Previously displayed character

main()
{
   unsigned c = 1;      // Currently highlighted character
   unsigned key;        // Command from keyboard

   disp_open();                        // Initialize display package
   showChart();                        // Display ASCII chart
   movecursor(c);                      // Initialize cursor position
   while ((key = getch()) != ESC) {    // Get user command
      switch (key) {
         case 'H': c -= 32; break;     // Move up
         case 'P': c += 32; break;     // Move down
         case 'K': c--; break;         // Move left
         case 'M': c++; break;         // Move right
      }
      c %= 256;         // Make sure c is in range 0 ... 255
      movecursor(c);    // Highlight and report on character c
   } 
   disp_move(24, 0);    // Move cursor to last line before ending
   disp_close();        // Close the display package
}

void showChart(void)
{
   unsigned c = 0;      // for-loop control variable

// Clear display

   disp_move(0, 0);
   disp_eeop();

// Display box and ASCII characters inside

   disp_box(3, DISP_NORMAL, TOPROW, TOPCOL, BOTROW, BOTCOL);
   for (c = 0; c <= 255; c++)
      showASCII(c, DISP_NORMAL);

// Display other text on screen

   disp_move(TOPROW - 1, TOPCOL);
   disp_printf("Cursor keys move; <Esc> quits");
   disp_move(BOTROW + 1, TOPCOL);
   disp_printf("Learning C++");
   disp_move(BOTROW + 1, BOTCOL - 22);
   disp_printf("ASCII Chart by Tom Swan");
}

// Display one ASCII character using a display attribute
void showASCII(unsigned c, int attrib)
{
   disp_pokew(rowasc(c), colasc(c), (attrib * 256) + c);
}

// Return row number for character c
int rowasc(unsigned c)
{
   return(2 + TOPROW + ((c / 32) * 2));
}

// Return column number for character c
int colasc(unsigned c)
{
   return(2 + TOPCOL + ((c % 32) * 2));
}

// Unhighlight old character and highlight a new one
void movecursor(unsigned c)
{
   showASCII(oldc, DISP_NORMAL);
   showASCII(c, DISP_REVERSEVIDEO);
   oldc = c;
   showvalues(c);
}

// Display character and its ASCII value in decimal and hex
void showvalues(unsigned c)
{
   disp_move(TOPROW - 1, TOPCOL + 35);
   disp_printf("Character =    ASCII = %d  %#x", c, c);
   disp_eeol();
   disp_pokew(TOPROW - 1, TOPCOL + 47, (DISP_NORMAL * 256) + c);
}


// Copyright (c) 1990 by Tom Swan. All rights reserved
// Revision 1.00    Date: 08/23/1990   Time: 08:57 am
