// NETWORK.C -- High-level network routines for LANTEST
// Copyright (c) 1994 by Allen Brunson  version 1.00  06/30/94

#ifdef   __TURBOC__    // If a Borland compiler
#include <alloc.h>     // malloc(), free()
#else
#include <malloc.h>    // malloc(), free()
#endif

#include <conio.h>     // getch(), kbhit()
#include <dos.h>       // delay()
#include <fcntl.h>     // file attribute defines
#include <io.h>        // close(), open()
#include <sys\stat.h>  // file attribute attributes
#include <stdio.h>     // sprintf()
#include <stdlib.h>    // atoi()
#include <string.h>    // strcpy()
#include <time.h>      // needed for randomize()
#include "lantest.h"   // LANTEST-specific defines


/****************************************************************************/
/*                                                                          */
/***  Network data                                                        ***/
/*                                                                          */
/****************************************************************************/

struct LANTESTPKT recvPkt;                         // Receive packet
struct IPXADDRFULL recvAddr;                       // Receive address
char   recvAddrStr[30];                            // Receive address string

struct LANTESTPKT sendPkt;                         // Send packet

struct IPXADDRFULL ipxAddrBroad;                   // Address for broadcasts
char   nameStr[NAMELEN + 1] = "NoName";            // Name string
struct USER user[USERTOTAL];                       // User table

byte   flurryMode = FALSE;                         // Flurry mode flag
unsigned long int flurryPackets = 0;               // Flurry packets received

#ifndef DEBUG                                      // If DEBUG not defined
void   *memPtr = NULL;                             // Memory pointer
#else                                              // If DEBUG defined
struct IPXDATA ipxData;                            // Used for NETLIB data
#endif

#ifdef ROUTE                                       // If router-finding
word   netCnt = 0;                                 // Count of networks
word   netsDisplayed = 0;                          // Needed for routeWait()
#ifndef DEBUG                                      // If DEBUG not defined
struct IPXADDRFULL *netPtr = NULL;                 // Network memory pointer
#else                                              // If DEBUG defined
struct IPXADDRFULL network[IPXNETCNT];             // Network structure
#endif
#endif                                             // End if router-finding


/****************************************************************************/
/*                                                                          */
/***  nameSet()                                                           ***/
/*                                                                          */
/****************************************************************************

This procedure sets the local user's name.                                  */

void nameSet(void)                                 // Begin nameSet()
  {
    word i = 0;                                    // String index
    word len;                                      // String length
    char *name;                                    // Pointer to name

    message(emptyStr);                             // Print blank line

    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past NAME command
    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    name = &cmdStr[i];                             // Set name string

    len = strlen(name);                            // Get string length

    if (!len)                                      // If no name given
      {
        message("You must specify a name "         // Display error
         "after the NAME command.");               //  message
        return;                                    // Return to caller
      }

    if (len > NAMELEN) name[NAMELEN] = 0;          // Truncate if too long

    strcpy(nameStr, &cmdStr[i]);                   // Copy name string
    strcpy(user[0].name, nameStr);                 // Save as first name
    sprintf(str, "Name set to \"%s\"", nameStr);   // Format string
    message(str);                                  // Display it
    message(emptyStr);                             // Print a blank line
  }                                                // End nameSet()


/****************************************************************************/
/*                                                                          */
/***  nStart()                                                            ***/
/*                                                                          */
/****************************************************************************

This procedure sets up the network data and displays opening messages.      */

void nStart(void)                                  // Begin nStart()
  {
    char addrStr[30];                              // Address string
    word rval;                                     // Function return value

    message("Starting up IPX communications ..."); // Display opening message

    #ifdef __TURBOC__                              // If a Borland compiler
    randomize();                                   // Init random numbers
    #endif

    #ifdef DEBUG                                   // If DEBUG is defined

    rval = ipxStart(IPXRECVCNT, IPXSENDCNT,        // Start up LANLIB
     IPXDATASIZE, IPXSOCKET, &ipxData,
     sizeof ipxData);

    #else                                          // If DEBUG isn't defined

    sprintf(str, "Allocating %d bytes of "         // Format message
     "memory ...", IPXMEMSIZE);
    message(str);                                  // Print message

    memPtr = malloc(IPXMEMSIZE);                   // Allocate memory

    if (memPtr == NULL)                            // If alloc failed
      {
        message("Error allocating memory.");       // Print error
        endProgram = TRUE;                         // Set end program flag
        return;                                    // Return
      }

    #ifdef ROUTE                                   // If router-finding
    #ifndef DEBUG                                  // If DEBUG not defined

    netPtr = malloc(sizeof (struct IPXADDRFULL) *  // Allocate network memory
     IPXNETCNT);

    if (netPtr == NULL)                            // If alloc failed
      {
        message("Error allocating net memory.");   // Display error message
        endProgram = TRUE;                         // Set end program flag
        return;                                    // Return
      }

    #endif                                         // End if not DEBUG
    #endif                                         // End if router-finding

    rval = ipxStart(IPXRECVCNT, IPXSENDCNT,        // Start up LANLIB
     IPXDATASIZE, IPXSOCKET, memPtr, IPXMEMSIZE);

    #endif                                         // End if not DEBUG

    if (rval)                                      // If ipxStart() error
      {
        err(rval);                                 // Display error
        endProgram = TRUE;                         // Set end program flag
        return;                                    // Return to caller
      }

    err(ipxAddrLocal(&user[0].addr));              // Get local address

    ipxAddrStr(addrStr, &user[0].addr);            // Convert address to str
    sprintf(str, "Local address is %s", addrStr);  // Set up string
    message(str);                                  // Print message

    user[0].inUse = TRUE;                          // First entry is this PC
    strcpy(user[0].name, nameStr);                 // Set name string

    // When uncommented, the following lines show some fooling
    //  around with ipxByteSwap() and ipxAddrSocket().  I couldn't
    //  think of any other place to demonstrate them ...

//  rval = user[0].addr.socket;                    // Save socket in rval
//  rval = ipxByteSwap(rval);                      // Swap to Intel format

//  ipxAddrSocket(&user[0].addr, 0x7135);          // Somewhere I had to use
//  ipxAddrSocket(&user[0].addr, rval);            //  ipxAddrSocket() ...

    ipxAddrCpy(&ipxAddrBroad, &user[0].addr);      // Set up ipxAddrBroad
    ipxAddrBrd(&ipxAddrBroad);                     //  for broadcasting

    rval = ipxLibVer();                            // Get version number
    sprintf(str, "LANLIB version is %d.%02d",      // Set up string
     rval >> 8, rval & 0x00FF);
    message(str);                                  // Print message
    message(emptyStr);                             // Blank line

    #ifdef ROUTE                                   // If router-finding

    message("Use the NAME command to set your "    // Display name,
     "name,");                                     //  route, and ping
    message("then use ROUTE to locate other "      //  reminders
     "network segments,");
    message("and finally PING to find other "
     "users.");

    #else                                          // If not router-finding

    message("Use the NAME command to set your "    // Display name and
     "name,");                                     //  ping reminders
    message("then use PING to find other users.");
    #endif

    message(emptyStr);                             // Print a blank line
  }                                                // End nStart()


/****************************************************************************/
/*                                                                          */
/***  nStop()                                                             ***/
/*                                                                          */
/****************************************************************************

This procedure shuts down LANLIB.                                           */

void nStop(void)                                   // Begin nStop()
  {
    message(emptyStr);                             // Print blank line
    message("Stopping IPX communications ...");    // Display final message
    err(ipxStop());                                // Stop IPX

    #ifndef DEBUG                                  // If DEBUG not defined
    if (memPtr != NULL) free(memPtr);              // Free NETLIB memory
    #endif                                         // End if not DEBUG

    #ifdef ROUTE                                   // If router-finding
    #ifndef DEBUG                                  // If DEBUG not defined
    if (netPtr != NULL) free(netPtr);              // Free network memory
    #endif                                         // End if not DEBUG
    #endif                                         // End if router-finding
  }                                                // End nStop()


/****************************************************************************/
/*                                                                          */
/***  recvBroadcast()                                                     ***/
/*                                                                          */
/****************************************************************************

This procedure processes a received broadcast packet.                       */

void recvBroadcast(void)                           // Begin recvBroadcast()
  {
    sprintf(str, "Broadcast from %s: %s",          // Format string
     recvPkt.name, recvPkt.msg);
    message(str);                                  // Display string

    strcpy(cmdStr, recvPkt.msg);                   // Interpret this string
    inputFlag = 2;                                 //  as a command
  }                                                // End recvBroadcast()


/****************************************************************************/
/*                                                                          */
/***  recvMessage()                                                       ***/
/*                                                                          */
/****************************************************************************

This procedure processes a received message packet.                         */

void recvMessage(void)                             // Begin recvMessage()
  {
    sprintf(str, "Message from %s: %s",            // Format string
     recvPkt.name, recvPkt.msg);
    message(str);                                  // Display string

    strcpy(cmdStr, recvPkt.msg);                   // Interpret this string
    inputFlag = 2;                                 //  as a command
  }                                                // End recvMessage()


/****************************************************************************/
/*                                                                          */
/***  recvPacket()                                                        ***/
/*                                                                          */
/****************************************************************************

This procedure is the front end for processing incoming packets.            */

void recvPacket(void)                              // Begin recvPacket()
  {
    word pktSize;                                  // Received packet size
    word error;                                    // Error value
    word rval;                                     // Return value

    rval = ipxRecvPkt(&recvPkt, sizeof recvPkt,    // Get waiting
     &recvAddr, &pktSize, &error);                 //  packet
    err(error);                                    // Display error message

    if (!rval || rval == ipxeIPXNOTSTARTED)        // Return if none
      return;

    ipxAddrStr(recvAddrStr, &recvAddr);            // Convert address to str

    if (recvPkt.signature != SIGNATURE)            // If signature wrong
      {
        sprintf(str, "Packet with unknown "        // Format error string
         "signature received from %s",
         recvAddrStr);
        message(str);                              // Display it
        return;                                    // Return
      }

    userSave();                                    // Record this user

    if (recvPkt.type == pFLURRY)                   // If flurry packet
      {
        flurryPackets++;                           // Increment flurry count

        if (!(flurryPackets % 1000))               // If 1000 have gone by
          {
            sprintf(str,                           // Format string
             "Flurry packet total: %lu",
             flurryPackets);

            message(str);                          // Display string
          }

        return;                                    // Return
      }

    sprintf(str, "Received packet from "           // Regular packet message
     "%s, size %d bytes",
      recvAddrStr, pktSize);
    message(str);                                  // Display message

    switch (recvPkt.type)                          // Decision on packet type
      {
        case pBROADCAST:                           // Broadcast packet
          recvBroadcast();                         // Process it
          break;

        case pMESSAGE:                             // Message packet
          recvMessage();                           // Process it
          break;

        case pPING:                                // Ping packet
          recvPing();                              // Process the ping
          break;

        case pPINGRESPONSE:                        // Ping response
          recvPingResponse();                      // Process the response
          break;

        default:                                   // Unknown type
          sprintf(str, "Unknown packet type from " // Format string
           "%s", recvAddrStr);
          message(str);                            // Display message
          break;
      }
  }                                                // End recvPacket()


/****************************************************************************/
/*                                                                          */
/***  recvPing()                                                          ***/
/*                                                                          */
/****************************************************************************

This procedure processes received ping packets.                             */

void recvPing(void)                                // Begin recvPing()
  {
    message("Got ping packet");                    // Display message

    if (!ipxAddrCmp(&user[0].addr, &recvAddr))     // Return if sender is
      return;                                      //  this PC

    // Wait up to one full second so the PC doing
    //  the pinging isn't deluged with replies all
    //  arriving at the exact same instant.

    #ifdef __TURBOC__                              // If a Borland compiler
    delay(random(1000));                           // Wait for up to a second
    #endif

    sprintf(str, "Sending ping response to %s",    // Format string
     recvAddrStr);
    message(str);                                  // Display string

    sendPkt.signature = SIGNATURE;                 // Set up send packet
    sendPkt.type = pPINGRESPONSE;
    sendPkt.ipxRecvMax = ipxRecvMax;
    sendPkt.ipxSendMax = ipxSendMax;
    strcpy(sendPkt.name, nameStr);

    ipxSendPkt(&sendPkt,                           // Send ping response
     sizeof sendPkt - (MSGLEN + 1), &recvAddr);                                   //  to sender
  }                                                // End recvPing()


/****************************************************************************/
/*                                                                          */
/***  recvPingResponse()                                                  ***/
/*                                                                          */
/****************************************************************************

This procedure processes received ping responses.                           */

void recvPingResponse(void)                        // Begin recvPingResponse()
  {
    message("Got ping response");                  // Display message
  }                                                // End recvPingResponse()


/****************************************************************************/
/*                                                                          */
/***  routeDisplay()                                                      ***/
/*                                                                          */
/****************************************************************************

This procedure shows all known networks.                                    */

void routeDisplay(void)                            // Begin routeDisplay()
  {
    static char lf = 0x0A;                         // End of line char

    char addrStr[42];                              // Address string
    char *fileStr;                                 // Filename string
    word i = 0;                                    // Loop counter
    int handle;                                    // File handle

    #ifdef ROUTE                                   // If router-finding
    word lines;                                    // Lines displayed
    #endif

    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past NET command
    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    fileStr = &cmdStr[i];                          // Set filename string

    i = strlen(fileStr);                           // Get string length

    if (!i)                                        // If length is zero
      {
        fileStr = NULL;                            // No file name
      }
    else                                           // If file name given
      {
        while (fileStr[i-1] == ' ')                // Remove any trailing
          {fileStr[i-1] = 0; i--;}                 //  spaces from the name

        strupr(fileStr);                           // String to uppercase
      }

    #ifdef ROUTE                                   // If ROUTE is defined

    if (netCnt == 0)                               // If no networks yet
      {
        message("Router finder not yet called.");  // Display message
        return;                                    // Return
      }

    #endif                                         // End if ROUTE is defined

    if (fileStr)                                   // If using a file
      {
        handle = open(fileStr,                     // Open the file
         O_CREAT | O_TRUNC | O_TEXT | O_RDWR,
         S_IREAD | S_IWRITE);

        if (handle < 0)                            // If open failed
          {
            sprintf(str, "Error opening file %s",  // Format message string
             fileStr);
            message(str);                          // Print the string
            return;                                // Return to caller
          }
      }

    message(emptyStr);
    message("     Network  Node         Sckt Immediate");
    message("     -------  ----         ---- ---------");
    message(emptyStr);

    #ifdef ROUTE                                   // If ROUTE is defined

    lines = 3;                                     // Set lines so far

    for (i = 0; i < netCnt; i++)                   // Loop for all networks
      {
        #ifdef DEBUG                               // If DEBUG defined
        ipxAddrStrLong(addrStr, &network[i]);      // Convert addr to string
        #else
        ipxAddrStrLong(addrStr, &netPtr[i]);       // Convert addr to string
        #endif

        sprintf(str, "%3d: %s", i + 1, addrStr);   // Format message string
        message(str);                              // Display it
        lines++;                                   // Update lines so far

        if (fileStr)                               // If outputting to file
          {
            write(handle, str, strlen(str));       // Write the string
            write(handle, &lf, 1);                 // End the line
          }

        if (lines >= (rows - 4))                   // If screen is full
          {
            message("Press any key to continue ...");

            while (!kbhit()); getch(); lines = 0;  // Wait for a keypress
          }
      }

    #else                                          // If ROUTE isn't defined

    ipxAddrStrLong(addrStr, &ipxAddrBroad);        // Convert address to str
    sprintf(str, "  1: %s", addrStr);              // Format message string
    message(str);                                  // Display it

    if (fileStr)                                   // If outputting to file
      {
        write(handle, str, strlen(str));           // Write the string
        write(handle, &lf, 1);                     // End the line
      }

    #endif                                         // End if ROUTE

    if (fileStr)                                   // If writing to file
      {
        close(handle);                             // Close file

        message(emptyStr);                         // Print a blank line
        sprintf(str, "Network data written to %s", // Format message string
         fileStr);
        message(str);                              // Display the string
      }

    message(emptyStr);                             // Print a blank line
  }                                                // End routeDisplay()


/****************************************************************************/
/*                                                                          */
/***  routeFind()                                                         ***/
/*                                                                          */
/****************************************************************************

This procedure calls ipxRouteFind() to locate network segments.             */

#ifdef ROUTE                                       // If router-finding
void routeFind(void)                               // Begin routeFind()
  {
    word rval;                                     // Error return value

    netsDisplayed = 0;                             // Reset displayed count

    message(emptyStr);                             // Print blank line
    message("Finding networks (Esc to abort) ...");// Display message

    #ifdef DEBUG                                   // If DEBUG is defined
    rval = ipxRouteFind(&network[0], IPXNETCNT,    // Find the networks
     routeWait, &netCnt);
    #else                                          // If DEBUG isn't defined
    rval = ipxRouteFind(netPtr, IPXNETCNT,         // Find the networks
     routeWait, &netCnt);
    #endif

    if (rval)                                      // If an error
      {
        message(emptyStr);                         // Go to next line
        err(rval);                                 // Display the error
      }

    sprintf(str, "Total networks found: %u",
     netCnt);
    message(str);                                  // Display nets found
    message(emptyStr);                             // Print blank line
  }                                                // End routeFind()
#endif                                             // End if ROUTE


/****************************************************************************/
/*                                                                          */
/***  routeWait()                                                         ***/
/*                                                                          */
/****************************************************************************

This procedure is called repeatedly by ipxRouteFind() while it's doing its
lengthy router checks.                                                     */

#ifdef ROUTE                                       // If router-finding
word routeWait(word netsFound)                     // Begin routeWait()
  {
    word key = 0;                                  // Keyboard input

    if (netsFound != netsDisplayed)                // If different than
      {                                            //  most recent check
        netsDisplayed = netsFound;                 // Update netsDisplayed

        sprintf(str, "Networks so far: %u",        // Format message string
         netsFound);
        message(str);                              // Display message
        messageBack();                             // Move back to same line
      }

    if (kbhit())                                   // If a key pressed
      {
        key = getch();                             // Get the key
        if (!key) key = 0x7500 + getch();          // Get extended key
      }

    if (key == 0x001B)                             // If key was Esc
      return TRUE;                                 // Abort route finding
    else                                           // If not Esc
      return FALSE;                                // Continue route finding
  }                                                // End routeWait()
#endif                                             // End if ROUTE


/****************************************************************************/
/*                                                                          */
/***  sendBroadcast()                                                     ***/
/*                                                                          */
/****************************************************************************

This procedure broadcasts a message to all users.                           */

void sendBroadcast(void)                           // Begin sendBroadcast()
  {
    word i = 0;                                    // String index
    word len;                                      // String length
    char *sendStr;                                 // String to send

    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past B command
    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    sendStr = &cmdStr[i];                          // Set send string

    len = strlen(sendStr);                         // Get string length

    if (!len)                                      // If no string
      {
        message("You must enter a message to "     // Display error
         "broadcast after the B command.");        //  message
        return;                                    // Return to caller
      }

    if (len > MSGLEN) sendStr[MSGLEN] = 0;         // Truncate if too long

    sendPkt.signature = SIGNATURE;                 // Set up packet for
    sendPkt.type = pBROADCAST;                     //  send
    sendPkt.ipxRecvMax = ipxRecvMax;
    sendPkt.ipxSendMax = ipxSendMax;
    strcpy(sendPkt.name, nameStr);
    strcpy(sendPkt.msg, sendStr);

    for (i = 0; i < USERTOTAL; i++)                // Loop for all users
      {
        if (!user[i].inUse) continue;              // Don't do empty slots

        err(ipxSendPkt(&sendPkt, sizeof sendPkt,   // Send the packet
         &user[i].addr));                          //  to this user
      }
  }                                                // End sendBroadcast()


/****************************************************************************/
/*                                                                          */
/***  sendErr()                                                           ***/
/*                                                                          */
/****************************************************************************

This procedure checks for send errors.  If one is found, an error message
is printed.                                                                 */

void sendErr(void)                                 // Begin sendErr()
  {
    word rval;                                     // Return value
    word pktSize;                                  // Size of packet
    word error;                                    // Error value
    char sendAddrStr[30];                          // Send address str
    struct IPXADDRFULL sendAddrErr;                // IPX address

    user[0].ipxRecvMax = ipxRecvMax;               // Update this PC's
    user[0].ipxSendMax = ipxSendMax;               //  statistics

    rval = ipxSendChk();                           // Check for send errors
    if (!rval || rval == ipxeIPXNOTSTARTED)        // Return if none
      return;

    rval = ipxSendErr(&sendPkt, sizeof sendPkt,    // Get the undelivered
     &sendAddrErr, &pktSize, &error);              //  packet

    ipxAddrStr(sendAddrStr, &sendAddrErr);         // Convert addr to string
    sprintf(str, "Error sending packet to %s",     // Format string
     sendAddrStr);
    message(str);                                  // Display string
    err(error);                                    // Display error
  }                                                // End sendErr()


/****************************************************************************/
/*                                                                          */
/***  sendFlurry()                                                        ***/
/*                                                                          */
/****************************************************************************

This procedure will send "flurry" packets to all known users if flurry mode
is on.  Note: You should not duplicate this kind of thing in your own
program, except for testing purposes.                                       */

void sendFlurry(void)                              // Begin sendFlurry()
  {
    static word i = 0;                             // Loop counter

    if (flurryMode == FALSE) return;               // Return if no flurries

    sendPkt.signature = SIGNATURE;                 // Set signature
    sendPkt.type = pFLURRY;                        // Set packet type
    sendPkt.ipxRecvMax = ipxRecvMax;               // Set max recv ECB count
    sendPkt.ipxSendMax = ipxSendMax;               // Set max send ECB count
    strcpy(sendPkt.name, nameStr);                 // Copy user name

    i++; if (i >= USERTOTAL) i = 0;                // Increment i

    if (user[i].inUse == FALSE) return;            // Don't do empty slots

    err(ipxSendPkt(&sendPkt,                       // Send packet to this
     sizeof sendPkt - (MSGLEN + 1),                //  user
     &user[i].addr));
  }                                                // End sendFlurry()


/****************************************************************************/
/*                                                                          */
/***  sendFlurryMode()                                                    ***/
/*                                                                          */
/****************************************************************************

This procedure turns flurry mode on or off or resets the flurry packet
count.                                                                      */

char badFlurryStr[] = "You must enter ON, OFF, "   // Error string
 "or RESET after the FLURRY command.";

void sendFlurryMode(void)                          // Begin sendFlurryMode()
  {
    word i = 0;                                    // String index
    char *parmStr;                                 // Parameter string

    strupr(cmdStr);                                // String to uppercase

    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past F command
    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    parmStr = &cmdStr[i];                          // Set parameter string

    i = strlen(parmStr);                           // Get parm str length
    while (parmStr[i - 1] == ' ' && i >= 1)        // Remove trailing
      {parmStr[i - 1] = 0; i--;}                   //  spaces

    if (!i)                                        // If no parms
      {message(badFlurryStr); return;}             // Return

    if (!strcmp(parmStr, "ON"))                    // If ON parameter given
      {
        flurryMode = ON;                           // Turn on flurry mode
        message("Flurry mode turned on.");         // Display message
        return;                                    // Return
      }

    if (!strcmp(parmStr, "OFF"))                   // If OFF parameter given
      {
        flurryMode = OFF;                          // Turn off flurry mode
        message("Flurry mode turned off.");        // Display message
        return;                                    // Return
      }

    if (!strcmp(parmStr, "RESET"))                 // If RESET given
      {
        flurryPackets = 0;                         // Reset packet count
        message("Flurry packet count "             // Display message
         "set to zero.");
        return;                                    // Return
      }

    message(badFlurryStr);                         // Display error message
  }                                                // End sendFlurryMode()


/****************************************************************************/
/*                                                                          */
/***  sendMessage()                                                       ***/
/*                                                                          */
/****************************************************************************

This procedure sends a message to one user.                                 */

void sendMessage(void)                             // Begin sendMessage()
  {
    word i = 0;                                    // String index
    word len;                                      // String length
    char *sendStr;                                 // String to send
    int  userNum;                                  // User number

    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past M command
    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    userNum = atoi(&cmdStr[i]);                    // Get user number

    if (userNum <= 0 || userNum > USERTOTAL ||     // If user number out of
     !user[userNum - 1].inUse)                     //  range or not in use
      {
        message("Invalid user number.  "           // Display message
         "Enter D for list."); 
        return;                                    // Return
      }

    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past user number
    while (cmdStr[i] == ' ') i++;                  // Go past spaces

    sendStr = &cmdStr[i];                          // Set string to send

    len = strlen(sendStr);                         // Get string length

    if (!len)                                      // If no string
      {
        message("You must enter a message to "     // Display error
         "send after the user number.");           //  message
        return;                                    // Return to caller
      }

    if (len > MSGLEN) sendStr[MSGLEN] = 0;         // Truncate if too long

    sendPkt.signature = SIGNATURE;                 // Set up packet for
    sendPkt.type = pMESSAGE;                       //  sending
    sendPkt.ipxRecvMax = ipxRecvMax;
    sendPkt.ipxSendMax = ipxSendMax;
    strcpy(sendPkt.name, nameStr);
    strcpy(sendPkt.msg, sendStr);

    err(ipxSendPkt(&sendPkt, sizeof sendPkt,       // Send the packet
     &user[userNum - 1].addr));

    sprintf(str, "Message sent to user %d",        // Format message string
     userNum);

    message(str);                                  // Display it
  };                                               // End sendBroadcast()


/****************************************************************************/
/*                                                                          */
/***  sendPing()                                                          ***/
/*                                                                          */
/****************************************************************************

This procedure sends out an "is anybody there?" request.                    */

void sendPing(void)                                // Begin sendPing()
  {
    #ifdef ROUTE                                   // If ROUTE is defined
    word i;                                        // Loop counter
    #endif

    userClear();                                   // Clear out user table

    sendPkt.signature = SIGNATURE;                 // Set packet signature
    sendPkt.type = pPING;                          // Set packet type
    sendPkt.ipxRecvMax = ipxRecvMax;
    sendPkt.ipxSendMax = ipxSendMax;
    strcpy(sendPkt.name, nameStr);                 // Set user name

    message("Sending ping ...");                   // Display message

    #ifdef ROUTE                                   // If ROUTE is defined

    if (netCnt == 0)                               // If no routes found
      {
        err(ipxSendPkt(&sendPkt,                   // Send the ping
         sizeof sendPkt - (MSGLEN + 1),            //  on local address only
         &ipxAddrBroad));
      }
    else                                           // If routes found
      {
        for (i = 0; i < netCnt; i++)               // Loop for all networks
          {
            #ifdef DEBUG                           // If DEBUG is defined
            err(ipxSendPkt(&sendPkt,               // Send ping on this
             sizeof sendPkt - (MSGLEN + 1),        //  network
             &network[i]));
            #else                                  // If DEBUG isn't defined
            err(ipxSendPkt(&sendPkt,               // Send ping on this
             sizeof sendPkt - (MSGLEN + 1),        //  network
             &netPtr[i]));
            #endif
          }
      }

    #else                                          // If ROUTE not defined

    err(ipxSendPkt(&sendPkt,                       // Send the ping
     sizeof sendPkt - (MSGLEN + 1),                //  on local address only
     &ipxAddrBroad));

    #endif                                         // End if ROUTE
  }                                                // End sendPing()


/****************************************************************************/
/*                                                                          */
/***  statDisplay()                                                       ***/
/*                                                                          */
/****************************************************************************

This procedure displays statistics on maximum ECB/packet pair usage or
resets the statistics to zeroes if the RESET parameter is given.            */

void statDisplay(void)                             // Begin statDisplay()
  {
    word recvCnt = IPXRECVCNT;                     // Receive ECB/buffers
    word sendCnt = IPXSENDCNT;                     // Send ECB/buffers
    word  i = 0;                                    // String index
    char *parmStr;                                 // Parameter str pointer

    strupr(cmdStr);                                // String to uppercase

    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past STAT command
    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    parmStr = &cmdStr[i];                          // Set parameter string

    i = strlen(parmStr);                           // Get parm str length
    while (parmStr[i - 1] == ' ' && i >= 1)        // Remove trailing
     {parmStr[i - 1] = 0; i--;}                    //  spaces

    if (!i)                                        // If no parms
      {
        sprintf(str, "Max receive ECBs in use: "   // Format receive message
         "%03d of %03d", ipxRecvMax, recvCnt);
        message(str);                              // Display it

        sprintf(str, "   Max send ECBs in use: "   // Format send message
         "%03d of %03d", ipxSendMax, sendCnt);
        message(str);                              // Display it

        return;                                    // Return
      }

    if (!strcmp(parmStr, "RESET"))                 // If RESET given
      {
        ipxRecvMax = 0; ipxSendMax = 0;            // Reset statistics
        message("Statistics reset.");              // Display message
        return;                                    // Return
      }

    message("Unknown STAT parameter.");            // Display message
  }                                                // End statDisplay()


/****************************************************************************/
/*                                                                          */
/***  userClear()                                                         ***/
/*                                                                          */
/****************************************************************************

This procedure clears out the user table.                                   */

void userClear(void)                               // Begin userClear()
  {
    word i;                                        // Loop counter

    for (i = 1; i < USERTOTAL; i++)                // Clear all entries
      user[i].inUse = FALSE;                       //  except the first
  }                                                // End userClear()


/****************************************************************************/
/*                                                                          */
/***  userDisplay()                                                       ***/
/*                                                                          */
/****************************************************************************

This procedure displays the list of known LANTEST users.                    */

void userDisplay(void)                             // Begin userDisplay()
  {
    word i;                                        // Loop counter
    word lines;                                    // Lines displayed
    word time;                                     // Time to each user
    char ipxAddr[60];                              // Address string
    char local;                                    // Local PC or not

    message(emptyStr);
    message("     Network  Node         Sckt Immediate     "
            "Rcv Snd Time   Name");
    message("     -------  ----         ---- ---------     "
            "--- --- ----   ----");
    message(emptyStr);

    lines = 3;                                     // Set lines so far

    for (i = 0; i < USERTOTAL; i++)                // Loop for all users
      {
        if (user[i].inUse == FALSE) continue;      // Continue if empty

        if (i == 0) local = '*'; else local = ' '; // Asterisk if first one

        ipxAddrStrLong(ipxAddr, &user[i].addr);    // Convert address to str
        err(ipxAddrImmed(&user[i].addr, &time));   // Get transport time

        sprintf(str, "%3d: %s  %03d %03d %04d  "   // Format message string
         "%c%s", i + 1, ipxAddr,
         user[i].ipxRecvMax, user[i].ipxSendMax,
         time, local, user[i].name);

        message(str);                              // Print string
        lines++;                                   // Update lines so far

        if (lines >= (rows - 4))                   // If screen is full
          {
            message("Press any key to continue ...");

            while (!kbhit()); getch(); lines = 0;  // Wait for a keypress
          }
      }

    message(emptyStr);                             // Print blank line
  }                                                // End userDisplay()


/****************************************************************************/
/*                                                                          */
/***  userSave()                                                          ***/
/*                                                                          */
/****************************************************************************

This procedure copies user data from the current received packet to the user
table.  If the user is already in the table, the information is updated.    */

void userSave(void)                                // Begin userSave()
  {
    word i;                                        // Loop counter

    for (i = 0; i < USERTOTAL; i++)                // Loop for all entries
      {
        if (!ipxAddrCmp(&recvAddr, &user[i].addr)) // Find first slot that
          break;                                   //  either matches this
                                                   //  address or is empty,
        if (!user[i].inUse) break;                 //  whichever comes first
      }

    if (i >= USERTOTAL)                            // If no free slots
      {
        message("User table is full.");            // Display error
        return;
      }

    user[i].inUse = TRUE;                          // Indicate entry is used
    user[i].ipxRecvMax = recvPkt.ipxRecvMax;       // Save receive statistic
    user[i].ipxSendMax = recvPkt.ipxSendMax;       // Save send statistic
    ipxAddrCpy(&user[i].addr, &recvAddr);          // Save address
    strcpy(user[i].name, recvPkt.name);            // Save name
  }                                                // End userSave()
