
/****************************************************************
*******\
 *  it_msg.c -- iRMX II inter-task data message latency
measurement.
 *  First run the code of two tasks serially (no messages sent).
Then
 *  create two tasks and a "mailbox" and measure how much extra
time is
 *  needed to send a fixed number of messages from task 1 to task
2.
 *      Compiler: iC-286 V4.1 (LARGE model). Q4 1989, by R. P. Kar
\****************************************************************
*******/

#include <stdio.h>
#include <rmxc.h>

#define MAX_LOOPS 200000L

unsigned long  strt_sec, end_sec;
selector       task1_t, task2_t, mbox_t;
unsigned       pri, el_time, msg_length, status;
unsigned long  count1, count2;
float          it_msg_time;
char           msg_buf[10] = "MESSAGE\0",
               recv_buf[];

/* "union" used to decompose a pointer into segment:offset */
typedef struct {unsigned offset; selector sel;} ptr_s;
union { unsigned *pointer; ptr_s ptr; } ptr_u;

/* This task sends data messages, to task 2 that is waiting to
receive */
void task1()
{
  unsigned loc_status;
  
  for (count1 = 0; count1 < MAX_LOOPS; count1++)
  {  /* Put a serial # on the message */
     msg_buf[8] = (unsigned char)count1 / 256;
     rqsenddata( mbox_t, msg_buf, 10, &loc_status );
  }
  printf("Task 1 exiting....\n");
  rqdeletetask(NULL, &status);                           /* delete
self */
}

/* This task receives the data messages */
void task2()
{
  unsigned loc_status;
  
  for (count2 = 0; count2 < MAX_LOOPS; count2++)
    msg_length = rqreceivedata( mbox_t, recv_buf, 0xffff,
&loc_status );
  printf("    Last message received... %s %u (length %u)\n",
recv_buf,
                          (unsigned)recv_buf[8], msg_length ); 
  rqdeletetask(NULL, &status);                          /* delete
self */
}

/***************************  MAIN PROGRAM 
***************************/
/* First parameter to "rqcreatemailbox" ==> data mailbox, FIFO
queues */
#define MBOX_FLAG 0x0020

main()
{
printf("   Inter-task message latency measurement\n");
printf("      Sending %D data messages...\n\n", MAX_LOOPS);

/* Set up a mailbox for inter-task data communication */
mbox_t = rqcreatemailbox( MBOX_FLAG, &status ); 
if (status != 0) printf("rqcreatemailbox error\n");

/* Measure serial execution time of tasks 1,2 (without messages)
*/
 
strt_sec = rqgettime(&status);                  /* Start of timing
period */
  for (count1 = 0; count1 < MAX_LOOPS; count1++)
  {  /* Put a serial # on the message */
     msg_buf[8] = (unsigned char)count1 / 256;
     /* rqsenddata( mbox_t, msg_buf, 10, &loc_status ); */
  }
  for (count2 = 0; count2 < MAX_LOOPS; count2++)
    /* msg_length = rqreceivedata( mbox_t, recv_buf, 0xffff,
&loc_status ) */;
end_sec = rqgettime(&status);                   /* End of timing
period */

el_time = (unsigned)(end_sec - strt_sec);

/* Place a pointer to any variable in union "ptr_u", so the data
segment
   of this program becomes known.
 */
ptr_u.pointer = &status;

/* Get main program's priority level */
pri = rqgetpriority (NULL, &status);

task1_t = rqcreatetask (pri+2, task1, ptr_u.ptr.sel, 0L, 512, 0,
&status);
if (status != 0) printf("rqcreatetask error\n");

/* Task 2 is created with a higher priority than task 1. This
ensures that if
 * it is waiting at a mailbox for a message from task 1, it will
be scheduled
 * as soon as the message is sent.
 */
task2_t = rqcreatetask (pri+1, task2, ptr_u.ptr.sel, 0L, 512, 0,
&status);

strt_sec = rqgettime(&status);                  /* Start of timing
period */

/* Set main program's priority below task 1,2 so they run to
completion */
rqsetpriority( (selector)0, pri+3, &status );
rqsleep( 0, &status );

end_sec = rqgettime(&status);                  /* End of timing
period */

/* Set main program back to initial priority */
rqsetpriority( (selector)0, pri, &status );

el_time = (unsigned)(end_sec - strt_sec) - el_time;

it_msg_time =  ( (float)el_time * 1000000.0 ) / (float)MAX_LOOPS
;
printf("    Inter-task message latency + task switch time = %6.1f
microsecs\n",
         it_msg_time);

/* Delete mailbox */
rqdeletemailbox( mbox_t, &status );

dqexit(0);
}


