/*  HASH.C   Michael O'Shea 70274,3174
**
**  This implements a hash table example
**
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#define FALSE 0
#define TRUE 1
#define KEY_SIZE 20      /* Example key made up of up to 20 characters */

/****************************************************************
** The following record contains only one field, the key field. *
** Extra fields can be added to this struct                     *
****************************************************************/

typedef struct {
	char name[KEY_SIZE+1];
	} RECORD;

/****************************************************************
** The following struct implements a node that contains a       *
** RECORD and a pointer to another node.  This is the basic     *
** link in a linked list.                                       *
****************************************************************/

typedef struct node {
	RECORD record;
	struct node *next;
	} NODE;

/***************************************************************
** The following typedefs simplify some syntax in declarations *
***************************************************************/

typedef int BOOLEAN;
typedef NODE *LINK;   /* LINK is a pointer to a NODE */
typedef char *KEY_TYPE;    /* This is the type of the record key */
			   /* see RECORD typedef above           */

/***************************************************************
** Application functions.                                      *
***************************************************************/

int main(void);
void display_node(LINK node);
BOOLEAN keys_equal(KEY_TYPE key1, KEY_TYPE key2);
int hash_value(KEY_TYPE key);
void print_menu(void);
void add_name(void);
void find_name(void);
void update_name(void);
void delete_name(void);
void print_table(void);

/***************************************************************
** Linked list functions.                                      *
***************************************************************/

void destroy_node(LINK node);
LINK create_node(void);
LINK init_list(void);
BOOLEAN list_empty(LINK listhead);
void destroy_list(LINK listhead);
void clear_list(LINK listhead);
void add_node(LINK listhead, LINK node);
void print_list(LINK listhead);
LINK find_node(LINK listhead, KEY_TYPE search_key, LINK *prevnode);
void delete_node(LINK prevnode, LINK node);

/***************************************************************
** Global hash table                                           *
***************************************************************/

#define TABLE_CNT 100
LINK list[TABLE_CNT];

/***************************************************************
** Main initializes the hash table by creating an empty        *
** linked list for each entry in the hash table.  the pointer  *
** to the head of each list is stored in the hash table.       *
** Main completes by releasing the memory in each linked list  *
** before terminating.                                         *
***************************************************************/

int main()
{
	int i;
	int choice=0;

	/* Initialize hash table */

	for (i=0;i<TABLE_CNT;i++)
	{
		list[i] = init_list();
	}

	while (choice!='x')
	{
		print_menu();
		choice=tolower(getch());
		printf("\n");
		switch (choice)
		{
			case 'a' : add_name();break;
			case 'f' : find_name();break;
			case 'u' : update_name();break;
			case 'd' : delete_name();break;
			case 'p' : print_table();break;
			case 'x' : printf("Done\n");break;
			default  : printf("Invalid command\n");
		}
	}

	/* Destroy linked lists in hash table */

	for (i=0;i<TABLE_CNT;i++)
	{
		destroy_list(list[i]);
	}

	return 0;
}

void add_name()
{
	LINK new_node;
	char new_name[KEY_SIZE+1];
	printf("Enter name: ");
	gets(new_name);
	printf("\n");
	new_node=create_node();
	strcpy(new_node->record.name,new_name);
	add_node(list[hash_value((KEY_TYPE)new_name)],new_node);
}

void find_name()
{
	LINK prevnode;
	LINK hold;
	char search_name[KEY_SIZE+1];
	printf("Enter name to find: ");
	gets(search_name);
	printf("\n");
	if ((hold=find_node(list[hash_value((KEY_TYPE)search_name)],search_name,&prevnode))!=NULL)
	{
		printf("Found! = ");
		display_node(hold);
		printf("\n\n");
	}
	else
	{
		printf("%s was not found\n\n",search_name);
	}
}

void update_name()
{
	LINK prevnode;
	LINK hold;
	LINK new_node;
	char work_name[KEY_SIZE+1];
	printf("Enter name to update: ");
	gets(work_name);
	printf("\n");
	if ((hold=find_node(list[hash_value((KEY_TYPE)work_name)],work_name,&prevnode))!=NULL)
	{
		printf("Updating = ");
		display_node(hold);
		printf("\nEnter updated name: ");
		gets(work_name);
		delete_node(prevnode,hold);
		new_node=create_node();
		strcpy(new_node->record.name,work_name);
		add_node(list[hash_value((KEY_TYPE)work_name)],new_node);
		printf("\n\n");
	}
	else
	{
		printf("%s was not found\n\n",work_name);
	}
}

void delete_name()
{
	LINK prevnode;
	LINK hold;
	char search_name[KEY_SIZE+1];
	printf("Enter name to delete: ");
	gets(search_name);
	printf("\n");
	if ((hold=find_node(list[hash_value((KEY_TYPE)search_name)],search_name,&prevnode))!=NULL)
	{
		printf("Deleting = ");
		display_node(hold);
		delete_node(prevnode,hold);
		printf("\n\n");
	}
	else
	{
		printf("%s was not found\n\n",search_name);
	}
}

void print_table()
{
	int i;
	for (i=0;i<TABLE_CNT;i++)
	{
		if (!list_empty(list[i]))
		{
			printf("%d ",i);
			print_list(list[i]);
			printf("\n");
		}
	}
	printf("\n");
}

BOOLEAN keys_equal(KEY_TYPE key1, KEY_TYPE key2)
{
	return (!strcmpi(key1,key2));
}

/************************************************************
**  This function returns a value from 0 to TABLE_SIZE-1    *
**  given a key value.  In this case the key is a string,   *
**  so the function first converts the string to an integer *
**  value by adding up the ASCII values of the characters   *
**  in the ket string.  If your key is already an integer,  *
**  you can simply return  mykey%TABLE_CNT                  *
**  The returned value is used as an index into the hash    *
**  table to determine which linked list to store the record*
**  in, or to find the record in.                           *
************************************************************/

int hash_value(KEY_TYPE key)
{
	int i;
	int len=strlen(key);
	int sum=0;
	for (i=0;i<len;i++)
		sum+=(int)key[i];

	return sum%TABLE_CNT;
}

/* This function creates an empty linked list.  The head points to the
** tail and the tail points to itself. It returns a pointer to the head
** of the list */

LINK init_list()
{
	LINK head;
	LINK tail;
	if ((head=(LINK)malloc(sizeof(NODE)))==NULL ||
	    (tail=(LINK)malloc(sizeof(NODE)))==NULL)
	{
		printf("Out of memory allocating list node\n");
		exit(1);
	}
	head->next=tail;
	tail->next=tail;
	return head;
}


/* This function adds a previously allocated an filled node
** to the linked list pointed to by listhead */

void add_node(LINK listhead, LINK new_node)
{
	new_node->next=listhead->next;
	listhead->next=new_node;
}


/* This function deletes a node from a linked list.  It is passed a
** pointer to the node and a pointer the node previous to the one
** being deleted. */

void delete_node(LINK prevnode, LINK node)
{
	prevnode->next=node->next;
	destroy_node(node);
}


/*  This function allocates and returns a pointer to a
**  new node. */

LINK create_node()
{
	LINK node;
	if ((node=(LINK)malloc(sizeof(NODE)))==NULL)
	{
		printf("Out of memory allocating list node\n");
		exit(1);
	}

	return node;
}


/* This function frees the memory associated with a node */

void destroy_node(LINK node)
{
	free(node);
}


/* This functiion frees the memory associated with each node
** in a particular linked list*/

void destroy_list(LINK listhead)
{
	LINK hold;
	LINK cursor=listhead->next;

	/* when cursor->next==cursor you have reached the tail node
	   as it points to itself... */

	while (cursor->next!=cursor)
	{
		hold=cursor;
		cursor=cursor->next;
		destroy_node(hold);
	}

	/* destroy remaining head and tail nodes */
	destroy_node(cursor->next);
	destroy_node(listhead);
}


/* This empties a linked list, leaving the head and tail nodes */

void clear_list(LINK listhead)
{
	LINK hold;
	LINK cursor=listhead->next;

	while (cursor->next!=cursor)
	{
		hold=cursor;
		cursor=cursor->next;
		destroy_node(hold);
	}

	/* Make sure head is pointing to the tail node */
	listhead->next=cursor;
}


/* This function returns TRUE if a linked list has no nodes.
** If the head is pointing to a node which is pointing to itself
** you know it's pointing at the tail as the tail is the only node
** pointing to itself.  A list in this condition is empty. */

BOOLEAN list_empty(LINK listhead)
{
	return (listhead->next->next==listhead->next);
}


void print_list(LINK listhead)
{
	LINK cursor=listhead->next;
	while (cursor->next!=cursor)
	{
		display_node(cursor);
		cursor=cursor->next;
	}
}


/* This function returns the pointer to the found node and the pointer
** to the node previous to the found node (makes it easier to delete
** the found node - if necessary).  If not found both prevnode and
** the returned found node pointer are set to NULL */

LINK find_node(LINK listhead, KEY_TYPE search_key, LINK *prevnode)
{
	LINK cursor=listhead->next;
	LINK hold=listhead;

	while (cursor->next!=cursor)
	{
		if (keys_equal(search_key,cursor->record.name)==TRUE)
		{
			*prevnode=hold;
			return cursor;
		}
		hold=cursor;
		cursor=cursor->next;
	}
	*prevnode=NULL;
	return (LINK) NULL;
}


/*  This function would be specific to the record info you have.
**  In this case, the record only contains the one key field.
*/

void display_node(LINK node)
{
	printf("%s ",node->record.name);
}

void print_menu()
{
	printf("<A>dd     name\n");
	printf("<F>ind    name\n");
	printf("<U>pdate  name\n");
	printf("<D>elete  name\n");
	printf("<P>rint hash table\n");
	printf("e<X>it\n\n");
	printf("Enter choice: ");
}






