// ---------------------------------------------------------------------------
// Module:		FU_TBAR.C
//
// Description:	FreeUp's Toolbar callback procedure and its drawing functions
//					
//
// Last modified: 06/28/94
//
// (C) Copyright 1994 Thomas R. Grubbe, All Rights Reserved
// ---------------------------------------------------------------------------

#include <windowsx.h>
#include <shellapi.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dir.h>
#include <dos.h>
#include <ctl3d.h>
#include <math.h>
#include "freeup.h"
#include "fu_draw.h"
#include "fu_util.h"
#include "fu_file.h"
#include "fu_cmd.h"
#include "fu_tbar.h"
#include "drvtype.h"

#define BTNHEIGHT	40		// Main button dimensions
#define BTNWIDTH	40

#define DRVHEIGHT	16		// Drive button dimensions
#define DRVWIDTH	32

#define TROWS	15			// Number of rows & cols for the various
#define TCOLS	3			// toolbar buttons

#define DROWS	5			// Number of rows & cols for the various
#define DCOLS	2			// disk drive buttons

// Structure for manipulating & drawing the toolbar buttons
typedef struct tagTOOLBARSTRUCT		
{
	HDC			hdcMem;
	HBITMAP		hbmIcon;
	HBITMAP		hbmMem;
	int			nBitmapHeight;
	int			nBitmapWidth;
}TOOLBARSTRUCT;

// Toolbar buttons
typedef struct tagBtnInfo
{
	HWND hwnd;
	int Col;
	int Row;
}BtnInfo;

// Disk drive buttons
typedef struct tagDrvBtns
{
	HWND hwnd;
	int Col;
	int Row;
	int ID;
	char Name[3];
}DrvBtns;

typedef TOOLBARSTRUCT FAR * LPTOOLBARSTRUCT;

TOOLBARSTRUCT ToolBar;
TOOLBARSTRUCT DriveBtns;

BtnInfo Btn[TROWS];
DrvBtns Drives[26];



LRESULT CALLBACK _export ToolBarWndProc(HWND hwnd, UINT msg,
									   WPARAM wParam, LPARAM lParam);
BOOL InitToolBar(HINSTANCE hInstance,
				 int nBitmap,
				 int nRows,
				 int nColumns,
				 LPTOOLBARSTRUCT lpToolBar);
void DeInitToolBar(LPTOOLBARSTRUCT lpToolBar);
void ToolBar_DrawItem(const DRAWITEMSTRUCT FAR * lpDrawItem,
					  int Row, int Col,
					  LPTOOLBARSTRUCT lpTBStruct);
void DrawToolBarButton(int Idx, LPDRAWITEMSTRUCT lpDrawItem);
void DrawDriveButton(int Idx, LPDRAWITEMSTRUCT lpDrawItem);
void DrawFreeSpace(void);
void DrawCurrentDir(void);
void EnableButtons(BOOL State);
int CreateDriveButtons(HWND hwnd);

extern int ToolBarHeight;
extern int FileCount;
// extern int NumDrives;

int cxChar;
int cyChar;
int cyBorder;

LRESULT CALLBACK _export ToolBarWndProc(HWND hwnd, UINT msg,
									   WPARAM wParam, LPARAM lParam)
{
	LPDRAWITEMSTRUCT lpInfo;
	int row, col;
	static BtnTop = 2;
	int DrvTop;
    HDC hDC;

	switch (msg)
	{
		case WM_CREATE:
		{
			int TxtHeight, BdrHeight;
            DWORD dwStyle;
			TEXTMETRIC tm;
			int cx, i, nRet;
            char ErrTxt[128];

			hDC = GetDC(hwnd);
			SelectObject(hDC, GetStockObject(ANSI_VAR_FONT));
			GetTextMetrics(hDC, &tm);
			ReleaseDC(hwnd, hDC);

			TxtHeight = tm.tmHeight + tm.tmExternalLeading + tm.tmDescent;
			BdrHeight = GetSystemMetrics(SM_CYFRAME);
			cx = tm.tmAveCharWidth;

			cxChar = cx;
			cyChar = TxtHeight;
            cyBorder = BdrHeight;

            ToolBarHeight = BTNHEIGHT + DRVHEIGHT + TxtHeight + (BdrHeight * 3) + BtnTop;

			for (i=0; i<TROWS; i++)	{
				dwStyle = WS_CHILD | WS_VISIBLE | BS_OWNERDRAW;
				Btn[i].hwnd = CreateWindow("button", NULL, dwStyle,
								cx, BtnTop, BTNWIDTH, BTNHEIGHT,
								hwnd, 1000 + i, ghInstance, NULL);
				cx += BTNWIDTH;
				Btn[i].Col = 0;
                Btn[i].Row = i;
			}

			CreateDriveButtons(hwnd);

//			Drives[getdisk()].BtnState = TRUE;
//            FreeUpMsgBox(MB_OK, szAppName, "%s", ErrTxt);
			InitToolBar(ghInstance, IDC_BUTTONS, TROWS, TCOLS, &ToolBar);
			InitToolBar(ghInstance, IDC_DRIVES, DROWS, DCOLS, &DriveBtns);
                				
			return 0L;
		}
		case WM_DRAWITEM:
		{
			lpInfo = (LPDRAWITEMSTRUCT)lParam;

			if (lpInfo->CtlType != ODT_BUTTON)
				return 0L;

			if (wParam >= 1000)	{
				DrawToolBarButton(wParam - 1000, (LPDRAWITEMSTRUCT)lParam);
				return 0L;
			}
			if (wParam <= 125)	{
				DrawDriveButton(wParam - 100, (LPDRAWITEMSTRUCT)lParam);
				return 0L;
            }
			break; 
		}
		case WM_MEASUREITEM:
		{
			LPMEASUREITEMSTRUCT MeasureItem;

			MeasureItem = (LPMEASUREITEMSTRUCT)lParam;

            if (wParam <=125)
				MeasureItem->itemHeight = max(cyChar, DRVHEIGHT);
			return TRUE;
		}
		case WM_COMMAND:
			switch (wParam)
			{
				case IDC_VIEW:
					SendMessage(ghwndMain, WM_COMMAND, IDM_VIEW, 0L);
					return 0L;
				case IDC_COPY:
					SendMessage(ghwndMain, WM_COMMAND, IDM_COPY, 0L);
					return 0L;
				case IDC_MOVE:
					SendMessage(ghwndMain, WM_COMMAND, IDM_MOVE, 0L);
					return 0L;
				case IDC_DELETE:
					SendMessage(ghwndMain, WM_COMMAND, IDM_DELETE, 0L);
					return 0L;
				case IDC_RENAME:
					SendMessage(ghwndMain, WM_COMMAND, IDM_RENAME, 0L);
                	return 0L;
				case IDC_TREE:
					SendMessage(ghwndMain, WM_COMMAND, IDM_READDIRS, 0L);
					return 0L;
				case IDC_MKDIR:
					SendMessage(ghwndMain, WM_COMMAND, IDM_MKDIR, 0L);
					return 0L;
				case IDC_ALL:
				{
					int NumSelected;

					NumSelected = GetTaggedFileCount();
                    if (NumSelected == FileCount)
						SendMessage(ghwndMain, WM_COMMAND, IDM_UNTAGALL, 0L);
					else
						SendMessage(ghwndMain, WM_COMMAND, IDM_TAGALL, 0L);
					return 0L;
				}
				case IDC_ZIP:
					SendMessage(ghwndMain, WM_COMMAND, IDM_ZIPFILES, 0L);
					return 0L;
				case IDC_FINDT:
					SendMessage(ghwndMain, WM_COMMAND, IDM_FIND, 0L);
					return 0L;
				case IDC_EXIT:
					SendMessage(ghwndMain, WM_COMMAND, IDM_EXIT, 0L);
					return 0L;
                default:
				{
//                	ResetDriveButtons();
					// Is it a drive button?
                	if (wParam >= 100 && wParam <= 125)
						PostMessage(ghwndMain, WM_COMMAND, wParam - 100, 0L);
					return 0L;
                }
			}
		case WM_MOVE:
		{
			RECT r, rcMain, rcTbar;
            int cx, i;

			GetClientRect(ghwndMain, &rcMain);
			GetWindowRect(hwnd, &r);
            GetClientRect(hwnd, &rcTbar);
            r.left   = rcMain.left;
			r.top    = rcMain.top;
            r.right  = rcMain.right;
			r.bottom = rcMain.top + ToolBarHeight;

			SetWindowPos(hwnd, NULL, r.left, r.top, r.right, r.bottom, SWP_SHOWWINDOW);

			cx = cxChar;
			for (i=0; i<TROWS; i++)	{
				MoveWindow(Btn[i].hwnd, cx, BtnTop, BTNWIDTH, BTNHEIGHT, TRUE);
				cx += BTNWIDTH;
				Btn[i].Col = 0;
                Btn[i].Row = i;
			}

			cx = cxChar;
			DrvTop = ToolBarHeight - cyBorder - DRVHEIGHT;
			for (i=0; i<NumDrives; i++)	{
				MoveWindow(Drives[i].hwnd, cx, DrvTop, DRVWIDTH, DRVHEIGHT, TRUE);
				cx += DRVWIDTH + (cxChar * 4);
            }
			return 0L;
		}

		case WM_PAINT:
		{
            RECT rc, rcS, rd;
			PAINTSTRUCT ps;
			TEXTMETRIC tm;
            int i, LeftText;

			hDC = BeginPaint(hwnd, &ps);

			GetClientRect(hwnd, &rc);
            GetClientRect(ghwndSeparator, &rcS);
            GetTextMetrics(hDC, &tm);

			SelectObject(hDC, GetStockObject(GRAY_BRUSH));

			StatusRect(hDC, rc.left, rc.top, rc.right, rc.bottom, 0, SR_RAISED);

			FreeSpaceRect.left   = rc.left + cxChar + 1;
			FreeSpaceRect.top    = rc.top + BTNHEIGHT + cyBorder + BtnTop; // rc.bottom - TxtHeight - BdrHeight;
			FreeSpaceRect.right  = Center - (cxChar/2) - 1;
			FreeSpaceRect.bottom = FreeSpaceRect.top + cyChar;

			StatusRect(hDC, FreeSpaceRect.left, FreeSpaceRect.top,
					Center - (cxChar/2), FreeSpaceRect.top + cyChar, 0, SR_GROUP);

			CurrentDirRect.left   = Center + rcS.right + (cxChar/2) + 1;
			CurrentDirRect.top    = rc.top + BTNHEIGHT + cyBorder + BtnTop; // rc.bottom - TxtHeight - BdrHeight;
			CurrentDirRect.right  = rc.right - cxChar - 1;
			CurrentDirRect.bottom = CurrentDirRect.top + cyChar;
			  
			StatusRect(hDC, Center + rcS.right + (cxChar/2), CurrentDirRect.top,
					CurrentDirRect.right, CurrentDirRect.top + cyChar, 0, SR_GROUP);

			SelectObject(hDC, GetStockObject(ANSI_VAR_FONT));
			SetBkColor(hDC, RGB(192, 192, 192));

            LeftText = cxChar + DRVWIDTH + 1;
			for (i=0; i<NumDrives; i++)	{
				TextOut(hDC, LeftText, ToolBarHeight - cyBorder - cyChar,
						Drives[i].Name, lstrlen(Drives[i].Name));
				LeftText += DRVWIDTH + (cxChar * 4);
			}                            	

			DrawFreeSpace();
			DrawCurrentDir();

			EndPaint(hwnd, &ps);
			return 0L;
		}
		case WM_DESTROY:
			DeInitToolBar(&ToolBar);
			DeInitToolBar(&DriveBtns);
			DestroyWindow(ghwndToolBar);
			UnregisterClass("ToolBarClass", ghInstance);
            return 0L;
    }
	return DefWindowProc(hwnd, msg, wParam, lParam);
}

void DrawFreeSpace(void)
{
	HDC hDC;
	char CountBuff[40];
    char BytesBuff[40];
	char FreeBuff[40];
	unsigned long FreeDiskSpace;
	int TxtCenter, OldBkClr;
    WORD wExtent;
    RECT r;

	FreeDiskSpace = GetFreeDiskSpace(getdisk());

    lstrcpy(FreeBuff, ul2str(FreeDiskSpace));
	wsprintf(CountBuff, "Drive %c:", CurrentDrive + 'A');

    wsprintf(BytesBuff, "%s Bytes Free", FreeBuff);
   	hDC = GetDC(ghwndToolBar);

	SelectObject(hDC, GetStockObject(ANSI_VAR_FONT));

    CopyRect(&r, &FreeSpaceRect);

	wExtent = LOWORD(GetTextExtent(hDC, BytesBuff, lstrlen(BytesBuff)));
	OldBkClr = SetBkColor(hDC, RGB(192, 192, 192));

	r.top += 1;
    r.left += 1;
	TxtCenter = r.top + ((r.bottom - r.top)/2) - (cyChar/2);

	ExtTextOut(hDC, r.right - wExtent - 1, TxtCenter, ETO_CLIPPED | ETO_OPAQUE,
				&r, (LPSTR)BytesBuff, lstrlen(BytesBuff), NULL);
	ExtTextOut(hDC, r.left+1, TxtCenter, ETO_CLIPPED,
				&r, (LPSTR)CountBuff, lstrlen(CountBuff), NULL);

	SetBkColor(hDC, OldBkClr);
	ReleaseDC(ghwndToolBar, hDC);
}

void DrawCurrentDir(void)
{
	HDC hDC;
	int TxtCenter, OldBkClr;
    int Idx;
    char CurrentPath[128];
    WORD wExtent;
    RECT r;

	Idx = GetDirIndex();
	lstrcpy(CurrentPath, Tree[Idx].Path);
    strupr(CurrentPath);

   	hDC = GetDC(ghwndToolBar);

	SelectObject(hDC, GetStockObject(ANSI_VAR_FONT));

    CopyRect(&r, &CurrentDirRect);

	OldBkClr = SetBkColor(hDC, RGB(192, 192, 192));

	r.top += 1;
    r.left += 1;
	TxtCenter = r.top + ((r.bottom - r.top)/2) - (cyChar/2);

	ExtTextOut(hDC, r.left+1, TxtCenter, ETO_CLIPPED | ETO_OPAQUE,
				&r, (LPSTR)CurrentPath, lstrlen(CurrentPath), NULL);

	SetBkColor(hDC, OldBkClr);
	ReleaseDC(ghwndToolBar, hDC);
}


void ToolBar_DrawItem(const DRAWITEMSTRUCT FAR * lpDrawItem,
					  int Row, int Col,
					  LPTOOLBARSTRUCT lpTBStruct)
{
	HDC		hDC;
	RECT	rcBtn;

	if (lpDrawItem->itemID == (UINT)-1)
		return;

	hDC = lpDrawItem->hDC;
	CopyRect(&rcBtn, &lpDrawItem->rcItem);

	// Does it have input focus
	if (lpDrawItem->itemState & ODS_SELECTED)
		Col = 1;
	if (lpDrawItem->itemState & ODS_FOCUS && lpDrawItem->itemAction != ODA_SELECT)
    	Col = 0;

	// Draw the selected bitmap
	BitBlt(hDC,
		   rcBtn.left, rcBtn.top,
		   lpTBStruct->nBitmapWidth, lpTBStruct->nBitmapHeight,
		   lpTBStruct->hdcMem,
		   Col * lpTBStruct->nBitmapWidth,
		   Row * lpTBStruct->nBitmapHeight,
		   SRCCOPY);
}

BOOL InitToolBar(HINSTANCE hInstance,
				 int nBitmap,
				 int nRows,
				 int nColumns,
				 LPTOOLBARSTRUCT lpToolBar)
{
	HANDLE hRes;
	HANDLE hResMem;
	LPBITMAPINFOHEADER lpbi;
	DWORD FAR * lpColorTable;
	LPSTR lpBits;
	int bc;
	HDC hDC;

	if (!lpToolBar->hdcMem)	{
    	// Get a screen DC
		hDC = GetDC(NULL);
        // Create a memory DC compatible the screen
		lpToolBar->hdcMem = CreateCompatibleDC(hDC);
		// Release the screen DC
		ReleaseDC(NULL, hDC);

		if (!lpToolBar->hdcMem)
			return FALSE;

		lpToolBar->hbmMem = NULL;
    }

	hRes = FindResource(hInstance, MAKEINTRESOURCE(nBitmap), RT_BITMAP);
	if (!hRes)
		return FALSE;

	hResMem = LoadResource(hInstance, hRes);
	if (!hResMem)
		return FALSE;

	lpbi = (LPBITMAPINFOHEADER)LockResource(hResMem);
	if (!lpbi)
		return FALSE;

	lpColorTable = (DWORD FAR *)(lpbi + 1);

	lpBits = (LPSTR)(lpColorTable + 16);

//	bc = (lpBits[0] & 0xF0) >> 4;

//	lpColorTable[bc] = RGB(13, 13, 13);

	hDC = GetDC(NULL);

	lpToolBar->hbmIcon = CreateDIBitmap(hDC, lpbi, (DWORD)CBM_INIT, lpBits,
										(LPBITMAPINFO)lpbi, DIB_RGB_COLORS);
	ReleaseDC(NULL, hDC);

	// Determine toolbar button size
	lpToolBar->nBitmapHeight = (WORD)lpbi->biHeight / nRows;
	lpToolBar->nBitmapWidth  = (WORD)lpbi->biWidth  / nColumns;

	UnlockResource(hResMem);
	FreeResource(hResMem);

	if (!lpToolBar->hbmIcon)
		return FALSE;

	lpToolBar->hbmMem = SelectObject(lpToolBar->hdcMem, lpToolBar->hbmIcon);

	if (!lpToolBar->hbmMem)
		return FALSE;

	return TRUE;
}

void DeInitToolBar(LPTOOLBARSTRUCT lpToolBar)
{
	if (lpToolBar->hbmIcon)	{
		if (lpToolBar->hbmMem)
			SelectObject(lpToolBar->hdcMem, lpToolBar->hbmMem);
		lpToolBar->hbmMem = NULL;
		DeleteObject(lpToolBar->hbmIcon);
		lpToolBar->hbmIcon = NULL;
	}

	if (lpToolBar->hdcMem)	{
		DeleteDC(lpToolBar->hdcMem);
		lpToolBar->hdcMem = NULL;
    }
}

void EnableButtons(BOOL State)
{
	int i;
    HMENU hMenu;

	for (i=0; i<TROWS; i++)	{
		switch (i+1000)
		{
			case IDC_VIEW:
			case IDC_COPY:
			case IDC_MOVE:
			case IDC_FINDT:
				EnableMenuItem(GetSubMenu(ghwndMain, 0), i + 300, !State);
				EnableWindow(Btn[i].hwnd, State);
				break;
            default: break;
		}
	}
}

void DrawToolBarButton(int Idx, LPDRAWITEMSTRUCT lpDrawItem)
{
	int row, col;

	if (!IsWindowEnabled(Btn[Idx].hwnd))
		col = 2;
	else	{
		if (lpDrawItem->itemState & ODS_SELECTED)
			col = 1;
		else
			col = 0;
	}

	row = Btn[Idx].Row;
	if ((lpDrawItem->itemAction & ODA_DRAWENTIRE) ||
				(lpDrawItem->itemAction & ODA_SELECT))
		ToolBar_DrawItem((LPDRAWITEMSTRUCT)(lpDrawItem),
						  row, col,
						  &ToolBar);
}
void DrawDriveButton(int Idx, LPDRAWITEMSTRUCT lpDrawItem)
{
	int row, col;

	if (lpDrawItem->itemState & ODS_SELECTED)
		col = 1;
	else
		col = 0;

	if ((lpDrawItem->itemAction & ODA_DRAWENTIRE) || (lpDrawItem->itemAction & ODA_SELECT))
		ToolBar_DrawItem((LPDRAWITEMSTRUCT)(lpDrawItem),
						  Drives[Idx].Row, col,
 						  &DriveBtns);
}


int CreateDriveButtons(HWND hwnd)
{
	int i;
	UINT Type;
	char Name[3];
    char Path[128];
	DWORD dwStyle;
	FILE *fp; 
 	int DrvTop, cx, Idx = 0, Row = 0;

	DrvTop = ToolBarHeight - cyBorder - DRVHEIGHT;
	cx = cxChar;  

	lstrcpy(Path, StartUpDir);
	if (Path[lstrlen(Path) - 1] != '\\')
		lstrcat(Path, "\\");
	lstrcat(Path, "DRIVES.LOG");

	if ((fp = fopen(Path, "w")) == NULL)	{
		FreeUpMsgBox(MB_OK, szAppName, "Error creating DRIVES.LOG.");
        return (-1);
    }

	fprintf(fp, "NumDrives=%d\n", NumDrives);

	for (i=0; i<NumDrives; i++)	{
		Type = GetDriveTypeEx(i);

		if (Type != 0)	{
			switch(Type)
			{
				case EX_DRIVE_FLOPPY:
					if (i == 0)
						Row = 0;
					if (i == 1)
						Row = 1;
					break;
				case EX_DRIVE_FIXED:
				case EX_DRIVE_RAMDISK:
					Row = 2;
					break;
				case EX_DRIVE_CDROM:
					Row = 3;
					break;
				case EX_DRIVE_REMOTE:
					Row = 4;
					break;
			}

			fprintf(fp, "%d - Idx=%d, Type=%d, Row=%d\n", i, Idx, Type, Row);

			wsprintf(Drives[Idx].Name, "%c:", i + 'A');
			dwStyle = WS_CHILD | WS_VISIBLE | BS_OWNERDRAW;
			Drives[Idx].hwnd = CreateWindow("button", NULL, dwStyle,
								cx, DrvTop, DRVWIDTH, DRVHEIGHT,
								hwnd, 100 + i, ghInstance, NULL);
			cx += DRVWIDTH + (cxChar * 4);
            Drives[Idx].Row = Row;
			Drives[Idx].Col = 0;
			Drives[Idx].ID = i;
			Idx++;
		}
	}
	fclose(fp);
    return (Idx);
}
