// ---------------------------------------------------------------------------
// Module:		FU_FILE.C
//
// Description:	FreeUp's File owner-drawn listbox functions as well
//				file sorting and displaying.
//
// Last modified: 06/28/94
//
// (C) Copyright 1994 Thomas R. Grubbe, All Rights Reserved
// ---------------------------------------------------------------------------

#include <windowsx.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dos.h>
#include <dir.h>
#include "freeup.h"
#include "fu_util.h"
#include "fu_types.h"
#include "fu_file.h"
#include "fu_tbar.h"


FileInfoStruct _huge Files[MAXFILES];

BOOL FileDraw_DrawInit(HINSTANCE hInstance,
							  int IconID,
							  int Rows,
							  int Columns,
							  BOOL ShowIcons,
							  LPFILESDRAWSTRUCT lpFilesDrawStruct);
extern void FileDraw_OnMeasureItem(HWND hwnd,
								   MEASUREITEMSTRUCT FAR* lpMeasureItem,
								   LPFILESDRAWSTRUCT lpFilesDrawStruct);
void FileDraw_DrawTerm(LPFILESDRAWSTRUCT lpFilesDrawStruct);
void FileDraw_OnDrawItem(HWND hwnd,
						 const DRAWITEMSTRUCT FAR* lpDrawItem,
						 int Row, int Col,
                         int nIndex,
						 LPFILESDRAWSTRUCT lpFilesDrawStruct);
void CleanFileArray(void);
int GetFiles(char *Path, char *Mask);
void SetFileBoxTextHeight(HWND hwnd, HFONT hFont,
						  LPFILESDRAWSTRUCT lpFilesDrawStruct);
void FileOnDrawItem(HWND hwnd, DRAWITEMSTRUCT FAR* lpDrawItem,
					LPFILESDRAWSTRUCT lpFilesDrawStruct);
BOOL FileOnCommand(HWND hwnd, UINT id, HWND hwndCtl, UINT code);
void FileActionItem(HWND hwnd, DWORD dwData, WORD wItemNum);
void FileSystemChangeColor(void);
int GetFileType(char *Ext);
void InvertTagged(void);
static void near FastRect(HDC hDC, int x, int y, int cx, int cy);

static int NameCmp(const void *Str1, const void *Str2);
static int ExtCmp (const void *Str1, const void *Str2);
static int SizeCmp(const void *Str1, const void *Str2);
static int DateCmp(const void *Str1, const void *Str2);
void SortFiles(void);

// Globals for all to share!
int FileCount;
unsigned long DirSize;


#pragma argsused
BOOL FileOnCommand(HWND hwnd, UINT id, HWND hwndCtl, UINT code)
{
	int Idx;
	DWORD dwData;

	if (SendMessage(ghwndFileBox, LB_GETSELCOUNT, 0, 0L) > 0)
    	EnableButtons(TRUE);
	Idx = SendMessage(hwndCtl, LB_GETCURSEL, 0, 0L);
	dwData = (DWORD) SendMessage(hwndCtl, LB_GETITEMDATA, (WORD)Idx, 0L);
	switch (code)
	{
		case LBN_DBLCLK:
			FileActionItem(hwndCtl, (DWORD)dwData, (WORD)Idx);
			return TRUE;
		case LBN_SELCHANGE:
			DrawFileTotals();
			return TRUE;
	}
    return FALSE;
}

#pragma argsused
void FileActionItem(HWND hwnd, DWORD dwData, WORD wIndex)
{
	char FilePath[128];
    char CmdLine[256];

	if (FileCount == 0)
    	return;
	MakeFilePath(FilePath);
	lstrcat(FilePath, Files[wIndex].FileName);
	// Is it an executible file
	if (Files[wIndex].FileType == EXECUTIBLE)
		WinExec(FilePath, SW_SHOW);
	else	{
		if (GetProfileString("Extensions", Files[wIndex].Ext+1, "",
						 (LPSTR)CmdLine, sizeof(CmdLine)))	{
			CmdLine[strcspn(CmdLine, "^")] = '\0';
			lstrcat(CmdLine, FilePath);
			WinExec(CmdLine, SW_SHOW);
		}
        else
			FreeUpMsgBox(MB_OK | MB_ICONEXCLAMATION, szAppName,
					"No assiciation exists for \"%s\" files.",
					Files[wIndex].Ext);
	}
}


static DWORD near RGB2BGR(DWORD rgb)
{
	return RGB(GetBValue(rgb), GetGValue(rgb), GetRValue(rgb));
}

#pragma argsused
extern void FileDraw_OnMeasureItem(HWND hwnd,
								   MEASUREITEMSTRUCT FAR* lpMeasureItem,
								   LPFILESDRAWSTRUCT lpFilesDrawStruct)
{
	lpMeasureItem->itemHeight = max(lpFilesDrawStruct->nBitmapHeight,
									lpFilesDrawStruct->nTextHeight);
}
void SetFileBoxTextHeight(HWND hwnd, HFONT hFont,
						  LPFILESDRAWSTRUCT lpFilesDrawStruct)
{
	TEXTMETRIC tm;
	HANDLE     hOldFont;
	HDC        hDC;

	hDC = GetDC(hwnd);

    hOldFont = SelectObject(hDC, hFont);
	GetTextMetrics(hDC, &tm);
	SelectObject(hDC, hOldFont);
	ReleaseDC(hwnd, hDC);

	lpFilesDrawStruct->nTextHeight = tm.tmHeight;

	if (hwnd != NULL)
		SendMessage(hwnd, LB_SETITEMHEIGHT, 0,
					MAKELPARAM(lpFilesDrawStruct->nTextHeight, 0)); 
}


void FileOnDrawItem(HWND hwnd, DRAWITEMSTRUCT FAR* lpDrawItem,
					LPFILESDRAWSTRUCT lpFilesDrawStruct)
{
	int r, c, Idx;

	Idx = lpDrawItem->itemID;

	switch(Files[Idx].FileType)
	{
		case EXECUTIBLE: r = 4; c = 2; break;
		case DOCUMENT:   r = 4; c = 1; break;
		case GRAPHIC:    r = 5; c = 0; break;
		case ARCHIVE:    r = 5; c = 1; break;
		default:         r = 4; c = 0; break;
	}
/*
	if (Files[Idx].Tagged)	{
		r = 5;
		c = 2;
    }
*/
	FileDraw_OnDrawItem(hwnd,
						lpDrawItem,
						r,
						c,
						Idx,
						lpFilesDrawStruct);
}

#pragma argsused
void FileDraw_OnDrawItem(HWND hwnd,
						 const DRAWITEMSTRUCT FAR* lpDrawItem,
						 int Row, int Col,
                         int nIndex,
						 LPFILESDRAWSTRUCT lpFilesDrawStruct)
{
	HDC				hDC;
	WORD			wIndent, wTopBitmap, wTopText;
	RECT			r, FocusRect;
	char			szText[80], SizeStr[25], DateStr[25], TimeStr[10], AttrStr[10];
    int				Col1, Col2, Col3, Col4, Col5;
	DWORD			dwExtent;


	if (lpDrawItem->itemID == (UINT)-1)
		return;

	hDC = lpDrawItem->hDC;
	CopyRect(&r, &lpDrawItem->rcItem);
	CopyRect(&FocusRect, &lpDrawItem->rcItem);

	r.left    += 16; // += lpFilesDrawStruct->nBitmapWidth;
	wTopText   = r.top; // + ((r.bottom - r.top) / 2) - (lpFilesDrawStruct->nTextHeight);
	wTopBitmap = r.top + ((r.bottom - r.top) / 2) - (lpFilesDrawStruct->nBitmapHeight / 2);

	if (lpDrawItem->itemAction == ODA_FOCUS)
		goto DealWithFocus;
	else if (lpDrawItem->itemAction == ODA_SELECT)
		goto DealWithSelection;

/*
    // Draw the selected bitmap
	BitBlt(hDC,
		   0, wTopBitmap,
		   lpFilesDrawStruct->nBitmapWidth, lpFilesDrawStruct->nBitmapHeight,
		   lpFilesDrawStruct->hDC,
		   Col * lpFilesDrawStruct->nBitmapWidth,
		   Row * lpFilesDrawStruct->nBitmapHeight,
		   SRCCOPY);
*/


DealWithSelection:

	if (lpDrawItem->itemState & ODS_SELECTED)
	{
//    	if (lpDrawItem->itemAction != ODA_FOCUS)	{
			SetBkColor(hDC, GetSysColor(COLOR_HIGHLIGHT));
			SetTextColor(hDC, GetSysColor(COLOR_HIGHLIGHTTEXT));
//		}
//        else	{
//			SetBkColor(hDC, GetSysColor(COLOR_WINDOW));
//			SetTextColor(hDC, GetSysColor(COLOR_WINDOWTEXT));
//            FastRect(hDC, FocusRect.left, FocusRect.top, FocusRect.right, FocusRect.bottom);
//        }
	}
	else
	{
		SetBkColor(hDC, GetSysColor(COLOR_WINDOW));
		SetTextColor(hDC, GetSysColor(COLOR_WINDOWTEXT));
	}

	lstrcpy(szText,  Files[nIndex].FileName);
	lstrcpy(SizeStr, ul2str(Files[nIndex].Size));
	MakeDateStr(Files[nIndex].Date, DateStr);
	MakeTimeStr(Files[nIndex].Time, TimeStr);
	MakeAttrStr(Files[nIndex].Attrib, AttrStr);

	if (AttrStr[0] == '\0')
    	MessageBeep(MB_OK);

	if (LowerCase)
		dwExtent = LOWORD(GetTextExtent(hDC, "xxxxxxxx.xxxxxx", 16));
    else
		dwExtent = LOWORD(GetTextExtent(hDC, "XXXXXXXX.XXXXXX", 16));
	r.right = (int)dwExtent;
	Col1 = r.left + 1;
	wIndent = FocusRect.left + dwExtent;

	dwExtent = LOWORD(GetTextExtent(hDC, "0000,000,000", 12));
	wIndent += dwExtent;
    dwExtent = LOWORD(GetTextExtent(hDC, SizeStr, lstrlen(SizeStr)));
	Col2 = wIndent - (int)dwExtent;

	dwExtent = LOWORD(GetTextExtent(hDC, "0000/00/00", 10));
    wIndent += dwExtent;
	dwExtent = LOWORD(GetTextExtent(hDC, DateStr, lstrlen(DateStr)));
	Col3 = wIndent - (int)dwExtent;

	dwExtent = LOWORD(GetTextExtent(hDC, "0000:00pm", 10));
	wIndent += dwExtent;
	dwExtent = LOWORD(GetTextExtent(hDC, TimeStr, lstrlen(TimeStr)));
	Col4 = wIndent - (int)dwExtent;

	dwExtent = LOWORD(GetTextExtent(hDC, "xxrash", 6));
	wIndent += dwExtent;
	dwExtent = LOWORD(GetTextExtent(hDC, AttrStr, lstrlen(AttrStr)));
	Col5 = wIndent - (int)dwExtent;

	ExtTextOut(hDC, Col5, wTopText, ETO_CLIPPED | ETO_OPAQUE,
				&FocusRect, AttrStr, lstrlen(AttrStr), NULL);
	ExtTextOut(hDC, Col4, wTopText, ETO_CLIPPED | ETO_OPAQUE,
				NULL, TimeStr, lstrlen(TimeStr), NULL);
	ExtTextOut(hDC, Col3, wTopText, ETO_CLIPPED | ETO_OPAQUE,
				NULL, DateStr, lstrlen(DateStr), NULL);
	ExtTextOut(hDC, Col2, wTopText, ETO_CLIPPED | ETO_OPAQUE,
				NULL, SizeStr, lstrlen(SizeStr), NULL);
	ExtTextOut(hDC, Col1, wTopText, ETO_CLIPPED | ETO_OPAQUE,
				NULL, szText, lstrlen(szText), NULL);

	if (lpDrawItem->itemState & ODS_FOCUS && lpDrawItem->itemAction != ODA_SELECT)	{
DealWithFocus:
//		DrawFocusRect(hDC, &FocusRect);
	}
	BitBlt(hDC, 0, wTopBitmap,
		   lpFilesDrawStruct->nBitmapWidth, lpFilesDrawStruct->nBitmapHeight,
		   lpFilesDrawStruct->hDC,
		   Col * lpFilesDrawStruct->nBitmapWidth,
		   Row * lpFilesDrawStruct->nBitmapHeight,
		   SRCCOPY);

}





BOOL FileDraw_DrawInit(HINSTANCE hInstance,
							  int IconID,
							  int Rows,
							  int Columns,
							  BOOL ShowIcons,
							  LPFILESDRAWSTRUCT lpFilesDrawStruct)
{
	HANDLE				hRes;
	HANDLE				hResMem;
	LPBITMAPINFOHEADER	lpBitInfo;
	DWORD FAR *			lpColorTable;
	LPSTR				lpBits;
	int					BkColor;
    HDC					hDC;

	if (!lpFilesDrawStruct->hDC)	{
		hDC = GetDC(NULL);
		lpFilesDrawStruct->hDC = CreateCompatibleDC(hDC);
		ReleaseDC(NULL, hDC);

		if (!lpFilesDrawStruct->hDC)	{
        	FreeUpMsgBox(MB_OK, szAppName, "Could not create a compatible device context");
			return FALSE;
        }

		lpFilesDrawStruct->hIconMem = NULL;
	}

	hRes = FindResource(hInstance, MAKEINTRESOURCE(IconID), RT_BITMAP);
	if (!hRes)	{
    	FreeUpMsgBox(MB_OK, szAppName, "Could not find bitmap resource");
		return FALSE;
    }

	hResMem = LoadResource(hInstance, hRes);
	if (!hRes)	{
    	FreeUpMsgBox(MB_OK, szAppName, "Error loading bitmap resource");
		return FALSE;
    }

	// Figure out the bitmaps background color
	lpBitInfo = (LPBITMAPINFOHEADER)LockResource(hResMem);
	if (!lpBitInfo)	{
    	FreeUpMsgBox(MB_OK, szAppName, "Error locking resource");
		return FALSE;
    }

	lpColorTable = (DWORD FAR *)(lpBitInfo + 1);
	lpBits = (LPSTR)(lpColorTable + 16);
	BkColor = (lpBits[0] & 0xf0) >> 4;

	lpColorTable[BkColor] = RGB2BGR(GetSysColor(COLOR_WINDOW));

	hDC = GetDC(NULL);

	lpFilesDrawStruct->hFileIcons = CreateDIBitmap(hDC, lpBitInfo, (DWORD)CBM_INIT,
										lpBits, (LPBITMAPINFO)lpBitInfo, DIB_RGB_COLORS);
	ReleaseDC(NULL, hDC);

	lpFilesDrawStruct->nBitmapHeight = (WORD)lpBitInfo->biHeight / Rows;
	lpFilesDrawStruct->nBitmapWidth  = (WORD)lpBitInfo->biWidth  / Columns;

	UnlockResource(hResMem);
	FreeResource(hResMem);

	if (!lpFilesDrawStruct->hFileIcons)	{
		FreeUpMsgBox(MB_OK, szAppName, "Error creating bitmap icon");
		return FALSE;
    }

	lpFilesDrawStruct->hIconMem = SelectObject(lpFilesDrawStruct->hDC,
											   lpFilesDrawStruct->hFileIcons);
	if (!lpFilesDrawStruct->hIconMem)	{
    	FreeUpMsgBox(MB_OK, szAppName, "Error obtaining memory for bitmap icon");
		return FALSE;
    }

    lpFilesDrawStruct->ShowIcons = ShowIcons;

	return TRUE;
}

void FileDraw_DrawTerm(LPFILESDRAWSTRUCT lpFilesDrawStruct)
{
	if (lpFilesDrawStruct->hFileIcons)	{
		if (lpFilesDrawStruct->hIconMem)
			SelectObject(lpFilesDrawStruct->hDC, lpFilesDrawStruct->hIconMem);
		lpFilesDrawStruct->hIconMem = NULL;
		DeleteObject(lpFilesDrawStruct->hFileIcons);
		lpFilesDrawStruct->hDC = NULL;
	}
	if (lpFilesDrawStruct->hDC)	{
		DeleteDC(lpFilesDrawStruct->hDC);
		lpFilesDrawStruct->hDC = NULL;
	}
}

static VOID near FastRect(HDC hDC, int x, int y, int cx, int cy)
{
    RECT rc;

    rc.left = x;
    rc.right = x+cx;
    rc.top = y;
    rc.bottom = y+cy;
    ExtTextOut(hDC,x,y,ETO_OPAQUE,&rc,NULL,0,NULL);
}


// -------------------------- Collect file information ---------------------------- 

int GetFiles(char *Path, char *Mask)
{
	int fp, Idx;
	struct find_t f;
	int attr;
	char Name[9], Ext[5];

	attr = _A_NORMAL + _A_SYSTEM + _A_HIDDEN + _A_RDONLY + _A_ARCH;

//    CleanFileArray();

	FileCount = 0;
	DirSize = 0L;
	TotalTaggedFiles = 0;
    TotalTaggedBytes = 0L;

	if (chdir(Path) != 0)	{
    	MessageBeep(MB_OK);
		return (-1);
	}
    SendMessage(ghwndFileBox, WM_SETREDRAW, FALSE, 0L);
    LoadWaitCursor();
	fp = _dos_findfirst(Mask, attr, &f);
	while (!fp)	{
		DirSize += f.size;
        if (LowerCase)
			lstrcpy(Files[FileCount].FileName, strlwr(f.name));
		else
			lstrcpy(Files[FileCount].FileName, f.name);

		_splitpath(f.name, "", "", Name, Ext);
		lstrcpy(Files[FileCount].Name, Name);
		lstrcpy(Files[FileCount].Ext, Ext);

		Files[FileCount].FileType = GetFileType(Ext);
		Files[FileCount].Size     = (unsigned long)f.size;
		Files[FileCount].Date     = f.wr_date;
		Files[FileCount].Time     = f.wr_time;
		Files[FileCount].Attrib   = f.attrib;
        Files[FileCount].Tagged   = FALSE;

		FileCount++;
		fp = _dos_findnext(&f);
	}
	Idx = GetDirIndex();
	Tree[Idx].DirSize = DirSize;
//	SendMessage(ghwndDirBox, LB_SETCURSEL, Idx, 0L); 

	SendMessage(ghwndFileBox, WM_SETREDRAW, TRUE, 0L);
	SortFiles();
	LoadArrowCursor();
	UpdateStatusBars();
	return (0);
}


int GetFileType(char *Ext)
{
	register int i;

    for (i=0; i<NUMEXTENSIONS; i++)	{
		if (!_fstricmp(Ext, Type[i].Ext)) return (Type[i].FileType);
    }
	return (DEFAULTTYPE);
}

void CleanFileArray(void)
{
	register int i;

	for (i=0; i<=FileCount; i++)	{
		Files[i].FileName[0] = '\0';
		Files[i].Name[0]     = '\0';
		Files[i].Ext[0]      = '\0';
		Files[i].Date        = 0;
		Files[i].Time        = 0;
        Files[i].Size        = 0L;
		Files[i].Attrib      = '\0';
		Files[i].FileType    = 0;
	}
}

// -------------------------- Sort and compare functions ----------------------------
static int NameCmp(const void *Str1, const void *Str2)
{
	FileInfoStruct _huge *N1 = (FileInfoStruct _huge *)Str1;
	FileInfoStruct _huge *N2 = (FileInfoStruct _huge *)Str2;

	if (!strcmp((char *)N1->Name, (char *)N2->Name))
		return( (strcmp((char *)N1->Ext, (char *)N2->Ext) > 0) ?
				SortOrder : -SortOrder);
	else
		return( (_fstricmp((LPSTR)N1->Name, (LPSTR)N2->Name) > 0) ?
				SortOrder : -SortOrder);
}
static int ExtCmp(const void *Str1, const void *Str2)
{
	FileInfoStruct _huge *E1 = (FileInfoStruct _huge *)Str1;
	FileInfoStruct _huge *E2 = (FileInfoStruct _huge *)Str2;

	if (!strcmp((char *)E1->Ext, (char *)E2->Ext))
		return( (strcmp((char *)E1->Name, (char *)E2->Name) > 0) ?
				SortOrder : -SortOrder);
	else
		return( (strcmp((char *)E1->Ext, (char *)E2->Ext) > 0) ?
				SortOrder : -SortOrder);
}
static int SizeCmp(const void *Str1, const void *Str2)
{
	FileInfoStruct _huge *S1 = (FileInfoStruct _huge *)Str1;
	FileInfoStruct _huge *S2 = (FileInfoStruct _huge *)Str2;

	if (S1->Size == S2->Size)
		return( (_fstricmp((char *)S1->Name, (char *)S2->Name) > 0) ?
				SortOrder : -SortOrder);
	else
		return ((S1->Size > S2->Size) ? SortOrder : -SortOrder);
}
static int DateCmp(const void *Str1, const void *Str2)
{
	FileInfoStruct _huge *D1 = (FileInfoStruct _huge *)Str1;
	FileInfoStruct _huge *D2 = (FileInfoStruct _huge *)Str2;

	if (D1->Date == D2->Date)
		return ((D1->Time > D2->Time) ? SortOrder : -SortOrder);
	else
    	return ((D1->Date > D2->Date) ? SortOrder : -SortOrder);
}

void SortFiles(void)
{
	register int i;

    if (FileCount > 0)	{
		switch (SortBy)
		{
			case SORTNAME:
				qsort((FileInfoStruct _huge *)Files, FileCount, sizeof(FileInfoStruct), NameCmp);
				break;
			case SORTEXT:
				qsort((FileInfoStruct _huge *)Files, FileCount, sizeof(FileInfoStruct), ExtCmp);
				break;
			case SORTSIZE:
				qsort((FileInfoStruct _huge *)Files, FileCount, sizeof(FileInfoStruct), SizeCmp);
				break;
			case SORTDATE:
				qsort((FileInfoStruct _huge *)Files, FileCount, sizeof(FileInfoStruct), DateCmp);
				break;
		}
    }
	
	SendMessage(ghwndFileBox, LB_RESETCONTENT, 0, 0L);
	SendMessage(ghwndFileBox, WM_SETREDRAW, TRUE, 0L);

	for (i=0; i<FileCount; i++)	{
		SendMessage(ghwndFileBox, LB_ADDSTRING, 0, (long)i);
		if (Files[i].Tagged)
			SendMessage(ghwndFileBox, LB_SELITEMRANGE, TRUE, MAKELONG(i, i));
    }
}

void InvertTagged(void)
{
	register int i;

	for (i=0; i<FileCount; i++)	{
		if (SendMessage(ghwndFileBox, LB_GETSEL, (WORD)i, 0L))	{
			SendMessage(ghwndFileBox, LB_SELITEMRANGE, FALSE, MAKELONG(i, i));
            Files[i].Tagged = FALSE;
			TotalTaggedFiles--;
			TotalTaggedBytes -= Files[i].Size;
        }
        else	{
			SendMessage(ghwndFileBox, LB_SELITEMRANGE, TRUE, MAKELONG(i, i));
			Files[i].Tagged = TRUE;
			TotalTaggedFiles++;
			TotalTaggedBytes += Files[i].Size;
        }
	}
	DrawFileTotals();
}