// ---------------------------------------------------------------------------
// Module:		FU_CMD.C
//
// Description:	Most of the file and directory functions that correspond
//				to the menu commands are found here. Also here are the
//				dialog procedures and support functions.
//
// Last modified: 06/28/94
//
// (C) Copyright 1994 Thomas R. Grubbe, All Rights Reserved
// ---------------------------------------------------------------------------

#include <windowsx.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dos.h>
#include <dir.h>
#include <io.h>
#include <fcntl.h>
#include <ctype.h>
#include <ctl3d.h>
#include "freeup.h"
#include "fu_util.h"
#include "fu_file.h"
#include "fu_cmd.h"


#define DELFILE			0x10		// flags for ActionProc
#define DELTAGGED		0x20
#define DELALL			0x30
#define DELSUBDIR		0x40
#define DELFILESDIR		0x50
#define MOVEFILE		0x60
#define MOVETAGGED		0x70
#define MOVEALL			0x80
#define COPYFILE		0x6f
#define COPYTAGGED		0x7f
#define COPYALL			0x8f

#define REN_DIREC		0xa0		// flags for renaming
#define REN_SINGLE		0xa1
#define REN_MULTIPLE	0xa2

extern int SetUpFreeUp(void);
extern void WriteDataFile(void);
void MakeFilePath(char *PathName);
BOOL CALLBACK _export ActionDlgProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK _export DeleteDlgProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK _export MkDirDlgProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK _export AttribDlgProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK _export CopyMoveDlgProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK _export OverWriteDlgProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK _export FindTextDlgProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK _export CustomNameDlgProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK _export ReNameDlgProc(HWND, UINT, WPARAM, LPARAM);

int ViewEdit(int MenuID);			// Callable functions from WinMain()
int Delete(void);
int MkDir(void);
int Attributes(void);
int DefOptions(void);
int CopyMove(int MenuID);
int FindText(void);
int CustomName(void);
int ReName(void);
void SrcLineCount(void);

static int DeleteFiles(void);		// Support routines
static int DeleteDirectory(void);
static int CopyFiles(void);
static int CopySingleFile(char *Src, char *Dest);
static int MoveFiles(void);
static BOOL OverWrite(int Idx, char *DestFilePath);
static void ClearGlobalStructs(void);
static void PaintOverWritePrompt(void);
static void CleanBuff(char *str);
static BOOL DoReNaming(char *DestStr, int Idx, WORD Method);

static int NumSelected = 0;				// Total files selected				
static HWND hwndAction;					// Action Dlg items
static HWND hwndDelete;
static HWND hwndOverWrite;
static int ActionCmd;					// See the #defines
static BOOL AskOnOverWrite;				// Overwrite all flag
static BOOL CaseSensitiveSearch = FALSE;
static WORD RenMethod;					// See #defines
FileInfoStruct GlobalSrcFile;			// For CopyMove() operations
FileInfoStruct GlobalDestFile;

static char DestPath[128];				// For CopyMove()

//
// ---------------------- FreeUp Callable File Functioms --------------------------
//
int ViewEdit(int MenuID)
{
	int Idx;
	int nRet;
    char FilePath[128];
	char CmdLine[255];

    if (MenuID == IDM_VIEW)	{
		if (DefaultViewer == NULL || DefaultViewer[0] == '\0')
			return (0);
	}
    else if (MenuID == IDM_EDIT)	{
		if (DefaultEditor == NULL || DefaultEditor[0] == '\0')
			return (0);
	}

	Idx = SendMessage(ghwndFileBox, LB_GETCURSEL, 0, 0L);

	MakeFilePath(FilePath);
	lstrcat(FilePath, Files[Idx].FileName);

    if (MenuID == IDM_VIEW)
		wsprintf(CmdLine, "%s %s", DefaultViewer, FilePath);
	else if (MenuID == IDM_EDIT)
		wsprintf(CmdLine, "%s %s", DefaultEditor, FilePath);

	nRet = WinExec(CmdLine, SW_SHOW);

	if (nRet < 32)	{
		FreeUpMsgBox(MB_OK, szAppName, "WinExec Failed. Exit code %d", nRet);
		return (IDERR);
    }
	return (1);
}


int ReName(void)
{
	FARPROC ReNameProc;
	int nRet, Idx;

	NumSelected = GetTaggedFileCount();

	ReNameProc = MakeProcInstance((FARPROC)ReNameDlgProc, ghInstance);
	nRet = DialogBox(ghInstance, "ReNameDlg", ghwndMain, (DLGPROC)ReNameProc);
	FreeProcInstance(ReNameProc);

	if (nRet != IDOK)
		return (IDCANCEL);

	// Re-Scan directories if renaming directories
	if (RenMethod == REN_DIREC)	{
		SendMessage(ghwndMain, WM_COMMAND, IDM_READDIRS, 0L);
		return (nRet);
    }

	SortFiles();

    return (nRet);
}

int Delete(void)
{
	int nRet, Idx1, Idx2;
	FARPROC DelProc, ActionProc;

	NumSelected = GetTaggedFileCount();

	if (FileCount > 0 && NumSelected == 0)
		ActionCmd = DELFILESDIR;
	else if (FileCount == 0 && NumSelected == 0)
    	ActionCmd = DELSUBDIR;
	else if (NumSelected == FileCount)
		ActionCmd = DELALL;
	else if (NumSelected == 1)
		ActionCmd = DELFILE;
	else if (NumSelected > 1)
		ActionCmd = DELTAGGED;

	Idx2 = GetDirIndex();

	DelProc = MakeProcInstance((FARPROC)DeleteDlgProc, ghInstance);
	nRet = DialogBox(ghInstance, "DeleteDlg", ghwndMain, (DLGPROC)DelProc);
	FreeProcInstance(DelProc);

	if (nRet != IDOK)
		return (IDCANCEL);

	ActionProc = MakeProcInstance((FARPROC)ActionDlgProc, ghInstance);
	nRet = DialogBox(ghInstance, "ActionDlg", ghwndMain, (DLGPROC)ActionProc);
	FreeProcInstance(ActionProc);

	if (nRet == IDERR)	{
		SendMessage(ghwndDirBox, LB_SETCURSEL, (WORD)Idx2, 0L);
		InvalidateRect(ghwndDirBox, NULL, TRUE);
	}
    return (nRet);
}

int MkDir(void)
{
	register int i;
	int nRet, TopIdx, Idx, dummy;
	FARPROC MkDirProc;

	MkDirProc = MakeProcInstance((FARPROC)MkDirDlgProc, ghInstance);
	nRet = DialogBox(ghInstance, "MkDirDlg", ghwndMain, (DLGPROC)MkDirProc);
	FreeProcInstance(MkDirProc);

	if (nRet == IDCANCEL)
		return (IDCANCEL);
	else if (nRet == IDERR)
    	return (IDERR);

	Idx = GetDirIndex();
	SendMessage(ghwndDirBox, WM_SETREDRAW, FALSE, 0L);
	TopIdx = SendMessage(ghwndDirBox, LB_GETTOPINDEX, 0, 0L);
	LoadWaitCursor();

	if (LowerCase)
		strlwr(Tree[Count].Name);
	else
		strupr(Tree[Count].Name);

	FlushMsgLoop();
	for (i=0; i<=Count; i++)	{
		Tree[i].IsParent = FALSE;
		Tree[i].LastChild = FALSE;
		PathCh(Tree[i].Path, lstrlen(Tree[i].Path), '\\', '/');
	}

	SortDir((TreeStruct _huge *)Tree, Count);

	wsprintf(Tree[0].Path, "%c:\\", (
			(LowerCase) ? CurrentDrive + 'a' : CurrentDrive + 'A'));

	lstrcpy(Tree[0].Name, Tree[0].Path);
	Tree[0].LastChild = FALSE;
	Tree[0].Level = 0;

	AddListBoxItems(FALSE);

	SendMessage(ghwndDirBox, WM_SETREDRAW, TRUE, 0L);
	SendMessage(ghwndDirBox, LB_SETTOPINDEX, TopIdx, 0L);
	SendMessage(ghwndDirBox, LB_SETCURSEL, Idx, 0L);
//	InvalidateRect(ghwndDirBox, NULL, TRUE);
	LoadArrowCursor();
    GetFiles(Tree[Idx].Path, "*.*");
	WriteDataFile();

	return (nRet);
}
int Attributes(void)
{
	int nRet, TopIdx, Idx1;
	FARPROC AttribProc;

	NumSelected = GetTaggedFileCount();

    Idx1 = SendMessage(ghwndFileBox, LB_GETCURSEL, 0, 0L);
	AttribProc = MakeProcInstance((FARPROC)AttribDlgProc, ghInstance);
	nRet = DialogBox(ghInstance, "AttribDlg", ghwndMain, (DLGPROC)AttribProc);
	FreeProcInstance(AttribProc);

	if (nRet == IDCANCEL)
		return (IDCANCEL);

	TopIdx = SendMessage(ghwndFileBox, LB_GETTOPINDEX, 0, 0L);
	SendMessage(ghwndFileBox, LB_SETTOPINDEX, (WORD)TopIdx, 0L);
	SendMessage(ghwndFileBox, LB_SELITEMRANGE, TRUE, MAKELONG(Idx1, Idx1));
    InvalidateRect(ghwndFileBox, NULL, TRUE);
    return (nRet);
}

int DefOptions(void)
{
	char CurPath[128];
    int nRet, Top, Idx;

	getcwd(CurPath, 128);
	LogPath(StartUpDir);

	if ((nRet = SetUpFreeUp()) == -1)	{
		FreeUpMsgBox(MB_OK | MB_ICONSTOP, "Error",
				"Could not find \"FUSETUP.DLL\"!");
        LogPath(CurPath);
		return (IDERR);
	}
	LogPath(CurPath);

	if (nRet == IDOK)	{
		Idx = GetDirIndex();
		Top = SendMessage(ghwndDirBox, LB_GETTOPINDEX, 0, 0L);	
		InitFreeUp();
		AddListBoxItems(FALSE);
		SendMessage(ghwndDirBox, LB_SETCURSEL, (WORD)Idx, 0L);
		SendMessage(ghwndDirBox, LB_SETTOPINDEX, (WORD)Top, 0L);
		GetFiles(Tree[Idx].Path, "*.*");
		UpdateStatusBars();
	}

    return (nRet);
}

int CopyMove(int MenuID)
{
	int nRet, Idx1, Idx2;
	FARPROC CopyMoveProc, ActionProc;
    char ErrorTitle[80];

	ClearGlobalStructs();

	NumSelected = GetTaggedFileCount();

	if (NumSelected == 0 || FileCount == 0)	{
		if (MenuID == IDM_COPY)
			lstrcpy(ErrorTitle, "Copy Error");
		else
        	lstrcpy(ErrorTitle, "Move Error");
		FreeUpMsgBox(MB_OK | MB_ICONINFORMATION, (LPSTR)ErrorTitle,
					 "No files are selected");
		return (IDCANCEL);
	}

	if (NumSelected == FileCount)
		ActionCmd = MOVEALL;
	else if (NumSelected == 1)
		ActionCmd = MOVEFILE;
	else if (NumSelected > 1)
		ActionCmd = MOVETAGGED;

	if (MenuID == IDM_COPY)
		ActionCmd |= 0x0f;

	CopyMoveProc = MakeProcInstance((FARPROC)CopyMoveDlgProc, ghInstance);
	nRet = DialogBox(ghInstance, "CopyMoveDlg", ghwndMain, (DLGPROC)CopyMoveProc);
	FreeProcInstance(CopyMoveProc);

	if (nRet == IDCANCEL)
		return (IDCANCEL);

	ActionProc = MakeProcInstance((FARPROC)ActionDlgProc, ghInstance);
	nRet = DialogBox(ghInstance, "ActionDlg", ghwndMain, (DLGPROC)ActionProc);
	FreeProcInstance(ActionProc);

	if (nRet == IDERR)
    	return (IDERR);
	return (nRet);
}
// ================== Support functions for this module ====================
static int DeleteFiles(void)
{
	int Idx1, Idx2, nRet, nRet2, TopIdx, BotIdx;
    int NumFiles;
	char FilePath[128];
	char PathName[128];
    BOOL LastIdxTagged = FALSE;
    register int i, j;

 	Idx1 = SendMessage(ghwndFileBox, LB_GETCURSEL, 0, 0L);
	Idx2 = GetDirIndex();

	SetWindowText(hwndAction, "Delete");
	SetWindowText(GetDlgItem(hwndAction, IDC_ACTIONPRMPT), "Deleting File:");

	lstrcpy(PathName, Tree[Idx2].Path);		// save pathname for GetFiles()

	if (SendMessage(ghwndFileBox, LB_GETSEL, FileCount - 1, 0L) ||
					Idx1 == FileCount - 1)
    	LastIdxTagged = TRUE;

    NumFiles = FileCount;
	for (i=0; i<NumFiles; i++)	{
		if (SendMessage(ghwndFileBox, LB_GETSEL, (WORD)i, 0L))	{

			lstrcpy(FilePath, Tree[Idx2].Path);
			if (FilePath[lstrlen(FilePath) - 1] != '\\')
				lstrcat(FilePath, "\\");
			lstrcat(FilePath, Files[i].FileName);
            strupr(FilePath);

			SetWindowText(GetDlgItem(hwndAction, IDC_ACTIONFILES), "Deleting File:");
			SetWindowText(GetDlgItem(hwndAction, IDC_ACTIONFILES), FilePath);
			if (remove(Files[i].FileName) != 0)	{
				FreeUpMsgBox(MB_OK | MB_ICONINFORMATION, "Error", "Error deleting %s.\n\rFile is probably read-only.", Files[i].FileName);
			}
			FileCount--;
            TotalFiles--;
		    // Update tree size info
			Tree[Idx2].DirSize -= Files[i].Size;
			DirSize -= Files[i].Size;
		}
	}

	TopIdx = SendMessage(ghwndFileBox, LB_GETTOPINDEX, 0, 0L);
	SendMessage(ghwndDirBox, LB_SETCURSEL, Idx2, 0L);
	TreeModified = TRUE;

	GetFiles(PathName, "*.*");		// Re-Scan directory

    BotIdx = SendMessage(ghwndFileBox, LB_GETCOUNT, 0, 0L) - 1;

	if (TopIdx < Idx1)
		SendMessage(ghwndFileBox, LB_SETTOPINDEX, TopIdx, 0L);

	if (LastIdxTagged && FileCount > 0)	{
		SendMessage(ghwndFileBox, LB_SELITEMRANGE, TRUE,
					MAKELONG(BotIdx, BotIdx));
    }
	else	{
		SendMessage(ghwndFileBox, LB_SELITEMRANGE, TRUE,
					MAKELONG(Idx1, Idx1));
    }
	if (Idx2 == 0)
		InvalidateRect(ghwndFileBox, NULL, TRUE);

    UpdateStatusBars();

	return 0;
}

static int DeleteDirectory()
{
	int nRet, TopIdx, i, Idx;
	char Path[128];
    BOOL LastDir = FALSE;

	Idx = GetDirIndex();
	lstrcpy(Path, Tree[Idx].Path);

	if (Idx == 0)	{
		FreeUpMsgBox(MB_OK | MB_ICONINFORMATION, "Delete Directory",
						"Cannot delete the root directory.");
		return (-1);
	}
	if (Tree[Idx].IsParent)	{
		FreeUpMsgBox(MB_OK | MB_ICONINFORMATION, "Delete Directory",
						"Cannot delete a parent directory.");
		return (-1);
	}
	if (FileCount != 0)	{
		FreeUpMsgBox(MB_OK | MB_ICONINFORMATION, "Delete Directory",
						"directory %s is not empty.", Path);
		return (-1);
    }
	if (Idx == Count)
		LastDir = TRUE;
	if (LastDir)
		chdir(Tree[Idx-1].Path);
	else
		chdir(Tree[Idx+1].Path);
	TopIdx = SendMessage(ghwndDirBox, LB_GETTOPINDEX, 0, 0L);
	SendMessage(ghwndDirBox, WM_SETREDRAW, FALSE, 0L);

	SetWindowText(hwndAction, "Delete");
	SetWindowText(GetDlgItem(hwndAction, IDC_ACTIONPRMPT), "Deleting Directory:");
	SetWindowText(GetDlgItem(hwndAction, IDC_ACTIONFILES), Tree[Idx].Path);

	if (rmdir(Path) != 0)	{
		FreeUpMsgBox(MB_OK | MB_ICONINFORMATION, "Error", "Error deleting %s\n\r"
												 "Directory not empty.", Path);
		return (-1);
	}

	if (Tree[Idx].LastChild && Tree[Idx-1].IsParent)	{
		Tree[Idx-1].LastChild = FALSE;
		Tree[Idx-1].IsParent = FALSE;
    }

    FlushMsgLoop();
    LoadWaitCursor();
	SendMessage(ghwndDirBox, LB_DELETESTRING, Idx, 0L);
	for (i=Idx; i<=Count; i++)	{
    	if (i != Count)	{
			lstrcpy(Tree[i].Name, Tree[i+1].Name);
			lstrcpy(Tree[i].Path, Tree[i+1].Path);
			lstrcpy(Tree[i].CustomName, Tree[i+1].CustomName);
			Tree[i].DirSize   = Tree[i+1].DirSize;
			Tree[i].Parent    = Tree[i+1].Parent;
			Tree[i].SubCnt    = Tree[i+1].SubCnt;
			Tree[i].Level     = Tree[i+1].Level;
			Tree[i].LastChild = Tree[i+1].LastChild;
			Tree[i].IsParent  = Tree[i+1].IsParent;
		}
		else	{
			_fmemset(Tree[i].Name, '\0', lstrlen(Tree[i].Name));
			_fmemset(Tree[i].Path, '\0', lstrlen(Tree[i].Path));
			_fmemset(Tree[i].CustomName, '\0', lstrlen(Tree[i].CustomName));
			Tree[i].DirSize   = 0L;
			Tree[i].Parent    = 0;
			Tree[i].SubCnt    = 0;
			Tree[i].Level     = 0;
			Tree[i].LastChild = TRUE;
			Tree[i].IsParent  = FALSE;
		}
	}

	for (i=0; i<=Count; i++)	{
		Tree[i].IsParent = FALSE;
		Tree[i].LastChild = FALSE;
	}
	LoadArrowCursor();
	Count--;
	AddListBoxItems(FALSE);

	SendMessage(ghwndDirBox, WM_SETREDRAW, TRUE, 0L);
	SendMessage(ghwndDirBox, LB_SETTOPINDEX, TopIdx, 0L);

    if (LastDir)	{
		SendMessage(ghwndDirBox, LB_SETCURSEL, Count, 0L);
		GetFiles(Tree[Count].Path, "*.*");
	}
	else	{
		SendMessage(ghwndDirBox, LB_SETCURSEL, Idx, 0L);
		GetFiles(Tree[Idx].Path, "*.*");
	}
    UpdateStatusBars();
	WriteDataFile();

	return (0);
}

static int CopyFiles(void)
{
	register int i;
	int Idx1, Idx2, nRet, OvrRet=0, CopyCnt=1;
	char Prompt[80], CurDir[128];
	char ErrMsg[256];
    unsigned long DiskBytesFree;
    char SrcPathName[128], DestPathName[128];

	SetWindowText(hwndAction, "Copy");
	if (NumSelected == 1)
		SetWindowText(GetDlgItem(hwndAction, IDC_ACTIONPRMPT), "Copying File:");

	if (DestPath == NULL || DestPath[0] == '\0')	{
		FreeUpMsgBox(MB_OK | MB_ICONSTOP, "Copy Error",
					 "No destination directory specified.");
		return (IDERR);
	}

	if (DestPath[1] != ':')	{
		FreeUpMsgBox(MB_OK | MB_ICONSTOP, "Copy Error",
					 "Invalid destination directory name:\r\n\t%s", DestPath);
		return (IDERR);
	}

	getcwd(CurDir, 128);
	if (LogPath(DestPath) == IDERR)	{
		wsprintf(ErrMsg, "Directory %s does not exist. Would you like to create it?", DestPath);
		if (FreeUpMsgBox(MB_YESNO | MB_ICONQUESTION, "Copy Warning",
						(LPSTR)ErrMsg) == IDYES)	{
			if (mkdir(DestPath) != 0)	{
				FreeUpMsgBox(MB_OK | MB_ICONEXCLAMATION, "MkDir Error",
							 "Error Creating %s", (LPSTR)DestPath);
                LogPath(CurDir);
				return (IDERR);
			}
		}
	}
	LogPath(CurDir);

	DiskBytesFree = GetFreeDiskSpace(DestPath[0] - 'A');
	if (TotalTaggedBytes > DiskBytesFree)	{
		wsprintf(ErrMsg, "Not enough space on drive %c:\ to copy %d files.",
        		 DestPath[0], NumSelected);
		FreeUpMsgBox(MB_OK | MB_ICONSTOP, "Copy Error", ErrMsg);
		return (IDERR);
    }
        			
            	
    // Do it!
	for (i=0; i<FileCount; i++)	{
		if (SendMessage(ghwndFileBox, LB_GETSEL, (WORD)i, 0L))	{
			if (NumSelected > 1)	{
            	wsprintf(Prompt, "Copying %d of %d files:", CopyCnt++, NumSelected);
				SetWindowText(GetDlgItem(hwndAction, IDC_ACTIONPRMPT), (LPSTR)Prompt);
			}
			MakeFilePath(SrcPathName);
            lstrcpy(DestPathName, DestPath);
			if (DestPathName[lstrlen(DestPathName)-1] != '\\')
				lstrcat(DestPathName, "\\");
			lstrcat(SrcPathName, Files[i].FileName);
			lstrcat(DestPathName, Files[i].FileName);
			strupr(SrcPathName);
			strupr(DestPathName);
			SetWindowText(GetDlgItem(hwndAction, IDC_ACTIONFILES),
						  (LPSTR)DestPathName);

			OvrRet = OverWrite(i, DestPathName);
			if (OvrRet == IDCANCEL)
            	break;
			if (OvrRet == IDC_YESBTN || OvrRet == IDC_YESALLBTN)	{
				if ((nRet = CopySingleFile(SrcPathName, DestPathName)) == IDERR)	{
	            	wsprintf(ErrMsg, "Error attempting to copy:\r\n\t%s\r\nTo:\r\n\t%s",
							 SrcPathName, DestPathName);
					FreeUpMsgBox(MB_OK | MB_ICONSTOP, "Copy Error", ErrMsg);
					return (IDERR);
				}
				else if (nRet == IDCANCEL)
					return (IDCANCEL);
				TotalFiles++;
    		}
			UpdateStatusBars();
		}
	}
	return (IDOK);
}
static int CopySingleFile(char *Src, char *Dest)
{
	FILE *fpSrc, *fpDest;
	int c, Handle;
	unsigned Date, Time, Attr;

	if (!lstrcmp(Src, Dest))	{
		FreeUpMsgBox(MB_OK | MB_ICONSTOP, "Copy Error",
					 "File cannot be copied onto itself!");
			return (IDCANCEL);
    }
	if ((fpSrc = fopen(Src, "rb")) == NULL)		// Open source file
		return (IDERR);
	if (setvbuf(fpSrc, NULL, _IOFBF, 32767)) // 16384))
		return (IDERR);

	if ((fpDest = fopen(Dest, "wb")) == NULL)	// create destination file
		return (IDERR);
	if (setvbuf(fpDest, NULL, _IOFBF, 32767)) // 16384))
		return (IDERR);

	FlushMsgLoop();
	// Copy source to dest
	while ((c = fgetc(fpSrc)) != EOF)
		fputc(c, fpDest);

	// Close 'em up
	fclose(fpSrc);
	fclose(fpDest);

	// Set the correct file attributes
	_dos_getfileattr(Src, &Attr);
	_dos_setfileattr(Dest, Attr);

	// Get the source date and time
	_dos_open(Src, O_RDONLY, &Handle);
	_dos_getftime(Handle, &Date, &Time);
	_dos_close(Handle);

	// Set dest date and time
	_dos_open(Dest, O_RDONLY, &Handle);
	_dos_setftime(Handle, Date, Time);
	_dos_close(Handle);

	return (0);
}

static int MoveFiles()
{
	register int i;
	int Idx1, Idx2, nRet, OvrRet, MoveCnt=1;
	char Prompt[80], CurDir[128];
    char ErrMsg[256];
	char SrcPathName[128], DestPathName[128];
    int NumFiles;

	SetWindowText(hwndAction, "Move");
	if (NumSelected == 1)
		SetWindowText(GetDlgItem(hwndAction, IDC_ACTIONPRMPT), "Moving File:");

	if (DestPath == NULL || DestPath[0] == '\0')	{
		FreeUpMsgBox(MB_OK | MB_ICONSTOP, "Move Error",
					 "No destination directory specified.");
		return (IDERR);
	}
	if (DestPath[1] != ':')	{
		FreeUpMsgBox(MB_OK | MB_ICONSTOP, "Move Error",
					 "Invalid destination directory name:\r\n\t%s", DestPath);
		return (IDERR);
	}

	getcwd(CurDir, 128);
	if (LogPath(DestPath) == IDERR)	{
		wsprintf(ErrMsg, "Directory %s does not exist. Would you like to create it?", DestPath);
		if ((nRet = FreeUpMsgBox(MB_YESNO | MB_ICONQUESTION, "Move Warning",
						(LPSTR)ErrMsg)) == IDYES)	{
			if (mkdir(DestPath) != 0)	{
				FreeUpMsgBox(MB_OK | MB_ICONEXCLAMATION, "MkDir Error",
							 "Error Creating %s", (LPSTR)DestPath);
                LogPath(CurDir);
				return (IDERR);
			}
		}
		else if (nRet == IDNO)	{
        	LogPath(CurDir);
			return (IDCANCEL);
		}
	}
	LogPath(CurDir); 

	Idx2 = GetDirIndex();
	lstrcpy(SrcPathName, Tree[Idx2].Path);
	strupr(SrcPathName);
	if (SrcPathName[0] != DestPath[0])	{
		FreeUpMsgBox(MB_OK | MB_ICONINFORMATION, "Move Error",
					 "Cannot move files to another drive.\n\rSelect \"Copy\", and delete the old file.\n\r");
		return (IDERR);
    }

	NumFiles = FileCount;
	// Do it!   	
	for (i=0; i<NumFiles; i++)	{
		if (SendMessage(ghwndFileBox, LB_GETSEL, (WORD)i, 0L))	{
			if (NumSelected > 1)	{
            	wsprintf(Prompt, "Moving %d of %d files:", MoveCnt++, NumSelected);
				SetWindowText(GetDlgItem(hwndAction, IDC_ACTIONPRMPT), (LPSTR)Prompt);
			}
			MakeFilePath(SrcPathName);
			lstrcpy(DestPathName, DestPath);
			if (DestPathName[lstrlen(DestPathName)-1] != '\\')
				lstrcat(DestPathName, "\\");
			lstrcat(SrcPathName, Files[i].FileName);
			lstrcat(DestPathName, Files[i].FileName);
			strupr(SrcPathName);
			strupr(DestPathName);
			SetWindowText(GetDlgItem(hwndAction, IDC_ACTIONFILES),
						  (LPSTR)DestPathName);

			OvrRet = OverWrite(i, DestPathName);
			if (OvrRet == IDCANCEL)
            	break;
            if (OvrRet == IDC_YESBTN || OvrRet == IDC_YESALLBTN)	{
	            // Move files via rename()
				if ((nRet = rename(SrcPathName, DestPathName)) != 0)	{
	            	wsprintf(ErrMsg, "Error attempting to move:\r\n\t%s\r\nTo:\r\n\t%s",
							 SrcPathName, DestPathName);
					FreeUpMsgBox(MB_OK | MB_ICONSTOP, "Move Error", ErrMsg);
					return (IDERR);
				}
            }

			DrawDirTotals();
            FileCount--;
		}
	}
	GetFiles(Tree[Idx2].Path, "*.*");
	SendMessage(ghwndFileBox, LB_SELITEMRANGE, TRUE, MAKELONG(0, 0));
    return (IDOK);
}
static int OverWrite(int Idx, char *DestFilePath)
{
	FARPROC OverWriteProc;
	int  nRet;
	char Msg[128];
	int attr;
    struct find_t f;

    attr = _A_SYSTEM + _A_HIDDEN + _A_RDONLY + _A_ARCH;

	if (!AskOnOverWrite)
		return (IDC_YESBTN);

	if (!_dos_findfirst(DestFilePath, attr, &f))	{
		lstrcpy(GlobalSrcFile.FileName, Files[Idx].FileName);
		GlobalSrcFile.Size   = Files[Idx].Size;
		GlobalSrcFile.Date   = Files[Idx].Date;
		GlobalSrcFile.Time   = Files[Idx].Time;
		GlobalSrcFile.Attrib = Files[Idx].Attrib;

		lstrcpy(GlobalDestFile.FileName, f.name);
		GlobalDestFile.Size   = f.size;
		GlobalDestFile.Date   = f.wr_date;
		GlobalDestFile.Time   = f.wr_time;
		GlobalDestFile.Attrib = f.attrib;

		OverWriteProc = MakeProcInstance((FARPROC)OverWriteDlgProc, ghInstance);
		nRet = DialogBox(ghInstance, "OverWriteDlg", ghwndMain, (DLGPROC)OverWriteProc);
		FreeProcInstance(OverWriteProc);

        SetFocus(hwndAction);
		return (nRet);
	}
	return (IDC_YESBTN);
}


void MakeFilePath(char *PathName)
{
	int Idx;

	Idx = GetDirIndex();
	lstrcpy(PathName, Tree[Idx].Path);
	if (PathName[lstrlen(PathName)-1] != '\\')
		lstrcat(PathName, "\\");
}

static void ClearGlobalStructs(void)
{
	memset(GlobalSrcFile.FileName, '\0', sizeof(GlobalSrcFile.FileName));
	GlobalSrcFile.Size   = 0L;
	GlobalSrcFile.Date   = 0;
	GlobalSrcFile.Time   = 0;
	GlobalSrcFile.Attrib = 0;

	memset(GlobalDestFile.FileName, '\0', sizeof(GlobalSrcFile.FileName));
	GlobalDestFile.Size   = 0L;
	GlobalDestFile.Date   = 0;
	GlobalDestFile.Time   = 0;
	GlobalDestFile.Attrib = 0;
}

static void PaintOverWritePrompt(void)
{
	char DestFilePath[128];
    char NameStr1[15], NameStr2[15];
	char SizeStr1[25], DateStr1[10], TimeStr1[10], AttrStr1[6];
	char SizeStr2[25], DateStr2[10], TimeStr2[10], AttrStr2[6];
	char SrcPrmpt[80];
    char DestPrmpt[80];

	if (!AskOnOverWrite)
    	return;
	lstrcpy(DestFilePath, DestPath);
	if (DestFilePath[lstrlen(DestFilePath)-1] != '\\')
		lstrcat(DestFilePath, "\\");
	lstrcat(DestFilePath, GlobalDestFile.FileName);
	strupr(DestFilePath);

	lstrcpy(SizeStr1, ul2str(GlobalDestFile.Size));
	MakeDateStr(GlobalDestFile.Date,   DateStr1);
	MakeTimeStr(GlobalDestFile.Time,   TimeStr1);
	MakeAttrStr(GlobalDestFile.Attrib, AttrStr1);

	lstrcpy(SizeStr2, ul2str(GlobalSrcFile.Size));
	MakeDateStr(GlobalSrcFile.Date,   DateStr2);
	MakeTimeStr(GlobalSrcFile.Time,   TimeStr2);
	MakeAttrStr(GlobalSrcFile.Attrib, AttrStr2);

	lstrcpy(NameStr1, GlobalDestFile.FileName);
	lstrcpy(NameStr2,  GlobalSrcFile.FileName);
	strupr(NameStr1);
    strupr(NameStr2);
	wsprintf(DestPrmpt, "\t%-15s %12s %9s %8s %5s", NameStr1,
											   SizeStr1,
											   DateStr1,
											   TimeStr1,
											   AttrStr1);
	wsprintf(SrcPrmpt,  "\t%-15s %12s %9s %8s %5s", NameStr2,
											   SizeStr2,
											   DateStr2,
											   TimeStr2,
											   AttrStr2);
	if (!(ActionCmd&0x0f))
		SetWindowText(GetDlgItem(hwndOverWrite, IDC_ISCOPYORMOVE), "Move:");
	else
		SetWindowText(GetDlgItem(hwndOverWrite, IDC_ISCOPYORMOVE), "Copy:");
	SetWindowText(GetDlgItem(hwndOverWrite, IDC_SRCFILENAME), (LPSTR)DestFilePath);
	SetWindowText(GetDlgItem(hwndOverWrite, IDC_SRCFILE), (LPSTR)SrcPrmpt);
	SetWindowText(GetDlgItem(hwndOverWrite, IDC_DESTFILE), (LPSTR)DestPrmpt);
}

int FindText(void)
{
	FARPROC FindTextProc;
	int Idx, nRet, i, j;
	char PathName[128];
	char buff[512], *bf;
	char SearchString[80];
	BOOL Found = FALSE;
	BOOL EndOFile = FALSE;
	int fp;

	NumSelected = GetTaggedFileCount();
	if (NumSelected == 0)	{
    	FreeUpMsgBox(MB_OK, szAppName, "No files selected for text search.");
		return (IDCANCEL);
    }

	FindTextProc = MakeProcInstance((FARPROC)FindTextDlgProc, ghInstance);
	nRet = DialogBox(ghInstance, "FindTextDlg", ghwndMain, (DLGPROC)FindTextProc);
	FreeProcInstance(FindTextProc);

	if (nRet != IDOK)
		return (IDCANCEL);

	lstrcpy(SearchString, GlobalSearchText);

	if (!CaseSensitiveSearch)
		strupr(SearchString);

	FlushMsgLoop();
	LoadWaitCursor();
	for (i=0; i<FileCount; i++)	{
		if (SendMessage(ghwndFileBox, LB_GETSEL, (WORD)i, 0L))	{
			SendMessage(ghwndFileBox, LB_SETTOPINDEX, (WORD)i, 0L);
			MakeFilePath(PathName);
			lstrcat(PathName, Files[i].FileName);
			strupr(PathName);

			if ((fp = _open(PathName, O_RDONLY | O_BINARY)) == -1)
				FreeUpMsgBox(MB_OK, szAppName, "Error opening %s", PathName);

			while (!EndOFile)	{
				memset(buff, '\0', sizeof(buff));
				if (_read(fp, buff, sizeof(buff)) == 0)
					EndOFile = TRUE;

                // Get rid of control chars
				for (j=0; j<sizeof(buff); j++)	{
					if (iscntrl(buff[j]))
						buff[j] = '.';
				}
				buff[j] = '\0';

				if (!CaseSensitiveSearch)
					strupr(buff);
				if (strstr(buff, SearchString))	{
					Found = TRUE;
					goto found;
				}
			}
found:
			_close(fp);

			if (!Found)
				SendMessage(ghwndFileBox, LB_SELITEMRANGE, FALSE, MAKELONG(i, i));

			Found = FALSE;
            EndOFile = FALSE;

            DrawFileTotals();
		}
		FlushMsgLoop();
	}
	LoadArrowCursor();
	MessageBeep(MB_OK);
    UpdateStatusBars();
	return (nRet);
}

static void CleanBuff(char *str)
{
	register int i;

	for (i=0; i<sizeof(str); i++)	{
		if (iscntrl(str[i]))
			str[i] = '.';
	}
	str[i] = '\0';
}
		
int CustomName(void)
{
	FARPROC CustomNameProc;
	int nRet;

	CustomNameProc = MakeProcInstance((FARPROC)CustomNameDlgProc, ghInstance);
	nRet = DialogBox(ghInstance, "CustomNameDlg", ghwndMain, (DLGPROC)CustomNameProc);
	FreeProcInstance(CustomNameProc);

	if (nRet == IDOK)	{
		InvalidateRect(ghwndDirBox, NULL, TRUE);
		WriteDataFile();
	}
    return (nRet);
}


static BOOL DoReNaming(char *DestStr, int Idx, WORD Method)
{
	char OldFileName[13];
	char NewFileName[13];
	char OldName[9];
	char NewName[9];
	char OldExt[5];
	char NewExt[5];
	char DestName[9];
	char DestExt[5]; 
	char Msg[128];
    char OldFilePath[128], NewFilePath[128];
	register int i;
	int MaxStr, ErrCode, dummy = 0;
   	

	if (DestStr == NULL || DestStr[0] == '\0')
		return TRUE;

	if (Method == REN_DIREC)
		lstrcpy(OldFileName, Tree[Idx].Name);
	else
		lstrcpy(OldFileName, Files[Idx].FileName);

	strupr(OldFileName);

	fnsplit(OldFileName, "", "", OldName, OldExt);
    fnsplit(DestStr, "", "", DestName, DestExt);
/*
	wsprintf(Msg, "Old:%s   %s\n\rDest:%s   %s", OldName, OldExt, DestName, DestExt);
	FreeUpMsgBox(MB_OK, "Debugging", Msg); 
*/
    MaxStr = max(lstrlen(OldName), lstrlen(DestName));

	for (i=0; i<=MaxStr; i++)	{
		switch (DestName[i])
		{
			case '*':
				lstrcpy(NewName, OldName);
                i = MaxStr;
				break;
			case '?':
				NewName[i] = OldName[i];
                break;
			default:
				NewName[i] = DestName[i];
				break;
		}
	}

	MaxStr = max(lstrlen(OldExt), lstrlen(DestExt));

	for (i=0; i<=MaxStr; i++)	{
		switch (DestExt[i])
		{
			case '*':
				lstrcpy(NewExt, OldExt);
				i = MaxStr;
                break;
			case '?':
				NewExt[i] = OldExt[i];
                break;
			default:
				NewExt[i] = DestExt[i];
				break;
		}
	}

	wsprintf(OldFileName, "%s%s", OldName, OldExt);
	wsprintf(NewFileName, "%s%s", NewName, NewExt);


	if (rename(OldFileName, NewFileName) != 0)	{
    	if (Method != REN_MULTIPLE)	{
			wsprintf(Msg, "Error: %s\n\rCurrent:%s\n\rError renaming: %s\n\rTo: %s",
					 sys_errlist[errno], getcwd("", 66), OldFileName, NewFileName);
			FreeUpMsgBox(MB_OK | MB_ICONINFORMATION, "Rename Error", Msg);
		}
		return (FALSE);
	}
	return (TRUE); 
}


void SrcLineCount(void)
{
	int Idx, i, nFiles = 0;
	char PathName[128];
    char buff[128], TotalLines[45];
	unsigned long NumSrcLines = 0L;
    FILE *fp;

	NumSelected = GetTaggedFileCount();
	if (NumSelected == 0)
		return;

    LoadWaitCursor();
	for (i=0; i<FileCount; i++)	{
		if (SendMessage(ghwndFileBox, LB_GETSEL, (WORD)i, 0L))	{
			MakeFilePath(PathName);
			lstrcat(PathName, Files[i].FileName);
			strupr(PathName);

			if ((fp = fopen(PathName, "r")) == NULL)
				FreeUpMsgBox(MB_OK, szAppName, "Error opening %s", PathName);

			while (!feof(fp))	{
				if (fgets(buff, 128, fp))
					NumSrcLines++;
			}
			fclose(fp);
            nFiles++;
		}
	}
	LoadArrowCursor();
    lstrcpy(TotalLines, ul2str(NumSrcLines));
	wsprintf(buff, "%d Files scanned.\r\n%u Lines of source code.", nFiles, NumSrcLines); // TotalLines);
	MessageBeep(MB_OK);
	FreeUpMsgBox(MB_OK, szAppName, buff);
}

// ========================== Command Dialog Procedures ==============================
#pragma argsused
BOOL CALLBACK _export ActionDlgProc(HWND hDlg, UINT msg,
									WPARAM wParam, LPARAM lParam)
{
	HFONT	hFont;
	LOGFONT	lFont;
	static int nRet;
	int Idx2;
    static BOOL FirstCall;

	switch (msg)
	{
		case WM_INITDIALOG:
			CenterDialog(hDlg);
			hFont = (HFONT)NULL;
			if ((hFont = (HFONT)SendMessage(hDlg, WM_GETFONT, 0, 0L)) != 0)
            {
				if (GetObject(hFont, sizeof(LOGFONT), (LPSTR)&lFont))
                {
					lFont.lfWeight = FW_NORMAL;
					if ((hFont = CreateFontIndirect((LPLOGFONT)&lFont)) != 0)	{
						SendMessage(GetDlgItem(hDlg, IDC_ACTIONFILES), WM_SETFONT, (WPARAM)hFont, FALSE);
                    }
				}
			}
			hwndAction = hDlg;
			FirstCall = TRUE;
            AskOnOverWrite = TRUE;
			return TRUE;
		case WM_COMMAND:
			switch (wParam)
            {
				case DELSUBDIR:
					nRet = DeleteDirectory();
					break;
				case DELFILESDIR:
					if (DeleteFiles() != IDERR)
						nRet = DeleteDirectory();
					break;
				case DELFILE:
				case DELTAGGED:
				case DELALL:
					nRet = DeleteFiles();
					break;
				case MOVEFILE:
				case MOVEALL:
				case MOVETAGGED:
					nRet = MoveFiles();
					break;
				case COPYFILE:
				case COPYALL:
				case COPYTAGGED:
					nRet = CopyFiles();
					break;
			}
			SendMessage(hDlg, WM_CLOSE, 0, 0L);
		case WM_CTLCOLOR:
		{
			return Ctl3dCtlColorEx(msg, wParam, lParam);
		}
		case WM_PAINT:
		{
			HDC hDC;
			PAINTSTRUCT ps;

			hDC = BeginPaint(hDlg, &ps);
			EndPaint(hDlg, &ps);

        	if (FirstCall)	{
				PostMessage(hDlg, WM_COMMAND, ActionCmd, 0L);
				FirstCall = FALSE;
			}
			return (BOOL)NULL;
		}
		case WM_CLOSE:
			DeleteObject(hFont);
			FirstCall = TRUE;
			AskOnOverWrite = TRUE;
            if (nRet == IDCANCEL)
				EndDialog(hDlg, IDCANCEL);
			else
				EndDialog(hDlg, nRet);
			return TRUE;
    }
	return FALSE;
}

#pragma argsused
BOOL CALLBACK _export DeleteDlgProc(HWND hDlg, UINT msg,
									WPARAM wParam, LPARAM lParam)
{
	HFONT hFont;
	LOGFONT lFont;
	char Prompt[80];
	char PathName[128];
    int Idx1, Idx2;

	switch (msg)
	{
		case WM_INITDIALOG:
			hFont = (HFONT)NULL;
			if ((hFont = (HFONT)SendMessage(hDlg, WM_GETFONT, 0, 0L)) != 0)
            {
				if (GetObject(hFont, sizeof(LOGFONT), (LPSTR)&lFont))
                {
					lFont.lfWeight = FW_NORMAL;
					if ((hFont = CreateFontIndirect((LPLOGFONT)&lFont)) != 0)	{
						SendMessage(GetDlgItem(hDlg, IDC_DELEDIT), WM_SETFONT, (WPARAM)hFont, FALSE);
					}
				}
			}

			hwndDelete = hDlg;
			Idx2 = GetDirIndex();
			switch (ActionCmd)
			{
            	case DELSUBDIR:
					SetWindowText(GetDlgItem(hDlg, IDC_DELPROMPT), "Delete the Directory:");
					lstrcpy(PathName, Tree[Idx2].Path);
                    strupr(PathName);
					SetWindowText(GetDlgItem(hDlg, IDC_DELEDIT), PathName);
					break;
				case DELFILESDIR:
					SetWindowText(GetDlgItem(hDlg, IDC_DELPROMPT), "Delete ALL Files and then delete the Directory:");
					lstrcpy(PathName, Tree[Idx2].Path);
                    strupr(PathName);
					SetWindowText(GetDlgItem(hDlg, IDC_DELEDIT), PathName);
					break;
				case DELFILE:
					SetWindowText(GetDlgItem(hDlg, IDC_DELPROMPT), "Delete the File:");
					Idx1 = SendMessage(ghwndFileBox, LB_GETCURSEL, 0, 0L);
					Idx2 = SendMessage(ghwndDirBox,  LB_GETCURSEL, 0, 0L);
                    lstrcpy(PathName, Tree[Idx2].Path);
					if (PathName[lstrlen(PathName)-1] != '\\')
						lstrcat(PathName, "\\");
					lstrcat(PathName, Files[Idx1].FileName);
                    strupr(PathName);
					SetWindowText(GetDlgItem(hDlg, IDC_DELEDIT), PathName);
                    break;
            	case DELTAGGED:
					wsprintf(Prompt, "Delete %d Files from directory:", NumSelected);
					SetWindowText(GetDlgItem(hDlg, IDC_DELPROMPT), (LPSTR)Prompt);
					lstrcpy(PathName, Tree[Idx2].Path);
                    strupr(PathName);
					SetWindowText(GetDlgItem(hDlg, IDC_DELEDIT), PathName);
					break;
				case DELALL:
					SetWindowText(GetDlgItem(hDlg, IDC_DELPROMPT), "Delete ALL Files from directory:");
					lstrcpy(PathName, Tree[Idx2].Path);
                    strupr(PathName);
					SetWindowText(GetDlgItem(hDlg, IDC_DELEDIT), PathName);
					break;
			}
			return TRUE;
		case WM_COMMAND:
			switch (wParam)
			{
				case IDOK:
					if (ActionCmd == DELFILESDIR)
						SendMessage(ghwndMain, WM_COMMAND, IDM_TAGALL, 0L);
					DeleteObject(hFont);
					EndDialog(hDlg, IDOK);
					return TRUE;
				case IDCANCEL:
					SendMessage(hDlg, WM_CLOSE, 0, 0L);
					return TRUE;
			}
			break;
		case WM_CTLCOLOR:
		{
			return Ctl3dCtlColorEx(msg, wParam, lParam);
		}
		case WM_CLOSE:
        	DeleteObject(hFont);
			EndDialog(hDlg, IDCANCEL);
			return TRUE;
	}
    return FALSE;
}


#pragma argsused
BOOL CALLBACK _export MkDirDlgProc(HWND hDlg, UINT msg,
									WPARAM wParam, LPARAM lParam)
{
	HFONT hFont;
	LOGFONT lFont;
	char NewDirName[14];
	char CurDir[128];
    char CurName[14];
    char NewFullPathName[256];
    int Idx1, Idx2;

	switch (msg)
	{
		case WM_INITDIALOG:
			hFont = (HFONT)NULL;
			if ((hFont = (HFONT)SendMessage(hDlg, WM_GETFONT, 0, 0L)) != 0)
            {
				if (GetObject(hFont, sizeof(LOGFONT), (LPSTR)&lFont))
                {
					lFont.lfWeight = FW_NORMAL;
					if ((hFont = CreateFontIndirect((LPLOGFONT)&lFont)) != 0)	{
//						SendMessage(GetDlgItem(hDlg, IDC_MDPROMPT1), WM_SETFONT, (WPARAM)hFont, FALSE);
//						SendMessage(GetDlgItem(hDlg, IDC_MDPROMPT2), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_CURRNAME), WM_SETFONT, (WPARAM)hFont, FALSE);
					}
				}
			}
			SendMessage(GetDlgItem(hDlg, IDC_MKDIREDIT), EM_LIMITTEXT, 12, 0l);
			Idx2 = GetDirIndex();
            lstrcpy(CurName, Tree[Idx2].Name);
			SetWindowText(GetDlgItem(hDlg, IDC_CURRNAME), strupr(CurName));

			return TRUE;
		case WM_COMMAND:
			switch (wParam)
			{
				case IDOK:
					GetWindowText(GetDlgItem(hDlg, IDC_MKDIREDIT), (LPSTR)NewDirName, sizeof(NewDirName));
					Idx2 = GetDirIndex();

					lstrcpy(CurDir, Tree[Idx2].Path);
					if (CurDir[lstrlen(CurDir)-1] != '\\')
						lstrcat(CurDir, "\\");
					lstrcat(CurDir, NewDirName);

					if (mkdir(CurDir) == 0)	{
						Count++;
						lstrcpy(Tree[Count].Name, NewDirName);
						lstrcpy(Tree[Count].Path, CurDir);
						Tree[Count].DirSize = 0L;
						Tree[Count].Parent = Idx2;
						Tree[Count].Level = Tree[Idx2].Level + 1;
						Tree[Count].LastChild = FALSE;
						Tree[Count].IsParent  = FALSE;

						DeleteObject(hFont);
						EndDialog(hDlg, IDOK);
                    }
					else	{
						FreeUpMsgBox(MB_OK | MB_ICONEXCLAMATION,
							"MkDir Error", "Error creating directory:\r\n%s", CurDir);
						SendMessage(hDlg, WM_CLOSE, 0, 0L);
                    }
                    return TRUE;
				case IDCANCEL:
					SendMessage(hDlg, WM_CLOSE, 0, 0L);
					return TRUE;
			}
			break;
		case WM_CTLCOLOR:
		{
			return Ctl3dCtlColorEx(msg, wParam, lParam);
		}
		case WM_CLOSE:
        	DeleteObject(hFont);
			EndDialog(hDlg, IDCANCEL);
			return TRUE;
	}
    return FALSE;
}

#pragma argsused
BOOL CALLBACK _export AttribDlgProc(HWND hDlg, UINT msg,
									WPARAM wParam, LPARAM lParam)
{
	HFONT hFont;
	LOGFONT lFont;
    char FileName[25];
	int Idx1, Idx2, i;
	unsigned Attrib=0;

	switch (msg)
	{
		case WM_INITDIALOG:
			hFont = (HFONT)NULL;
			if ((hFont = (HFONT)SendMessage(hDlg, WM_GETFONT, 0, 0L)) != 0)
            {
				if (GetObject(hFont, sizeof(LOGFONT), (LPSTR)&lFont))
                {
					lFont.lfWeight = FW_NORMAL;
					if ((hFont = CreateFontIndirect((LPLOGFONT)&lFont)) != 0)	{
						SendMessage(GetDlgItem(hDlg, IDC_RDONLY), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_ARCHIVE), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_SYSTEM), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_HIDDEN), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_FILENAME), WM_SETFONT, (WPARAM)hFont, FALSE);
					}
				}
			}
			if (NumSelected > 1)	{
				SetWindowText(GetDlgItem(hDlg, IDC_PROMPT1), "Change attributes for selected");
				SetWindowText(GetDlgItem(hDlg, IDC_PROMPT2), "files:");
				wsprintf(FileName, "%d Files", NumSelected);
				SetWindowText(GetDlgItem(hDlg, IDC_FILENAME), FileName);
			}
			else	{
				Idx1 = SendMessage(ghwndFileBox, LB_GETCURSEL, 0, 0L);
				SetWindowText(GetDlgItem(hDlg, IDC_FILENAME), Files[Idx1].FileName);
				Attrib = Files[Idx1].Attrib;

				if (Attrib & _A_RDONLY)
					CheckDlgButton(hDlg, IDC_RDONLY, IsDlgButtonChecked(hDlg, IDC_RDONLY) ? 0 : 1);
				if (Attrib & _A_ARCH)
					CheckDlgButton(hDlg, IDC_ARCHIVE, IsDlgButtonChecked(hDlg, IDC_ARCHIVE) ? 0 : 1);
				if (Attrib & _A_SYSTEM)
					CheckDlgButton(hDlg, IDC_SYSTEM, IsDlgButtonChecked(hDlg, IDC_SYSTEM) ? 0 : 1);
				if (Attrib & _A_HIDDEN)
					CheckDlgButton(hDlg, IDC_HIDDEN, IsDlgButtonChecked(hDlg, IDC_HIDDEN) ? 0 : 1);
			}
			return TRUE;
		case WM_COMMAND:
			switch (wParam)
			{
				case IDOK:
					if (IsDlgButtonChecked(hDlg, IDC_RDONLY))
						Attrib |= _A_RDONLY;
					if (IsDlgButtonChecked(hDlg, IDC_ARCHIVE))
						Attrib |= _A_ARCH;
					if (IsDlgButtonChecked(hDlg, IDC_SYSTEM))
						Attrib |= _A_SYSTEM;
					if (IsDlgButtonChecked(hDlg, IDC_HIDDEN))
						Attrib |= _A_HIDDEN;

					for (i=0; i<FileCount; i++)	{
                    	if (SendMessage(ghwndFileBox, LB_GETSEL, (WORD)i, 0L))	{
							if (_dos_setfileattr(Files[i].FileName, Attrib) != 0)
								FreeUpMsgBox(MB_OK | MB_ICONEXCLAMATION, "Error",
										"Unable to modify file attribute for: %s", Files[i].FileName);
							else
								Files[i].Attrib = Attrib;
						}
                    }
					DeleteObject(hFont);
					EndDialog(hDlg, IDOK);
                    return TRUE;
				case IDCANCEL:
					SendMessage(hDlg, WM_CLOSE, 0, 0L);
					return TRUE;
				case IDC_RDONLY:
				case IDC_ARCHIVE:
				case IDC_SYSTEM:
				case IDC_HIDDEN:
					CheckDlgButton(hDlg, wParam,
						IsDlgButtonChecked(hDlg, wParam) ? 0 : 1);
					return TRUE;
			}
			break;
		case WM_CTLCOLOR:
		{
			return Ctl3dCtlColorEx(msg, wParam, lParam);
		}
		case WM_CLOSE:
        	DeleteObject(hFont);
			EndDialog(hDlg, IDCANCEL);
			return TRUE;
	}
	return FALSE;
}

#pragma argsused
BOOL CALLBACK _export CopyMoveDlgProc(HWND hDlg, UINT msg,
									  WPARAM wParam, LPARAM lParam)
{
	HFONT	hFont;
	LOGFONT	lFont;
	static int nRet;
	char Actn[10], Prompt[80];
    char Fname[14];
	int Idx1, Idx2;


	switch (msg)
	{
		case WM_INITDIALOG:
			hFont = (HFONT)NULL;
			if ((hFont = (HFONT)SendMessage(hDlg, WM_GETFONT, 0, 0L)) != 0)
            {
				if (GetObject(hFont, sizeof(LOGFONT), (LPSTR)&lFont))
                {
					lFont.lfWeight = FW_NORMAL;
					if ((hFont = CreateFontIndirect((LPLOGFONT)&lFont)) != 0)	{
						SendMessage(GetDlgItem(hDlg, IDC_FNAMEPRMPT), WM_SETFONT, (WPARAM)hFont, FALSE);
                    }
				}
			}

			if (ActionCmd&0x0f)	{
				lstrcpy(Actn, "Copy");
				SetWindowText(hDlg, "Copy");
            }
			else	{
				lstrcpy(Actn, "Move");
				SetWindowText(hDlg, "Move");
			}

			switch (ActionCmd)
			{
				case COPYFILE:
				case MOVEFILE:
					Idx1 = SendMessage(ghwndFileBox, LB_GETCURSEL, 0, 0L);
					lstrcpy(Fname, Files[Idx1].FileName);
					wsprintf(Prompt, "%s %s to directory:", Actn, strupr(Fname));
					SetWindowText(GetDlgItem(hDlg, IDC_FNAMEPRMPT), (LPSTR)Prompt);
					break;
				case COPYTAGGED:
				case MOVETAGGED:
					wsprintf(Prompt, "%s %d files to directory:", Actn, NumSelected);
					SetWindowText(GetDlgItem(hDlg, IDC_FNAMEPRMPT), (LPSTR)Prompt);
					break;
				case COPYALL:
				case MOVEALL:
					wsprintf(Prompt, "%s ALL files to directory:", Actn);
					SetWindowText(GetDlgItem(hDlg, IDC_FNAMEPRMPT), (LPSTR)Prompt);
					break;
			}

            SetWindowText(GetDlgItem(hDlg, IDC_DESTPATH), (LPSTR)DestPath);
			return TRUE;
		case WM_COMMAND:
			switch (wParam)
			{
				case IDOK:
					GetWindowText(GetDlgItem(hDlg, IDC_DESTPATH),
                    			  (LPSTR)DestPath, sizeof(DestPath));
                	DeleteObject(hFont);
                	EndDialog(hDlg, IDOK);
                    break;
				case IDCANCEL:
					DeleteObject(hFont);
					EndDialog(hDlg, IDCANCEL);
					break;
			}
            return TRUE;
		case WM_CTLCOLOR:
		{
			return Ctl3dCtlColorEx(msg, wParam, lParam);
		}
		case WM_PAINT:
			return (BOOL)NULL;
		case WM_CLOSE:
			DeleteObject(hFont);
            if (nRet == IDCANCEL)
				EndDialog(hDlg, IDCANCEL);
			else
				EndDialog(hDlg, nRet);
			return TRUE;
    }
	return FALSE;
}
#pragma argsused
BOOL CALLBACK _export OverWriteDlgProc(HWND hDlg, UINT msg,
									   WPARAM wParam, LPARAM lParam)
{
	HFONT	hFont;
	LOGFONT	lFont;
    int nRet;

	switch (msg)
	{
		case WM_INITDIALOG:
			hFont = (HFONT)NULL;
			if ((hFont = (HFONT)SendMessage(hDlg, WM_GETFONT, 0, 0L)) != 0)
            {
				if (GetObject(hFont, sizeof(LOGFONT), (LPSTR)&lFont))
                {
					lFont.lfWeight = FW_NORMAL;
					if ((hFont = CreateFontIndirect((LPLOGFONT)&lFont)) != 0)	{
						SendMessage(GetDlgItem(hDlg, IDC_SRCFILENAME), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_SRCFILE), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_DESTFILE), WM_SETFONT, (WPARAM)hFont, FALSE);
                    }
				}
			}
			hwndOverWrite = hDlg;
            PaintOverWritePrompt();
			return TRUE;
		case WM_COMMAND:
			switch (wParam)
			{
				case IDC_YESBTN:
                	DeleteObject(hFont);
                	EndDialog(hDlg, IDC_YESBTN);
					break;
				case IDC_YESALLBTN:
                	AskOnOverWrite = FALSE;
					DeleteObject(hFont);
                	EndDialog(hDlg, IDC_YESBTN);
					break;
				case IDC_NOBTN:
					DeleteObject(hFont);
                	EndDialog(hDlg, IDC_NOBTN);
					break;
				case IDCANCEL:
					DeleteObject(hFont);
					EndDialog(hDlg, IDCANCEL);
					break;
			}
			return TRUE;
		case WM_CTLCOLOR:
		{
			return Ctl3dCtlColorEx(msg, wParam, lParam);
		}
		case WM_PAINT:
			return (BOOL)NULL;
		case WM_CLOSE:
			DeleteObject(hFont);
			EndDialog(hDlg, IDCANCEL);
			return TRUE;
    }
	return FALSE;
}

#pragma argsused
BOOL CALLBACK _export FindTextDlgProc(HWND hDlg, UINT msg,
									   WPARAM wParam, LPARAM lParam)
{
	HFONT	hFont;
	LOGFONT	lFont;
    int nRet;

	switch (msg)
	{
		case WM_INITDIALOG:
			hFont = (HFONT)NULL;
			if ((hFont = (HFONT)SendMessage(hDlg, WM_GETFONT, 0, 0L)) != 0)
            {
				if (GetObject(hFont, sizeof(LOGFONT), (LPSTR)&lFont))
                {
					lFont.lfWeight = FW_NORMAL;
					if ((hFont = CreateFontIndirect((LPLOGFONT)&lFont)) != 0)	{
						SendMessage(GetDlgItem(hDlg, IDC_FINDPRMPT), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_CASECHKBOX), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_WHOLECHKBOX), WM_SETFONT, (WPARAM)hFont, FALSE);
                    }
				}
			}
			SetWindowText(GetDlgItem(hDlg, IDC_FINDEDIT), (LPSTR)GlobalSearchText);
			if (CaseSensitiveSearch)
            	CheckDlgButton(hDlg, IDC_CASECHKBOX, 1);
			return TRUE;
		case WM_COMMAND:
			switch (wParam)
			{
				case IDOK:
                	CaseSensitiveSearch = (BOOL)IsDlgButtonChecked(hDlg, IDC_CASECHKBOX) ? 1 : 0;
					GetWindowText(GetDlgItem(hDlg, IDC_FINDEDIT), (LPSTR)GlobalSearchText, sizeof(GlobalSearchText));
                	DeleteObject(hFont);
                	EndDialog(hDlg, IDOK);
					break;
				case IDCANCEL:
					DeleteObject(hFont);
					EndDialog(hDlg, IDCANCEL);
					break;
				case IDC_CASECHKBOX:
				case IDC_WHOLECHKBOX:
					CheckDlgButton(hDlg, wParam,
							IsDlgButtonChecked(hDlg, wParam) ? 0 : 1);
                    break;
			}
			return TRUE;
		case WM_CTLCOLOR:
		{
			return Ctl3dCtlColorEx(msg, wParam, lParam);
		}
		case WM_PAINT:
			return (BOOL)NULL;
		case WM_CLOSE:
			DeleteObject(hFont);
			EndDialog(hDlg, IDCANCEL);
			return TRUE;
    }
	return FALSE;
}

#pragma argsused
BOOL CALLBACK _export CustomNameDlgProc(HWND hDlg, UINT msg,
									   WPARAM wParam, LPARAM lParam)
{
	HFONT	hFont;
	LOGFONT	lFont;
	int nRet;
    static int Idx;

	switch (msg)
	{
		case WM_INITDIALOG:
			hFont = (HFONT)NULL;
			if ((hFont = (HFONT)SendMessage(hDlg, WM_GETFONT, 0, 0L)) != 0)
            {
				if (GetObject(hFont, sizeof(LOGFONT), (LPSTR)&lFont))
                {
					lFont.lfWeight = FW_NORMAL;
					if ((hFont = CreateFontIndirect((LPLOGFONT)&lFont)) != 0)	{
						SendMessage(GetDlgItem(hDlg, IDC_CUSTNAMEPRMPT), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_FULLPATHNAME), WM_SETFONT, (WPARAM)hFont, FALSE);
                    }
				}
			}

			Idx = SendMessage(ghwndDirBox, LB_GETCURSEL, 0, 0L);
			SetWindowText(GetDlgItem(hDlg, IDC_FULLPATHNAME), (LPSTR)Tree[Idx].Path);
			SetWindowText(GetDlgItem(hDlg, IDC_CUSTNAMEEDIT), (LPSTR)Tree[Idx].CustomName);
			return TRUE;
		case WM_COMMAND:
			switch (wParam)
			{
				case IDOK:
					GetWindowText(GetDlgItem(hDlg, IDC_CUSTNAMEEDIT), (LPSTR)Tree[Idx].CustomName, sizeof(Tree[Idx].CustomName));
                	DeleteObject(hFont);
                	EndDialog(hDlg, IDOK);
					break;
				case IDCANCEL:
					DeleteObject(hFont);
					EndDialog(hDlg, IDCANCEL);
					break;
			}
			return TRUE;
		case WM_CTLCOLOR:
		{
			return Ctl3dCtlColorEx(msg, wParam, lParam);
		}
		case WM_PAINT:
			return (BOOL)NULL;
		case WM_CLOSE:
			DeleteObject(hFont);
			EndDialog(hDlg, IDCANCEL);
			return TRUE;
    }
	return FALSE;
}

#pragma argsused
BOOL CALLBACK _export ReNameDlgProc(HWND hDlg, UINT msg,
									   WPARAM wParam, LPARAM lParam)
{
	HFONT	hFont;
	LOGFONT	lFont;
	int nRet, i;
	static int Idx, fIdx;
	char OldName[13];
	char NewName[13];

 	switch (msg)
	{
		case WM_INITDIALOG:
			hFont = (HFONT)NULL;
			if ((hFont = (HFONT)SendMessage(hDlg, WM_GETFONT, 0, 0L)) != 0)
            {
				if (GetObject(hFont, sizeof(LOGFONT), (LPSTR)&lFont))
                {
					lFont.lfWeight = FW_NORMAL;
					if ((hFont = CreateFontIndirect((LPLOGFONT)&lFont)) != 0)	{
						SendMessage(GetDlgItem(hDlg, IDC_RENPROMPT1), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_RENPROMPT2), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_OLDNAME), WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(GetDlgItem(hDlg, IDC_RENHELPER), WM_SETFONT, (WPARAM)hFont, FALSE);
                    }
				}
			}

			Idx = GetDirIndex();

            SendMessage(GetDlgItem(hDlg, IDC_EDITNAME), EM_LIMITTEXT, 12, 0L);

			if (Idx == 0 && NumSelected == 0)	{
				FreeUpMsgBox(MB_OK | MB_ICONEXCLAMATION, "Rename Error", "You cannot rename the Root directory");
				SendMessage(hDlg, WM_COMMAND, IDCANCEL, 0L);
			}

			if (NumSelected == 0)	{
				lstrcpy(OldName, Tree[Idx].Name);
                SetWindowText(hDlg, "Rename Sub-Directory");
				SetWindowText(GetDlgItem(hDlg, IDC_OLDNAME), (LPSTR)strupr(OldName));
				SetWindowText(GetDlgItem(hDlg, IDC_RENHELPER), "Enter new Sub-Directory name.");
                RenMethod = REN_DIREC;
			}
			else if (NumSelected > 1)	{
				wsprintf(OldName, "%d Files", NumSelected);
				SetWindowText(hDlg, "Rename Multiple Files"); 
				SetWindowText(GetDlgItem(hDlg, IDC_OLDNAME), strupr(OldName));
				SetWindowText(GetDlgItem(hDlg, IDC_RENHELPER), "Enter new filespec (eg. *.DOC)");
                RenMethod = REN_MULTIPLE;
			}
			else	{
				fIdx = SendMessage(ghwndFileBox, LB_GETCURSEL, 0, 0L);
                lstrcpy(OldName, Files[fIdx].FileName);
				SetWindowText(GetDlgItem(hDlg, IDC_OLDNAME), strupr(OldName));
                RenMethod = REN_SINGLE;
            }
			return TRUE;
		case WM_COMMAND:
			switch (wParam)
			{
				case IDOK:
					GetWindowText(GetDlgItem(hDlg, IDC_EDITNAME), (LPSTR)NewName, sizeof(NewName));
					switch (RenMethod)
					{
						case REN_DIREC:
							if (!DoReNaming(NewName, GetDirIndex(), RenMethod))	{
								DeleteObject(hFont);
								EndDialog(hDlg, IDCANCEL);
                                return TRUE;
                            }
							break;
						case REN_SINGLE:
							i = SendMessage(ghwndFileBox, LB_GETCURSEL, 0, 0L);
							if (!DoReNaming(NewName, i, RenMethod))	{
								DeleteObject(hFont);
								EndDialog(hDlg, IDCANCEL);
                                return TRUE;
							}
							break;
						case REN_MULTIPLE:
							for (i=0; i<GetTotalFilesCount(); i++)	{
                            	if (SendMessage(ghwndFileBox, LB_GETSEL, i, 0L))	{
									if (!DoReNaming(NewName, i, RenMethod))
										if (FreeUpMsgBox(MB_YESNOCANCEL | MB_ICONEXCLAMATION,
												 "Rename Error", "Error Renaming to %s\r\nContinue?",
												  NewName) != IDYES)
											DeleteObject(hFont);
											EndDialog(hDlg, IDCANCEL);
								}
							}
							break;
					}

                	DeleteObject(hFont);
                	EndDialog(hDlg, IDOK);
					break;
				case IDCANCEL:
					DeleteObject(hFont);
					EndDialog(hDlg, IDCANCEL);
					break;
			}
			return TRUE;
		case WM_CTLCOLOR:
		{
			return Ctl3dCtlColorEx(msg, wParam, lParam);
		}
		case WM_PAINT:
			return (BOOL)NULL;
		case WM_CLOSE:
			DeleteObject(hFont);
			EndDialog(hDlg, IDCANCEL);
			return TRUE;
    }
	return FALSE;
}

